/* netpoke.c
 * $Id: netpoke.c,v 1.1.1.1 2000/12/05 15:19:35 nap Exp $
 */

/* (c) Copyright 2000 M.I.T.
   Permission is hereby granted, without written agreement or
   royalty fee, to use, copy, modify, and distribute this
   software and its documentation for any purpose, provided that
   the above copyright notice and the following three paragraphs
   appear in all copies of this software.

   IN NO EVENT SHALL M.I.T. BE LIABLE TO ANY PARTY FOR DIRECT,
   INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
   ARISING OUT OF THE USE OF THIS SOFTARE AND ITS DOCUMENTATION,
   EVEN IF M.I.T. HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
   DAMAGE.

   M.I.T. SPECIFICALLY DISCLAIMS ANY WARRANTIES INCLUDING, BUT
   NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT.

   THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS AND M.I.T. HAS
   NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
   ENHANCEMENTS, OR MODIFICATIONS. */

#include <string.h>
#include <stdio.h>
#include <errno.h>
#include <stdlib.h>

#include <unistd.h>
//#include <sys/time.h>

#include "pcap.h"

#define  LIBNET_BIG_ENDIAN 1
#include <libnet.h>

#include "util.h"
#include "gmt2local.c"


#define HELP_STRING \
"Usage: %s [<options> ...] <filename>\n" \
"Parameters:\n" \
"    <filename>    Tcpdump input file (use \"-\" for standard input)\n" \
"Options:\n" \
"    -h            Print this message\n" \
"    -z            Zero MAC addresses\n" \
"    -n            Negate MAC addresses\n" \
"    -t            Include timing packets in output stream\n" \
"    -v            Verbose\n" \
"    -V            Super Verbose\n"\
"    -T            Print local and original times\n"\
"    -u            Show Debug messages\n"\
"    -p            Don't send packets with src and dst ip in outside network through router\n" \
"    -D <hours>    Number of hours to add or subtract (for daylight savings)\n"\
"    -r <n>        If packet not sent successfully , retry <n> times\n"\
"    -d <devname>  Network device name\n" \
"    -i <int_dev>  Internal Network device name (defaults to default device or <ext_dev> if set)\n" \
"    -e <ext_dev>  External Network device name (defaults to default device or <int_dev> if set)\n" \
"    -N <int_net>  Internal Network Subnet *.*.*.* in decimal, * for wildcard, (in quotes)\n" \
"    -I <ETH_INT>  Set destination MAC address internal network Router\n"\
"    -E <ETH_EXT>  Set destination MAC address external network Router\n"\
"    -s <x>        Speedup factor (defaults to 1.0)\n"

#define OPT_STRING "hzntvVTupD:r:d:i:e:N:I:E:s:"

#define FILE_BUFFER_SIZE (8 * 1024 * 1024)


typedef enum {
    KEEP_ENET_ADDRESSES,
    NEGATE_ENET_ADDRESSES,
    ZERO_ENET_ADDRESSES,
    SET_ENET_ADDRESSES
} MAC_address_option;


/* Global variables */

int    swap_bytes;
long   usleep_overhead, min_usleep;

/* Utility routines */

static void
help(char *program)
{
    fprintf(stderr, HELP_STRING, program);
    exit(1);
}


static void
error(char *msg)
{
    fprintf(stderr, " error: %s\n", msg);
    exit(1);
}


#define WAIT_TRIALS 18
#define MSEC        1000
#define MIN_WAIT    (5 * MSEC)
#define MAX_WAIT    (100 * MSEC)

static long
find_usleep_overhead(void)
{
    long wait, max, overhead;
    int  n;
    struct timeval start, end;

    max = 0;
    wait = MIN_WAIT;

    n = WAIT_TRIALS;
    while (n--) {
	gettimeofday(&start, NULL);
	usleep(wait);
	gettimeofday(&end, NULL);
	overhead = timeval_diff(end, start) - wait;
	if (overhead > max)
	    max = overhead;
	wait += (MAX_WAIT - MIN_WAIT) / (WAIT_TRIALS - 1);
    }

    return max;
}

/* Key */

static void
wait_till(struct timeval later)
{
    struct timeval now, delay;
    long diff;

    gettimeofday(&now, NULL);
    delay = later;
    timeval_sub(&delay, now);

    /* Use sleep() to get within 2 sec. */
    while (delay.tv_sec > 1) {
	sleep(delay.tv_sec - 1);
	gettimeofday(&now, NULL);
	delay = later;
	timeval_sub(&delay, now);
    }

    diff = timeval_diff(later, now);

    /* Use usleep() to get within 10 usec. */
    while (diff > min_usleep) {
	usleep(diff - usleep_overhead - 10);
	gettimeofday(&now, NULL);
	diff = timeval_diff(later, now);
    }

    /* Spin to get within 2 usec. */
    while (diff > 1) {
	gettimeofday(&now, NULL);
	diff = timeval_diff(later, now);
    }
}


/* Do things to MAC addresses. */

static void
zero_MAC_address(u_char *address)
{
    int n;

    n = ETHER_ADDR_LEN;
    while (n--)
	*address++ = 0;
}


static void
negate_MAC_address(u_char *address)
{
    int n;

    n = ETHER_ADDR_LEN;
    while (n--)
	*address++ = ~ *address;
}


static void
set_MAC_address(u_char *address, u_char *dst_MAC_address)
{
  int n;
  //    fprintf(stderr, "MAC address");
    n = ETHER_ADDR_LEN;
    while (n--){
      //      fprintf(stderr, ":%x", *dst_MAC_address);
      *address++ = *dst_MAC_address++;
    }
    //    fprintf(stderr, "\n");
    


}


static void
send_timing_packet(struct libnet_link_int *lp, char *devname,
		   struct timeval when, float speedup)
{
    u_char zeros[ETHER_ADDR_LEN], buf[ETH_H + IP_H + TCP_H];

    int    src_ip, dest_ip, n;

    memset(zeros, 0, sizeof zeros);
    memset(buf, 0, sizeof buf);
    
    src_ip = when.tv_sec;
    if (when.tv_usec >= 500000)
	src_ip++;
    dest_ip = * (int *) &speedup;

    libnet_build_ethernet(
	zeros,              /* destination MAC address */
	zeros,              /* source MAC address      */
	ETHERTYPE_IP,       /* ethernet frame type     */
	NULL,               /* pointer to payload      */
	0,                  /* payload length          */
	buf                 /* where to build packet   */
    );

    libnet_build_ip(
        TCP_H,              /* packet length                  */
	0,                  /* IP tos                         */
	0,                  /* IP id                          */
	0,                  /* fragmentation flags and offset */
	64,                 /* time to live                   */
	IPPROTO_TCP,        /* protocol                       */
	src_ip,             /* source IP address              */
	dest_ip,            /* destination IP address         */
	NULL,               /* pointer to payload             */
	0,                  /* payload length                 */
	buf + ETH_H         /* where to build packet          */
    );

    libnet_build_tcp(
        0,                  /* source port            */
	0,                  /* destination port       */
	0L,                 /* sequence number        */
	0L,                 /* acknowledgement number */
	                    /* control bits           */
	TH_FIN | TH_SYN | TH_RST | TH_PUSH | TH_ACK | TH_URG,
	1024,               /* advertised window size */
	0,                  /* urgent pointer         */
	NULL,               /* pointer to payload     */
	0,                  /* payload size           */
	buf + ETH_H + IP_H  /* where to build packet  */
    );

    libnet_do_checksum(buf + ETH_H, IPPROTO_IP,  IP_H);
    libnet_do_checksum(buf + ETH_H, IPPROTO_TCP, TCP_H);

    n = libnet_write_link_layer(lp, devname, buf, ETH_H + TCP_H + IP_H);

    if (n == -1)
	fprintf(stderr, "Error writing timing packet\n");
    else if (n != ETH_H + TCP_H + IP_H)
	fprintf(stderr, "Problem writing timing packet; only wrote %d bytes\n", n);
}

static void
fprint_timeval(FILE *fp, struct timeval tv)
{
    int hour, min, sec;

    sec = tv.tv_sec % 60;
    tv.tv_sec /= 60;
    min = tv.tv_sec % 60;
    tv.tv_sec /= 60;
    hour = tv.tv_sec % 24;

    fprintf(fp, "%02d:%02d:%02d.%06ld", hour, min, sec, tv.tv_usec);
}

int
main(int argc, char **argv)
{
  char           ch, msg[2048], *filename, *devname, *int_dev, *ext_dev, *filebuffer;
  u_char        *src_bp, *dst_bp, *int_net_bp;
  FILE          *fp;
  int            flatout, mung_time, incl_timing, super_verbose, verbose, num_packets, miss_packets = 0, n;
  int            debug, retry, retry_n =1, times, hours =0;
  int            network_class = 4,dont_send_purely_ext_packets_through = 0, set_ext_MAC_address = 0, set_int_MAC_address = 0;
  u_int32_t      int_net=0;
  double         speedup;
  unsigned char *pkt_data;
  char          *tmpstr, b_send_to_inside = 1, is_purely_ext_packet= 0;
  long           wait = 1;
  int i = 0; //for loops
  u_char int_MAC_address[ETHER_ADDR_LEN], ext_MAC_address[ETHER_ADDR_LEN];

  MAC_address_option mac_address_mode;

  /* from pcap.h */
  struct pcap_file_header  header;
  struct pcap_pkthdr       pkt_hdr;

  /* from libnet.h */
  struct libnet_ethernet_hdr *ethheader;
  struct ip *ip_packet;
  struct libnet_link_int     *lp, *lp_int, *lp_ext;
  /* from /usr/include/netinet/in.h */
  struct sockaddr_in sockaddr_in;

  /* from /usr/include/sys/time.h */
  struct timeval file_start_time, net_start_time;


  /* Init globals. */
  swap_bytes = 0;
  usleep_overhead = find_usleep_overhead();
  min_usleep = 2 * usleep_overhead;
  num_packets = 0;

  /* Init parameters. */
  devname = NULL;
  int_dev = NULL;
  ext_dev = NULL;
  mac_address_mode = KEEP_ENET_ADDRESSES;
  speedup = 1.0;
  flatout = 0;
  mung_time = 0;
  incl_timing = 0;
  verbose = 0;
  super_verbose = 0;
  debug = 0;
  times = 0;

  /* Parse command line. */
  while ((ch = getopt(argc, argv, OPT_STRING)) != EOF)
    switch (ch) {
    case 'h':
      help(argv[0]);
      break;
    case 'z':
      mac_address_mode = ZERO_ENET_ADDRESSES;
      break;
    case 'n':
      mac_address_mode = NEGATE_ENET_ADDRESSES;
      break;
    case 't':
      incl_timing = 1;
      break;
    case 'v':
      verbose = 1;
      break;
    case 'V':
      super_verbose = 1;
      break;
    case 'T':
      times = 1;
      break;
    case 'u':
      debug = 1;
      break;
    case 'p':
      dont_send_purely_ext_packets_through = 1;
      break;
    case 'D':
      hours = atoi(optarg);
      break;
    case 'r':
      retry_n = atoi(optarg);
      break;
    case 'd':
      devname = optarg;
      break;
    case 'i':
      int_dev = optarg;
      break;
    case 'e':
      ext_dev = optarg;
      break;
    case 'N':
      int_net = ip_string_to_addr(optarg);
      tmpstr = optarg;      
      for(i = 0; i < 4; i++){
	if(*tmpstr == '*'){
	  network_class = i;
	  break;
	}else{
	  tmpstr = strpbrk(tmpstr, ".");
	  tmpstr++;
	}
      }
      /*
	fprintf(stderr, "network_class:%d", network_class);
	fprintf(stderr, "internal subnet:%d.%d.%d.%d\n",
	*(int_net_bp), *(int_net_bp+1), *(int_net_bp+2), *(int_net_bp+3));
	*/
      break;
    case 'I':
      mac_address_mode = SET_ENET_ADDRESSES;
      set_ext_MAC_address = 1;
      string_to_MAC_address(int_MAC_address, optarg);
      break;
    case 'E':
      mac_address_mode = SET_ENET_ADDRESSES;
      set_int_MAC_address = 1;
      string_to_MAC_address(ext_MAC_address, optarg);
      break;
    case 's':
      speedup = atof(optarg);
      if (speedup > 0.0)
	speedup = 1.0 / speedup;
      else {
	speedup = 0.0;
	flatout = 1;
      }
      mung_time = speedup != 1.0;
      break;
    default:
      help(argv[0]);
    }
  if (optind != argc - 1)
    help(argv[0]);
  filename = argv[optind++];






  /* Open input file. */
  if (strcmp(filename, "-") == 0)
    fp = stdin;
  else {
    fp = fopen(filename, "rb");
    if (! fp) {
      sprintf(msg, "problem opening %s (%s)", filename, strerror(errno));
      error(msg);
    }
  }

  /* Setup file buffer. */
  setbuf(stdout, NULL);

  filebuffer = malloc(FILE_BUFFER_SIZE);
  if (! filebuffer)
    error("out of memory");
  if (setvbuf(fp, filebuffer, _IOFBF, FILE_BUFFER_SIZE))
    error("problem seting up input file buffer");

  /* Read input file header. */
  if (fread(&header, sizeof(struct pcap_file_header), 1, fp) != 1) {
    sprintf(msg, "problem reading header in %s", filename);
    error(msg);
  }
  if (header.magic != PCAP_MAGIC_NUMBER) {
    swap_bytes = 1;
    byteswap_pcap_file_header(&header);
    if (header.magic != PCAP_MAGIC_NUMBER) {
      sprintf(msg, "bad dump file format in %s", filename);
      error(msg);
    }
  }
  if (header.linktype != DLT_EN10MB) {
    sprintf(msg, "bad link type %d in %s", header.linktype, filename);
    error(msg);
  }

  if(super_verbose){
    fprintf(stderr, "opening network connection");
  }
  // Open network connection.
  if (!int_dev || !ext_dev){
    //attempt to open one interface for all packets
    if(!devname){
      if (libnet_select_device(&sockaddr_in, (u_char **) &devname, msg) != 1){
	error(msg);
      }
    }
    lp = libnet_open_link_interface(devname, msg);
    if (! lp)
      error(msg);
  }else{
    //attempt to open two interfaces for both internal and external networks.
    lp_int = libnet_open_link_interface(int_dev, msg);
    if (! lp_int)
      error(msg);
    lp_ext = libnet_open_link_interface(ext_dev, msg);
    if (! lp_ext)
      error(msg); 
  }

  /* Open network connection.
     if (! devname)
     if (libnet_select_device(&sockaddr_in, (u_char **) &devname, msg) != 1)
     error(msg);
     lp = libnet_open_link_interface(devname, msg);
     if (! lp)
     error(msg);
  */
  /* Setup packet data memory. */
  pkt_data = malloc(header.snaplen);
  if (! pkt_data)
    error("out of memory");
  ethheader = (void *) pkt_data;

  //    setup internal network rep, swap necessary bytes
  int_net_bp = (u_char*)&int_net;
#if linux
  byteswap(&int_net, 4);
#endif 
    
  
  /* Loop over input file.  */
  while (1) {
    /* Read packet header. */
    if (fread(&pkt_hdr, sizeof(struct pcap_pkthdr), 1, fp) != 1) {
      if (feof(fp))
	break;
      else if (ferror(fp)) {
	sprintf(msg, "problem reading packet header (%s)", strerror(errno));
	error(msg);
      } else {
	fprintf(stderr, "problem reading packet %d header\n", num_packets);
	continue;
      }
    }
    if (swap_bytes)
      byteswap_pcap_packet_header(&pkt_hdr);
    if (pkt_hdr.caplen > header.snaplen)
      pkt_hdr.caplen = header.snaplen;

    /* Read packet data. */
    if (fread(pkt_data, pkt_hdr.caplen, 1, fp) != 1) {
      if (feof(fp))
	break;
      else if (ferror(fp)) {
	sprintf(msg, "problem reading packet data (%s)", strerror(errno));
	error(msg);
      } else {
	fprintf(stderr, "problem reading packet %d data\n", num_packets);
	fprintf(stderr, "    pkt_hdr.caplen = %d\n", pkt_hdr.caplen);
	continue;
      }
    }


   

    /* Note time of first packet. */
    if (! num_packets) {
      file_start_time = pkt_hdr.ts;
      if (incl_timing)
	send_timing_packet(lp, devname,
			   file_start_time,
			   (float) (1.0 / speedup));
      gettimeofday(&net_start_time, NULL);
    }

    /* Wait till it's time for this packet to go. */
    if (! flatout) {
      struct timeval when;

      when = pkt_hdr.ts;
      timeval_sub(&when, file_start_time);
      if (mung_time)
	timeval_mul(&when, speedup);
      timeval_add(&when, net_start_time);
      wait_till(when);
    }

    


    if(debug){
      //print ethernet header
      fprintf(stdout, "eth:  ");
      src_bp = (u_char *) ethheader;
      for(i = 6; i < 12; i++){
	if(*(src_bp+i) < 16){
	  fprintf(stdout, "0");
	}
	fprintf(stdout, "%x:", (*(src_bp+i)));
      }
      fprintf(stdout, " > ");
      for(i = 0; i < 6; i++){
	if(*(src_bp+i) < 16){
	  fprintf(stdout, "0");
	}
	fprintf(stdout, "%x:", (*(src_bp+i)));
      }
      fprintf(stdout, " ");
      for(i = 12; i < 14; i++){
	if(*(src_bp+i) < 16){
	  fprintf(stdout, "0");
	}
	fprintf(stdout, "%x:", (*(src_bp+i)));
      }
      fprintf(stdout, " ");
    }

    
    //print time
    if(times){
      int32_t thiszone;
      int daylightsavings;
      struct timeval local;
      register int s;
      thiszone = gmt2local(0);
    
      daylightsavings = hours*3600;

      s = (pkt_hdr.ts.tv_sec + thiszone + daylightsavings) % 86400;
      fprintf(stdout, "orig: %02d:%02d:%02d.%06u ",
	      s / 3600, (s % 3600) / 60, s % 60, (u_int32_t)pkt_hdr.ts.tv_usec);
      
      gettimeofday(&local, NULL);
      s = (local.tv_sec + thiszone) % 86400;
      fprintf(stdout, "loc: %02d:%02d:%02d.%06u ",
	      s / 3600, (s % 3600) / 60, s % 60, (u_int32_t)local.tv_usec);
      if(!verbose || devname){
	fprintf(stdout, "\n");
      }
    }
    
    retry = retry_n;  
    while(retry){
      // Send the packet down the wire.  If destined towards the outside, send on int_dev, otherwise send on ext_dev
      if(int_dev && ext_dev){

	

	//check if it is an ip packet
#if linux
	byteswap(&ethheader->ether_type, sizeof(ethheader->ether_type));
#endif
	//	fprintf(stderr, "%x,%x\n", ETHERTYPE_IP, ethheader->ether_type);
	if(ethheader->ether_type == ETHERTYPE_IP){//if type ip, then send to inside or outside network
	  //switch it back
#if linux
	  byteswap(&ethheader->ether_type, sizeof(ethheader->ether_type));
#endif
	  //get ip header
	  ip_packet = (struct ip *)(pkt_data + sizeof(struct libnet_ethernet_hdr));;
	  //get ip addresses
	  src_bp = (u_char *)(&(ip_packet->ip_src.s_addr));
	  dst_bp = (u_char *)(&(ip_packet->ip_dst.s_addr));

	  
	


	  b_send_to_inside = 0; //defaults to send to outside unless we find that it does not belong there.
	  for(i = 0; i < network_class; i++){
	    if(*(dst_bp+i) != *(int_net_bp+i)){
	      b_send_to_inside = 1;
	      if(super_verbose){
		fprintf(stderr, "ip: %d.%d.%d.%d , %d.%d.%d.%d\n",
			(*int_net_bp), (*(int_net_bp+1)), (*(int_net_bp+2)), (*(int_net_bp+3)),
			(*dst_bp), (*(dst_bp+1)), (*(dst_bp+2)), (*(dst_bp+3)));
	      }
	      is_purely_ext_packet=0;
	      if(dont_send_purely_ext_packets_through){
		for(i = 0; i < network_class; i++){
		  if(*(src_bp+i) != *(int_net_bp+i)){
		    if(super_verbose){
		      fprintf(stderr, "ip: %d.%d.%d.%d > %d.%d.%d.%d\n",
			      (*src_bp), (*(src_bp+1)), (*(src_bp+2)), (*(src_bp+3)),
			      (*dst_bp), (*(dst_bp+1)), (*(dst_bp+2)), (*(dst_bp+3)));
		    }
		    is_purely_ext_packet=1;
		    b_send_to_inside = 0;
		    break;
		  }
		}
	      }
	      break;
	    }
	  }

	  
	  /* Optionally mung MAC addresses. */
	  switch (mac_address_mode) {
	  case KEEP_ENET_ADDRESSES:
	    break;
	  case NEGATE_ENET_ADDRESSES:
	    negate_MAC_address(ethheader->ether_dhost);
	    negate_MAC_address(ethheader->ether_shost);
	    break;
	  case ZERO_ENET_ADDRESSES:
	    zero_MAC_address(ethheader->ether_dhost);
	    zero_MAC_address(ethheader->ether_shost);
	    break;
	  case SET_ENET_ADDRESSES:
	    if(b_send_to_inside){
	      if(set_int_MAC_address){
		//	      fprintf(stderr, "setting internalMAC address\n");
		set_MAC_address(ethheader->ether_dhost, int_MAC_address);
	      }
	    }else{
	      if(set_ext_MAC_address  && !is_purely_ext_packet){
		//	      fprintf(stderr, "setting externalMAC address\n");
		set_MAC_address(ethheader->ether_dhost, ext_MAC_address);
	      }
	    }
	    break;
	  }

	  //now send to inside or outside
	  if(b_send_to_inside){
	  
	    //ip_packet->ip_dst.s_addr = (u_int32_t) hex_string_to_u_int32_t("7070a8c0");
	  
	    if(verbose && !(retry < retry_n)){
	      fprintf(stdout, "in:  ");
	    }
	    n = libnet_write_link_layer(lp_int, int_dev, pkt_data, pkt_hdr.caplen);
	  }else{

	    //ip_packet->ip_dst.s_addr = (u_int32_t) hex_string_to_u_int32_t("107010ac");
	    if(verbose && !(retry < retry_n)){
	      fprintf(stdout, "out: ");
	    }
	    n = libnet_write_link_layer(lp_ext, ext_dev, pkt_data, pkt_hdr.caplen);
	  }
	  fprintf(stderr, "data : (%x) header: (%x)", pkt_data, &ip_packet->ip_dst.s_addr);
	  if(verbose && !(retry < retry_n)){
	    fprintf(stdout, "ip: %d.%d.%d.%d > %d.%d.%d.%d\n",
		    (*src_bp), (*(src_bp+1)), (*(src_bp+2)), (*(src_bp+3)),
		    (*dst_bp), (*(dst_bp+1)), (*(dst_bp+2)), (*(dst_bp+3)));
	  }
	}else{//type not ip, just send to outside
	  if(verbose && !(retry < retry_n)){
	    fprintf(stdout, "out: ip: not ip packet\n");
	  }
	  n = libnet_write_link_layer(lp_ext, ext_dev, pkt_data, pkt_hdr.caplen);
	 
	}
	if(super_verbose){
	  fprintf(stderr, "packet size:%d, header.snaplen = %d\n", pkt_hdr.caplen, header.snaplen);
	}
      }else{//only one device, send it there.
	n = libnet_write_link_layer(lp, devname, pkt_data, pkt_hdr.caplen);
      }
    


      if (n == -1) {
	retry--;
	if(!retry){
	  char str[1024];

	  fprintf(stderr, "Error writing packet: %s\n", ll_strerror(errno));
	  fprintf(stderr, "  Packet number %d\n", num_packets + 1);
	  //	  cftime(str, "%Ex %EX", (const time_t *) &pkt_hdr.ts);
	  fprintf(stderr, "  Timestamp     %06ld.%06ld (%s.%06ld)\n",
		  pkt_hdr.ts.tv_sec,
		  pkt_hdr.ts.tv_usec,
		  str,
		  pkt_hdr.ts.tv_usec);
	  fprintf(stderr, "**************PACKET FAILED TO BE SENT!!!!!!***************\n");
	  miss_packets++;
	  break;
	}
	if(super_verbose){
	  fprintf(stderr, "Retrying.....\n");
	}
	usleep(wait);	
      } else if ((unsigned) n != pkt_hdr.caplen){
	fprintf(stderr, "Problem writing packet %d, only wrote %d bytes\n",
		num_packets + 1, n);
	retry = 0;
      }else{
	if(super_verbose){
	  if(retry < retry_n){
	    fprintf(stderr, "**************PACKET WAS SENT!!!!!!***************\n");
	  }
	}
	retry = 0;
      }
    }
    num_packets++;
  }
    
  fprintf(stderr, "sent %d packets, missed %d packets", num_packets, miss_packets);

  if (fclose(fp)) {
    sprintf(msg, "problem closing %s (%s)", filename, strerror(errno));
    error(msg);
  }
  free(filebuffer);
  free(pkt_data);
  
  /* Done. */
  exit(0);
}





