/* netpoke.c
 * $Id: netpoke2.c,v 1.1.1.1 2000/12/05 15:19:35 nap Exp $
 */

/* (c) Copyright 2000 M.I.T.
   Permission is hereby granted, without written agreement or
   royalty fee, to use, copy, modify, and distribute this
   software and its documentation for any purpose, provided that
   the above copyright notice and the following three paragraphs
   appear in all copies of this software.

   IN NO EVENT SHALL M.I.T. BE LIABLE TO ANY PARTY FOR DIRECT,
   INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
   ARISING OUT OF THE USE OF THIS SOFTARE AND ITS DOCUMENTATION,
   EVEN IF M.I.T. HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
   DAMAGE.

   M.I.T. SPECIFICALLY DISCLAIMS ANY WARRANTIES INCLUDING, BUT
   NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT.

   THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS AND M.I.T. HAS
   NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
   ENHANCEMENTS, OR MODIFICATIONS. */

#include <string.h>
#include <stdio.h>
#include <errno.h>
#include <stdlib.h>

#include <unistd.h>
#include <sys/time.h>

#include "pcap.h"

#define  LIBNET_BIG_ENDIAN 1
#include <libnet.h>

#include "util.h"


#define HELP_STRING \
"Usage: %s [<options> ...] <filename>\n" \
"Parameters:\n" \
"    <filename>    Tcpdump input file (use \"-\" for standard input)\n" \
"Options:\n" \
"    -h            Print this message\n" \
"    -z            Zero MAC addresses\n" \
"    -n            Negate MAC addresses\n" \
"    -t            Include timing packets in output stream\n" \
"    -v            Verbose\n" \
"    -d <devname>  Network device name\n" \
"    -s <x>        Speedup factor (defaults to 1.0)\n"


#define OPT_STRING "hzntvd:s:"

#define FILE_BUFFER_SIZE (8 * 1024 * 1024)


typedef enum {
    KEEP_ENET_ADDRESSES,
    NEGATE_ENET_ADDRESSES,
    ZERO_ENET_ADDRESSES
} MAC_address_option;


/* Global variables */

int    swap_bytes;
long   usleep_overhead, min_usleep;

/* Utility routines */


static void
help(char *program)
{
    fprintf(stderr, HELP_STRING, program);
    exit(1);
}


static void
error(char *msg)
{
    fprintf(stderr, " error: %s\n", msg);
    exit(1);
}


#define WAIT_TRIALS 18
#define MSEC        1000
#define MIN_WAIT    (5 * MSEC)
#define MAX_WAIT    (100 * MSEC)

static long
find_usleep_overhead(void)
{
    long wait, max, overhead;
    int  n;
    struct timeval start, end;

    max = 0;
    wait = MIN_WAIT;

    n = WAIT_TRIALS;
    while (n--) {
	gettimeofday(&start, NULL);
	usleep(wait);
	gettimeofday(&end, NULL);
	overhead = timeval_diff(end, start) - wait;
	if (overhead > max)
	    max = overhead;
	wait += (MAX_WAIT - MIN_WAIT) / (WAIT_TRIALS - 1);
    }

    return max;
}

/* Key */

static void
wait_till(struct timeval later)
{
    struct timeval now, delay;
    long diff;

    gettimeofday(&now, NULL);
    delay = later;
    timeval_sub(&delay, now);

    /* Use sleep() to get within 2 sec. */
    while (delay.tv_sec > 1) {
	sleep(delay.tv_sec - 1);
	gettimeofday(&now, NULL);
	delay = later;
	timeval_sub(&delay, now);
    }

    diff = timeval_diff(later, now);

    /* Use usleep() to get within 10 usec. */
    while (diff > min_usleep) {
	usleep(diff - usleep_overhead - 10);
	gettimeofday(&now, NULL);
	diff = timeval_diff(later, now);
    }

    /* Spin to get within 2 usec. */
    while (diff > 1) {
	gettimeofday(&now, NULL);
	diff = timeval_diff(later, now);
    }
}


/* Do things to MAC addresses. */

static void
zero_MAC_address(u_char *address)
{
    int n;

    n = ETHER_ADDR_LEN;
    while (n--)
	*address++ = 0;
}


static void
negate_MAC_address(u_char *address)
{
    int n;

    n = ETHER_ADDR_LEN;
    while (n--)
	*address++ = ~ *address;
}

static void
send_timing_packet(struct libnet_link_int *lp, char *devname,
		   struct timeval when, float speedup)
{
    u_char zeros[ETHER_ADDR_LEN], buf[ETH_H + IP_H + TCP_H];
    int    src_ip, dest_ip, n;

    memset(zeros, 0, sizeof zeros);
    memset(buf, 0, sizeof buf);
    
    src_ip = when.tv_sec;
    if (when.tv_usec >= 500000)
	src_ip++;
    dest_ip = * (int *) &speedup;

    libnet_build_ethernet(
	zeros,              /* destination MAC address */
	zeros,              /* source MAC address      */
	ETHERTYPE_IP,       /* ethernet frame type     */
	NULL,               /* pointer to payload      */
	0,                  /* payload length          */
	buf                 /* where to build packet   */
    );

    libnet_build_ip(
        TCP_H,              /* packet length                  */
	0,                  /* IP tos                         */
	0,                  /* IP id                          */
	0,                  /* fragmentation flags and offset */
	64,                 /* time to live                   */
	IPPROTO_TCP,        /* protocol                       */
	src_ip,             /* source IP address              */
	dest_ip,            /* destination IP address         */
	NULL,               /* pointer to payload             */
	0,                  /* payload length                 */
	buf + ETH_H         /* where to build packet          */
    );

    libnet_build_tcp(
        0,                  /* source port            */
	0,                  /* destination port       */
	0L,                 /* sequence number        */
	0L,                 /* acknowledgement number */
	                    /* control bits           */
	TH_FIN | TH_SYN | TH_RST | TH_PUSH | TH_ACK | TH_URG,
	1024,               /* advertised window size */
	0,                  /* urgent pointer         */
	NULL,               /* pointer to payload     */
	0,                  /* payload size           */
	buf + ETH_H + IP_H  /* where to build packet  */
    );

    libnet_do_checksum(buf + ETH_H, IPPROTO_IP,  IP_H);
    libnet_do_checksum(buf + ETH_H, IPPROTO_TCP, TCP_H);

    n = libnet_write_link_layer(lp, devname, buf, ETH_H + TCP_H + IP_H);

    if (n == -1)
	fprintf(stderr, "Error writing timing packet\n");
    else if (n != ETH_H + TCP_H + IP_H)
	fprintf(stderr, "Problem writing timing packet; only wrote %d bytes\n", n);
}

static void
fprint_timeval(FILE *fp, struct timeval tv)
{
    int hour, min, sec;

    sec = tv.tv_sec % 60;
    tv.tv_sec /= 60;
    min = tv.tv_sec % 60;
    tv.tv_sec /= 60;
    hour = tv.tv_sec % 24;

    fprintf(fp, "%02d:%02d:%02d.%06ld", hour, min, sec, tv.tv_usec);
}

int
main(int argc, char **argv)
{
    char           ch, msg[2048], *filename, *devname, *filebuffer;
    FILE          *fp;
    int            flatout, mung_time, incl_timing, verbose, num_packets, n;
    double         speedup;
    unsigned char *pkt_data;

    MAC_address_option mac_address_mode;

    /* from pcap.h */
    struct pcap_file_header  header;
    struct pcap_pkthdr       pkt_hdr;

    /* from libnet.h */
    struct libnet_ethernet_hdr *ethheader;
    struct libnet_link_int     *lp;

    /* from /usr/include/netinet/in.h */
    struct sockaddr_in sockaddr_in;

    /* from /usr/include/sys/time.h */
    struct timeval file_start_time, net_start_time;


    /* Init globals. */
    swap_bytes = 0;
    usleep_overhead = find_usleep_overhead();
    min_usleep = 2 * usleep_overhead;
    num_packets = 0;

    /* Init parameters. */
    devname = NULL;
    mac_address_mode = KEEP_ENET_ADDRESSES;
    speedup = 1.0;
    flatout = 0;
    mung_time = 0;
    incl_timing = 0;
    verbose = 0;

    /* Parse command line. */
    while ((ch = getopt(argc, argv, OPT_STRING)) != EOF)
	switch (ch) {
	  case 'h':
	    help(argv[0]);
	    break;
	  case 'z':
	    mac_address_mode = ZERO_ENET_ADDRESSES;
	    break;
	  case 'n':
	    mac_address_mode = NEGATE_ENET_ADDRESSES;
	    break;
	  case 't':
	    incl_timing = 1;
	    break;
	  case 'v':
	    verbose = 1;
	    break;
	  case 'd':
	    devname = optarg;
	    break;
	  case 's':
	    speedup = atof(optarg);
	    if (speedup > 0.0)
		speedup = 1.0 / speedup;
	    else {
		speedup = 0.0;
		flatout = 1;
	    }
	    mung_time = speedup != 1.0;
	    break;
	  default:
	    help(argv[0]);
	}
    if (optind != argc - 1)
	help(argv[0]);
    filename = argv[optind++];

    /* Open input file. */
    if (strcmp(filename, "-") == 0)
	fp = stdin;
    else {
	fp = fopen(filename, "rb");
	if (! fp) {
	    sprintf(msg, "problem opening %s (%s)", filename, strerror(errno));
	    error(msg);
	}
    }

    /* Setup file buffer. */
    setbuf(stdout, NULL);

    filebuffer = malloc(FILE_BUFFER_SIZE);
    if (! filebuffer)
	error("out of memory");
    if (setvbuf(fp, filebuffer, _IOFBF, FILE_BUFFER_SIZE))
	error("problem seting up input file buffer");

    /* Read input file header. */
    if (fread(&header, sizeof(struct pcap_file_header), 1, fp) != 1) {
	sprintf(msg, "problem reading header in %s", filename);
	error(msg);
    }
    if (header.magic != PCAP_MAGIC_NUMBER) {
	swap_bytes = 1;
	byteswap_pcap_file_header(&header);
	if (header.magic != PCAP_MAGIC_NUMBER) {
	    sprintf(msg, "bad dump file format in %s", filename);
	    error(msg);
	}
    }
    if (header.linktype != DLT_EN10MB) {
	sprintf(msg, "bad link type %d in %s", header.linktype, filename);
	error(msg);
    }

    /* Open network connection. */
    if (! devname)
	if (libnet_select_device(&sockaddr_in, (u_char **) &devname, msg) != 1)
	    error(msg);
    lp = libnet_open_link_interface(devname, msg);
    if (! lp)
	error(msg);

    /* Setup packet data memory. */
    pkt_data = malloc(header.snaplen);
    if (! pkt_data)
	error("out of memory");
    ethheader = (void *) pkt_data;
    
    /* Loop over input file.  */
    while (1) {
	/* Read packet header. */
	if (fread(&pkt_hdr, sizeof(struct pcap_pkthdr), 1, fp) != 1) {
	    if (feof(fp))
		break;
	    else if (ferror(fp)) {
		sprintf(msg, "problem reading packet header (%s)", strerror(errno));
		error(msg);
	    } else {
		fprintf(stderr, "problem reading packet %d header\n", num_packets);
		continue;
	    }
	}
	if (swap_bytes)
	    byteswap_pcap_packet_header(&pkt_hdr);
	if (pkt_hdr.caplen > header.snaplen)
	    pkt_hdr.caplen = header.snaplen;

	/* Read packet data. */
	if (fread(pkt_data, pkt_hdr.caplen, 1, fp) != 1) {
	    if (feof(fp))
		break;
	    else if (ferror(fp)) {
		sprintf(msg, "problem reading packet data (%s)", strerror(errno));
		error(msg);
	    } else {
		fprintf(stderr, "problem reading packet %d data\n", num_packets);
		fprintf(stderr, "    pkt_hdr.caplen = %d\n", pkt_hdr.caplen);
		continue;
	    }
	}

	/* Optionally mung MAC addresses. */
	switch (mac_address_mode) {
	  case KEEP_ENET_ADDRESSES:
	    break;
	  case NEGATE_ENET_ADDRESSES:
	    negate_MAC_address(ethheader->ether_dhost);
	    negate_MAC_address(ethheader->ether_shost);
	    break;
	  case ZERO_ENET_ADDRESSES:
	    zero_MAC_address(ethheader->ether_dhost);
	    zero_MAC_address(ethheader->ether_shost);
	    break;
	}

	/* Note time of first packet. */
	if (! num_packets) {
	    file_start_time = pkt_hdr.ts;
	    if (incl_timing)
		send_timing_packet(lp, devname,
				   file_start_time,
				   (float) (1.0 / speedup));
	    gettimeofday(&net_start_time, NULL);
	}

	/* Wait till it's time for this packet to go. */
	if (! flatout) {
	    struct timeval when;

	    when =  pkt_hdr.ts;
	    timeval_sub(&when, file_start_time);
	    if (mung_time)
		timeval_mul(&when, speedup);
	    timeval_add(&when, net_start_time);
	    wait_till(when);
	}

	/* Send the packet down the wire. */
	n = libnet_write_link_layer(lp, devname, pkt_data, pkt_hdr.caplen);
	if (n == -1) {
	    char str[1024];

	    fprintf(stderr, "Error writing packet: %s\n", ll_strerror(errno));
	    fprintf(stderr, "  Packet number %d\n", num_packets + 1);
	    /*	    cftime(str, "%Ex %EX", (const time_t *) &pkt_hdr.ts); */
	    fprintf(stderr, "  Timestamp     %06ld.%06ld (%s.%06ld)\n",
		    pkt_hdr.ts.tv_sec,
		    pkt_hdr.ts.tv_usec,
		    str,
		    pkt_hdr.ts.tv_usec);
	} else if ((unsigned) n != pkt_hdr.caplen)
	    fprintf(stderr, "Problem writing packet %d, only wrote %d bytes\n",
		    num_packets + 1, n);

	if (verbose)
	    fprintf(stdout, "%d\n", pkt_hdr.caplen);

	num_packets++;
    }
    
    /* Clean up. */
    if (libnet_close_link_interface(lp)) {
	sprintf(msg, "problem closing %s", devname);
	error(msg);
    }
    if (fclose(fp)) {
	sprintf(msg, "problem closing %s (%s)", filename, strerror(errno));
	error(msg);
    }
    free(filebuffer);
    free(pkt_data);

    /* Done. */
    exit(0);
}
