/* tcpsize.c
 * $Id: tcpsize.c,v 1.1.1.1 2000/12/05 15:19:35 nap Exp $
 */

#include <string.h>
#include <stdio.h>
#include <errno.h>
#include <stdlib.h>

#include <pcap.h>

#include "util.h"


#define HELP_STRING \
"Usage: %s <filename>\n" \
"Parameters:\n" \
"    <filename>    Tcpdump input file (use \"-\" for standard input)\n"

/* Utility routines */

static void
help(char *program)
{
    fprintf(stderr, HELP_STRING, program);
    exit(1);
}


static void
error(char *msg)
{
    fprintf(stderr, " error: %s\n", msg);
    exit(1);
}

int
main(int argc, char **argv)
{
    char  msg[2048], *filename, *dummy;
    FILE *fp;
    long  num_packets, size;

    /* from pcap.h */
    struct pcap_file_header  header;
    struct pcap_pkthdr       pkt_hdr;

    /* Setup dummy buffer */
    dummy = malloc(128 * 1024);
    if (! dummy)
	error("Memory exhausted");

    /* Parse command line. */
    if (argc != 2)
	help(argv[0]);
    filename = argv[1];

    /* Open input file. */
    if (strcmp(filename, "-") == 0)
	fp = stdin;
    else {
	fp = fopen(filename, "rb");
	if (! fp) {
	    sprintf(msg, "problem opening %s (%s)", filename, strerror(errno));
	    error(msg);
	}
    }

    /* Read input file header. */
    if (fread(&header, sizeof(struct pcap_file_header), 1, fp) != 1) {
	sprintf(msg, "problem reading header in %s", filename);
	error(msg);
    }
    if (header.magic != PCAP_MAGIC_NUMBER) {
	sprintf(msg, "bad dump file format in %s", filename);
	error(msg);
    }
    if (header.linktype != DLT_EN10MB) {
	sprintf(msg, "bad link type %d in %s", header.linktype, filename);
	error(msg);
    }
    
    /* Loop over input file.  */
    num_packets = size = 0;
    while (1) {
	/* Read packet header. */
	if (fread(&pkt_hdr, sizeof(struct pcap_pkthdr), 1, fp) != 1) {
	    if (feof(fp))
		break;
	    else if (ferror(fp)) {
		sprintf(msg, "problem reading packet header (%s)", strerror(errno));
		error(msg);
	    } else {
		fprintf(stderr, "problem reading packet %ld header\n", num_packets);
		continue;
	    }
	}

	/* Skip packet data. */
	/*	if (fseek(fp, (long) pkt_hdr.caplen, SEEK_CUR)) { */
	if (fread(dummy, pkt_hdr.caplen, 1, fp) != 1) {
	    if (feof(fp))
		break;
	    else if (ferror(fp)) {
		sprintf(msg, "problem reading packet data (%s)", strerror(errno));
		error(msg);
	    } else {
		fprintf(stderr, "problem reading packet %ld data\n", num_packets);
		fprintf(stderr, "    pkt_hdr.caplen = %d\n", pkt_hdr.caplen);
		continue;
	    }
	}

	num_packets++;
	size += (long) pkt_hdr.len;
    }
    
    printf("Number of packets:     %ld\n", num_packets);
    printf("Total size of packets: %ld\n", size);

    /* Clean up. */
    if (fclose(fp)) {
	sprintf(msg, "problem closing %s (%s)", filename, strerror(errno));
	error(msg);
    }

    /* Done. */
    exit(0);
}
