/* 

	subscan.c
	
	scans a subnet for computers listening on a given port, 
	and does it quick.
	
	for use with tcpip.h (high level socket functions)

	author: jjohnson@eagle.ptialaska.net
		presonic@irc
		
	compile: gcc -o subscan subscan.c
	
	this example makes use of:
	
		i_connect()		(non blocking i/o)
		i_disconnect()
		i_nslookup()

*/

#include <stdio.h>
#include <unistd.h>		/* STDIN_FILENO,STDERR_FILENO */
#include <sys/types.h>		/* FD_ZERO,FD_SET, etc */
#include <sys/time.h>		/* struct timeval */
#include <errno.h>
      
#define	SHOW_SUCCESS		/* print out successful connections */
#define	SHOW_REFUSED		/* print out refused connections */
#define	SHOW_UNREACH		/* print out unreachable connections */
#define SHOW_UNKNOWN		/* print out unknown connection error */

#define TIMEOUT 60		/* seconds before select() can timeout */
				/* note: the longer the timeout, the more accurate
					 this scanner is.  60 is good. */
#include "tcpip.h"

main(int argc, char *argv[])
{
	int i, sockfd[257], maxsockfd=0, sockopt;
	char ip[30], subnet[30], *ptr,string[256];
	fd_set wset;
	/* socklen_t socklen; */
	unsigned int socklen;
	struct timeval time;
	FILE *log;

	if(argc != 4)
	{
		printf("usage: %s <ip|hostname> <port> <log>\n",argv[0]);
		exit(-1);
	}
	
	/* validate arguments */

	strncpy(subnet,i_nslookup(argv[1],RETURN_SUBNET),sizeof(subnet));
	
	if(strcmp(subnet,"ERROR") == 0)
	{
		printf("Error finding subnet with given argument.  (bad dns name?)\n");
		printf("usage: %s <ip|hostname> <port> <log>\n",argv[0]);
		exit(-1);
	}
	
	printf("Scanning subnet:\t%s\nPort:\t\t\t%s\nAppending to log file:\t%s\nScanning...\n",subnet,argv[2],argv[3]);
	
	/* linux has a 256 fd per process limit by default.  we dont need these anyways */
	for(i=0 ; i < 256 ; ++i)
	{
		if(i == STDERR_FILENO) continue;	/* dont close stderr just yet. */
		close(i);
	}
	
	if((log = fopen(argv[3],"a")) == NULL)
	{
		perror("error opening log file");
		exit(-1);
	}
	
	close(STDERR_FILENO);	/* done opening file, we'll close this now. */
	
	sprintf(string,"START of scan on subnet %s for port %d (%s)\n\0",subnet,atoi(argv[2]),argv[1]);
	fputs(string,log);

	FD_ZERO(&wset);
	
	/* connect all sockets */
	
	for(i=1 ; i != 256 ; ++i)
	{
		sprintf(ip,"%s.%d\0",subnet,i);
		if((sockfd[i] = i_connect(ip,atoi(argv[2]),NONBLOCKING)) < 0)
		{
			string[0] = '\0';
			if(sockfd[i] == SOCKET_ERR)
				sprintf(string,"*** socket() error with sockfd[%d]\n\0");
			else if(sockfd[i] == CONNECT_ERR)
				sprintf(string,"*** connect() error with sockfd[%d]\n\0");
			if(strlen(string) > 0) fputs(string,log);
			continue;
		}
		usleep(400);
	}

	while(1)
	{
		/* build fd_set, create maxsockfd */
		for(i = 1 ; i != 256 ; ++i)
		{
			if(getsockopt(sockfd[i],SOL_SOCKET,SO_ERROR,&sockopt,&socklen) == 0)
			{
				FD_SET(sockfd[i], &wset);
				if(sockfd[i] > maxsockfd) maxsockfd = sockfd[i];
			}
		}
		
		time.tv_sec = TIMEOUT;
		if(select(maxsockfd + 1,NULL,&wset,NULL,&time) == 0)
		{
		
			sprintf(string,"END of scan on subnet %s for port %d\n\0",subnet,atoi(argv[2]));
			fputs(string,log);
			fclose(log);
			
			exit(-1);
		}
		
		for(i=1 ; i != 256; ++i)
		{
			if(FD_ISSET(sockfd[i],&wset))
			{
				/* remove the sockfd from wset */
				FD_CLR(sockfd[i],&wset);
				
				socklen = 4; 	/* sizeof(int) */
				sockopt = -1;
				
				getsockopt(sockfd[i],SOL_SOCKET,SO_ERROR,&sockopt,&socklen);
				
				i_disconnect(sockfd[i]);	/* dont need it anymore */
                                
				string[0] = '\0';
				
				if(sockopt == 0)
				{
					#ifdef SHOW_SUCCESS
					sprintf(string,"SUCCESSFUL CONNECTION: %s.%d(%s)\n",subnet,i,argv[2]);
					#endif	
				}
				else if(sockopt == ECONNREFUSED)
				{
					#ifdef SHOW_REFUSED
					sprintf(string,"REFUSED CONNECTION: %s.%d(%s)\n",subnet,i,argv[2]);
					#endif
				}
				else if(sockopt == ENETUNREACH)
				{
					#ifdef SHOW_UNREACH
					sprintf(string,"UNREACHABLE CONNECTION: %s.%d(%s)\n",subnet,i,argv[2]);
					#endif
				}
				else if(sockopt == EHOSTUNREACH)
				{
					#ifdef SHOW_UNREACH
					sprintf(string,"UNREACHABLE CONNECTION: %s.%d(%s)\n",subnet,i,argv[2]);
					#endif
				}
				else 
				{
					#ifdef SHOW_UNKNOWN
					sprintf(string,"UNKNOWN ERROR #%d: %s.%d(%s)\n",sockopt,subnet,i,argv[2]);
					#endif
				}
				if(strlen(string) > 0) fputs(string,log);

			}
		}
	}
}
