//----------------------------------------------------------------------------
//   Implementation of TBaseDib class
//----------------------------------------------------------------------------
#include "avi2mpg2.h"

extern bool AbortOperation;

char EOF_STR[] = "Premature end of file";
char INV_HDR[] = "Unsupported image header";
char INV_CDEP[] = "Unsupported color depth";
char INV_BPLAN[] = "Unsupported number of bit planes";
char NO_MEM[] = "Insufficient memory to load image";
char NO_COMPB[] = "RLE Compressed BMP files are not supported";
char JERR_GIF_BUG[] = "GIF output error";

char JERR_GIF_CODESIZE[] = "Invalid GIF codesize";
char JERR_GIF_COLORSPACE[] = "Invalid GIF colorspace";
char JERR_GIF_IMAGENOTFOUND[] = "Not enough images in GIF file";
char JERR_GIF_NOT[] = "Not a GIF file";
char JWRN_GIF_BADDATA[] = "Corrupt data in GIF file";
char JWRN_GIF_CHAR[] = "Invalid char in GIF file, ignoring";
char JWRN_GIF_ENDCODE[] = "Premature end of GIF image";
char JWRN_GIF_NOMOREDATA[] = "Ran out of GIF bits";

char JERR_TGA_BADCMAP[] = "Unsupported Targa colormap format";

char JERR_TGA_BADPARMS[] = "Invalid or unsupported Targa file";
char JERR_TGA_COLORSPACE[] = "Targa output must be grayscale or RGB";


//
// size of scan in bytes =
//   Pixel Width * bits per pixel rounded up to a DWORD boundary
//
inline long ScanBytes(int pixWidth, int bitsPixel) {
  return (((long)pixWidth*bitsPixel+31) / 32) * 4;
}

TBaseDib::TBaseDib()
{
  Info = 0;
  Bits = 0;
  NumClrs = 0;
  Mode = DIB_RGB_COLORS;
  W = 0;
  H = 0;
  BitsPerPixel = 0;
  IsResHandle = false;
  EndScan = 0;
  strcpy(ErrStr, "");
  strcpy(fName, "");
  Handle = NULL;
  Handle2 = NULL;
}

void TBaseDib::YieldTime()
{
//  MSG msg;

//  int i;
//  for (i = 0; i < 100000; i++);

//  if (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))
//  {
//    TranslateMessage(&msg);
//    DispatchMessage(&msg);
//  }
}

bool TBaseDib::CreateDib(bool isBMP_BI_BITFIELDS)
{
  BITMAPINFOHEADER InfoHeader;
  int  colorAlloc;

  if (Handle)
  {
    if ((Info->bmiHeader.biBitCount != BitsPerPixel) ||
        (Info->bmiHeader.biWidth != W) ||
        (Info->bmiHeader.biHeight != H))
    {
      ::GlobalUnlock(Handle);
      ::GlobalFree(Handle);
    }
    else
      return true;
  }
  switch (BitsPerPixel)
  {
	 case 1:
		NumClrs = 2;
		break;
	 case 4:
		NumClrs = 16;
		break;
	 case 8:
		NumClrs = 256;
		break;
    case 15:
    case 16:
	 case 24:
    case 32:
		NumClrs = 0;
		break;
	 default:
		strcpy(ErrStr, INV_CDEP);
		return false;
  }
  InfoHeader.biSize = sizeof(BITMAPINFOHEADER);
  InfoHeader.biWidth = W;
  InfoHeader.biHeight = H;
  if (BitsPerPixel == 15)
    InfoHeader.biBitCount = 16;
  else
    InfoHeader.biBitCount = (WORD) BitsPerPixel;
  InfoHeader.biPlanes = 1;
  InfoHeader.biXPelsPerMeter = 0;
  InfoHeader.biYPelsPerMeter = 0;
  InfoHeader.biClrUsed = 0;
  InfoHeader.biClrImportant = 0;  // 0 = all colors
  if (isBMP_BI_BITFIELDS)
    InfoHeader.biCompression = BI_BITFIELDS;
  else
    InfoHeader.biCompression = BI_RGB;
  InfoHeader.biSizeImage = ScanBytes(W, InfoHeader.biBitCount) * H;

  if (isBMP_BI_BITFIELDS)
    colorAlloc = 3 * sizeof(DWORD);
  else
    colorAlloc = NumClrs * sizeof(RGBQUAD); // size of color tables
  BitsAllocated = (long) InfoHeader.biSize + (long) colorAlloc + (long) InfoHeader.biSizeImage;

  if (Handle)
    Handle = GlobalFree(Handle);
  Handle = GlobalAlloc(GMEM_MOVEABLE | GMEM_ZEROINIT, BitsAllocated);
  if (Handle)
  {
	 Info = (LPBITMAPINFO)::GlobalLock(Handle);
	 if (Info)
	 {
		Info->bmiHeader = InfoHeader;
		switch (InfoHeader.biBitCount)
		{
		  case 1 :
			 xSizeNoPad = W >> 3;
			 if (W % 8 != 0)
				xSizeNoPad++;
			 break;
		  case 4 :
			 xSizeNoPad = W >> 1;
			 if (W % 2 != 0)
				xSizeNoPad++;
			 break;
		  case 8 :
			 xSizeNoPad = W;
			 break;
        case 15:
        case 16:
          xSizeNoPad = W << 1;
          break;
		  case 24:
			 xSizeNoPad = W * 3;
			 break;
        case 32:
          xSizeNoPad = W << 2;
          break;
		}
		xSize = xSizeNoPad;
		while ((xSize & 3) != 0)
		  xSize++;
		Mode = DIB_RGB_COLORS;
		Bits = (char far *) Info + ((int)InfoHeader.biSize + colorAlloc);
      if (NumClrs)
        Colors = (TRgbQuad *) Info + (int) InfoHeader.biSize;
      else
        Colors = NULL;
		return true;
	 }
  }
  strcpy(ErrStr, NO_MEM);
  return false;
}

bool TBaseDib::IsJPEG()
{
  unsigned char buffer[1024];
  int state, i;

  if (fread(&buffer, 1, sizeof(buffer), fpin) != sizeof(buffer))
	 return false;

  state = 0;
  i = 0;
  while ((i < 1024) && (state != 2))
  {
	 switch (state)
	 {
		case 0 : if (buffer[i] == 0xFF)
					  state++;
					break;
		case 1 : if (buffer[i] == 0xD8)
					  state++;
					else
					  state = 0;
					break;
//		case 2 : if (buffer[i] == 0xFF)
//					  state++;
//					else
//					  state--;
//					break;
//		case 3 : if (buffer[i] >= 0xE0)
//					  state++;
//					else
//					  state = 0;
//					break;
	 }
	 i++;
  }
  if (state == 2)
  {
	 fseek(fpin, i - 2, SEEK_SET);
	 return true;
  }
  return false;
}


bool TBaseDib::IsTarga()
{
  unsigned char theader[18];
  int maplen, firstentry;

  if (fread(&theader, 1, sizeof(theader), fpin) != sizeof(theader))
	 return false;

  maplen = GET_2B(theader, 5);
  firstentry = GET_2B(theader, 3);

  if ((theader[1] > 1) ||		/* cmaptype must be 0 or 1 */
		(theader[16] < 8 || theader[16] > 32))
	 return false;

  if (theader[2] > 8)
	 theader[2] -= (unsigned char) 8;

  /* Now should have subtype 1, 2, or 3 */
  switch (theader[2]) {
	 case 1:			/* Colormapped image */
			  if (theader[16] != 8 || theader[1] != 1)
				 return false;
			  break;
	 case 2:			/* RGB image */
           switch (theader[16])
           {
             case 15:
             case 16:
             case 24:
             case 32:
               break;
             default:
				   return false;
           }
			  break;
	 case 3:			/* Grayscale image */
			  if (theader[16] != 8)
				 return false;
			  break;
	 default:
			  return false;
  }

  if (theader[2] == 1)
  {
	 if (maplen > 0)
	 {
		if (maplen > 256 || firstentry)
		  return false;
	 }
	 else
	 {
		if (theader[1])		/* but you promised a cmap! */
		  return false;
	 }
  }
  return true;
}

//
// Select a read file routine based on file type
//

bool TBaseDib::LoadFile(const char* name)
{
  unsigned char bytes[8];
  bool retval;

  AnimatedGIF = false;
  GIFFrameNum = 0;
  GIFTotalFrames = 0;
  fType = TYPE_NONE;
  if ((fpin = fopen(name, "rb")) == NULL)
  {
	 sprintf(ErrStr, "Error opening file %s", name);
	 return false;
  }

  if (fread(&bytes, 1, 8, fpin) != 8)
  {
    fclose(fpin);
	 sprintf(ErrStr, "%s is an empty or invalid file", name);
	 return false;
  }
  fseek(fpin, 0, SEEK_SET);
  strcpy(fName, name);
  if (bytes[0] == 10)
  {
    fType = TYPE_PCX;
	 retval = LoadPCXFile();
  }
  else
    if ((bytes[0] == 'B') && (bytes[1] == 'M'))
    {
      fType = TYPE_BMP;
	   retval = LoadBMPFile();
    }
    else
      if ((bytes[0] == 'G') && (bytes[1] == 'I') && (bytes[2] == 'F'))
      {
        fType = TYPE_GIF;
	     retval = LoadGIFFile();
      }
      else
        if ((bytes[0] == 0xFF) && (bytes[1] == 0xD8) &&
		      (bytes[2] == 0xFF) && (bytes[3] >= 0xE0))
        {
          fType = TYPE_JPG;
	       retval = LoadJPEGFile();
        }
        else
          if (IsTarga())
          {
				fseek(fpin, 0, SEEK_SET);
            fType = TYPE_TGA;
 	         retval = LoadTGAFile();
          }
          else
          {
            fseek(fpin, 0, SEEK_SET);
            if (IsJPEG())
            {
              fType = TYPE_JPG;
	           retval = LoadJPEGFile();
            }
            else
            {
              sprintf(ErrStr, "%s is an unknown file type", name);
              retval = false;
            }
          }
  fclose(fpin);
  return retval;
}

/* Fetch the next code_size bits from the GIF data */
/* We assume code_size is less than 16 */

int TBaseDib::GetCode()
{
  int offs, ret, count, accum;

  while ((GIFCurBit + GIFCodeSize) > GIFLastBit)
  {
    /* Time to reload the buffer */
    if (GIFOutOfBlocks)
      return GIFEndCode;	/* fake something useful */
    /* preserve last two bytes of what we have -- assume code_size <= 16 */
    GIFCodeBuf[0] = GIFCodeBuf[GIFLastByte - 2];
    GIFCodeBuf[1] = GIFCodeBuf[GIFLastByte - 1];
    /* Load more bytes; set flag if we reach the terminator block */
    if ((count = getc(fpin)) == EOF)
    {
	   strcpy(ErrStr, EOF_STR);
      longjmp(SetJmpBuffer, 1);
    }
    if (!count)
    {
      GIFOutOfBlocks = true;
      return GIFEndCode;	/* fake something useful */
    }
    else
    {
      if (fread(&GIFCodeBuf[2], 1, count, fpin) != count)
      {
        GIFOutOfBlocks = true;
        return GIFEndCode;	/* fake something useful */
      }
    }
    /* Reset counters */
    GIFCurBit = (GIFCurBit - GIFLastBit) + 16;
    GIFLastByte = 2 + count;
	 GIFLastBit = GIFLastByte * 8;
  }

  /* Form up next 24 bits in accum */
  offs = GIFCurBit >> 3;	/* byte containing cur_bit */
  accum = GIFCodeBuf[offs + 2];
  accum <<= 8;
  accum |= GIFCodeBuf[offs + 1];
  accum <<= 8;
  accum |= GIFCodeBuf[offs];

  /* Right-align cur_bit in accum, then mask off desired number of bits */
  accum >>= (GIFCurBit & 7);
  ret = ((int) accum) & ((1 << GIFCodeSize) - 1);

  GIFCurBit += GIFCodeSize;
  return ret;
}

/* Read an LZW-compressed byte */

int TBaseDib::LZWReadByte()
{
  int code;		/* current working code */
  int incode;			/* saves actual input code */

  /* First time, just eat the expected Clear code(s) and return next code, */
  /* which is expected to be a raw byte. */
  if (GIFFirstTime)
  {
    GIFFirstTime = FALSE;
    code = GIFClearCode;	/* enables sharing code with Clear case */
  }
  else
  {
    /* If any codes are stacked from a previously read symbol, return them */
    if (GIFSp > GIFSymbolStack)
      return (int) *(-- GIFSp);
    /* Time to read a new symbol */
    code = GetCode();
  }

  if (code == GIFClearCode)
  {
    /* Reinit state, swallow any extra Clear codes, and */
    /* return next code, which is expected to be a raw byte. */
    ReInitLZW();
    do
    {
      code = GetCode();
    } while (code == GIFClearCode);
    if (code > GIFClearCode)  /* make sure it is a raw byte */
		code = 0;			/* use something valid */
    /* make firstcode, oldcode valid! */
    GIFFirstCode = GIFOldCode = code;
    return code;
  }

  if (code == GIFEndCode)
  {
    /* Skip the rest of the image, unless GetCode already read terminator */
    if (!GIFOutOfBlocks)
    {
      if (!SkipDataBlocks())
        longjmp(SetJmpBuffer, 1);
      GIFOutOfBlocks = true;
    }
	 /* Pad data with 0's */
    return 0;			/* fake something usable */
  }

  /* Got normal raw byte or LZW symbol */
  incode = code;		/* save for a moment */

  if (code >= GIFMaxCode) /* special case for not-yet-defined symbol */
  {
    /* code == max_code is OK; anything bigger is bad data */
    if (code > GIFMaxCode)
      incode = 0;		/* prevent creation of loops in symbol table */
    /* this symbol will be defined as oldcode/firstcode */
    *(GIFSp++) = (unsigned char) GIFFirstCode;
	 code = GIFOldCode;
  }

  /* If it's a symbol, expand it into the stack */
  while (code >= GIFClearCode)
  {
    *(GIFSp++) = GIFSymbolTail[code]; /* tail is a byte value */
    code = GIFSymbolHead[code]; /* head is another LZW symbol */
  }
  /* At this point code just represents a raw byte */
  GIFFirstCode = code;	/* save for possible future use */

  /* If there's room in table, */
  if ((code = GIFMaxCode) < LZW_TABLE_SIZE)
  {
    /* Define a new symbol = prev sym + head of this sym's expansion */
	 GIFSymbolHead[code] = (unsigned short int) GIFOldCode;
	 GIFSymbolTail[code] = (unsigned char) GIFFirstCode;
    GIFMaxCode++;
    /* Is it time to increase code_size? */
    if ((GIFMaxCode >= GIFLimitCode) &&
	     (GIFCodeSize < MAX_LZW_BITS))
    {
      GIFCodeSize++;
      GIFLimitCode <<= 1;	/* keep equal to 2^code_size */
    }
  }

  GIFOldCode = incode;	/* save last input symbol for future use */
  return GIFFirstCode;	/* return first byte of symbol's expansion */
}

/* (Re)initialize LZW state; shared code for startup and Clear processing */

void TBaseDib::ReInitLZW()
{
  GIFCodeSize = GIFInputCodeSize + 1;
  GIFLimitCode = GIFClearCode << 1;	/* 2^code_size */
  GIFMaxCode = GIFClearCode + 2;	/* first unused code value */
  GIFSp = GIFSymbolStack;		/* init stack to empty */
}

/* Initialize for a series of LZWReadByte (and hence GetCode) calls */

void TBaseDib::InitLZWCode()
{
  /* GetCode initialization */
  GIFLastByte = 2;		/* make safe to "recopy last two bytes" */
  GIFLastBit = 0;		/* nothing in the buffer */
  GIFCurBit = 0;		/* force buffer load on first call */
  GIFOutOfBlocks = false;

  /* LZWReadByte initialization: */
  /* compute special code values (note that these do not change later) */
  GIFClearCode = 1 << GIFInputCodeSize;
  GIFEndCode = GIFClearCode + 1;
  GIFFirstTime = true;
  ReInitLZW();
}

/* Ignore any extension blocks */

bool TBaseDib::SkipDataBlocks()
{
  char buf[256];
  int count;

  if ((count = getc(fpin)) == EOF)
  {
	 strcpy(ErrStr, EOF_STR);
    return false;
  }
  while (count > 0)
  {
    if (fread(&buf, 1, count, fpin) != count)
    {
	   strcpy(ErrStr, EOF_STR);
      return false;
    }
    if ((count = getc(fpin)) == EOF)
    {
	   strcpy(ErrStr, EOF_STR);
      return false;
    }
  }
  return true;
}

/* Check to see if this is an animated GIF file */

bool TBaseDib::CheckGIFAnim()
{
  char hdrbuf[768];		/* workspace for reading control blocks */
  int i, l, t, w, h;

  /* Scan until we reach start of desired image. */
  for (;;)
  {
    if ((i = getc(fpin)) == EOF)
      return false;

    if (i == ';')		/* GIF terminator?? */
      return false;

    if (i == '!')		/* Extension */
    {
      /* read extension label byte */
      if ((i = getc(fpin)) == EOF)
        return false;

      if (SkipDataBlocks())
        continue;
      else
        return false;
    }

    if (i != ',') 	/* Not an image separator? */
      continue;

    /* Read and decipher Local Image Descriptor */
    if (fread(hdrbuf, 1, 9, fpin) != 9)
      return false;

    l = GET_2B(hdrbuf, 0);
    t = GET_2B(hdrbuf, 2);
	 w = GET_2B(hdrbuf, 4);
    h = GET_2B(hdrbuf, 6);
    if ((l + w > logWidth) || (t + h > logHeight))
      return false;

    /* Read local colormap if header indicates it is present */
    if (hdrbuf[8] & 0x80)
    {
      i = (2 << (hdrbuf[8] & 0x07)) * 3;
      if (i > 768)
        return false;

      if (fread(hdrbuf, 1, i, fpin) != i)
        return false;
	 }

    if ((hdrbuf[0] = (unsigned char) getc(fpin)) == EOF) /* get min-code-size byte */
      return false;

    if (hdrbuf[0] < 2 || hdrbuf[0] >= MAX_LZW_BITS)
      return false;

    /* Reached desired image, so break out of loop */
    break;
  }
  return true;
}

/* Get a GIF image header */

bool TBaseDib::GetGIFImageHeader()
{
  char hdrbuf[10];		/* workspace for reading control blocks */
  int i;

  GIFDelayTime = 0;
  TransparentGIF = false;
  GIFUserInputFlag = false;
  GIFDisposal = 0;
  TransparentIdx = 0;
  LCLColorTable = false;
  LCLClrUsed = 0;
  /* Scan until we reach start of desired image. */
  for (;;)
  {
    if ((i = getc(fpin)) == EOF)
    {
      strcpy(ErrStr, EOF_STR);
      return false;
    }

    if (i == ';')		/* GIF terminator?? */
    {
	   strcpy(ErrStr, JERR_GIF_IMAGENOTFOUND);
      return false;
    }

    if (i == '!')		/* Extension */
    {
      /* read extension label byte */
      if ((i = getc(fpin)) == EOF)
      {
        strcpy(ErrStr, EOF_STR);
        return false;
      }
      if (i == 0xF9)
      {
        if (fread(hdrbuf, 1, 6, fpin) != 6)
        {
          strcpy(ErrStr, EOF_STR);
          return false;
        }
        TransparentGIF = hdrbuf[1] & 0x01;
        GIFUserInputFlag = (hdrbuf[1] & 0x02) >> 1;
        GIFDisposal = (hdrbuf[1] & 0x1C) >> 2;
        GIFDelayTime = GET_2B(hdrbuf, 2);
        TransparentIdx = hdrbuf[4];
      }
      else
      {
        if (SkipDataBlocks())
          continue;
        else
          return false;
      }
    }

    if (i != ',') 	/* Not an image separator? */
      continue;

    /* Read and decipher Local Image Descriptor */
    if (fread(&GIFImgDescriptor, 1, sizeof(GIFImgDescriptor), fpin) != sizeof(GIFImgDescriptor))
    {
      strcpy(ErrStr, EOF_STR);
      return false;
    }
    picLeft = GIFImgDescriptor.leftPosition;
    picTop = GIFImgDescriptor.topPosition;
	 picWidth = GIFImgDescriptor.width;
    picHeight = GIFImgDescriptor.height;
    if ((picLeft + picWidth > logWidth) ||
        (picTop + picHeight > logHeight))
    {
      strcpy(ErrStr, INV_HDR);
      return false;
    }
	 IsInterlaced = (GIFImgDescriptor.packedFields & 0x40);

    /* Read local colormap if header indicates it is present */
    LCLColorTable = (GIFImgDescriptor.packedFields & 0x80);
    if (LCLColorTable)
    {
		LCLClrUsed = 2 << (GIFImgDescriptor.packedFields & 0x07);
      biClrUsed = LCLClrUsed;
      for (i = 0; i < biClrUsed; i++)
      {
        if (fread(&hdrbuf[0], 1, 3, fpin) != 3)
        {
          strcpy(ErrStr, EOF_STR);
          return false;
        }
        ColorMap[i].rgbRed = hdrbuf[0];
        ColorMap[i].rgbGreen = hdrbuf[1];
        ColorMap[i].rgbBlue = hdrbuf[2];
      }
	 }
    else
    {
      biClrUsed = GBLClrUsed;
      memcpy(&ColorMap[0], &GBLColorMap[0], sizeof(RGBQUAD) * biClrUsed);
    }

    if ((GIFInputCodeSize = getc(fpin)) == EOF) /* get min-code-size byte */
    {
      strcpy(ErrStr, EOF_STR);
      return false;
    }
    if (GIFInputCodeSize < 2 || GIFInputCodeSize >= MAX_LZW_BITS)
    {
      strcpy(ErrStr, JERR_GIF_CODESIZE);
      return false;
    }
    /* Reached desired image, so break out of loop */
    break;
  }

  /* Prepare to read selected image: first initialize LZW decompressor */
  InitLZWCode();
  if (IsInterlaced)
  {
	 GIFPass2Offset = (picHeight + 7) / 8;
	 GIFPass3Offset = GIFPass2Offset + (picHeight + 3) / 8;
	 GIFPass4Offset = GIFPass3Offset + (picHeight + 1) / 4;
  }
  switch (biClrUsed)
  {
    case 2:
      BitsPerPixel = 1;
      picBitsPerPixel = 1;
      break;
    case 16:
      BitsPerPixel = 4;
      picBitsPerPixel = 4;
      break;
    default:
      BitsPerPixel = 8;
      picBitsPerPixel = 8;
  }
  return true;
}

/* Read a GIF file header */

bool TBaseDib::ReadGIFHeader()
{
  unsigned char buf[3];
  int i;

  if (fread(&GIFHeader, 1, sizeof(GIFHeader), fpin) != sizeof(GIFHeader))
  {
	 strcpy(ErrStr, EOF_STR);
	 return false;
  }

  /* Decipher Logical Screen Descriptor */
  picWidth = GIFHeader.width;
  logWidth = picWidth;
  picHeight = GIFHeader.height;
  logHeight = picHeight;
  GBLClrUsed = 2 << (GIFHeader.packedFields & 0x07);
  if (GBLClrUsed == 0)
    GBLClrUsed = 255;
  /* Read global colormap if header indicates it is present */
  GBLColorTable = (GIFHeader.packedFields & 0x80) != 0;
  if (GBLColorTable)
  {
    BackGroundClr = GIFHeader.backGroundColor;
    for (i = 0; i < GBLClrUsed; i++)
    {
      if (fread(&buf[0], 1, 3, fpin) != 3)
      {
        strcpy(ErrStr, EOF_STR);
        return false;
      }
      GBLColorMap[i].rgbRed = buf[0];
      GBLColorMap[i].rgbGreen = buf[1];
      GBLColorMap[i].rgbBlue = buf[2];
    }
  }
  else
  {
    BackGroundClr = 0;
    for (i = 0; i < GBLClrUsed; i++)
    {
      GBLColorMap[i].rgbRed = (unsigned char) i;
      GBLColorMap[i].rgbGreen = (unsigned char) i;
      GBLColorMap[i].rgbBlue = (unsigned char) i;
    }
  }
  biClrUsed = GBLClrUsed;
  return GetGIFImageHeader();
}

bool TBaseDib::ReadTGAPixel8()
{
  int i;

  /* Duplicate previously read pixel? */
  if (TGADupPixelCount > 0)
  {
    TGADupPixelCount--;
    return true;
  }

  /* Time to read RLE block header? */
  if (--TGABlockCount < 0) /* decrement pixels remaining in block */
  {
    if ((i = (unsigned char) getc(fpin)) == EOF)
    {
      strcpy(ErrStr, EOF_STR);
      return false;
    }
    if (i & 0x80)		/* Start of duplicate-pixel block? */
    {
      TGADupPixelCount = i & 0x7F; /* number of dups after this one */
      TGABlockCount = 0;	/* then read new block header */
    }
    else
      TGABlockCount = i & 0x7F; /* number of pixels after this one */
  }

  /* Read next pixel */
  i = getc(fpin);
  TGAPixel[0] = i;
  if (i == EOF)
  {
    strcpy(ErrStr, EOF_STR);
    return false;
  }
  else
    return true;
}

bool TBaseDib::ReadTGAPixel16()
{
  int i;

  /* Duplicate previously read pixel? */
  if (TGADupPixelCount > 0)
  {
    TGADupPixelCount--;
    return true;
  }

  /* Time to read RLE block header? */
  if (--TGABlockCount < 0) /* decrement pixels remaining in block */
  {
    if ((i = (unsigned char) getc(fpin)) == EOF)
    {
      strcpy(ErrStr, EOF_STR);
      return false;
    }
    if (i & 0x80)		/* Start of duplicate-pixel block? */
    {
      TGADupPixelCount = i & 0x7F; /* number of dups after this one */
      TGABlockCount = 0;	/* then read new block header */
    }
    else
      TGABlockCount = i & 0x7F; /* number of pixels after this one */
  }

  /* Read next pixel */
  if (fread(&TGAPixel[0], 1, 2, fpin) != 2)
  {
    strcpy(ErrStr, EOF_STR);
    return false;
  }
  else
    return true;
}

bool TBaseDib::ReadTGAPixel24()
{
  int i;

  /* Duplicate previously read pixel? */
  if (TGADupPixelCount > 0)
  {
    TGADupPixelCount--;
    return true;
  }

  /* Time to read RLE block header? */
  if (--TGABlockCount < 0) /* decrement pixels remaining in block */
  {
    if ((i = (unsigned char) getc(fpin)) == EOF)
    {
      strcpy(ErrStr, EOF_STR);
      return false;
    }
    if (i & 0x80)		/* Start of duplicate-pixel block? */
    {
      TGADupPixelCount = i & 0x7F; /* number of dups after this one */
      TGABlockCount = 0;	/* then read new block header */
    }
    else
      TGABlockCount = i & 0x7F; /* number of pixels after this one */
  }

  /* Read next pixel */
  if (fread(&TGAPixel[0], 1, 3, fpin) != 3)
  {
    strcpy(ErrStr, EOF_STR);
    return false;
  }
  else
    return true;
}

bool TBaseDib::ReadTGAPixel32()
{
  int i;

  /* Duplicate previously read pixel? */
  if (TGADupPixelCount > 0)
  {
    TGADupPixelCount--;
    return true;
  }

  /* Time to read RLE block header? */
  if (--TGABlockCount < 0) /* decrement pixels remaining in block */
  {
    if ((i = (unsigned char) getc(fpin)) == EOF)
    {
      strcpy(ErrStr, EOF_STR);
      return false;
    }
    if (i & 0x80)		/* Start of duplicate-pixel block? */
    {
      TGADupPixelCount = i & 0x7F; /* number of dups after this one */
      TGABlockCount = 0;	/* then read new block header */
    }
    else
      TGABlockCount = i & 0x7F; /* number of pixels after this one */
  }

  /* Read next pixel */
  if (fread(&TGAPixel[0], 1, 4, fpin) != 4)
  {
    strcpy(ErrStr, EOF_STR);
    return false;
  }
  else
    return true;
}

static const unsigned char c5to8bits[32] = {
	 0,   8,  16,  25,  33,  41,  49,  58,
   66,  74,  82,  90,  99, 107, 115, 123,
  132, 140, 148, 156, 165, 173, 181, 189,
  197, 206, 214, 222, 230, 239, 247, 255
};

/* Read a TGA file header */

bool TBaseDib::ReadTGAHeader()
{
  int i, j, imageType;

  if (fread(&tgaHeader, 1, sizeof(tgaHeader), fpin) != sizeof(tgaHeader))
  {
    strcpy(ErrStr, EOF_STR);
    return false;
  }

  biClrUsed = tgaHeader.colorMapLength;
  picWidth = tgaHeader.imageWidth;
  picHeight = tgaHeader.imageHeight;
  BitsPerPixel = tgaHeader.pixelDepth;
  picBitsPerPixel = BitsPerPixel;
  IsBottomUp = ((tgaHeader.imageDescriptor & 0x20) == 0);	/* bit 5 set => top-down */
  if (tgaHeader.colorMapType > 1 ||		/* color map type must be 0 or 1 */
      BitsPerPixel < 8 || BitsPerPixel > 32)
  {
    strcpy(ErrStr, JERR_TGA_BADPARMS);
    return false;
  }

  imageType = tgaHeader.imageType;
  if (imageType > 8)
  {
    /* It's an RLE-coded file */
	 RLECompressed = true;
	 TGABlockCount = TGADupPixelCount = 0;
	 imageType -= 8;
  }
  else
    RLECompressed = false;

  /* Now should have image type 1, 2, or 3 */
  switch (imageType)
  {
    case 1:			/* color mapped image */
	   if (BitsPerPixel != 8 || tgaHeader.colorMapType != 1)
      {
        strcpy(ErrStr, JERR_TGA_BADPARMS);
        return false;
      }
	   break;
    case 2:			/* RGB image */
	   switch (BitsPerPixel)
      {
        case 15:
	     case 16:
	     case 24:
        case 32:
          break;
        default:
          strcpy(ErrStr, JERR_TGA_BADPARMS);
          return false;
	   }
	   break;
    case 3:			/* Grayscale image */
      if (BitsPerPixel != 8)
      {
        strcpy(ErrStr, JERR_TGA_BADPARMS);
        return false;
      }
		for (i = 0; i < 256; i++)
		{
		  ColorMap[i].rgbRed = (unsigned char) i;
		  ColorMap[i].rgbGreen = (unsigned char) i;
		  ColorMap[i].rgbBlue = (unsigned char) i;
		}
	   break;
    default:
      strcpy(ErrStr, JERR_TGA_BADPARMS);
      return false;
  }
  if (tgaHeader.idLength)
  {
    if (fread(TGAIDField, 1, tgaHeader.idLength, fpin) != tgaHeader.idLength)
    {
      strcpy(ErrStr, EOF_STR);
      return false;
    }
    TGAIDField[tgaHeader.idLength] = 0;
  }

  if (imageType == 1)
  {
	 if (biClrUsed > 0)
    {
		if (biClrUsed > 256 || tgaHeader.firstEntryIdx != 0)
      {
        strcpy(ErrStr, JERR_TGA_BADCMAP);
        return false;
      }
	   switch (tgaHeader.colorMapEntrySize)
      {
        case 15:
        case 16:
          for (i = 0; i < biClrUsed; i++)
          {
            if (fread(&j, 1, 2, fpin) != 2)
            {
              strcpy(ErrStr, EOF_STR);
              return false;
            }
            ColorMap[i].rgbBlue = c5to8bits[j & 0x001F];
            ColorMap[i].rgbGreen = c5to8bits[(j & 0x03E0) >> 5];
            ColorMap[i].rgbRed = c5to8bits[(j & 0x7C00) >> 10];
          }
          break;
        case 24:
          for (i = 0; i < biClrUsed; i++)
          {
            if (fread(&ColorMap[i], 1, sizeof(RGBTRIPLE), fpin) != sizeof(RGBTRIPLE))
            {
              strcpy(ErrStr, EOF_STR);
              return false;
            }
          }
          break;
        case 32:
          if (fread(ColorMap, 1, sizeof(RGBQUAD) * biClrUsed, fpin) != sizeof(RGBQUAD) * biClrUsed)
          {
            strcpy(ErrStr, EOF_STR);
            return false;
          }
          break;
        default:
          strcpy(ErrStr, JERR_TGA_BADCMAP);
          return false;
      }
	 }
    else
    {
		if (tgaHeader.colorMapType)		/* but you promised a color map! */
      {
        strcpy(ErrStr, JERR_TGA_BADPARMS);
        return false;
      }
	 }
  }
  else
  {
	 if (biClrUsed > 0)
    {
      int lngth;
    	switch (tgaHeader.colorMapEntrySize)
	   {
		  case 15:
		  case 16:
          lngth = 2;
			 break;
		  case 24:
          lngth = 3;
			 break;
		  case 32:
          lngth = 4;
			 break;
		  default:
			 lngth = 1;
	   }
		if (fseek(fpin, biClrUsed * lngth, SEEK_CUR))
      {
        strcpy(ErrStr, EOF_STR);
        return false;
      }
    }
  }
  if (imageType == 3)
    biClrUsed = 256;
  return true;
}

bool TBaseDib::ReadBMPHeader()
{
  unsigned char bmpfileheader[14];

  /* Read the bitmap file header */
  if (fread(&bmpfileheader, 1, sizeof(bmpfileheader), fpin) != sizeof(bmpfileheader))
  {
	 strcpy(ErrStr, EOF_STR);
	 return false;
  }

  /* The infoheader might be 12 bytes (OS/2 1.x), 40 bytes (Windows),
	* or 64 bytes (OS/2 2.x).  Check the first 4 bytes to find out which.
	*/
  if (fread(&bmpinfoheader, 1, 4, fpin) != 4)
  {
	 strcpy(ErrStr, EOF_STR);
	 return false;
  }

  BMPHeaderSize = GET_4B(bmpinfoheader,0);
  if (BMPHeaderSize < 12 || BMPHeaderSize > 64)
  {
	 strcpy(ErrStr, INV_HDR);
	 return false;
  }

  if (fread(&bmpinfoheader[4], 1, BMPHeaderSize - 4, fpin) != BMPHeaderSize - 4)
  {
	 strcpy(ErrStr, EOF_STR);
	 return false;
  }

  switch (BMPHeaderSize)
  {
	 case 12:
		/* Decode OS/2 1.x header (Microsoft calls this a BITMAPCOREHEADER) */
		picWidth = GET_2B(bmpinfoheader, 4);
		picHeight = GET_2B(bmpinfoheader, 6);
      if (picHeight < 0)
      {
        IsBottomUp = false;
        picHeight = abs(picHeight);
      }
      else
        IsBottomUp = true;
		BMPPlanes = GET_2B(bmpinfoheader, 8);
		BitsPerPixel = GET_2B(bmpinfoheader, 10);
      RLECompressed = false;
      biClrUsed = 0;
		break;
	 case 40:
	 case 64:
		/* Decode Windows 3.x header or OS/2 2.x header */
		picWidth = GET_4B(bmpinfoheader,4);
		picHeight = GET_4B(bmpinfoheader,8);
      if (picHeight < 0)
      {
        IsBottomUp = false;
        picHeight = abs(picHeight);
      }
      else
        IsBottomUp = true;
		BMPPlanes = GET_2B(bmpinfoheader,12);
		BitsPerPixel = GET_2B(bmpinfoheader,14);
      biCompression = GET_4B(bmpinfoheader, 16);
		RLECompressed = ((biCompression == BI_RLE4) | (biCompression == BI_RLE8));
      if (RLECompressed)
      {
        if ((BitsPerPixel != 4) && (BitsPerPixel != 8) || (!IsBottomUp))
        {
		    strcpy(ErrStr, INV_HDR);
		    return false;
        }
      }
		biClrUsed = GET_4B(bmpinfoheader,32);
		break;
	 default:
		strcpy(ErrStr, INV_HDR);
		return false;
  }

  switch (BitsPerPixel)
  {
	 case 1:
	 case 4:
	 case 8:
      if (!biClrUsed)
        switch (BitsPerPixel)
        {
          case 1:
            biClrUsed = 2;
            break;
          case 4:
            biClrUsed = 16;
            break;
          case 8:
            biClrUsed = 256;
            break;
        }
      break;
    case 16:
    case 24:
    case 32:
      break;
    default:
      strcpy(ErrStr, INV_CDEP);
      return false;
  }

  if (biClrUsed > 256)
    biClrUsed = 256;
  picBitsPerPixel = BitsPerPixel;
  if (BMPHeaderSize == 12)
	 mapentrysize = 3;		/* OS/2 uses RGBTRIPLE colormap */
  else
	 mapentrysize = 4;		/* Windows uses RGBQUAD colormap */
  if (BMPPlanes != 1)
  {
	 strcpy(ErrStr, INV_BPLAN);
	 return false;
  }
  return true;
}

bool TBaseDib::ReadPCXByte()
{
  if (count > 1)
  {
    count--;
    return true;
  }
  if (row_data_idx >= ROW_BUF_SIZE)
  {
    row_data_idx = 0;
    if (fread(&row_data, 1, ROW_BUF_SIZE, fpin) == 0)
      return false;
  }
  value = row_data[row_data_idx];
  row_data_idx++;

  /* Check for a repeat count, indicated by the two high bits being set */
  if ((value & 0x00C0) == 0x00C0)
  {
    count = 0x3F & value;
    if (row_data_idx >= ROW_BUF_SIZE)
    {
      row_data_idx = 0;
      if (fread(&row_data, 1, ROW_BUF_SIZE, fpin) == 0)
        return false;
    }
    value = row_data[row_data_idx];
    row_data_idx++;
  }
  else
    count = 1;
  return true;
}

/*
**	Bit masks for planar decoding
*/
unsigned char	LowMasks[4] = { 0x01, 0x02, 0x04, 0x08 };
unsigned char HighMasks[4] = { 0x10, 0x20, 0x40, 0x80 };

RGBQUAD DefaultPalette[16] = {
	{   0,   0,   0, 0 },
	{   0,   0, 255, 0 },
	{   0, 255,   0, 0 },
	{   0, 255, 255, 0 },
	{ 255,   0,   0, 0 },
	{ 255,   0, 255, 0 },
	{ 255, 255,   0, 0 },
	{ 255, 255, 255, 0 },
	{  85,  85, 255, 0 },
	{  85,  85,  85, 0 },
	{   0, 170,   0, 0 },
	{ 170,   0,   0, 0 },
	{  85, 255, 255, 0 },
	{ 255,  85, 255, 0 },
	{ 255, 255,  85, 0 },
	{ 255, 255, 255, 0 }
};

//
// Read the header from a PCX file
//

bool TBaseDib::ReadPCXHeader()
{
  /*	Read in the header information, and verify it */
  if (fread(&pcxHeader, 1, sizeof(pcxHeader), fpin) != sizeof(pcxHeader))
  {
	 strcpy(ErrStr, EOF_STR);
	 return false;
  }

  if (pcxHeader.manufacturer != 10)
  {
	 strcpy(ErrStr, INV_HDR);
	 return false;
  }

  picWidth = 1 + pcxHeader.maxX - pcxHeader.minX;
  picHeight = 1 + pcxHeader.maxY - pcxHeader.minY;
  BitsPerPixel = pcxHeader.bitsPerPixel * pcxHeader.planes;
  picBitsPerPixel = BitsPerPixel;

  switch (BitsPerPixel)
  {
	 case 1:
      biClrUsed = 2;
      break;
	 case 4:
      biClrUsed = 16;
      break;
	 case 8:
      biClrUsed = 256;
      break;
	 case 24:
      biClrUsed = 0;
		break;
	 default:
		strcpy(ErrStr, INV_CDEP);
		return false;
  }
  return true;
}

bool TBaseDib::LoadBMPFile()
{
  return false;
}

bool TBaseDib::LoadPCXFile()
{
  return false;
}

bool TBaseDib::LoadGIFFile()
{
  return false;
}

bool TBaseDib::LoadTGAFile()
{
  return false;
}

bool TBaseDib::LoadJPEGFile()
{
  return false;
}

bool TBaseDib::CreateDib2(int w, int h, int bitsPerPixel)
{
  BITMAPINFOHEADER InfoHeader;
  int  colorAlloc;

  W2 = w;
  H2 = h;
  BitsPerPixel2 = bitsPerPixel;
  switch (BitsPerPixel2)
  {
	 case 1:
		NumClrs2 = 2;
		break;
	 case 4:
		NumClrs2 = 16;
		break;
	 case 8:
		NumClrs2 = 256;
		break;
    case 15:
    case 16:
	 case 24:
    case 32:
		NumClrs2 = 0;
		break;
	 default:
		strcpy(ErrStr, INV_CDEP);
		return false;
  }
  InfoHeader.biSize = sizeof(BITMAPINFOHEADER);
  InfoHeader.biWidth = W2;
  InfoHeader.biHeight = H2;
  InfoHeader.biBitCount = (WORD) BitsPerPixel2;
  InfoHeader.biPlanes = 1;
  InfoHeader.biXPelsPerMeter = 0;
  InfoHeader.biYPelsPerMeter = 0;
  InfoHeader.biClrUsed = 0;
  InfoHeader.biClrImportant = 0;  // 0 = all colors
  InfoHeader.biCompression = BI_RGB;
  InfoHeader.biSizeImage = ScanBytes(W2, InfoHeader.biBitCount) * H2;

  colorAlloc = NumClrs2 * sizeof(RGBQUAD); // size of color tables
  BitsAllocated2 = (long) InfoHeader.biSize + (long) colorAlloc + (long) InfoHeader.biSizeImage;

  Handle2 = GlobalAlloc(GMEM_MOVEABLE | GMEM_ZEROINIT, BitsAllocated2);
  if (Handle2)
  {
	 Info2 = (LPBITMAPINFO)::GlobalLock(Handle2);
	 if (Info2)
	 {
		Info2->bmiHeader = InfoHeader;
		switch (InfoHeader.biBitCount)
		{
		  case 1 :
			 xSizeNoPad2 = W2 >> 3;
			 if (W2 % 8 != 0)
				xSizeNoPad2++;
			 break;
		  case 4 :
			 xSizeNoPad2 = W2 >> 1;
			 if (W2 % 2 != 0)
				xSizeNoPad2++;
			 break;
		  case 8 :
			 xSizeNoPad2 = W2;
			 break;
        case 15:
        case 16:
          xSizeNoPad2 = W2 << 1;
          break;
		  case 24:
			 xSizeNoPad2 = W2 * 3;
			 break;
        case 32:
          xSizeNoPad2 = W2 << 2;
          break;
		}
		xSize2 = xSizeNoPad2;
		while ((xSize2 & 3) != 0)
		  xSize2++;
		Bits2 = (char far *) Info2 + ((int)InfoHeader.biSize + colorAlloc);
		return true;
	 }
  }
  strcpy(ErrStr, NO_MEM);
  return false;
}

void TBaseDib::SwitchDibs()
{
  ::GlobalUnlock(Handle);
  ::GlobalFree(Handle);
  W = W2;
  H = H2;
  Handle = Handle2;
  Info = Info2;
  BitsPerPixel = BitsPerPixel2;
  NumClrs = NumClrs2;
  xSize = xSize2;
  xSizeNoPad = xSizeNoPad2;
  Bits = Bits2;
  BitsAllocated = BitsAllocated2;
  mapentrysize = 4;
  if (NumClrs)
    Colors = (TRgbQuad *) &Info->bmiColors[0];
  else
    Colors = NULL;
}

static int oneBitMasks[8] = {0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x01};
static int oneBitShift[8] = {7, 6, 5, 4, 3, 2, 1, 0};

bool TBaseDib::ChangeDepth(int bitsPerPixel)
{
  bool retcode;

  if ((bitsPerPixel != 24) && (bitsPerPixel != 32))
    return false;
  if (BitsPerPixel == bitsPerPixel)
    return true;
  if (!CreateDib2(W, H, bitsPerPixel))
    return false;
  if (bitsPerPixel == 24)
    retcode = To24Bits();
  else
    retcode = To32Bits();
  if (!retcode)
  {
    if (Handle2)
    {
      ::GlobalUnlock(Handle2);
      ::GlobalFree(Handle2);
    }
  }
  else
    SwitchDibs();
  return retcode;
}

bool TBaseDib::To24Bits()
{
  unsigned char *srcbits, *srcbase;
  unsigned char *destbits, *destbase;
  RGBQUAD *colorMap;
  int i, j, k;

  srcbits = (unsigned char *) GetBits();
  destbits = (unsigned char *) Bits2;
  if (NumClrs)
    colorMap = (RGBQUAD *) (&Info->bmiColors[0]);

  i = 0;
  while ((i < H) && (!AbortOperation))
  {
    srcbase = srcbits + (i * xSize);
    destbase = destbits + (i * xSize2);
    switch (BitsPerPixel)
    {
      case 1:
        for (j = 0; j < W; j++)
        {
          k = j % 8;
          k = (srcbase[0] & oneBitMasks[k]) >> oneBitShift[k];
          if (j % 8 == 7)
            srcbase++;
          destbase[0] = colorMap[k].rgbBlue;
          destbase[1] = colorMap[k].rgbGreen;
          destbase[2] = colorMap[k].rgbRed;
          destbase += 3;
        }
        break;

      case 4:
        for (j = 0; j < W; j++)
        {
          if (j % 2)
          {
            k = srcbase[0] & 0x0F;
            srcbase++;
          }
          else
            k = (srcbase[0] & 0xF0) >> 4;
          destbase[0] = colorMap[k].rgbBlue;
          destbase[1] = colorMap[k].rgbGreen;
          destbase[2] = colorMap[k].rgbRed;
          destbase += 3;
        }
        break;

      case 8:
        for (j = 0; j < W; j++)
        {
          k = srcbase[0];
          destbase[0] = colorMap[k].rgbBlue;
          destbase[1] = colorMap[k].rgbGreen;
          destbase[2] = colorMap[k].rgbRed;
          srcbase++;
          destbase += 3;
        }
        break;

      case 16:
        for (j = 0; j < W; j++)
        {
          k = srcbase[0] | (srcbase[1] << 8);
          destbase[0] = (k & 0x001F) << 3;
          destbase[1] = (k & 0x03E0) >> 2;
          destbase[2] = (k & 0x7C00) >> 7;
          srcbase += 2;
          destbase += 3;
        }
        break;

      case 32:
        for (j = 0; j < W; j++)
        {
          destbase[0] = srcbase[0];
          destbase[1] = srcbase[1];
          destbase[2] = srcbase[2];
          srcbase += 4;
          destbase += 3;
        }
    }
    YieldTime();
    i++;
  }
  return !AbortOperation;
}

bool TBaseDib::To32Bits()
{
  unsigned char *srcbits, *srcbase;
  unsigned char *destbits, *destbase;
  RGBQUAD *colorMap;
  int i, j, k;

  srcbits = (unsigned char *) GetBits();
  destbits = (unsigned char *) Bits2;
  if (NumClrs)
    colorMap = (RGBQUAD *) (&Info->bmiColors[0]);

  i = 0;
  while ((i < H) && (!AbortOperation))
  {
    srcbase = srcbits + (i * xSize);
    destbase = destbits + (i * xSize2);
    switch (BitsPerPixel)
    {
      case 1:
        for (j = 0; j < W; j++)
        {
          k = j % 8;
          k = (srcbase[0] & oneBitMasks[k]) >> oneBitShift[k];
          if (j % 8 == 7)
            srcbase++;
          destbase[0] = colorMap[k].rgbBlue;
          destbase[1] = colorMap[k].rgbGreen;
          destbase[2] = colorMap[k].rgbRed;
          destbase += 4;
        }
        break;

      case 4:
        for (j = 0; j < W; j++)
        {
          if (j % 2)
          {
            k = srcbase[0] & 0x0F;
            srcbase++;
          }
          else
            k = (srcbase[0] & 0xF0) >> 4;
          destbase[0] = colorMap[k].rgbBlue;
          destbase[1] = colorMap[k].rgbGreen;
          destbase[2] = colorMap[k].rgbRed;
          destbase += 4;
        }
        break;

      case 8:
        for (j = 0; j < W; j++)
        {
          k = srcbase[0];
          destbase[0] = colorMap[k].rgbBlue;
          destbase[1] = colorMap[k].rgbGreen;
          destbase[2] = colorMap[k].rgbRed;
          srcbase++;
          destbase += 4;
        }
        break;

      case 16:
        for (j = 0; j < W; j++)
        {
          k = srcbase[0] | (srcbase[1] << 8);
          destbase[0] = (k & 0x001F) << 3;
          destbase[1] = (k & 0x03E0) >> 2;
          destbase[2] = (k & 0x7C00) >> 7;
          srcbase += 2;
          destbase += 4;
        }
        break;

      case 24:
        for (j = 0; j < W; j++)
        {
          destbase[0] = srcbase[0];
          destbase[1] = srcbase[1];
          destbase[2] = srcbase[2];
          srcbase += 3;
          destbase += 4;
        }
    }
    YieldTime();
    i++;
  }
  return !AbortOperation;
}

/*
void TBaseDib::ScaleLine(unsigned char *sp, unsigned char *dp, int sw, int dw)
{
  int i, x0, y01, y02, y03, y11, y12, y13, v1, v2, v3;
  long count, total1, total2, total3;

  count = total1 = total2 = total3 = 0;
  x0 = 0;
  if (sw == dw)
    memcpy(dp, sp, sw * 3);
  else
    if (dw > sw)
    {
      y01 = (int)(sp[0]);
      y02 = (int)(sp[1]);
      y03 = (int)(sp[2]);
      y11 = (int)(sp[3]);
      y12 = (int)(sp[4]);
      y13 = (int)(sp[5]);
      for (i = 0; i < dw; i++)
      {
        *dp++ = (unsigned char)(y01 + (int)((count * (long)(y11 - y01)) / (long)dw));
        *dp++ = (unsigned char)(y02 + (int)((count * (long)(y12 - y02)) / (long)dw));
        *dp++ = (unsigned char)(y03 + (int)((count * (long)(y13 - y03)) / (long)dw));
        count += sw;
        if (count >= dw)
        {
          count -= dw;
          x0+=3;
          y01 = y11;
          y02 = y12;
          y03 = y13;
          if (x0 < (sw-1) * 3)
          {
            y11 = sp[x0+3];
            y12 = sp[x0+4];
            y13 = sp[x0+5];
          }
        }
      }
    }
    else
    {
      for (i = 0; i < sw; i++)
      {
        count += dw;
        if (count >= sw)
        {
          count -= sw;
          v1 = (int)(*sp++);
          v2 = (int)(*sp++);
          v3 = (int)(*sp++);
          total1 += ((long)v1 * ((long)dw - count)) / (long)dw;
          total2 += ((long)v2 * ((long)dw - count)) / (long)dw;
          total3 += ((long)v3 * ((long)dw - count)) / (long)dw;
          *dp++ = (unsigned char)((total1 * (long)dw) / (long)sw);
          *dp++ = (unsigned char)((total2 * (long)dw) / (long)sw);
          *dp++ = (unsigned char)((total3 * (long)dw) / (long)sw);
          total1 = ((long)v1 * count) / (long)dw;
          total2 = ((long)v2 * count) / (long)dw;
          total3 = ((long)v3 * count) / (long)dw;
        }
        else
        {
          total1 += *sp++;
          total2 += *sp++;
          total3 += *sp++;
        }
      }
    }
}

bool TBaseDib::Rescale(int w, int h)
{
  unsigned char *srcbits, *destbits, *srcbase, *destbase;
  int i, j;
  long *tots, count;
  unsigned char *vp, *vp2;
  int lines, bitsPerPixel;

  if (BitsPerPixel != 24)
    if (!ChangeDepth(24))
      return false;

  if (!CreateDib2(w, h, 24))
    return false;

  srcbits = (unsigned char *) GetBits();
  destbits = (unsigned char *) Bits2;
  vp = NULL;
  vp2 = NULL;

  count = 0;
  if (H == H2)
  {
    lines = 0;
    while ((lines < H) && (!AbortOperation))
    {
      srcbase = srcbits + (lines * xSize);
      destbase = destbits + (lines * xSize2);
      ScaleLine(srcbase, destbase, W, W2);
      YieldTime();
      lines++;
    }
  }
  else
    if (H < H2)
    {
      vp = (unsigned char *) malloc(xSize2);
      vp2 = (unsigned char *)malloc(xSize2);
      ScaleLine(srcbits, vp, W, W2);
      srcbase = srcbits + xSize;
      ScaleLine(srcbase, vp2, W, W2);

      j = 2;
      lines = 0;
      while ((lines < H2) && (!AbortOperation))
      {
        destbase = destbits + (lines * xSize2);
        for (i = 0; i < W2 * 3; i += 3)
        {
          *destbase++ = (unsigned char)((int)(vp[i]) +
            (int)((count * ((long)(vp2[i]) - (long)(vp[i]))) / (long)H2));
          *destbase++ = (unsigned char)((int)(vp[i + 1]) +
            (int)((count * ((long)(vp2[i + 1]) - (long)(vp[i + 1]))) / (long)H2));
          *destbase++ = (unsigned char)((int)(vp[i + 2]) +
            (int)((count * ((long)(vp2[i + 2]) - (long)(vp[i + 2]))) / (long)H2));
        }
        if ((count += H) >= H2)
        {
          count -= H2;
          destbase = vp;
          vp = vp2;
          vp2 = destbase;
          if (j < H)
          {
            srcbase = srcbits + (j++ * xSize);
            ScaleLine(srcbase, vp2, W, W2);
          }
          else
            memcpy(vp2, vp, xSize2);
        }
        YieldTime();
        lines++;
      }
    }
    else
    {
      tots = (long *)malloc(xSize2 * sizeof(long));
      memset((char *)tots, 0, xSize2 * sizeof(long));
      vp = (unsigned char *)malloc(xSize2);
      j = 0;
      lines = 0;
      while ((lines < H) && (!AbortOperation))
      {
        srcbase = srcbits + (lines * xSize);
        ScaleLine(srcbase, vp, W, W2);
        count += H2;
        if (count >= H)
        {
          count -= H;
          destbase = destbits + (j * xSize2);
          for (i = 0; i < W2 * 3; i += 3)
          {
            tots[i] += ((long)(vp[i]) * ((long)H2 - count)) / (long)H2;
            tots[i + 1] += ((long)(vp[i + 1]) * ((long)H2 - count)) / (long)H2;
            tots[i + 2] += ((long)(vp[i + 2]) * ((long)H2 - count)) / (long)H2;
            *destbase++ = (unsigned char)((tots[i] * (long)H2) / (long)H);
            *destbase++ = (unsigned char)((tots[i + 1] * (long)H2) / (long)H);
            *destbase++ = (unsigned char)((tots[i + 2] * (long)H2) / (long)H);
            tots[i] = ((long)(vp[i]) * count) / (long)H2;
            tots[i + 1] = ((long)(vp[i + 1]) * count) / (long)H2;
            tots[i + 2] = ((long)(vp[i + 2]) * count) / (long)H2;
          }
          j++;
        }
        else
        {
          for (i = 0; i < W2 * 3; i += 3)
          {
            tots[i] += vp[i];
            tots[i + 1] += vp[i + 1];
            tots[i + 2] += vp[i + 2];
          }
        }
        YieldTime();
        lines++;
      }
    }
  if (vp)
    free(vp);
  if (vp2)
    free(vp2);

  if (!AbortOperation)
  {
    SwitchDibs();
    return true;
  }
  else
    return false;
}

*/


void TBaseDib::ScaleLine(unsigned char *sp, unsigned char *dp, int sw, int dw)
{
  int i, x0, y01, y02, y03, y11, y12, y13, v1, v2, v3;
  long count, total1, total2, total3;

  count = total1 = total2 = total3 = 0;
  x0 = 0;
  if (sw == dw)
    memcpy(dp, sp, sw * 4);
  else
    if (dw > sw)
    {
      y01 = (int)(sp[0]);
      y02 = (int)(sp[1]);
      y03 = (int)(sp[2]);
      y11 = (int)(sp[4]);
      y12 = (int)(sp[5]);
      y13 = (int)(sp[6]);
      for (i = 0; i < dw; i++)
      {
        *dp++ = (unsigned char)(y01 + (int)((count * (long)(y11 - y01)) / (long)dw));
        *dp++ = (unsigned char)(y02 + (int)((count * (long)(y12 - y02)) / (long)dw));
        *dp++ = (unsigned char)(y03 + (int)((count * (long)(y13 - y03)) / (long)dw));
        *dp++ = 0;
        count += sw;
        if (count >= dw)
        {
          count -= dw;
          x0+=4;
          y01 = y11;
          y02 = y12;
          y03 = y13;
          if (x0 < (sw-1) * 4)
          {
            y11 = sp[x0+4];
            y12 = sp[x0+5];
            y13 = sp[x0+6];
          }
        }
      }
    }
    else
    {
      for (i = 0; i < sw; i++)
      {
        count += dw;
        if (count >= sw)
        {
          count -= sw;
          v1 = (int)(*sp++);
          v2 = (int)(*sp++);
          v3 = (int)(*sp++);
          sp++;
          total1 += ((long)v1 * ((long)dw - count)) / (long)dw;
          total2 += ((long)v2 * ((long)dw - count)) / (long)dw;
          total3 += ((long)v3 * ((long)dw - count)) / (long)dw;
          *dp++ = (unsigned char)((total1 * (long)dw) / (long)sw);
          *dp++ = (unsigned char)((total2 * (long)dw) / (long)sw);
          *dp++ = (unsigned char)((total3 * (long)dw) / (long)sw);
          *dp++ = 0;
          total1 = ((long)v1 * count) / (long)dw;
          total2 = ((long)v2 * count) / (long)dw;
          total3 = ((long)v3 * count) / (long)dw;
        }
        else
        {
          total1 += *sp++;
          total2 += *sp++;
          total3 += *sp++;
          sp++;
        }
      }
    }
}

bool TBaseDib::Rescale(int w, int h, int realW, int realH)
{
  unsigned char *srcbits, *destbits, *srcbase, *destbase;
  int i, j;
  long *tots, count;
  unsigned char *vp, *vp2;
  int lines;

  if (BitsPerPixel != 32)
    if (!ChangeDepth(32))
      return false;

  if (!CreateDib2(w, h, 32))
    return false;

  srcbits = (unsigned char *) GetBits();
  destbits = (unsigned char *) Bits2;
  vp = NULL;
  vp2 = NULL;

  count = 0;
  if (H == H2)
  {
    lines = 0;
    while ((lines < H) && (!AbortOperation))
    {
      srcbase = srcbits + (lines * xSize);
      destbase = destbits + (lines * xSize2);
      ScaleLine(srcbase, destbase, W, W2);
      YieldTime();
      lines++;
    }
  }
  else
    if (H < H2)
    {
      vp = (unsigned char *) malloc(xSize2);
      vp2 = (unsigned char *)malloc(xSize2);
      ScaleLine(srcbits, vp, W, W2);
      srcbase = srcbits + xSize;
      ScaleLine(srcbase, vp2, W, W2);

      j = 2;
      lines = 0;
      while ((lines < H2) && (!AbortOperation))
      {
        destbase = destbits + (lines * xSize2);
        for (i = 0; i < W2 * 4; i += 4)
        {
          *destbase++ = (unsigned char)((int)(vp[i]) +
            (int)((count * ((long)(vp2[i]) - (long)(vp[i]))) / (long)H2));
          *destbase++ = (unsigned char)((int)(vp[i + 1]) +
            (int)((count * ((long)(vp2[i + 1]) - (long)(vp[i + 1]))) / (long)H2));
          *destbase++ = (unsigned char)((int)(vp[i + 2]) +
            (int)((count * ((long)(vp2[i + 2]) - (long)(vp[i + 2]))) / (long)H2));
          *destbase++ = 0;
        }
        if ((count += H) >= H2)
        {
          count -= H2;
          destbase = vp;
          vp = vp2;
          vp2 = destbase;
          if (j < H)
          {
            srcbase = srcbits + (j++ * xSize);
            ScaleLine(srcbase, vp2, W, W2);
          }
          else
            memcpy(vp2, vp, xSize2);
        }
        YieldTime();
        lines++;
      }
    }
    else
    {
      tots = (long *)malloc(xSize2 * sizeof(long));
      memset((char *)tots, 0, xSize2 * sizeof(long));
      vp = (unsigned char *)malloc(xSize2);
      j = 0;
      lines = 0;
      while ((lines < H) && (!AbortOperation))
      {
        srcbase = srcbits + (lines * xSize);
        ScaleLine(srcbase, vp, W, W2);
        count += H2;
        if (count >= H)
        {
          count -= H;
          destbase = destbits + (j * xSize2);
          for (i = 0; i < W2 * 4; i += 4)
          {
            tots[i] += ((long)(vp[i]) * ((long)H2 - count)) / (long)H2;
            tots[i + 1] += ((long)(vp[i + 1]) * ((long)H2 - count)) / (long)H2;
            tots[i + 2] += ((long)(vp[i + 2]) * ((long)H2 - count)) / (long)H2;
            *destbase++ = (unsigned char)((tots[i] * (long)H2) / (long)H);
            *destbase++ = (unsigned char)((tots[i + 1] * (long)H2) / (long)H);
            *destbase++ = (unsigned char)((tots[i + 2] * (long)H2) / (long)H);
            *destbase++ = 0;
            tots[i] = ((long)(vp[i]) * count) / (long)H2;
            tots[i + 1] = ((long)(vp[i + 1]) * count) / (long)H2;
            tots[i + 2] = ((long)(vp[i + 2]) * count) / (long)H2;
          }
          j++;
        }
        else
        {
          for (i = 0; i < W2 * 4; i += 4)
          {
            tots[i] += vp[i];
            tots[i + 1] += vp[i + 1];
            tots[i + 2] += vp[i + 2];
          }
        }
        YieldTime();
        lines++;
      }
    }
  if (vp)
    free(vp);
  if (vp2)
    free(vp2);

  if (!AbortOperation)
  {
    SwitchDibs();
    if (W != realW || H != realH)
      return PadDib(realW, realH);
    else
      return true;
  }
  else
    return false;
}

bool TBaseDib::PadDib(int w, int h)
{
  unsigned char *srcbits, *destbits, *prow, *dest;
  int i, j, k, x, y, x1, y1;

  if (!CreateDib2(w, h, 32))
    return false;

  srcbits = (unsigned char *) GetBits();
  destbits = (unsigned char *) Bits2;

  x = (w - W) >> 1;
  x1 = x + W - 1;
  y = (h - H) >> 1;
  y1 = y + H - 1;

  k = 0;
  for (i = 0; i < h; i++)
  {
    dest = destbits + (i * xSize2);
    if (i < y || i > y1)
      memset(dest, 0, xSize2);
    else
    {
      prow = srcbits + (k * xSize);
      for (j = 0; j < w; j++)
      {
        if (j < x || j > x1)
          dest[0] = dest[1] = dest[2] = 0;
        else
        {
          dest[0] = prow[0];
          dest[1] = prow[1];
          dest[2] = prow[2];
          prow += 4;
        }
        dest += 4;
      }
      k++;
    }
  }
  SwitchDibs();
  return true;
}

bool TBaseDib::Crop(int x, int y, int w, int h)
{
  unsigned char *srcbits, *destbits, *prow, *dest;
  int i, j, k, w1, h1, x1, x2, y1, y2, dX;

  if (!CreateDib2(w, h, 32))
    return false;

  srcbits = (unsigned char *) GetBits();
  destbits = (unsigned char *) Bits2;

  w1 = W - x;
  h1 = H - y;
  if (w1 < w)
  {
    x1 = (w - w1) >> 1;
    x2 = x1 + w1 - 1;
  }
  else
  {
    x1 = 0;
    x2 = w - 1;
  }
  if (h1 < h)
  {
    y1 = (h - h1) >> 1;
    y2 = y1 + h1 - 1;
  }
  else
  {
    y1 = 0;
    y2 = h - 1;
  }

  k = H - y - 1;
  dX = x << 2;
  for (i = h - 1; i >= 0; i--)
  {
    dest = destbits + (i * xSize2);
    if (i < y1 || i > y2)
      memset(dest, 0, xSize2);
    else
    {
      prow = srcbits + (k * xSize + dX);
      for (j = 0; j < w; j++)
      {
        if (j < x1 || j > x2)
          dest[0] = dest[1] = dest[2] = 0;
        else
        {
          dest[0] = prow[0];
          dest[1] = prow[1];
          dest[2] = prow[2];
          prow += 4;
        }
        dest += 4;
      }
      k--;
    }
  }
  SwitchDibs();
  return true;
}

void TBaseDib::Deinterlace()
{
  unsigned char *bits, *prow, *upper, *lower;
  int i, j;

  bits = (unsigned char *) GetBits();

  for (i = 0; i < H; i++)
  {
    prow = bits + (i * xSize);
    /* Only do interpolation if the row:
       (1) Isn't one we want to keep
       (2) Has both an upper and a lower row
       Otherwise, just duplicate the source row
    */
    if (!((i % 2 == 1) || (i - 1 < 0) || (i + 1 >= H)))
    {
      upper = prow - xSize;
      lower = prow + xSize;
      for (j = 0; j < xSize; j++)
        prow[j] = (upper[j] + lower[j]) >> 1;
    }
  }
}

bool TBaseDib::Invert()
{
  unsigned char *srcbits, *destbits;
  int i, j;

  if (!CreateDib2(W, H, BitsPerPixel))
    return false;

  srcbits = (unsigned char *) GetBits();
  destbits = (unsigned char *) Bits2;

  j = H - 1;
  for (i = 0; i < H; i++)
    memcpy(destbits + (i * xSize2), srcbits + (j-- * xSize), xSize);
  SwitchDibs();
  return true;
}

bool TBaseDib::CreateBlankDib(int w, int h)
{
  W = w;
  H = h;
  BitsPerPixel = 32;
  return CreateDib(false);
}

void TBaseDib::CopyBaseVars(TBaseDib *dib)
{
  strcpy(ErrStr, dib->ErrStr);
  picWidth = dib->picWidth;
  picHeight = dib->picHeight;
  EndScan = dib->EndScan;
  picBitsPerPixel = dib->picBitsPerPixel;
  strcpy(fName, dib->fName);
  fType = dib->fType;
  memcpy(bmpinfoheader, dib->bmpinfoheader, 64);
  biClrUsed = dib->biClrUsed;
  GBLClrUsed = dib->GBLClrUsed;
  LCLClrUsed = dib->LCLClrUsed;
  biCompression = dib->biCompression;
  BMPPlanes = dib->BMPPlanes;
  BMPHeaderSize = dib->BMPHeaderSize;
  mapentrysize = dib->mapentrysize;
  redMask = dib->redMask;
  greenMask = dib->greenMask;
  blueMask = dib->blueMask;
  pcxHeader = dib->pcxHeader;
  tgaHeader = dib->tgaHeader;
  GIFHeader = dib->GIFHeader;
  GIFImgDescriptor = dib->GIFImgDescriptor;
  memcpy(ColorMap, dib->ColorMap, sizeof(RGBQUAD) * 256);
  memcpy(GBLColorMap, dib->GBLColorMap, sizeof(RGBQUAD) * 256);
  GBLColorTable = dib->GBLColorTable;
  LCLColorTable = dib->LCLColorTable;
  BackGroundClr = dib->BackGroundClr;
  GIFCodeSize = dib->GIFCodeSize;
  GIFLimitCode = dib->GIFLimitCode;
  GIFMaxCode = dib->GIFMaxCode;
  GIFSp = dib->GIFSp;
  GIFLastByte = dib->GIFLastByte;
  GIFLastBit = dib->GIFLastBit;
  GIFCurBit = dib->GIFCurBit;
  GIFOutOfBlocks = dib->GIFOutOfBlocks;
  GIFClearCode = dib->GIFClearCode;
  GIFEndCode = dib->GIFEndCode;
  GIFFirstTime = dib->GIFFirstTime;
  IsInterlaced = dib->IsInterlaced;
  GIFInputCodeSize = dib->GIFInputCodeSize;
  AnimatedGIF = dib->AnimatedGIF;
  GIFDelayTime = dib->GIFDelayTime;
  TransparentGIF = dib->TransparentGIF;
  TransparentIdx = dib->TransparentIdx;
  GIFUserInputFlag = dib->GIFUserInputFlag;
  GIFDisposal = dib->GIFDisposal;
  GIFFrameNum = dib->GIFFrameNum;
  GIFTotalFrames = dib->GIFTotalFrames;
  logWidth = dib->logWidth;
  logHeight = dib->logHeight;
  picLeft = dib->picLeft;
  picTop = dib->picTop;
  GIFPass2Offset = dib->GIFPass2Offset;
  GIFPass3Offset = dib->GIFPass3Offset;
  GIFPass4Offset = dib->GIFPass4Offset;
  GIFOldCode = dib->GIFOldCode;
  GIFFirstCode = dib->GIFFirstCode;
  CurGIFPos = dib->CurGIFPos;
  IsBottomUp = dib->IsBottomUp;
  memcpy(TGAIDField, dib->TGAIDField, 256);
  TGABlockCount = dib->TGABlockCount;
  TGADupPixelCount = dib->TGADupPixelCount;
  RLECompressed = dib->RLECompressed;
  JPGColorSpace = dib->JPGColorSpace;
}

bool TBaseDib::CopyFromDib(TBaseDib *dib)
{
  W = dib->W;
  H = dib->H;
  BitsPerPixel = dib->BitsPerPixel;
  CopyBaseVars(dib);

  if (!CreateDib(false))
    return false;

  if (NumClrs)
    memcpy(Info->bmiColors, dib->GetInfo()->bmiColors, sizeof(RGBQUAD) * NumClrs);

  memcpy(GetBits(), dib->GetBits(), Info->bmiHeader.biSizeImage);
  return true;
}
