//----------------------------------------------------------------------------
//   Implementation of TFullDib class
//----------------------------------------------------------------------------
#include "avi2mpg2.h"
#include "jpeglib.h"

extern bool AbortDisplay;
extern bool ShowAnimGIF;
extern char EOF_STR[];
extern char INV_HDR[];
extern char INV_CDEP[];
extern char INV_BPLAN[];
extern char NO_MEM[];
extern char NO_COMPB[];
extern unsigned char	LowMasks[];
extern unsigned char HighMasks[];
extern RGBQUAD DefaultPalette[];

static unsigned char *GIFSaveBuffer;
static TFullDib *thisDib;

struct dib_error_mgr {
  struct jpeg_error_mgr pub;	/* "public" fields */
  jmp_buf setjmp_buffer;	/* for return to caller */
};

typedef struct dib_error_mgr *dib_error_ptr;

/*
 * Here's the routine that will replace the standard error_exit method:
 */

METHODDEF void dib_error_exit (j_common_ptr cinfo)
{
  dib_error_ptr diberr;

  /* cinfo->err really points to a tview_error_mgr struct, so coerce pointer */
  diberr = (dib_error_ptr) cinfo->err;

  /* Return control to the setjmp point */
  longjmp(diberr->setjmp_buffer, 1);
}


METHODDEF void dib_output_message (j_common_ptr cinfo)

{
  char buffer[JMSG_LENGTH_MAX];

  /* Create the message */
  (*cinfo->err->format_message) (cinfo, buffer);

  /* Send it to stderr, adding a newline */
  strcpy(thisDib->ErrStr, buffer);
}


//
// size of scan in bytes =
//   Pixel Width * bits per pixel rounded up to a DWORD boundary
//
inline long ScanBytes(int pixWidth, int bitsPixel) {
  return (((long)pixWidth*bitsPixel+31) / 32) * 4;
}


TFullDib::TFullDib(TWindow* parent)
            :TBaseDib()
{
  Parent = parent;
  thisDib = this;
  GIFSaveBuffer = NULL;
}

TFullDib::~TFullDib()
{
  if (GIFSaveBuffer)
  {
    delete[] GIFSaveBuffer;
    GIFSaveBuffer = NULL;
  }
}

//
// Read a BMP file into a Windows DIB
//

bool TFullDib::LoadBMPFile()
{
  int i, j, col, endAdder;
  int r, g, b;
  unsigned char huge *pcntr;
  void huge *pdib;
  unsigned char huge *pbase;
  DWORD redShift, greenShift, blueShift;
  int redScaleShift, greenScaleShift, blueScaleShift;

  if (!ReadBMPHeader())

    return false;


  if (RLECompressed)

    return LoadBMPRLEFile();


  W = picWidth;

  H = picHeight;

  if (!CreateDib(false))

    return false;


  pdib = GetBits();
  pbase = (unsigned char huge *)pdib;

  if (BitsPerPixel == 16)
  {
    if (biCompression == BI_BITFIELDS)
    {
      if (fread(&redMask, 1, sizeof(redMask), fpin) != sizeof(redMask))
      {
		  strcpy(ErrStr, EOF_STR);
        return false;
      }
      redShift = 0;
      while (((redMask >> redShift) & 0x0001) == 0)
        redShift++;
      i = redShift;
      redScaleShift = 0;
      while ((i < 16) && ((redMask >> i) & 0x0001))
      {
        i++;
        redScaleShift++;
      }
      redScaleShift -= 5;
      if (fread(&greenMask, 1, sizeof(greenMask), fpin) != sizeof(greenMask))
      {
		  strcpy(ErrStr, EOF_STR);
        return false;
      }
      greenShift = 0;
      while (((greenMask >> greenShift) & 0x0001) == 0)
        greenShift++;
      i = greenShift;
      greenScaleShift = 0;
      while ((i < 16) && ((greenMask >> i) & 0x0001))
      {
        i++;
        greenScaleShift++;
      }
      greenScaleShift -= 5;
      if (fread(&blueMask, 1, sizeof(blueMask), fpin) != sizeof(blueMask))
      {
		  strcpy(ErrStr, EOF_STR);
        return false;
      }
      blueShift = 0;
      while (((blueMask >> blueShift) & 0x0001) == 0)
        blueShift++;
      i = blueShift;
      blueScaleShift = 0;
      while ((i < 16) && ((blueMask >> i) & 0x0001))
      {
        i++;
        blueScaleShift++;
      }
      blueScaleShift -= 5;
    }
    else
    {
      redMask = 0x7C00;
      redShift = 10;
      redScaleShift = 0;
      greenMask = 0x03E0;
      greenShift = 5;
      greenScaleShift = 0;
      blueMask = 0x001F;
      blueShift = 0;
      blueScaleShift = 0;
    }
  }

  if (BitsPerPixel == 32)
  {
    if (biCompression == BI_BITFIELDS)
    {
      if (fread(&redMask, 1, sizeof(redMask), fpin) != sizeof(redMask))
      {
		  strcpy(ErrStr, EOF_STR);
        return false;
      }
      redShift = 0;
      while (((redMask >> redShift) & 0x0001) == 0)
        redShift++;
      i = redShift;
      redScaleShift = 0;
      while ((i < 32) && ((redMask >> i) & 0x0001))
      {
        i++;
        redScaleShift++;
      }
      redScaleShift -= 8;
      if (fread(&greenMask, 1, sizeof(greenMask), fpin) != sizeof(greenMask))
      {
		  strcpy(ErrStr, EOF_STR);
        return false;
      }
      greenShift = 0;
      while (((greenMask >> greenShift) & 0x0001) == 0)
        greenShift++;
      i = greenShift;
      greenScaleShift = 0;
      while ((i < 32) && ((greenMask >> i) & 0x0001))
      {
        i++;
        greenScaleShift++;
      }
      greenScaleShift -= 8;
      if (fread(&blueMask, 1, sizeof(blueMask), fpin) != sizeof(blueMask))
      {
		  strcpy(ErrStr, EOF_STR);
        return false;
      }
      blueShift = 0;
      while (((blueMask >> blueShift) & 0x0001) == 0)
        blueShift++;
      i = blueShift;
      blueScaleShift = 0;
      while ((i < 32) && ((blueMask >> i) & 0x0001))
      {
        i++;
        blueScaleShift++;
      }
      blueScaleShift -= 8;
    }
    else
    {
      redMask = 0x00FF0000;
      redShift = 16;
      redScaleShift = 0;
      greenMask = 0x0000FF00;
      greenShift = 8;
      greenScaleShift = 0;
      blueMask = 0x000000FF;
      blueShift = 0;
      blueScaleShift = 0;
    }
  }
  /* Read the colormap, if any */
  if (biClrUsed)
  {
	 switch (mapentrysize)
	 {
		case 3:
		  /* BGR format (occurs in OS/2 files) */
		  for (i = 0; i < biClrUsed; i++)
		  {
          if ((col = (unsigned char) getc(fpin)) == EOF)
          {
            strcpy(ErrStr, EOF_STR);
            return false;
          }
			 Info->bmiColors[i].rgbBlue = col;
          if ((col = (unsigned char) getc(fpin)) == EOF)
          {
            strcpy(ErrStr, EOF_STR);
            return false;
          }
			 Info->bmiColors[i].rgbGreen = col;
          if ((col = (unsigned char) getc(fpin)) == EOF)
          {
            strcpy(ErrStr, EOF_STR);
            return false;
          }
			 Info->bmiColors[i].rgbRed = col;
		  }
		  break;
		case 4:
		  /* BGR0 format (occurs in MS Windows files) */
		  if (fread(Info->bmiColors, 1, sizeof(RGBQUAD) * biClrUsed, fpin) != sizeof(RGBQUAD) * biClrUsed)
		  {
			 strcpy(ErrStr, EOF_STR);
			 return false;
		  }
		  break;
    }
  }

  if (IsBottomUp)
  {
    endAdder = 1;
    EndScan = 0;
  }
  else
  {
    endAdder = -1;
    EndScan = H - 1;
  }

  j = 0;
  /* Process data */
  switch (BitsPerPixel)
  {
    case 1:
    case 4:
    case 8:
      while ((j < H) && (!AbortDisplay))
      {
	     pcntr = pbase + (EndScan * xSize);
        if (fread(pcntr, xSize, 1, fpin) != 1)
        {
		    strcpy(ErrStr, EOF_STR);
          if (IsBottomUp)
            return EndScan != 0;
          else
            return EndScan != H - 1;
        }
		  YieldTime();
        j++;
	     EndScan += endAdder;
      }
      break;
    case 16:
      while ((j < H) && (!AbortDisplay))
      {
	     pcntr = pbase + (EndScan * xSize);
        if (fread(pcntr, xSize, 1, fpin) != 1)
        {
		    strcpy(ErrStr, EOF_STR);
          if (IsBottomUp)
            return EndScan != 0;
          else
            return EndScan != H - 1;
        }

		  for (col = 0; col < xSizeNoPad; col += 2)
        {
          i = pcntr[col] | (pcntr[col + 1] << 8);
		    b = (i & blueMask) >> blueShift;
		    g = (i & greenMask) >> greenShift;
		    r = (i & redMask) >> redShift;
          if (redScaleShift != 0)
          {
            if (redScaleShift > 0)
              r = r >> redScaleShift;
            else
              r = r << abs(redScaleShift);
          }
          if (greenScaleShift != 0)
          {
            if (greenScaleShift > 0)
              g = g >> greenScaleShift;
            else
              g = g << abs(greenScaleShift);
          }
          if (blueScaleShift != 0)
          {
            if (blueScaleShift > 0)
              b = b >> blueScaleShift;
            else
              b = b << abs(blueScaleShift);
          }
          i = (r << 10) | (g << 5) | b;
          pcntr[col] = i & 0x00FF;
          pcntr[col + 1] = i >> 8;
        }

		  YieldTime();
        j++;
	     EndScan += endAdder;
      }
      break;
    case 24:
      while ((j < H) && (!AbortDisplay))
      {
	     pcntr = pbase + (EndScan * xSize);
        if (fread(pcntr, xSize, 1, fpin) != 1)
        {
		    strcpy(ErrStr, EOF_STR);
          if (IsBottomUp)
            return EndScan != 0;
          else
            return EndScan != H - 1;
        }

		  YieldTime();
        j++;
	     EndScan += endAdder;
      }
      break;
    case 32:
      while ((j < H) && (!AbortDisplay))
      {
	     pcntr = pbase + (EndScan * xSize);
        if (fread(pcntr, xSize, 1, fpin) != 1)
        {
		    strcpy(ErrStr, EOF_STR);
          if (IsBottomUp)
            return EndScan != 0;
          else
            return EndScan != H - 1;
        }

		  for (col = 0; col < xSizeNoPad; col += 4)
        {
          i = pcntr[col] | (pcntr[col + 1] << 8) | (pcntr[col + 2] << 16) | (pcntr[col + 3] << 24);
		    b = (i & blueMask) >> blueShift;
		    g = (i & greenMask) >> greenShift;
		    r = (i & redMask) >> redShift;
          if (redScaleShift != 0)
          {
            if (redScaleShift > 0)
              r = r >> redScaleShift;
            else
              r = r << abs(redScaleShift);
          }
          if (greenScaleShift != 0)
          {
            if (greenScaleShift > 0)
              g = g >> greenScaleShift;
            else
              g = g << abs(greenScaleShift);
          }
          if (blueScaleShift != 0)
          {
            if (blueScaleShift > 0)
              b = b >> blueScaleShift;
            else
              b = b << abs(blueScaleShift);
          }
          pcntr[col] = (unsigned char) b;
          pcntr[col + 1] = (unsigned char) g;
		    pcntr[col + 2] = (unsigned char) r;
		    pcntr[col + 3] = 0;
        }

		  YieldTime();
        j++;
	     EndScan += endAdder;
      }
      break;
  }
  return true;
}

bool TFullDib::LoadBMPRLEFile()
{
  int i, j, col, byte1, byte2;
  unsigned char huge *pcntr;
  void huge *pdib;
  unsigned char huge *pbase;
  bool newLine;

  W = picWidth;

  H = picHeight;

  if (!CreateDib(false))

    return false;


  pdib = GetBits();
  pbase = (unsigned char huge *)pdib;

  /* Read the colormap, if any */
  if (biClrUsed)
  {
	 switch (mapentrysize)
	 {
		case 3:
		  /* BGR format (occurs in OS/2 files) */
		  for (i = 0; i < biClrUsed; i++)
		  {
          if ((col = (unsigned char) getc(fpin)) == EOF)
          {
            strcpy(ErrStr, EOF_STR);
            return false;
          }
			 Info->bmiColors[i].rgbBlue = col;
          if ((col = (unsigned char) getc(fpin)) == EOF)
          {
            strcpy(ErrStr, EOF_STR);
            return false;
          }
			 Info->bmiColors[i].rgbGreen = col;
          if ((col = (unsigned char) getc(fpin)) == EOF)
          {
            strcpy(ErrStr, EOF_STR);
            return false;
          }
			 Info->bmiColors[i].rgbRed = col;
		  }
		  break;
		case 4:
		  /* BGR0 format (occurs in MS Windows files) */
		  if (fread(Info->bmiColors, 1, sizeof(RGBQUAD) * biClrUsed, fpin) != sizeof(RGBQUAD) * biClrUsed)
		  {
			 strcpy(ErrStr, EOF_STR);
			 return false;
		  }
		  break;
    }
  }

  EndScan = 0;
  j = 0;
  /* Process data */
  switch (BitsPerPixel)
  {
    case 4:
      col = 0;
      newLine = false;
	   pcntr = pbase + (EndScan * xSize);
      while ((j < H) && (!AbortDisplay))
      {
        if ((byte1 = getc(fpin)) == EOF)
        {
		    strcpy(ErrStr, EOF_STR);
		    return EndScan != 0;
        }
        if ((byte2 = getc(fpin)) == EOF)
        {
		    strcpy(ErrStr, EOF_STR);
		    return EndScan != 0;
        }
        if (newLine)
        {
          if ((byte1) || (byte2))
          {
            col = 0;
            YieldTime();
            j++;
            EndScan++;
            pcntr = pbase + (EndScan * xSize);
          }
          newLine = false;
        }
        if (byte1)
        {
          for (i = 0; i < byte1; i++)
          {
            if (col % 2)
              pcntr[0] = (unsigned char) (pcntr[0] & 0xF0) | (byte2 & 0x0F);
            else
              pcntr[0] = (unsigned char) (pcntr[0] & 0x0F) | (byte2 & 0xF0);
            col++;
            if (col % 2 == 0)
              pcntr++;
            if (col >= picWidth)
            {
              if (i < byte1 - 1)
              {
                col = 0;
                YieldTime();
                j++;
                EndScan++;
                pcntr = pbase + (EndScan * xSize);
              }
              else
                newLine = true;
            }
          }
        }
        else
        {
          if (byte2 < 3)
          {
            switch (byte2)
            {
              case 0:
                col = 0;
                YieldTime();
                j++;
                EndScan++;
                pcntr = pbase + (EndScan * xSize);
                break;

              case 1:
                j = H;
                break;

              case 2:
                if ((byte1 = getc(fpin)) == EOF)
                {
                  strcpy(ErrStr, EOF_STR);
                  return EndScan != 0;
                }
                if ((byte2 = getc(fpin)) == EOF)
                {
                  strcpy(ErrStr, EOF_STR);
                  return EndScan != 0;
                }
                col += byte1;
                if (col >= picWidth)
                  col = picWidth - col;
                for (i = 0; i < byte2; i++)
                {
                  YieldTime();
                  j++;
                  EndScan++;
                }
                pcntr = pbase + (EndScan * xSize) + (col >> 1);
                break;
            }
          }
          else
          {
            for (i = 0; i < byte2; i++)
            {
              if (i % 2 == 0)
              {
                if ((byte1 = getc(fpin)) == EOF)
                {
                  strcpy(ErrStr, EOF_STR);
                  return EndScan != 0;
                }
              }
              if (col % 2)
                pcntr[0] = (unsigned char) (pcntr[0] & 0xF0) | (byte1 & 0x0F);
              else
                pcntr[0] = (unsigned char) (pcntr[0] & 0x0F) | (byte1 & 0xF0);
              col++;
              if (col % 2 == 0)
                pcntr++;
              if (col >= picWidth)
              {
                if (i < byte2 - 1)
                {
                  col = 0;
                  YieldTime();
                  j++;
                  EndScan++;
                  pcntr = pbase + (EndScan * xSize);
                }
                else
                  newLine = true;
              }
            }
            if (byte2 % 2)
              byte2++;
            if ((byte2 >> 1) % 2)
            {
              if ((byte1 = getc(fpin)) == EOF)
              {
                strcpy(ErrStr, EOF_STR);
                return EndScan != 0;
              }
            }
          }
        }
      }
      break;

    case 8:
      col = 0;
      newLine = false;
	   pcntr = pbase + (EndScan * xSize);
      while ((j < H) && (!AbortDisplay))
      {
        if ((byte1 = getc(fpin)) == EOF)
        {
		    strcpy(ErrStr, EOF_STR);
		    return EndScan != 0;
        }
        if ((byte2 = getc(fpin)) == EOF)
        {
		    strcpy(ErrStr, EOF_STR);
		    return EndScan != 0;
        }
        if (newLine)
        {
          if ((byte1) || (byte2))
          {
            col = 0;
            YieldTime();
            j++;
            EndScan++;
            pcntr = pbase + (EndScan * xSize);
          }
          newLine = false;
        }
        if (byte1)
        {
          for (i = 0; i < byte1; i++)
          {
            *pcntr++ = (unsigned char) byte2;
            col++;
            if (col >= picWidth)
            {
              if (i < byte1 - 1)
              {
                col = 0;
                YieldTime();
                j++;
                EndScan++;
                pcntr = pbase + (EndScan * xSize);
              }
              else
                newLine = true;
            }
          }
        }
        else
        {
          if (byte2 < 3)
          {
            switch (byte2)
            {
              case 0:
                col = 0;
                YieldTime();
                j++;
                EndScan++;
                pcntr = pbase + (EndScan * xSize);
                break;

              case 1:
                j = H;
                break;

              case 2:
                if ((byte1 = getc(fpin)) == EOF)
                {
                  strcpy(ErrStr, EOF_STR);
                  return EndScan != 0;
                }
                if ((byte2 = getc(fpin)) == EOF)
                {
                  strcpy(ErrStr, EOF_STR);
                  return EndScan != 0;
                }
                col += byte1;
                if (col >= picWidth)
                  col = picWidth - col;
                for (i = 0; i < byte2; i++)
                {
                  YieldTime();
                  j++;
                  EndScan++;
                }
                pcntr = pbase + (EndScan * xSize) + col;
                break;
            }
          }
          else
          {
            for (i = 0; i < byte2; i++)
            {
              if ((byte1 = getc(fpin)) == EOF)
              {
                strcpy(ErrStr, EOF_STR);
                return EndScan != 0;
              }
              *pcntr++ = (unsigned char) byte1;
              col++;
              if (col >= picWidth)
              {
                if (i < byte2 - 1)
                {
                  col = 0;
                  YieldTime();
                  j++;
                  EndScan++;
                }
                else
                  newLine = true;
              }
            }
            if (byte2 % 2)
            {
              if ((byte1 = getc(fpin)) == EOF)
              {
                strcpy(ErrStr, EOF_STR);
                return EndScan != 0;
              }
            }
          }
        }
      }
      break;
  }
  return true;
}

//
// Read a PCX file into a Windows DIB
//

bool TFullDib::LoadPCXFile()
{
  int	 i, k, col, maxrow, maxcol, plane;
  unsigned char bit, highbit, mask;
  int bytesPerScanLine;
  unsigned char huge *pcntr;
  void huge *pdib;
  unsigned char huge *pbase;

  if (!ReadPCXHeader())
    return false;

  W = picWidth;
  H = picHeight;
  if (!CreateDib(false))
    return false;

  switch (BitsPerPixel)
  {
    case 1:
    case 4:
      /* Not all PCX files have a valid palette */
      for (i = 0 ; i < NumClrs ; ++i)
        if (pcxHeader.pal1[i][RGB_RED] != 0 ||
            pcxHeader.pal1[i][RGB_GREEN] != 0 ||
            pcxHeader.pal1[i][RGB_BLUE] != 0)
          break;
      if (i == NumClrs)
        if (NumClrs == 16)
          /* Use default 16 color palette */
          for (i = 0; i < NumClrs; i++)
          {
            Info->bmiColors[i].rgbRed = DefaultPalette[i].rgbBlue;
            Info->bmiColors[i].rgbGreen = DefaultPalette[i].rgbGreen;
            Info->bmiColors[i].rgbBlue = DefaultPalette[i].rgbRed;
          }
        else
        {
          Info->bmiColors[0].rgbRed = DefaultPalette[0].rgbBlue;
          Info->bmiColors[0].rgbGreen = DefaultPalette[0].rgbGreen;
          Info->bmiColors[0].rgbBlue = DefaultPalette[0].rgbRed;
          Info->bmiColors[1].rgbRed = DefaultPalette[15].rgbBlue;
          Info->bmiColors[1].rgbGreen = DefaultPalette[15].rgbGreen;
          Info->bmiColors[1].rgbBlue = DefaultPalette[15].rgbRed;
        }
      else
        for (i = 0; i < NumClrs; i++)
        {
          Info->bmiColors[i].rgbRed = pcxHeader.pal1[i][RGB_BLUE];
          Info->bmiColors[i].rgbGreen = pcxHeader.pal1[i][RGB_GREEN];
          Info->bmiColors[i].rgbBlue = pcxHeader.pal1[i][RGB_RED];
        }
      break;
    case 8:
      if (fseek(fpin, -769L, SEEK_END))
      {
        strcpy(ErrStr, EOF_STR);
        return false;
      }
      i = getc(fpin);
      if (i == EOF)
      {
        strcpy(ErrStr, EOF_STR);
        return false;
      }
      value = i;
      if (value == 12)
        for (i = 0; i < NumClrs; i++)
        {
          if ((k = (unsigned char) getc(fpin)) == EOF)
          {
            strcpy(ErrStr, EOF_STR);
            return false;
          }
          Info->bmiColors[i].rgbRed = k;
          if ((k = (unsigned char) getc(fpin)) == EOF)
          {
            strcpy(ErrStr, EOF_STR);
            return false;
          }
          Info->bmiColors[i].rgbGreen = k;
          if ((k = (unsigned char) getc(fpin)) == EOF)
          {
            strcpy(ErrStr, EOF_STR);
            return false;
          }
          Info->bmiColors[i].rgbBlue = k;
        }
      else
      {
      }
      break;
  }

  pdib = GetBits();
  pbase = (unsigned char huge *)pdib;

  /* Process data */
  if (fseek(fpin, (long) sizeof(pcxHeader), SEEK_SET))
  {
    strcpy(ErrStr, EOF_STR);
    return false;
  }
  bytesPerScanLine = (long) pcxHeader.bytesPerLine * (long) pcxHeader.planes;
  col = 0;
  plane = 0;
  EndScan = 0;
  switch (BitsPerPixel)
  {
    case 1:
      maxcol = bytesPerScanLine * 8;
      break;
    case 4:
      maxcol = bytesPerScanLine * 2;
      break;
    case 24:
      maxcol = bytesPerScanLine / 3;
      if (pcxHeader.planes > 1)
        plane = 2;
  }
  maxrow = H - 1;
  row_data_idx = ROW_BUF_SIZE;
  count = 0;
  pcntr = pbase + (long) maxrow * xSize;

  switch (BitsPerPixel)
  {
    case 1:
      while ((EndScan <= maxrow) && (!AbortDisplay))
      {
        if (!ReadPCXByte())
          return EndScan != 0;
        pcntr[col >> 3] = value;
        col += 8;
        if (col >= maxcol)
        {
          col = 0;
          EndScan++;
          YieldTime();
          pcntr = pbase + (long) (maxrow - EndScan) * xSize;
        }
      }
      break;
    case 4:
      if (pcxHeader.planes > 1)
      {
        while ((EndScan <= maxrow) && (!AbortDisplay))
        {
          if (!ReadPCXByte())
            return EndScan != 0;
          bit = LowMasks[plane];
          highbit = HighMasks[plane];
          for (k = 0; k < 8; k += 2)
          {
            mask = (unsigned char) (0x80 >> k);
            if (value & mask)
              *pcntr |= highbit;
            mask >>= 1;
            if (value & mask)
              *pcntr |= bit;
            pcntr++;
          }
          col += 8;
          if (col >= maxcol)
          {
            col = 0;
            plane++;
            if (plane > 3)
            {
              EndScan++;
              YieldTime();
              plane = 0;
            }
            pcntr = pbase + (long) (maxrow - EndScan) * xSize;
          }
        }
      }
      else
      {
        while ((EndScan <= maxrow) && (!AbortDisplay))
        {
          if (!ReadPCXByte())
            return EndScan != 0;
          *pcntr++ = value;
          col += 2;
          if (col >= maxcol)
          {
            col = 0;
            EndScan++;
            YieldTime();
            pcntr = pbase + (long) (maxrow - EndScan) * xSize;
          }
        }
      }
      break;
    case 8:
      while ((EndScan <= maxrow) && (!AbortDisplay))
      {
        if (!ReadPCXByte())
          return EndScan != 0;
        *pcntr++ = value;
        col++;
        if (col >= bytesPerScanLine)
        {
          col = 0;
          EndScan++;
          YieldTime();
          pcntr = pbase + (long) (maxrow - EndScan) * xSize;
        }
      }
      break;
    case 24:
      if (pcxHeader.planes > 1)
      {
        while ((EndScan <= maxrow) && (!AbortDisplay))
        {
          if (!ReadPCXByte())
            return EndScan != 0;
          pcntr[plane] = value;
          pcntr += 3;
          col++;
          if (col >= maxcol)
          {
            col = 0;
            plane--;
            if (plane < 0)
            {
              EndScan++;
              YieldTime();
              plane = 2;
            }
            pcntr = pbase + (long) (maxrow - EndScan) * xSize;
          }
        }
      }
      else
      {
        while ((EndScan <= maxrow) && (!AbortDisplay))
        {
          if (!ReadPCXByte())
            return EndScan != 0;
          pcntr[plane] = value;
          plane++;
          if (plane > 3)
          {
            pcntr += 3;
            plane = 0;
            col++;
            if (col >= maxcol)
            {
              col = 0;
              EndScan++;
              YieldTime();
              plane = 2;
              pcntr = pbase + (long) (maxrow - EndScan) * xSize;
            }
          }
        }
      }
      break;
  }
  return true;
}


void TFullDib::DisposeGIF()
{
  unsigned char huge *pcntr;
  void huge *pdib;
  unsigned char huge *pbase;
  int i, j, k, l, startRow, stopRow;

  pdib = GetBits();
  pbase = (unsigned char huge *)pdib;
  startRow = logHeight - picHeight - picTop;
  stopRow = startRow + picHeight;
  switch (GIFDisposal)
  {
    case 0:
//      memset(pdib, 0, Info->bmiHeader.biSizeImage);
      break;
    case 1:
      break;
    case 2:
      switch (BitsPerPixel)
      {
        case 1:
          for (i = startRow; i < stopRow; i++)
          {
            pcntr = pbase + (i * xSize) + (picLeft / 8);
            for (j = picLeft; j < picWidth + picLeft; j++)
            {
              k = pcntr[0];
              l = 7 - (j % 8);
              k &= ~(1 << l);
              k |= (BackGroundClr << l);
              pcntr[0] = k;
              if (l == 0)
                pcntr++;
            }
          }
          break;
        case 4:
          for (i = startRow; i < stopRow; i++)
          {
            pcntr = pbase + (i * xSize) + (picLeft / 2);
            for (j = picLeft; j < picWidth + picLeft; j++)
            {
              k = pcntr[0];
              l = (1 - (j % 2)) << 2;
              k &= ~(0xF << l);
              k |= (BackGroundClr << l);
              pcntr[0] = k;
              if (l == 0)
                pcntr++;
            }
          }
          break;
        case 8:
          for (i = startRow; i < stopRow; i++)
          {
            pcntr = pbase + (i * xSize) + picLeft;
            memset(pcntr, BackGroundClr, picWidth);
          }
      }
      break;
    case 3:
      memcpy(pdib, GIFSaveBuffer, Info->bmiHeader.biSizeImage);
  }
}

//
// Read a GIF file into a Windows DIB
//

bool TFullDib::LoadGIFFile()
{
  int i, j;
  unsigned char huge *pcntr;
  void huge *pdib;
  unsigned char huge *pbase;

  if ((ShowAnimGIF) && (AnimatedGIF))
  {
    DisposeGIF();
    if (!ReadGIFHeader())
      return false;
  }
  else
  {
    FirstTimeGIF = true;
    if (!ReadGIFHeader())
      return false;

    W = logWidth;
    H = logHeight;

    if (!CreateDib(false))
      return false;

    if (GIFSaveBuffer)
    {
      delete[] GIFSaveBuffer;
      GIFSaveBuffer = NULL;
    }
  }

  for (i = 0; i < biClrUsed; i++)
  {
    Info->bmiColors[i].rgbBlue = ColorMap[i].rgbBlue;
	 Info->bmiColors[i].rgbGreen = ColorMap[i].rgbGreen;
	 Info->bmiColors[i].rgbRed = ColorMap[i].rgbRed;
  }

  pdib = GetBits();
  pbase = (unsigned char huge *)pdib;

  if (GBLColorTable)
  {
    switch (BitsPerPixel)
    {
      case 1:
        if (BackGroundClr)
        {
          for (i = 0; i < H; i++)
          {
            pcntr = pbase + (i * xSize);
            memset(pcntr, 0xFF, W / 8);
            if (W % 8)
            {
              pcntr += W / 8;
              *pcntr |= (0xFF << (8 - (W % 8)));
            }
          }
        }
        else
          memset(pbase, 0, Info->bmiHeader.biSizeImage);
        break;
      case 4:
        if (BackGroundClr)
        {
          j = (BackGroundClr << 4) | BackGroundClr;
          for (i = 0; i < H; i++)
          {
            pcntr = pbase + (i * xSize);
            memset(pcntr, j, W / 2);
            if (W % 2)
            {
              pcntr += W / 2;
              *pcntr = BackGroundClr << 4;
            }
          }
        }
        else
          memset(pbase, 0, Info->bmiHeader.biSizeImage);
        break;
      case 8:
        if (BackGroundClr)
        {
          memset(pbase, BackGroundClr, Info->bmiHeader.biSizeImage);
          if (W < xSize)
          {
            for (i = 0; i < H; i++)
            {
              pcntr = pbase + (i * xSize) + W;
              for (j = W; j < xSize; j++)
                *pcntr++ = 0;
            }
          }
        }
        else
          memset(pbase, 0, Info->bmiHeader.biSizeImage);
    }
  }

  if ((ShowAnimGIF) && (GIFDisposal == 3))
  {
    if (!GIFSaveBuffer)
      GIFSaveBuffer = new unsigned char[Info->bmiHeader.biSizeImage];
    memcpy(GIFSaveBuffer, pdib, Info->bmiHeader.biSizeImage);
  }

  if (!ReadGIFFile())
    return false;

  if (!AbortDisplay)
  {
    GIFFrameNum = 1;
    GIFTotalFrames = 1;
    if (fgetpos(fpin, &CurGIFPos))
      return true;

    if (ShowAnimGIF)
      AnimatedGIF = CheckGIFAnim();
  }
  return true;
}



bool TFullDib::LoadNextGIFFile()

{
  int i;

  if ((fpin = fopen(fName, "rb")) == NULL)
  {
    AnimatedGIF = false;
	 return false;
  }

  if (fsetpos(fpin, &CurGIFPos))
  {
    fclose(fpin);
    AnimatedGIF = false;
    return false;
  }

  DisposeGIF();

  if (GetGIFImageHeader())
  {
    for (i = 0; i < biClrUsed; i++)
    {
      Info->bmiColors[i].rgbBlue = ColorMap[i].rgbBlue;
      Info->bmiColors[i].rgbGreen = ColorMap[i].rgbGreen;
      Info->bmiColors[i].rgbRed = ColorMap[i].rgbRed;
    }

    if (GIFDisposal == 3)
    {
      if (!GIFSaveBuffer)
        GIFSaveBuffer = new unsigned char[Info->bmiHeader.biSizeImage];
      memcpy(GIFSaveBuffer, GetBits(), Info->bmiHeader.biSizeImage);
    }

    if (ReadGIFFile())
    {
      GIFFrameNum++;
      if (GIFFrameNum > GIFTotalFrames)
        GIFTotalFrames = GIFFrameNum;
      if (!fgetpos(fpin, &CurGIFPos))
      {
        fclose(fpin);
        return true;
      }
    }
  }
  if (!AbortDisplay)
  {
    fseek(fpin, 0, SEEK_SET);
    if (LoadGIFFile())
    {
      fclose(fpin);
      return true;
    }
  }
  AnimatedGIF = false;
  fclose(fpin);
  return false;
}


bool TFullDib::ReadGIFFile()
{
  int i, j, k, l, irow;
  unsigned char huge *pcntr;
  void huge *pdib;
  unsigned char huge *pbase;
  int rowminus1;

  /* Establish the setjmp return context for dib_error_exit to use. */
  if (setjmp(SetJmpBuffer))
	 return EndScan != 0;

  rowminus1 = logHeight - 1 - picTop;

  pdib = GetBits();
  pbase = (unsigned char huge *)pdib;
  /* Process data */
  EndScan = 0;
  if (TransparentGIF)
  {
    if (IsInterlaced)
    {
	   while ((EndScan < picHeight) && (!AbortDisplay))
	   {
		  if (EndScan < GIFPass2Offset)
		    irow = EndScan << 3;
		  else
		    if (EndScan < GIFPass3Offset)
			   irow = 4 + ((EndScan - GIFPass2Offset) << 3);
		    else
			   if (EndScan < GIFPass4Offset)
				  irow = 2 + ((EndScan - GIFPass3Offset) << 2);
			   else
				  irow = 1 + ((EndScan - GIFPass4Offset) << 1);
		  pcntr = (pbase + ((rowminus1 - irow) * xSize));
        switch (BitsPerPixel)
        {
          case 1:
            pcntr += (picLeft / 8);
            for (i = picLeft; i < picWidth + picLeft; i++)
            {
              k = LZWReadByte();
              if (k != TransparentIdx)
              {
                j = pcntr[0];
                l = 7 - (i % 8);
                j &= ~(1 << l);
                j |= (k << l);
                pcntr[0] = j;
                if (l == 0)
                  pcntr++;
              }
              else
                if (i % 8 == 7)
                  pcntr++;
            }
            break;
          case 4:
            pcntr += (picLeft / 2);
            for (i = picLeft; i < picWidth + picLeft; i++)
            {
              k = LZWReadByte();
              if (k != TransparentIdx)
              {
                j = pcntr[0];
                l = (1 - (i % 2)) << 2;
                j &= ~(0xF << l);
                j |= (k << l);
                pcntr[0] = j;
                if (l == 0)
                  pcntr++;
              }
              else
                if (i % 2)
                  pcntr++;
            }
            break;
          case 8:
            pcntr += picLeft;
            for (i = picLeft; i < picWidth + picLeft; i++)
            {
              k = LZWReadByte();
              if (k != TransparentIdx)
	             *pcntr = (unsigned char) k;
              pcntr++;
            }
        }
		  YieldTime();
		  EndScan++;
    	}
    }
    else
    {
	   while ((EndScan < picHeight) && (!AbortDisplay))
	   {
		  pcntr = (pbase + ((rowminus1 - EndScan) * xSize));
        switch (BitsPerPixel)
        {
          case 1:
            pcntr += (picLeft / 8);
            for (i = picLeft; i < picWidth + picLeft; i++)
            {
              k = LZWReadByte();
              if (k != TransparentIdx)
              {
                j = pcntr[0];
                l = 7 - (i % 8);
                j &= ~(1 << l);
                j |= (k << l);
                pcntr[0] = j;
                if (l == 0)
                  pcntr++;
              }
              else
                if (i % 8 == 7)
                  pcntr++;
            }
            break;
          case 4:
            pcntr += (picLeft / 2);
            for (i = picLeft; i < picWidth + picLeft; i++)
            {
              k = LZWReadByte();
              if (k != TransparentIdx)
              {
                j = pcntr[0];
                l = (1 - (i % 2)) << 2;
                j &= ~(0xF << l);
                j |= (k << l);
                pcntr[0] = j;
                if (l == 0)
                  pcntr++;
              }
              else
                if (i % 2)
                  pcntr++;
            }
            break;
          case 8:
            pcntr += picLeft;
            for (i = picLeft; i < picWidth + picLeft; i++)
            {
              k = LZWReadByte();
              if (k != TransparentIdx)
	             *pcntr = (unsigned char) k;
              pcntr++;
            }
        }
	     YieldTime();
	     EndScan++;
	   }
    }
  }
  else /* not transparent */
  {
    if (IsInterlaced)
    {
	   while ((EndScan < picHeight) && (!AbortDisplay))
	   {
		  if (EndScan < GIFPass2Offset)
		    irow = EndScan << 3;
		  else
		    if (EndScan < GIFPass3Offset)
			   irow = 4 + ((EndScan - GIFPass2Offset) << 3);
		    else
			   if (EndScan < GIFPass4Offset)
			     irow = 2 + ((EndScan - GIFPass3Offset) << 2);
			   else
		  		  irow = 1 + ((EndScan - GIFPass4Offset) << 1);
		  pcntr = (pbase + ((rowminus1 - irow) * xSize));
        switch (BitsPerPixel)
        {
          case 1:
            pcntr += (picLeft / 8);
            for (i = picLeft; i < picWidth + picLeft; i++)
            {
              k = LZWReadByte();
              j = pcntr[0];
              l = 7 - (i % 8);
              j &= ~(1 << l);
              j |= (k << l);
              pcntr[0] = j;
              if (l == 0)
                pcntr++;
            }
            break;
          case 4:
            pcntr += (picLeft / 2);
            for (i = picLeft; i < picWidth + picLeft; i++)
            {
              k = LZWReadByte();
              j = pcntr[0];
              l = (1 - (i % 2)) << 2;
              j &= ~(0xF << l);
              j |= (k << l);
              pcntr[0] = j;
              if (l == 0)
                pcntr++;
            }
            break;
          case 8:
            pcntr += picLeft;
            for (i = picLeft; i < picWidth + picLeft; i++)
	           *pcntr++ = (unsigned char) LZWReadByte();
        }
		  YieldTime();
		  EndScan++;
    	}
    }
    else
    {
	   while ((EndScan < picHeight) && (!AbortDisplay))
	   {
		  pcntr = (pbase + ((rowminus1 - EndScan) * xSize));
        switch (BitsPerPixel)
        {
          case 1:
            pcntr += (picLeft / 8);
            for (i = picLeft; i < picWidth + picLeft; i++)
            {
              k = LZWReadByte();
              j = pcntr[0];
              l = 7 - (i % 8);
              j &= ~(1 << l);
              j |= (k << l);
              pcntr[0] = j;
              if (l == 0)
                pcntr++;
            }
            break;
          case 4:
            pcntr += (picLeft / 2);
            for (i = picLeft; i < picWidth + picLeft; i++)
            {
              k = LZWReadByte();
              j = pcntr[0];
              l = (1 - (i % 2)) << 2;
              j &= ~(0xF << l);
              j |= (k << l);
              pcntr[0] = j;
              if (l == 0)
                pcntr++;
            }
            break;
          case 8:
            pcntr += picLeft;
            for (i = picLeft; i < picWidth + picLeft; i++)
	           *pcntr++ = (unsigned char) LZWReadByte();
        }
		  YieldTime();
		  EndScan++;
	   }
    }
  }
  FirstTimeGIF = false;
  return true;
}

//
// Read a TARGA file into a Windows DIB
//

bool TFullDib::LoadTGAFile()
{
  int row, picXSize;
  void huge *pdib;
  unsigned char huge *pbase;
  unsigned char huge *prow;

  if (!ReadTGAHeader())
    return false;


  if (RLECompressed)

    return LoadTGARLEFile();


  W = picWidth;

  H = picHeight;

  if (!CreateDib(false))

	 return false;

  pdib = GetBits();
  pbase = (unsigned char huge *)pdib;

  EndScan = 0;
  switch (BitsPerPixel)
  {
    case 8:
      memcpy(Info->bmiColors, ColorMap, sizeof(RGBQUAD) * biClrUsed);
      picXSize = picWidth;
      if (IsBottomUp)
      {
        row = 0;
	     while ((row < picHeight) && (!AbortDisplay))
	     {
          prow = pbase + (row * xSize);
          if (fread(prow, 1, picXSize, fpin) != picXSize)
          {
            strcpy(ErrStr, EOF_STR);
            return EndScan != 0;
          }
		    row++;
		    YieldTime();
		    EndScan++;
	     }
      }
      else
      {
        row = picHeight - 1;
	     while ((row >= 0) && (!AbortDisplay))
	     {
          prow = pbase + (row * xSize);
          if (fread(prow, 1, picXSize, fpin) != picXSize)
          {
            strcpy(ErrStr, EOF_STR);
            return EndScan != 0;
          }
		    row--;
		    YieldTime();
		    EndScan++;
	     }
      }
      break;
    case 15:
    case 16:
      picXSize = picWidth * 2;
      if (IsBottomUp)
      {
        row = 0;
	     while ((row < picHeight) && (!AbortDisplay))
	     {
          prow = pbase + (row * xSize);
          if (fread(prow, 1, picXSize, fpin) != picXSize)
          {
            strcpy(ErrStr, EOF_STR);
            return EndScan != 0;
          }
		    row++;
		    YieldTime();
		    EndScan++;
	     }
      }
      else
      {
        row = picHeight - 1;
	     while ((row >= 0) && (!AbortDisplay))
	     {
          prow = pbase + (row * xSize);
          if (fread(prow, 1, picXSize, fpin) != picXSize)
          {
            strcpy(ErrStr, EOF_STR);
            return EndScan != 0;
          }
		    row--;
		    YieldTime();
		    EndScan++;
	     }
      }
      break;
    case 24:
      picXSize = picWidth * 3;
      if (IsBottomUp)
      {
        row = 0;
	     while ((row < picHeight) && (!AbortDisplay))
	     {
          prow = pbase + (row * xSize);
          if (fread(prow, 1, picXSize, fpin) != picXSize)
          {
            strcpy(ErrStr, EOF_STR);
            return EndScan != 0;
          }
		    row++;
		    YieldTime();
		    EndScan++;
	     }
      }
      else
      {
        row = picHeight - 1;
	     while ((row >= 0) && (!AbortDisplay))
	     {
          prow = pbase + (row * xSize);
          if (fread(prow, 1, picXSize, fpin) != picXSize)
          {
            strcpy(ErrStr, EOF_STR);
            return EndScan != 0;
          }
		    row--;
		    YieldTime();
		    EndScan++;
	     }
      }
      break;
    case 32:
      picXSize = picWidth * 4;
      if (IsBottomUp)
      {
        row = 0;
	     while ((row < picHeight) && (!AbortDisplay))
	     {
          prow = pbase + (row * xSize);
          if (fread(prow, 1, picXSize, fpin) != picXSize)
          {
            strcpy(ErrStr, EOF_STR);
            return EndScan != 0;
          }
		    row++;
		    YieldTime();
		    EndScan++;
	     }
      }
      else
      {
        row = picHeight - 1;
	     while ((row >= 0) && (!AbortDisplay))
	     {
          prow = pbase + (row * xSize);
          if (fread(prow, 1, picXSize, fpin) != picXSize)
          {
            strcpy(ErrStr, EOF_STR);
            return EndScan != 0;
          }
		    row--;
		    YieldTime();
		    EndScan++;
	     }
      }
      break;
  }
  return true;
}

bool TFullDib::LoadTGARLEFile()
{
  int i, row, col;
  void huge *pdib;
  unsigned char huge *pbase;
  unsigned char huge *prow;

  W = picWidth;

  H = picHeight;

  if (!CreateDib(false))

	 return false;

  pdib = GetBits();
  pbase = (unsigned char huge *)pdib;

  EndScan = 0;
  switch (BitsPerPixel)
  {
    case 8:
      memcpy(Info->bmiColors, ColorMap, sizeof(RGBQUAD) * biClrUsed);
      if (IsBottomUp)
      {
        row = 0;
	     while ((row < picHeight) && (!AbortDisplay))
	     {
          prow = pbase + (row * xSize);
          col = 0;
          for (i = 0; i < picWidth; i++)
          {
            if (!ReadTGAPixel8())
            {
              strcpy(ErrStr, EOF_STR);
              return EndScan != 0;
            }
            prow[col++] = TGAPixel[0];
          }
		    row++;
		    YieldTime();
		    EndScan++;
	     }
      }
      else
      {
        row = picHeight - 1;
	     while ((row >= 0) && (!AbortDisplay))
	     {
          prow = pbase + (row * xSize);
          col = 0;
          for (i = 0; i < picWidth; i++)
          {
            if (!ReadTGAPixel8())
            {
              strcpy(ErrStr, EOF_STR);
              return EndScan != 0;
            }
            prow[col++] = TGAPixel[0];
          }
		    row--;
		    YieldTime();
		    EndScan++;
	     }
      }
      break;
    case 15:
    case 16:
      if (IsBottomUp)
      {
        row = 0;
	     while ((row < picHeight) && (!AbortDisplay))
	     {
          prow = pbase + (row * xSize);
          col = 0;
          for (i = 0; i < picWidth; i++)
          {
            if (!ReadTGAPixel16())
            {
              strcpy(ErrStr, EOF_STR);
              return EndScan != 0;
            }
            prow[col++] = TGAPixel[0];
            prow[col++] = TGAPixel[1];
          }
		    row++;
		    YieldTime();
		    EndScan++;
	     }
      }
      else
      {
        row = picHeight - 1;
	     while ((row >= 0) && (!AbortDisplay))
	     {
          prow = pbase + (row * xSize);
          col = 0;
          for (i = 0; i < picWidth; i++)
          {
            if (!ReadTGAPixel16())
            {
              strcpy(ErrStr, EOF_STR);
              return EndScan != 0;
            }
            prow[col++] = TGAPixel[0];
            prow[col++] = TGAPixel[1];
          }
		    row--;
		    YieldTime();
		    EndScan++;
	     }
      }
      break;
    case 24:
      if (IsBottomUp)
      {
        row = 0;
	     while ((row < picHeight) && (!AbortDisplay))
	     {
          prow = pbase + (row * xSize);
          col = 0;
          for (i = 0; i < picWidth; i++)
          {
            if (!ReadTGAPixel24())
            {
              strcpy(ErrStr, EOF_STR);
              return EndScan != 0;
            }
            prow[col++] = TGAPixel[0];
            prow[col++] = TGAPixel[1];
            prow[col++] = TGAPixel[2];
          }
		    row++;
		    YieldTime();
		    EndScan++;
	     }
      }
      else
      {
        row = picHeight - 1;
	     while ((row >= 0) && (!AbortDisplay))
	     {
          prow = pbase + (row * xSize);
          col = 0;
          for (i = 0; i < picWidth; i++)
          {
            if (!ReadTGAPixel24())
            {
              strcpy(ErrStr, EOF_STR);
              return EndScan != 0;
            }
            prow[col++] = TGAPixel[0];
            prow[col++] = TGAPixel[1];
            prow[col++] = TGAPixel[2];
          }
		    row--;
		    YieldTime();
		    EndScan++;
	     }
      }
      break;
    case 32:
      if (IsBottomUp)
      {
        row = 0;
	     while ((row < picHeight) && (!AbortDisplay))
	     {
          prow = pbase + (row * xSize);
          col = 0;
          for (i = 0; i < picWidth; i++)
          {
            if (!ReadTGAPixel32())
            {
              strcpy(ErrStr, EOF_STR);
              return EndScan != 0;
            }
            prow[col++] = TGAPixel[0];
            prow[col++] = TGAPixel[1];
            prow[col++] = TGAPixel[2];
            prow[col++] = TGAPixel[3];
          }
		    row++;
		    YieldTime();
		    EndScan++;
	     }
      }
      else
      {
        row = picHeight - 1;
	     while ((row >= 0) && (!AbortDisplay))
	     {
          prow = pbase + (row * xSize);
          col = 0;
          for (i = 0; i < picWidth; i++)
          {
            if (!ReadTGAPixel32())
            {
              strcpy(ErrStr, EOF_STR);
              return EndScan != 0;
            }
            prow[col++] = TGAPixel[0];
            prow[col++] = TGAPixel[1];
            prow[col++] = TGAPixel[2];
            prow[col++] = TGAPixel[3];
          }
		    row--;
		    YieldTime();
		    EndScan++;
	     }
      }
      break;
  }
  return true;
}

//
// Read a JPEG file into a Windows DIB
//

bool TFullDib::LoadJPEGFile()
{
  int i, j;
  struct jpeg_decompress_struct cinfo;
  struct dib_error_mgr jerr;
  JSAMPROW row_ptr;
  JSAMPARRAY buffer;
  void huge *pdib;
  unsigned char huge *pincr;
  int rowminus1;

  cinfo.err = jpeg_std_error(&jerr.pub);
  jerr.pub.error_exit = dib_error_exit;
  jerr.pub.output_message = dib_output_message;
  EndScan = 0;

  /* Establish the setjmp return context for dib_error_exit to use. */
  if (setjmp(jerr.setjmp_buffer))
  {
    if (!AbortDisplay)
	   /* Create the message */
	   cinfo.err->format_message((jpeg_common_struct *) &cinfo, ErrStr);
	 jpeg_destroy_decompress(&cinfo);
    if (AbortDisplay)
      return true;
	 return EndScan != 0;
  }

  jpeg_create_decompress(&cinfo);
  cinfo.preview_mode = false;
  jpeg_stdio_src(&cinfo, fpin);
  (void) jpeg_read_header(&cinfo, true);
  IsInterlaced = jpeg_has_multiple_scans(&cinfo);
  cinfo.quantize_colors = false;
  if ((cinfo.out_color_space != JCS_GRAYSCALE) && (cinfo.out_color_space != JCS_RGB))
    cinfo.out_color_space = JCS_RGB;
  JPGColorSpace = cinfo.jpeg_color_space;
  jpeg_start_decompress(&cinfo);

  W = cinfo.output_width;
  H = cinfo.output_height;
  if (cinfo.out_color_components == 1)
    BitsPerPixel = 8;
  else
    BitsPerPixel = 24;
  picBitsPerPixel = BitsPerPixel;
  if (!CreateDib(false))
  {
	 strcpy(ErrStr, NO_MEM);
	 jpeg_destroy_decompress(&cinfo);
	 return false;
  }

  rowminus1 = H - 1;
  buffer = &row_ptr;
  pdib = GetBits();
  pincr = (unsigned char huge *)pdib;

  EndScan = 0;
  j = 0;
  if (BitsPerPixel == 24)
  {
    while ((j < cinfo.output_height) && (!AbortDisplay))
    {
	   row_ptr = (JSAMPROW) (pincr + ((rowminus1 - j) * xSize));
	   (void) jpeg_read_scanlines(&cinfo, buffer, 1);
      YieldTime();
	   EndScan++;
      j++;
    }
  }
  else
  {
    for (i = 0; i < 256; i++)
    {
      Info->bmiColors[i].rgbRed = i;
      Info->bmiColors[i].rgbGreen = i;
      Info->bmiColors[i].rgbBlue = i;
    }
    while ((j < cinfo.output_height) && (!AbortDisplay))
    {
	   row_ptr = (JSAMPROW) (pincr + ((rowminus1 - j) * xSize));
	   (void) jpeg_read_scanlines(&cinfo, buffer, 1);
		YieldTime();
	   EndScan++;
      j++;
    }
  }
  jpeg_destroy_decompress(&cinfo);
  return true;
}


bool TFullDib::LoadFromBMP(BITMAPINFO *bitmapInfo)
{
  unsigned char *bits, *dest, *prow;
  int i, j, k, rowsize, row, endadder;
  RGBQUAD *colorTable;
  bool topDown = false;

  W = bitmapInfo->bmiHeader.biWidth;
  H = bitmapInfo->bmiHeader.biHeight;
  if (H < 0)
  {
    H = -H;
    topDown = true;
  }
  picBitsPerPixel = bitmapInfo->bmiHeader.biBitCount;
  BitsPerPixel = 32;

  if (!CreateDib(false))
  {
    strcpy(ErrStr, NO_MEM);
    return false;
  }

  if (topDown)
  {
    row = H - 1;
    endadder = -1;
  }
  else
  {
    row = 0;
    endadder = 1;
  }

  bits = (unsigned char *) bitmapInfo + bitmapInfo->bmiHeader.biSize;
  dest = (unsigned char *) GetBits();
  switch (picBitsPerPixel)
  {
    case 8:
      colorTable = (RGBQUAD *) bits;
      bits += sizeof(RGBQUAD) * bitmapInfo->bmiHeader.biClrUsed;

      rowsize = W;
      while (rowsize % 4)
        rowsize++;

      for (i = 0; i < H; i++)
      {
        prow = bits + (row * rowsize);
        for (j = 0; j < W; j++)
        {
          k = prow[j];
          *dest++ = colorTable[k].rgbBlue;
          *dest++ = colorTable[k].rgbGreen;
          *dest++ = colorTable[k].rgbRed;
          dest++;
        }
        row += endadder;
      }
      break;

    case 16:
      rowsize = W << 1;
      while (rowsize % 4)
        rowsize++;

      for (i = 0; i < H; i++)
      {
        prow = bits + (row * rowsize);
        for (j = 0; j < W; j++)
        {
          *dest++ = (prow[0] & 0x1F) << 3;
          *dest++ = ((prow[0] & 0xE0) >> 2) | ((prow[1] & 0x3) << 6);
          *dest++ = (prow[1] & 0x7C) << 1;
          dest++;
          prow += 2;
        }
        row += endadder;
      }
      break;

    case 24:
      rowsize = W * 3;
      while (rowsize % 4)
        rowsize++;

      for (i = 0; i < H; i++)
      {
        prow = bits + (row * rowsize);
        for (j = 0; j < W; j++)
        {
          *dest++ = *prow++;
          *dest++ = *prow++;
          *dest++ = *prow++;
          dest++;
        }
        row += endadder;
      }
      break;

    case 32:
      if (topDown)
      {
        rowsize = W << 2;
        for (i = 0; i < H; i++)
        {
          prow = bits + (row * rowsize);
          memcpy(dest, prow, rowsize);
          dest += rowsize;
          row += endadder;
        }
      }
      else
        memcpy(dest, bits, W * H * 4);
  }
  return true;
}

bool TFullDib::LoadFromDD(DDSURFACEDESC ddsd)
{
  unsigned char *bits, *dest, *prow;
  int i, j, rowsize;

  W = ddsd.dwWidth;
  H = ddsd.dwHeight;
  picBitsPerPixel = 32;
  BitsPerPixel = 32;

  if (!CreateDib(false))
  {
    strcpy(ErrStr, NO_MEM);
    return false;
  }

  rowsize = ddsd.lPitch;
  bits = (unsigned char *) ddsd.lpSurface;
  dest = (unsigned char *) GetBits();
  j = H - 1;
  for (i = 0; i < H; i++)
  {
    prow = bits + (j-- * rowsize);
    memcpy(dest, prow, rowsize);
    dest += rowsize;
  }
  return true;
}
