//----------------------------------------------------------------------------
//   apisamp - bbMPEG API win32 console sample app
//----------------------------------------------------------------------------

#include "windows.h"
#include "bbmpgapi.h"
#include <stdio.h>

HINSTANCE            dllPtr;          // dll handle
bbMPEGSetFocusEntry  bbMPEGSetFocus;  // bbMPEG setfocus function pointer
bbMPEGInitEntry      bbMPEGInit;      // bbMPEG init function pointer
bbMPEGMakeMPEGEntry  bbMPEGMakeMPEG;  // bbMPEG makempeg function pointer
bbMPEGShutdownEntry  bbMPEGShutdown;  // bbMPEG shutdown function pointer
char                *audioBuffer;     // local audio buffer
unsigned int         audioBufferSize; // local audio buffer size
char                *videoBuffer;     // local video buffer
unsigned int         width, height;   // output width and height

// test video frame, bbMPEG in big block letters
char logoBits[20][32] = {
  {1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1},
  {1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1},
  {1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1},
  {1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1},
  {1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1},
  {1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1},
  {1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1},
  {1,1,1,0,1,1,1,0,1,1,1,0,0,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,0,1,1,1},
  {1,1,1,0,1,1,1,0,1,1,1,0,1,0,1,0,1,0,1,0,1,0,1,1,1,0,1,1,1,1,1,1},
  {1,1,1,0,1,1,1,0,1,1,1,0,1,0,1,0,1,0,0,0,1,0,0,0,1,0,1,1,1,1,1,1},
  {1,1,1,0,0,0,1,0,0,0,1,0,1,0,1,0,1,0,1,1,1,0,1,1,1,0,1,0,0,1,1,1},
  {1,1,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,1,1,0,1,1,1,0,1,1,0,1,1,1},
  {1,1,1,0,0,0,1,0,0,0,1,0,1,0,1,0,1,0,1,1,1,0,0,0,1,0,0,0,0,1,1,1},
  {1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1},
  {1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1},
  {1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1},
  {1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1},
  {1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1},
  {1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1},
  {1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1}};

// example of the audio callback

int getAudio(unsigned int frameNum, unsigned int *numSamples, unsigned int *bufferSize, char **buffer)
{
  unsigned int i;
  short int *bufferData, j;

  memset(audioBuffer, 0, audioBufferSize); // zero the buffer

  // just generate a test tone
  j = -1024;
  bufferData = (short int*) audioBuffer;
  for (i = 1; i < (audioBufferSize >> 1); i++)
  {
    bufferData[i] = j;
    j += 4;
    if (j > 1024)
      j = -1024;
  }

  *numSamples = 1601;            // return the number of samples
  *bufferSize = audioBufferSize; // return the size in bytes of samples
  *buffer = audioBuffer;         // return a pointer to the audio data
  return bbErrNone;              // no errors
}

// example of the video callback

int getVideo(unsigned int frameNum, unsigned int *bytesInRow, char **buffer)
{
  unsigned int rowSize, i, j, k, l, m;
  char *rowPtr;

  rowSize = width << 2;  // row size is width * 4 (32 bpp)

  // return a series of test frames
  l = height - 1;  // start with the first row (bottom up bmp style)
  for (i = 0; i < 20; i++) // do 20 * 12 = 240 rows
  {
    rowPtr = videoBuffer + (l-- * rowSize); // pointer to current row
    m = 0; // current column
    for (j = 0; j < 32; j++)  // do 32 * 10 = 320 columns
    {
      if (logoBits[i][j]) // check color to display
      {
        for (k = 0; k < 10; k++) // do 10 columns in frameNum color
        {
          rowPtr[m++] = (char)frameNum; // blue byte
          rowPtr[m++] = (char)frameNum; // green byte
          rowPtr[m++] = (char)frameNum; // red byte
          m++;                          // alpha byte
        }
      }
      else
      {
        for (k = 0; k < 10; k++) // do 10 columns in inverse frameNum color
        {
          rowPtr[m++] = (char)(255 - (unsigned char)frameNum); // blue byte
          rowPtr[m++] = (char)(255 - (unsigned char)frameNum); // green byte
          rowPtr[m++] = (char)(255 - (unsigned char)frameNum); // red byte
          m++;                                                 // alpha byte
        }
      }
    }
    for (k = 0; k < 11; k++) // replicate 11 rows
      memcpy(videoBuffer + (l-- * rowSize), rowPtr, rowSize);
  }

  *bytesInRow = rowSize; // return row size = width * 4 (32 bpp)
  *buffer = videoBuffer; // return a pointer to the video data
  return bbErrNone;
}


void shutdown()
{
  if (audioBuffer)
    free(audioBuffer);
  if (videoBuffer)
    free(videoBuffer);
  if (bbMPEGShutdown)
    bbMPEGShutdown();  // call bbMPEG's shutdown routine
  if (dllPtr)
    FreeLibrary(dllPtr);
}

int main(int argc, char* argv[])
{
  char outputFilename[256];
  makeMPEGRecInfo makeMPEGInfo;

  dllPtr = NULL;
  bbMPEGInit = NULL;
  bbMPEGMakeMPEG = NULL;
  bbMPEGShutdown = NULL;
  bbMPEGSetFocus = NULL;
  videoBuffer = NULL;
  audioBuffer = NULL;
  width = 320;
  height = 240;

  if (argc > 1)
    strcpy(outputFilename, argv[1]);
  else
    strcpy(outputFilename, "test.mpg");

  dllPtr = LoadLibrary("bbmpeg.dll");  // load bbMPEG.dll
  if (!dllPtr)
  {
    printf("\nUnable to load dll bbMPEG.dll.\n");
    exit(1);
  }

  // get the address of bbMPEG's init routine
  bbMPEGInit = (bbMPEGInitEntry) GetProcAddress(dllPtr, "bbMPEGInit");
  if (!bbMPEGInit)
  {
    printf("\nCould not get bbMPEGInit function address from bbMPEG.dll.\n");
    shutdown();
    exit(1);
  }

  // get the address of bbMPEG's makempeg routine
  bbMPEGMakeMPEG = (bbMPEGMakeMPEGEntry) GetProcAddress(dllPtr, "bbMPEGMakeMPEG");
  if (!bbMPEGMakeMPEG)
  {
    printf("\nCould not get bbMPEGMakeMPEG function address from bbMPEG.dll.\n");
    shutdown();
    exit(1);
  }

  // get the address of bbMPEG's shutdown routine
  bbMPEGShutdown = (bbMPEGShutdownEntry) GetProcAddress(dllPtr, "bbMPEGShutdown");
  if (!bbMPEGShutdown)
  {
    printf("\nCould not get bbMPEGShutdown function address from bbMPEG.dll.\n");
    shutdown();
    exit(1);
  }

  // get the address of bbMPEG's setFocus routine
  bbMPEGSetFocus = (bbMPEGSetFocusEntry) GetProcAddress(dllPtr, "bbMPEGSetFocus");
  if (!bbMPEGSetFocus)
  {
    printf("\nCould not get bbMPEGSetFocus function address from bbMPEG.dll.\n");
    shutdown();
    exit(1);
  }

  if (bbMPEGInit() != bbErrNone)
  {
    printf("\nCould not initialize bbMPEG.dll.\n");
    shutdown();
    exit(1);
  }

  bbMPEGSetFocus(); // does not need to be called, just there if needed

  audioBufferSize = 1601; // 48000.0kHz / 29.97fps
  audioBufferSize *= 2;   // stereo, two bytes per sample
  audioBufferSize *= 2;   // 16 bit samples, two bytes per sample

  audioBuffer = (char*) malloc(audioBufferSize); // create audio buffer
  if (!audioBuffer)
  {
    printf("\nCould not get memory for audio buffer.\n");
    shutdown();
    exit(1);
  }

  videoBuffer = (char*) malloc(width * height * 4); // create video buffer
  if (!videoBuffer)
  {
    printf("\nCould not get memory for video buffer.\n");
    shutdown();
    exit(1);
  }

  makeMPEGInfo.doVideo = 1;        // set to zero if video encoding is not required
  makeMPEGInfo.width = width;      // width of video in pixels
  makeMPEGInfo.height = height;    // height of video in pixels
  makeMPEGInfo.startFrame = 0;     // starting frame number
  makeMPEGInfo.endFrame = 255;     // ending frame number, must be >= startFrame

  makeMPEGInfo.doAudio = 1;        // set to zero if audio encoding is not required
  makeMPEGInfo.audioRate = 48000;  // sample rate, valid values are 32000, 44100 and 48000
  makeMPEGInfo.stereo = 1;         // 1 = stereo, 0 = mono
  makeMPEGInfo.sampleSize = 16;    // sample size, MUST be 16 bits
  makeMPEGInfo.startSample = 0;    // first sample number
  makeMPEGInfo.endSample = 256 * 1601 - 1; // last sample number

  makeMPEGInfo.getVideo = getVideo;  // fill in the video callback field
  makeMPEGInfo.getAudio = getAudio;  // fill in the audio callback field

  makeMPEGInfo.outputFilename = outputFilename; // output MPEG filename

  bbMPEGMakeMPEG(&makeMPEGInfo);     // call the makeMPEG routine to create an MPEG

  shutdown();
  return 0;
}

