/*
 * btscanner - Displays the output of Bluetooth scans
 * Copyright (C) 2003 Pentest Limited
 * 
 * Written 2003 by Tim Hurman <timh at pentest.co.uk>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation;
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT OF THIRD PARTY
 * RIGHTS.  IN NO EVENT SHALL THE COPYRIGHT HOLDER(S) AND AUTHOR(S) BE LIABLE
 * FOR ANY CLAIM, OR ANY SPECIAL INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY
 * DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 * 
 * ALL LIABILITY, INCLUDING LIABILITY FOR INFRINGEMENT OF ANY PATENTS,
 * COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS, RELATING TO USE OF THIS SOFTWARE
 * IS DISCLAIMED.
 */

/*
 * bs_sdp.c: the formatting for for the SDP output.
 */

#include <sys/types.h>
#include <stdlib.h>
#include <pthread.h>
#include <unistd.h>
#include <stdio.h>
#include <signal.h>
#include <errno.h>
#include <string.h>
#include <curses.h>

#include "btscanner.h"

#define SCR_BUF_SZ 1024
static char UUID_str[MAX_LEN_UUID_STR];

void print_lang_attr(void *value, void *user)
{
	char buf[SCR_BUF_SZ];
	sdp_lang_attr_t *lang = (sdp_lang_attr_t *)value;
	struct proc_info *pi = (struct proc_info*)user;

	snprintf(buf, SCR_BUF_SZ, "  code_ISO639: 0x%02x\n", lang->code_ISO639);
	buf[SCR_BUF_SZ-1] = 0; bs_processlines(pi, buf);
	snprintf(buf, SCR_BUF_SZ, "  encoding:    0x%02x\n", lang->encoding);
	buf[SCR_BUF_SZ-1] = 0; bs_processlines(pi, buf);
	snprintf(buf, SCR_BUF_SZ, "  base_offset: 0x%02x\n", lang->base_offset);
	buf[SCR_BUF_SZ-1] = 0; bs_processlines(pi, buf);
}   

static void print_service_desc(void *value, void *user)
{
	char buf[SCR_BUF_SZ];
	char str[MAX_LEN_PROTOCOL_UUID_STR];
	sdp_data_t *p = (sdp_data_t *)value, *s;
	int i = 0, proto = 0;
	struct proc_info *pi = (struct proc_info*)user;
   
	for (; p; p = p->next, i++) {
		switch (p->dtd) {
		case SDP_UUID16:
		case SDP_UUID32:
		case SDP_UUID128:
			sdp_uuid2strn(&p->val.uuid, UUID_str, MAX_LEN_UUID_STR);
			sdp_proto_uuid2strn(&p->val.uuid, str, sizeof(str));
			proto = sdp_uuid_to_proto(&p->val.uuid);
			snprintf(buf, SCR_BUF_SZ, "  \"%s\" (0x%s)\n", str, UUID_str);
			buf[SCR_BUF_SZ-1] = 0; bs_processlines(pi, buf);
			break;
		case SDP_UINT8:
			if (proto == RFCOMM_UUID)
				snprintf(buf, SCR_BUF_SZ, "    Channel: %d\n", p->val.uint8);
			else
				snprintf(buf, SCR_BUF_SZ, "    uint8: 0x%x\n", p->val.uint8);
			buf[SCR_BUF_SZ-1] = 0; bs_processlines(pi, buf);
			break;
		case SDP_UINT16:
			if (proto == L2CAP_UUID) {
				if (i == 1)
					snprintf(buf, SCR_BUF_SZ, "    PSM: %d\n", p->val.uint16);
				else
					snprintf(buf, SCR_BUF_SZ, "    Version: 0x%04x\n",
					  p->val.uint16);
			} else if (proto == BNEP_UUID)
				if (i == 1)
					snprintf(buf, SCR_BUF_SZ, "    Version: 0x%04x\n",
					  p->val.uint16);
				else
					snprintf(buf, SCR_BUF_SZ, "    uint16: 0x%x\n",
					  p->val.uint16);
			else
				snprintf(buf, SCR_BUF_SZ, "    uint16: 0x%x\n", p->val.uint16);
			buf[SCR_BUF_SZ-1] = 0; bs_processlines(pi, buf);
			break;
		case SDP_SEQ16:
			strncpy(buf, "    SEQ16:", SCR_BUF_SZ);
			buf[SCR_BUF_SZ] = 0;
			for (s = p->val.dataseq; s; s = s->next) {
				snprintf(buf+strlen(buf), SCR_BUF_SZ, " %x", s->val.uint16);
				buf[SCR_BUF_SZ] = 0;
			}
			strncat(buf, "\n", SCR_BUF_SZ);
			buf[SCR_BUF_SZ] = 0;
			bs_processlines(pi, buf);
			break;
		case SDP_SEQ8:
			strncpy(buf, "    SEQ8:", SCR_BUF_SZ);
			buf[SCR_BUF_SZ] = 0;
			for (s = p->val.dataseq; s; s = s->next) {
				snprintf(buf+strlen(buf), SCR_BUF_SZ, " %x", s->val.uint8);
				buf[SCR_BUF_SZ] = 0;
			}
			strncat(buf, "\n", SCR_BUF_SZ);
			buf[SCR_BUF_SZ] = 0;
			bs_processlines(pi, buf);
			break;
		default:
			snprintf(buf, SCR_BUF_SZ, "    FIXME: dtd=0%x\n", p->dtd);
			buf[SCR_BUF_SZ-1] = 0; bs_processlines(pi, buf);
			break;
		}
	}
}



/* sdp service classes */
static void print_service_class(void *value, void *userData)
{
	char ServiceClassUUID_str[MAX_LEN_SERVICECLASS_UUID_STR];
	uuid_t *uuid = (uuid_t *)value;
	struct proc_info *pi = (struct proc_info*)userData;
	char buf[SCR_BUF_SZ];

	sdp_uuid2strn(uuid, UUID_str, MAX_LEN_UUID_STR); 
	sdp_svclass_uuid2strn(uuid, ServiceClassUUID_str,
	  MAX_LEN_SERVICECLASS_UUID_STR);
	snprintf(buf, SCR_BUF_SZ, "  \"%s\" (0x%s)\n",
	  ServiceClassUUID_str, UUID_str);
	buf[SCR_BUF_SZ-1] = 0; bs_processlines(pi, buf);
}

/* sdp access protos */
void print_access_protos(void *value, void *userData)
{
	sdp_list_t *protDescSeq = (sdp_list_t *)value;
	sdp_list_foreach(protDescSeq, print_service_desc, userData);
}

void print_profile_desc(void *value, void *userData)
{
	struct proc_info *pi = (struct proc_info*)userData;
	char buf[SCR_BUF_SZ];
	sdp_profile_desc_t *desc = (sdp_profile_desc_t *)value;
	char str[MAX_LEN_PROFILEDESCRIPTOR_UUID_STR];

	sdp_uuid2strn(&desc->uuid, UUID_str, MAX_LEN_UUID_STR);
	sdp_profile_uuid2strn(&desc->uuid, str,
	  MAX_LEN_PROFILEDESCRIPTOR_UUID_STR);

	snprintf(buf, SCR_BUF_SZ,"  \"%s\" (0x%s)\n", str, UUID_str);
	buf[SCR_BUF_SZ-1] = 0; bs_processlines(pi, buf);

	if (desc->version) {
		snprintf(buf, SCR_BUF_SZ,"    Version: 0x%04x\n", desc->version);
		buf[SCR_BUF_SZ-1] = 0; bs_processlines(pi, buf);
	}
}



/* print the sdp services */
int bs_print_services(struct proc_info *pi, search_context_t *ctxt)
{
	char buf[SCR_BUF_SZ];
	/* firstly print this one */
	sdp_list_t *list = 0, *proto = 0;
	sdp_data_t *d;

	if(NULL == ctxt) return 0;

	/* sdp_record_print */
	d = sdp_data_get(ctxt->rec, SDP_ATTR_SVCNAME_PRIMARY);
	if (d) {
		snprintf(buf, SCR_BUF_SZ, "Service Name:        %s\n", d->val.str);
		buf[SCR_BUF_SZ-1] = 0; bs_processlines(pi, buf);
	}
	d = sdp_data_get(ctxt->rec, SDP_ATTR_SVCDESC_PRIMARY);
	if (d) {
		snprintf(buf, SCR_BUF_SZ, "Service Description: %s\n", d->val.str);
		buf[SCR_BUF_SZ-1] = 0; bs_processlines(pi, buf);
	}
	d = sdp_data_get(ctxt->rec, SDP_ATTR_PROVNAME_PRIMARY);
	if (d) {
		snprintf(buf, SCR_BUF_SZ, "Service Provider:    %s\n", d->val.str);
		buf[SCR_BUF_SZ-1] = 0; bs_processlines(pi, buf);
	}

	/* normal output */
	snprintf(buf, SCR_BUF_SZ, "Service RecHandle:   0x%x\n", ctxt->rec->handle);
	buf[SCR_BUF_SZ-1] = 0; bs_processlines(pi, buf);
	if (sdp_get_service_classes(ctxt->rec, &list) == 0) {
		bs_processlines(pi, "Service Class ID List:\n");
		sdp_list_foreach(list, print_service_class, pi);
		sdp_list_free(list, free);
	}

	if (sdp_get_access_protos(ctxt->rec, &proto) == 0) {
		bs_processlines(pi, "Protocol Descriptor List:\n");
		sdp_list_foreach(proto, print_access_protos, pi);
		sdp_list_free(proto, (sdp_free_func_t)sdp_data_free);
	}

	if (sdp_get_lang_attr(ctxt->rec, &list) == 0) {
		bs_processlines(pi, "Language Base Attr List:\n");
		sdp_list_foreach(list, print_lang_attr, pi);
		sdp_list_free(list, free);
	}

	if (sdp_get_profile_descs(ctxt->rec, &list) == 0) {
		bs_processlines(pi, "Profile Descriptor List:\n");
		sdp_list_foreach(list, print_profile_desc, pi);
		sdp_list_free(list, free);
	}

	if(ctxt->sub || ctxt->next)
		bs_processlines(pi, "\n");

	/* print the sub contexts, if any */
	if (ctxt->sub)
		bs_print_services(pi, ctxt->sub);

	/* print the next contexts, if any */
	if (ctxt->next)
		bs_print_services(pi, ctxt->next);

	return 0;
}

