/*
 * Cheops Network User Interface
 *
 * Copyright (C) 1999, Adtran, Inc.
 * 
 * Distributed under the terms of the GNU GPL
 *
 */

#include <gtk/gtk.h>
#include <netinet/in.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <pwd.h>
#include "cheops.h"

#define SMTP_PORT 25

static char *service = "smtp";

static char *label = "Mail Server";

struct smtp_setup {
	/* e-mail address to request */
	char email[80];
	/* Do we try to verify? */
	int verify;

	GtkWidget *window;
	GtkWidget *emailw;
	GtkWidget *verifyw;
};

struct smtp_monitor {
	int s;			/* Socket for communication */
	int level;		/* level of error we should report */
	struct net_page *np;	/* Page our object is on */
	struct net_object *no;	/* Pointer to the object we're monitoring */
	int io_id;		/* ID for I/O attempts */
	struct smtp_setup *setup;		/* Setup data */
};

char *plugin_name() {
	return service;
}

char *plugin_label() {
	return label;
}

int plugin_type() {
	return TYPE_MONITOR;
}

int default_level() {
	return MONITOR_CRITICAL;
}

void reset(struct smtp_monitor *hm)
{
	if (hm->io_id > -1) {
		cancel_data(hm->io_id);
		hm->io_id = -1;
	}
	if (hm->s > -1) {
		close(hm->s);
		hm->s = -1;
	}

}

void stage4(void *data, int s, GdkInputCondition c)
{
	/* Called when the e-mail address is verified */
	struct smtp_monitor *hm = (struct smtp_monitor *)data;
	char buf[256];

	hm->io_id = -1;
#if 0
	printf("%s: Stage 4 and condition is %d: ", hm->no, c);
#endif
	memset(buf, 0, sizeof(buf));
	read(hm->s, buf, sizeof(buf));
	if (!strncmp(buf, "250", 3)) {
		monitor_report(hm->no, hm->np, MONITOR_NOMINAL, service, "Nominal condition");
	} else {
		monitor_report(hm->no, hm->np, hm->level, service, "Verification failed");
	}
	reset(hm);
}
void stage3(void *data, int s, GdkInputCondition c)
{
	extern int errno;
	/* Called when the mail server announces itself to us */
	struct smtp_monitor *hm = (struct smtp_monitor *)data;
	char buf[256];
	char buf2[256];
	int res;

	hm->io_id = -1;
#if 0
	printf("%s: Stage 3 and condition is %d!\n", c);
#endif
	memset(&buf, 0, sizeof(buf));
	read(hm->s, &buf, sizeof(buf) - 1);
	if (!strncmp(buf, "220", 3)) {
		if (hm->setup && hm->setup->verify) {
			snprintf(buf, sizeof(buf), "VRFY %s\n", hm->setup->email);
			if ((res = write(s, buf, strlen(buf))) != strlen(buf)) {
				printf("Result is %d (%d)\n", res, errno);
				monitor_report(hm->no, hm->np, hm->level, service, "Unable to request verification");
			} else {
				hm->io_id = wait_for_data(s, stage4, data);
				return;
			}
		} else 
			monitor_report(hm->no, hm->np, MONITOR_NOMINAL, service, "Nominal condition");
	} else {
		snprintf(buf2, sizeof(buf2), "Unexpected welcome: %s", buf);
		monitor_report(hm->no, hm->np, hm->level, service, buf2);
	}
	reset(hm);	
}

void stage2(void *data, int s, GdkInputCondition c)
{
	/* Called when connect() has completed */
	struct smtp_monitor *hm = (struct smtp_monitor *)data;
	char buf[256];
	int res;
	
	res = get_socket_error(s);
	
	hm->io_id = -1;

#if 0
	printf("%s: Stage 2 and condition is %d:",no, c);
#endif
	
	/* If the socket is connected (i.e. available for write), then
	   we write our request to it.  Assuming both go through, we 
	   then setup a callback to handle the reply, if one is available.
	   If not, we immediately report an error */
	
	if (!res) {
		hm->io_id = wait_for_data(s, stage3, data);
		return;
	} else {
		snprintf(buf, sizeof(buf), "Connection failed: %s", strerror(res));
		monitor_report(hm->no, hm->np, hm->level, service, buf);
	}
	reset(hm);
}

void monitor(struct net_object *no, int level, void **data, void *setupdata)
{
	struct smtp_monitor *hm = (struct smtp_monitor *)(*data);
	
	/* While the arguments to our monitoring function are straightforward
	   and the general idea (connect, write a request, read back and look
	   for our magic string) isn't that difficult, the function is somewhat
	   complicated by the requirement that we never make a blocking
	   I/O call.
	
	   This is the first step of the monitoring function.  First, we
	   allocate a monitor stucture to hold some important info like
	   the callback id's, our sockets, and the object and page. */
	
#if 0
	fprintf(stdout, "Monitoring SMTP!\n");
#endif
	if (!*data) {
#if 0
		fprintf(stdout, "Making new monitor structure for %s\n", no->hostname);
#endif
		hm = g_new(struct smtp_monitor, 1);
		hm->s = -1;
		hm->io_id = -1;
		hm->no = no;
		hm->np = no->np;
		hm->setup = setupdata;
		*data = hm;
	}
	hm->level = level;

	/* Remember, we're not guaranteed that we have delivered
	   a reply by this time, so we need to close the socket
	   if it isn't already closed after first removing any
	   pending io callbacks */
	
	reset(hm);
	   
	hm->s = socket(AF_INET, SOCK_STREAM, IPPROTO_IP);
	
	/* Immediately report an error if socket creation failed */
	
	if (hm->s < 0) {
		monitor_report(no, no->np, hm->level, service, "Unable to create socket (local)");
		return;
	}
	
	/* monitor.c provides this as a convenience function.  We try to asynchronously
	   connect the socket to the given address.  When it is available for writing
	   or an  error has occured, stage2 will be called */
	
	hm->io_id = try_to_connect(hm->s, no->ip_addr, htons(SMTP_PORT), hm, stage2);
}

void cleanup(void **data)
{
	/* Here, we clean up our http_monitor structure and 
	   remove any pending inputs we had, since we are no longer
	   being asked to work */
	   
	struct smtp_monitor *hm = (struct smtp_monitor *)(*data);
	if (hm->s > -1) {
		if (hm->io_id > -1) {
			cancel_data(hm->io_id);
			hm->io_id = -1;
		}
		close(hm->s);
	}
	g_free(hm);
	*data = NULL;
	/* For completeness */
#if 0
	fprintf(stdout, "Cleaned up!\n");
#endif
}

static void close_setup_window(GtkWidget *w)
{
	struct smtp_setup *s = (struct smtp_setup *)
		gtk_object_get_user_data(GTK_OBJECT(w));
	/* When they close the window, grab their info */
	strncpy(s->email, gtk_entry_get_text(GTK_ENTRY(s->emailw)), sizeof(s->email));
	s->verify = GTK_TOGGLE_BUTTON(s->verifyw)->active;
	
	/* Destroy the window itself */
	gtk_widget_hide(s->window);
	gtk_widget_destroy(s->window);
	s->window = NULL;
}

static void handle_toggle(GtkWidget *w)
{
	struct smtp_setup *s = (struct smtp_setup *)
		gtk_object_get_user_data(GTK_OBJECT(w));
	gtk_widget_set_sensitive(s->emailw, GTK_TOGGLE_BUTTON(w)->active);
}

void *setup(void *oldsetup, struct net_object *no)
{
	/* Allocate an smtp_setup structure, initialize some reasonable
	   defaults, display a window, and then return our structure.
	   
	   It's not important that the structure have final values in 
	   it when we return, since our callbacks will fill them in when
	   the user closes our setup window */
	   
	struct smtp_setup *s;
	
	GtkWidget *label;
	GtkWidget *bbox;
	GtkWidget *close;
	GtkWidget *vbox;
	struct passwd *pw;
	
	char buf[256];
	
	if (oldsetup)
		s = oldsetup;
	else {
		s = g_new(struct smtp_setup, 1);
		memset(s, 0, sizeof(struct smtp_setup));
		pw = getpwuid(getuid());
		if (pw) 
			strncpy(s->email, pw->pw_name, sizeof(s->email));
	}
	
	if (s->window) {
		gtk_widget_show(s->window);
		return s;
	}
	
	/* Make our setup box */
	snprintf(buf, sizeof(buf), "SMTP Monitor: %s", no->hostname);
	s->window = gtk_window_new(GTK_TOPLEVEL);
	gtk_widget_realize(s->window);
	gtk_window_set_title(GTK_WINDOW(s->window), buf);
	fix_icon(s->window->window);

	vbox = gtk_vbox_new(FALSE, 5);
	gtk_widget_show(vbox);

	label = gtk_label_new(buf);
	gtk_widget_show(label);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 5);

	s->emailw = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(s->emailw), s->email);
	gtk_widget_set_sensitive(s->emailw, FALSE);

	s->verifyw = gtk_check_button_new_with_label("Verify an e-mail address:");
	gtk_object_set_user_data(GTK_OBJECT(s->verifyw), s);
	gtk_signal_connect(GTK_OBJECT(s->verifyw), "clicked", GTK_SIGNAL_FUNC(handle_toggle), NULL);
	if (s->verify)
		gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(s->verifyw), TRUE);
	gtk_box_pack_start(GTK_BOX(vbox), s->verifyw, FALSE, FALSE, 5);
	gtk_widget_show(s->verifyw);

	gtk_box_pack_start(GTK_BOX(vbox), s->emailw, FALSE, FALSE, 5);
	gtk_widget_show(s->emailw);
	
	bbox = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(bbox);
	
	close = gtk_button_new_with_label("  Close  ");
	gtk_object_set_user_data(GTK_OBJECT(close), s);
	gtk_signal_connect(GTK_OBJECT(close), "clicked", GTK_SIGNAL_FUNC(close_setup_window), NULL);
	gtk_object_set_user_data(GTK_OBJECT(s->window), s);
	gtk_signal_connect(GTK_OBJECT(s->window), "delete_event", GTK_SIGNAL_FUNC(close_setup_window), NULL);
	gtk_widget_show(close);
	
	gtk_box_pack_end(GTK_BOX(bbox), close, FALSE, FALSE, 5);
	gtk_box_pack_end(GTK_BOX(vbox), bbox, FALSE, FALSE, 5);	
	
	gtk_container_border_width(GTK_CONTAINER(s->window), 10);
	gtk_container_add(GTK_CONTAINER(s->window), vbox);
	gtk_widget_show(s->window);
	return s;
}

void setup_cleanup(void *data)
{
	struct smtp_setup *setup = (struct smtp_setup *)data;
	/* If we have a setup window open, destroy it, then free
	   our structure */
	if (setup->window)
		gtk_widget_destroy(setup->window);
#if 0
	printf("Cleaning up setup structure (%p)!\n", setup);
#endif
	g_free(setup);
}

char *setup2str(void *data)
{
	static char buf[256];
	struct smtp_setup *s = (struct smtp_setup *)data;
	
	/* When monitor stuff is saved, the saving code calls this
	   routing to convert the setup structure into a string that it
	   can include in the configuration file. */
	
	snprintf(buf, sizeof(buf), "%d!%s", s->verify, s->email);
	return buf;
}

void *str2setup(char *buf, struct net_object *no)
{

	/* Hre, we read a setup string and return a setup pointer.  We use
	   "!" as delimiters because ^ is already used and ! are unlikely
	   to be found in the request */
	char *c;
	struct smtp_setup *s;
	struct passwd *pw;

	s = g_new(struct smtp_setup, 1);
	pw = getpwuid(getuid());
	if (pw) 
		strncpy(s->email,pw->pw_name,sizeof(s->email)); 
	else
		strcpy(s->email,"");
	
	s->window = NULL;

	c = strtok(buf, "!");
	if (c)
		s->verify = atoi(c);
	else
		s->verify = 0;
	c = strtok(NULL, "!");
	if (c)
		strncpy(s->email, c, sizeof(s->email));
	return s;
}
