/*
 *    File:      main.c 
 *    Purpose:   Program the PLL in the Ramsey-FM25A kit via PC
 *		 paralllel port.
 *    Author:    Nox
 *    Date:      December 27, 2001 
 * ---------------------------------------------------------------------
 * This code is indended to provide an alternative way to change
 * Operational Frequency on any Motorola MC145170-2 based FM transmitter
 * via the PC parallel port.
 *			
 * Parallel to PLL connections are as follows:
 * 					
 *     Parallel		to	 MC145170-2   or   PIC Removed
 *	 Port			    (IC)	    U2 Socket
 *  ----------------------------------------------------------
 *     Pin 2 (d0) .............. Pin 5 (Din) ..... Pin 3
 *     Pin 3 (d1) .............. Pin 7 (clk) ..... Pin 2
 *     Pin 4 (d2) .............. Pin 6 (!ena) .... Pin 12
 *     
 *     Pin 25<->20 soldered .... Pin 14 (gnd) .... Pin 14
 *
 *  Note: You must remove the PIC uP from Socket U2.  You can Insert the
 *        data lines coming form the Parallel port directly into the 
 *        socket holes.  One can use another 16 pin socket flipped upside
 * 	  down with component lead wire acting as conductive bridges 
between
 * 	  the parallel port signals and the FM25A circuit board.  The 
wires
 * 	  from the parallel port are solderd to the pins of the upsidedown
 * 	  socket.  This configuration allows you to easily switch back to
 * 	  the PIC microprocessor without a lot of hassle and mess.
 *
 *  Nox
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/io.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/fcntl.h>
#include <sys/stat.h>
#include "spipll.h"

#define PORTRANGE 8
#define MEMBUFSIZE 1024*256

int bogoloopus = 1000;
int iobase = 0;

/* 
------------------------------------------------------------------------- 
*/

/*
 * writes to SPI PLL device registers (used with MC145170-2) 
 * by Nox
 */

/* main program and command interpreter */
int main(int argc, char **argv)
{

int freq;
int new_freq;

int N_REG_B1;   // N register, bit 1
int N_REG_B2;   // N register, bit 2


unsigned int optch;
char	opts[] = "f:h";

/*
 * parse command line options
 *
 */
	while ( ((optch = getopt(argc, argv, opts)) != EOF) )
	{
		switch (optch)
		{
			case 'f':	// set frequency
			freq = atoi(optarg);
			break;

			case 'h':	// show help
			usage();
			terminate();
			break;

			default:
				fprintf(stdout,"spipll -h for help.\n");
				terminate();
				break;
			
		}
	}

/*
 * Setup parallel port address, default is 0x378
 *
 */
	probe_loopus();
	do_iobase("0x378");

/* do program PLL and exit */
	
        if (freq < 880 || freq > 1080 ) {
                fprintf(stdout, "Frequency out of range.\n");
		terminate();
                return;
        }

        if (freq >= 1024 && freq <= 1080) {
                new_freq = freq - 1024;
                N_REG_B1=0x04;
                N_REG_B2=new_freq;
        }

        if (freq >= 880 && freq <=1023) {
                new_freq = freq - 768;
                N_REG_B1=0x03;
                N_REG_B2=new_freq;
        }

/*
 *  Send out the data to pll
 */

        do_cs("");
        spi_outb(RAMSEY_C_REG);
        do_cs("0");
//        spi_outb(RAMSEY_NULL);
        do_cs("");
        spi_outb(RAMSEY_R_REG_B1);
        spi_outb(RAMSEY_R_REG_B2);
        spi_outb(RAMSEY_R_REG_B3);
        do_cs("0");
//       spi_outb(RAMSEY_NULL);
        do_cs("");
        spi_outb(N_REG_B1);
        spi_outb(N_REG_B2);
        do_cs("0");
//        spi_outb(RAMSEY_NULL);
        fprintf(stdout, "Frequency set to: %d00 KHz\n", freq);

	terminate();

	/* not reached */
	return 0;
}


/* 
------------------------------------------------------------------------- 
*/

/* sets IO base */
void do_iobase(void *arg)
{
        char *io_string = (char *) arg;
        int new_iobase = 0;

        sscanf(io_string, "%x", &new_iobase);
        if (new_iobase) {
                if (iobase) ioperm(iobase, PORTRANGE, 0);
                iobase = new_iobase;
                ioperm(iobase, PORTRANGE, 1);
                outb(SPI_CS, iobase+0);
        } else {
                fprintf(stdout, "Current iobase=0x%04x.\n", iobase);
        }
        return;
}      

/* 
------------------------------------------------------------------------- 
*/

/* deasserts/assers chip select line */
void do_cs(void *arg)
{
        char *param = (char *) arg;
        int csval = 0;

        if (strlen(param)) {
                sscanf(param, "%u", &csval);
                spi_cs(csval);
        } else {
                spi_cs(0);
                spi_bitdelay();
                spi_cs(1);
                spi_bitdelay();
        }
        return;
}

/* 
------------------------------------------------------------------------- 
*/

/* cleanly shuts down program */
void terminate()
{
        if (iobase) {
                spi_cs(0);
                ioperm(iobase, PORTRANGE, 0);
        }
        exit(0);
}

/* 
------------------------------------------------------------------------- 
*/

/* print program usage */
void usage()
{
        fprintf(stdout, "usage: spipll -h -f <freq>\n");
        fprintf(stdout, 
"----------------------------------------------\n");
	fprintf(stdout, "-f <frequency>, where frequency is 880 -> 
1080\n");
	fprintf(stdout, "-h help (this display)\n");
        return;
}

/* 
------------------------------------------------------------------------- 
*/

/* sleeps us microseconds */
void myusleep(long us)
{
	long i;
	long cycles = us*bogoloopus;
	
	for (i=0; i<cycles; i++) ;
	return;
}

/* 
------------------------------------------------------------------------- 
*/

/* probes machine speed */
void probe_loopus(void)
{
	struct timeval tv1, tv2;
	long i;
	long us;
	
	gettimeofday(&tv1, NULL);
	for (i=0; i<100000000; i++) ;
	gettimeofday(&tv2, NULL);
	us = (tv2.tv_sec - tv1.tv_sec) * 1000000
		+ (tv2.tv_usec - tv1.tv_usec);
	bogoloopus = (100000000/us);
}

/* 
------------------------------------------------------------------------- 
*/

/* sets chip select output 1=selected */
void spi_cs(int cs)
{
        char old;

        old = inb(iobase + 0);
        old |= SPI_CS;
        if (cs) old &= ~SPI_CS;
        outb(old, iobase + 0);
        return;
}

/* 
------------------------------------------------------------------------- 
*/

/* set SI line */
void spi_si(int mydata)
{
        char old;

        old = inb(iobase + 0);
        old &= ~SPI_SI;
        if (mydata) old |= SPI_SI;
        outb(old, iobase + 0);
        return;
}

/* 
------------------------------------------------------------------------- 
*/

/* gets SO state */
int spi_so()
{
        int ret;

        ret = !!(inb(iobase + 1) & SPI_SO);
        return ret;
}

/* 
------------------------------------------------------------------------- 
*/

/* sets SCK line */
void spi_sck(int sck)
{
        char old;

        old = inb(iobase + 0);
        old &= ~SPI_SCK;
        if (sck) old |= SPI_SCK;
        outb(old, iobase + 0);
        return;
}

/* 
------------------------------------------------------------------------- 
*/

/* standard delay function */
void spi_bitdelay()
{
        myusleep(5);
}

/* 
------------------------------------------------------------------------- 
*/

/* sends out one byte using SPI protocol */
void spi_outb(char mydata)
{
        int bit;
        int i;

        spi_sck(0);
        spi_si(0);
        spi_bitdelay();
        for (i=0; i<8; i++) {
                bit = ((mydata >> (7-i)) & 1);
                spi_si(bit);
                spi_bitdelay();
                spi_sck(1);
                spi_bitdelay();
                spi_sck(0);
                spi_bitdelay();
        }
        spi_si(0);
        spi_bitdelay();
        return;
}

/*-------------------------- end of main.----------------------------*/
