/* 
 *	Copyright (C) Chia-chen Kuo - April 2001
 *
 *  This file is part of DVD2AVI, a free MPEG-2 decoder
 *	
 *  DVD2AVI is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  DVD2AVI is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#define GLOBAL
extern "C"
{
#include "global.h"
#include "AC3Dec\ac3.h"
}

#define TRACK_HEIGHT	30
#define INIT_WIDTH		320
#define INIT_HEIGHT		240

#define MASKCOLOR			RGB(0, 6, 0)
#define MAX_LOADSTRING		100
#define INI_VERSION			1

#define SAVE_AVI		1
#define	SAVE_D2V		2
#define SAVE_WAV		3
#define	OPEN_D2V		4
#define OPEN_VOB		5
#define OPEN_WAV		6

#define PRIORITY_HIGH		1
#define PRIORITY_NORMAL		2
#define PRIORITY_LOW		3

BOOL PopFileDlg(PTSTR, HWND, int);
ATOM MyRegisterClass(HINSTANCE);
LRESULT CALLBACK WndProc(HWND, UINT, WPARAM, LPARAM);
LRESULT CALLBACK Statistics(HWND, UINT, WPARAM, LPARAM);
LRESULT CALLBACK VideoList(HWND, UINT, WPARAM, LPARAM);
LRESULT CALLBACK AudioList(HWND, UINT, WPARAM, LPARAM);
LRESULT CALLBACK Delay(HWND, UINT, WPARAM, LPARAM);
LRESULT CALLBACK ClipResize(HWND, UINT, WPARAM, LPARAM);
LRESULT CALLBACK Luminance(HWND, UINT, WPARAM, LPARAM);
LRESULT CALLBACK Normalization(HWND, UINT, WPARAM, LPARAM);
LRESULT CALLBACK About(HWND, UINT, WPARAM, LPARAM);
static DWORD DDColorMatch(LPDIRECTDRAWSURFACE, COLORREF);
static void ShowStatistics(bool);
static void DisableControl(int);
static void ClearTrack(void);
static void CheckINI(void);
static void CheckFlag(void);
static void Recovery(void);
static void RefreshWindow(void);
static void OpenVideoFile(HWND);
static void OpenAudioFile(HWND);
DWORD WINAPI ProcessWAV(LPVOID n);

OPENFILENAME ofn, sfn;
int INI_Version, INIT_X, INIT_Y, Priority_Flag;
FILE *INIFile, *BMPFile;
char szPath[_MAX_PATH], szTemp[_MAX_PATH], szWindowClass[MAX_LOADSTRING];
struct _finddata_t vfpfile, seqfile;

HINSTANCE hInst;
HANDLE hProcess, hThread;
DWORD threadId;
HWND hClipResizeDlg, hLumDlg, hNormDlg;

HDC hdc;
PAINTSTRUCT ps;
HBRUSH hBrush;

char Outfilename[MAX_FILE_NUMBER][_MAX_PATH];
int SoundDelay[MAX_FILE_NUMBER];

static unsigned char BMPHeader[54] =
{
	0x42, 0x4d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x36, 0x00, 0x00, 0x00, 0x28, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x01, 0x00, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x12, 0x0b,
	0x00, 0x00, 0x12, 0x0b, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{
	MSG msg;
	HACCEL hAccel;

	// Initialize global strings
	LoadString(hInstance, IDC_GUI, szWindowClass, MAX_LOADSTRING);
	MyRegisterClass(hInstance);

	// Perform application initialization
	hInst = hInstance;

	hWnd = CreateWindow(szWindowClass, "DVD2AVI", WS_OVERLAPPEDWINDOW & ~(WS_THICKFRAME|WS_MAXIMIZEBOX),
		CW_USEDEFAULT, 0, INIT_WIDTH, INIT_HEIGHT, NULL, NULL, hInstance, NULL);

	CheckINI();

	hTrack = CreateWindow(TRACKBAR_CLASS, NULL,
		WS_CHILD | WS_VISIBLE | WS_DISABLED | TBS_ENABLESELRANGE | TBS_NOTICKS | TBS_TOP,
		0, INIT_HEIGHT, INIT_WIDTH-4*TRACK_HEIGHT, TRACK_HEIGHT, hWnd, (HMENU) ID_TRACKBAR, hInst, NULL);
	SendMessage(hTrack, TBM_SETRANGE, (WPARAM) true, (LPARAM) MAKELONG(0, TRACK_PITCH));

	hLeftButton = CreateWindow("BUTTON", "[",
		WS_CHILD | WS_VISIBLE | WS_DLGFRAME | WS_DISABLED,
		INIT_WIDTH-4*TRACK_HEIGHT, INIT_HEIGHT,
		TRACK_HEIGHT, TRACK_HEIGHT, hWnd, (HMENU) ID_LEFT_BUTTON, hInst, NULL);

	hLeftArrow = CreateWindow("BUTTON", "<",
		WS_CHILD | WS_VISIBLE | WS_DLGFRAME | WS_DISABLED,
		INIT_WIDTH-3*TRACK_HEIGHT, INIT_HEIGHT,
		TRACK_HEIGHT, TRACK_HEIGHT, hWnd, (HMENU) ID_LEFT_ARROW, hInst, NULL);

	hRightArrow = CreateWindow("BUTTON", ">",
		WS_CHILD | WS_VISIBLE | WS_DLGFRAME | WS_DISABLED,
		INIT_WIDTH-2*TRACK_HEIGHT, INIT_HEIGHT,
		TRACK_HEIGHT, TRACK_HEIGHT, hWnd, (HMENU) ID_RIGHT_ARROW, hInst, NULL);

	hRightButton = CreateWindow("BUTTON", "]",
		WS_CHILD | WS_VISIBLE | WS_DLGFRAME | WS_DISABLED,
		INIT_WIDTH-TRACK_HEIGHT, INIT_HEIGHT,
		TRACK_HEIGHT, TRACK_HEIGHT, hWnd, (HMENU) ID_RIGHT_BUTTON, hInst, NULL);

	ResizeWindow(INIT_WIDTH, INIT_HEIGHT);
	MoveWindow(hWnd, INIT_X, INIT_Y, INIT_WIDTH+Edge_Width, INIT_HEIGHT+Edge_Height+TRACK_HEIGHT, true);
	ShowWindow(hWnd, nCmdShow);

	CheckFlag();

	// Load accelerators
	hAccel = LoadAccelerators(hInstance, (LPCTSTR)IDR_ACCELERATOR);

	// Main message loop
	while (GetMessage(&msg, NULL, 0, 0)) 
	{
		if (!TranslateAccelerator(hWnd, hAccel, &msg))
		{
			TranslateMessage(&msg);
			DispatchMessage(&msg);
		}
	}

	return msg.wParam;
}

// Processes messages for the main window
LRESULT CALLBACK WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	DWORD wmId, wmEvent;
	static HINSTANCE hLibrary;
	int i, j;

	switch (message)
	{
		case WM_CREATE:
			DragAcceptFiles(hWnd, true);

			PreScale_Ratio = 1.0; LumGain = 128;

			hDC = GetDC(hWnd);
			hBrush = CreateSolidBrush(MASKCOLOR);
			hMenu = GetMenu(hWnd);
			hProcess = GetCurrentProcess();

			for (i=0; i<MAX_FILE_NUMBER; i++)
				Infilename[i] = (char*)malloc(_MAX_PATH);

			for (i=0; i<8; i++)
			{
				p_block[i] = (short *)malloc(sizeof(short)*64 + 64);
				block[i]   = (short *)((long)p_block[i] + 64 - (long)p_block[i]%64);
			}

			p_fTempArray = (void *)malloc(sizeof(float)*128 + 64);
		    fTempArray = (void *)((long)p_fTempArray + 64 - (long)p_fTempArray%64);

			Initialize_REF_IDCT();
			Initialize_FPU_IDCT();

			// register VFAPI
			HKEY key; DWORD trash;

			if (RegCreateKeyEx(HKEY_CURRENT_USER, "Software\\VFPlugin", 0, "",
				REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &key, &trash) == ERROR_SUCCESS)
			{
				if (_getcwd(szBuffer, _MAX_PATH)!=NULL)
				{
					if (szBuffer[strlen(szBuffer)-1] != '\\')
						strcat(szBuffer, "\\");

					strcpy(szPath, szBuffer);

					if (_findfirst("DVD2AVI.vfp", &vfpfile) != -1L)
					{
						strcat(szBuffer, "DVD2AVI.vfp");

						RegSetValueEx(key, "DVD2AVI", 0, REG_SZ, (LPBYTE)szBuffer, strlen(szBuffer));
						CheckMenuItem(hMenu, IDM_VFAPI, MF_CHECKED);
					}

					RegCloseKey(key);
				}
			}

			// load DLL
			PlugIn_Flag = true;

			if ((hLibrary = LoadLibrary("OpenDVD.dll")) == NULL)
				PlugIn_Flag = false;
			else
			{
				if ((KeyOp = (pfnKeyOp) GetProcAddress(hLibrary, "KeyOp")) == NULL)
					PlugIn_Flag = false;

				if ((BufferOp = (pfnBufferOp) GetProcAddress(hLibrary, "BufferOp")) == NULL)
					PlugIn_Flag = false;
			}

			if (!PlugIn_Flag)
				DeleteMenu(GetSubMenu(hMenu,4), 3, MF_BYPOSITION);
			break;

		case WM_COMMAND:
			wmId    = LOWORD(wParam);
			wmEvent = HIWORD(wParam);

			// parse the menu selections
			switch (wmId)
			{
				case IDM_OPEN:
					DialogBox(hInst, (LPCTSTR)IDD_FILELIST, hWnd, (DLGPROC)VideoList);
					break;

				case IDM_PREVIEW:
					if (IsWindowEnabled(hTrack))
					{
						Display_Flag = true;

						EnableMenuItem(hMenu, IDM_OPEN, MF_GRAYED);
						EnableMenuItem(hMenu, IDM_SAVE, MF_GRAYED);
						EnableMenuItem(hMenu, IDM_LOAD_D2V, MF_GRAYED);
						EnableMenuItem(hMenu, IDM_SAVE_D2V, MF_GRAYED);
						EnableMenuItem(hMenu, IDM_STOP, MF_ENABLED);

						DisableControl(false);
						ShowStatistics(true);

						process.locate = LOCATE_RIP;

						if (WaitForSingleObject(hThread, INFINITE)==WAIT_OBJECT_0)
							hThread = CreateThread(NULL, 0, MPEG2Dec, 0, 0, &threadId);
					}
					break;

				case IDM_SAVE:
					if (PopFileDlg(szOutput, hWnd, SAVE_AVI))
					{
						AVI_Flag = true;
						Display_Flag = false;

						EnableMenuItem(hMenu, IDM_OPEN, MF_GRAYED);
						EnableMenuItem(hMenu, IDM_SAVE, MF_GRAYED);
						EnableMenuItem(hMenu, IDM_LOAD_D2V, MF_GRAYED);
						EnableMenuItem(hMenu, IDM_SAVE_D2V, MF_GRAYED);
						EnableMenuItem(hMenu, IDM_STOP, MF_ENABLED);

						DisableControl(true);
						ShowStatistics(true);

						process.locate = LOCATE_RIP;

						if (WaitForSingleObject(hThread, INFINITE)==WAIT_OBJECT_0)
							hThread = CreateThread(NULL, 0, MPEG2Dec, 0, 0, &threadId);
					}
					break;

				case IDM_PROCESS_WAV:
					DialogBox(hInst, (LPCTSTR)IDD_FILELIST, hWnd, (DLGPROC)AudioList);
					break;

				case IDM_SAVE_D2V:
					if (PopFileDlg(szOutput, hWnd, SAVE_D2V))
					{
						sprintf(szBuffer, "%s.d2v", szOutput);

						D2VFile = fopen(szBuffer, "w+");

						D2V_Flag = true;
						Display_Flag = false;

						EnableMenuItem(hMenu, IDM_OPEN, MF_GRAYED);
						EnableMenuItem(hMenu, IDM_SAVE, MF_GRAYED);
						EnableMenuItem(hMenu, IDM_LOAD_D2V, MF_GRAYED);
						EnableMenuItem(hMenu, IDM_SAVE_D2V, MF_GRAYED);
						EnableMenuItem(hMenu, IDM_STOP, MF_ENABLED);

						DisableControl(true);
						ShowStatistics(true);

						process.locate = LOCATE_RIP;

						if (WaitForSingleObject(hThread, INFINITE)==WAIT_OBJECT_0)
							hThread = CreateThread(NULL, 0, MPEG2Dec, 0, 0, &threadId);
					}
					break;

				case IDM_LOAD_D2V:
					if (PopFileDlg(szInput, hWnd, OPEN_D2V))
					{
D2V_PROCESS:
						D2VFile = fopen(szInput, "r");

						while (File_Limit)
						{
							File_Limit--;
							_close(Infile[File_Limit]);
						}

						fscanf(D2VFile, "DVD2AVIProjectFile\n%d\n", &File_Limit);

						i = File_Limit;
						while (i)
						{
							fscanf(D2VFile, "%d ", &j);
							fgets(Infilename[File_Limit-i], j+1, D2VFile);
							if ((Infile[File_Limit-i] = _open(Infilename[File_Limit-i], _O_RDONLY | _O_BINARY))==-1)
							{
								while (i<File_Limit)
								{
									_close(Infile[File_Limit-i-1]);
									i++;
								}

								File_Limit = 0;
								break;
							}

							i--;
						}

						Recovery();

						fscanf(D2VFile, "\nStream_Type=%d,%X,%X\n", &SystemStream_Flag, &lfsr0, &lfsr1);
						if (lfsr0 || lfsr1)
							KeyOp_Flag = KEY_INPUT;
						else
							KeyOp_Flag = KEY_OFF;

						fscanf(D2VFile, "iDCT_Algorithm=%d\n", &iDCT_Flag);
						fscanf(D2VFile, "YUVRGB_Scale=%d\n", &Scale_Flag);
						fscanf(D2VFile, "Luminance=%d,%d\n", &LumGain, &LumOffset);

						if (LumGain!=128 || LumOffset!=0)
						{
							CheckMenuItem(hMenu, IDM_LUMINANCE, MF_CHECKED);
							Luminance_Flag = true;				
						}
						else
						{
							CheckMenuItem(hMenu, IDM_LUMINANCE, MF_UNCHECKED);
							Luminance_Flag = false;
						}

						if (hLumDlg!=NULL)
						{
							DestroyWindow(hLumDlg);
							hLumDlg = CreateDialog(hInst, (LPCTSTR)IDD_LUMINANCE, hWnd, (DLGPROC)Luminance);
						}

						fscanf(D2VFile, "Picture_Size=%d,%d,%d,%d,%d,%d\n", &Clip_Top, &Clip_Bottom, 
							&Clip_Left, &Clip_Right, &Squeeze_Width, &Squeeze_Height);

						if (Clip_Top || Clip_Bottom || Clip_Left || Clip_Right || Squeeze_Width || Squeeze_Height)
						{
							CheckMenuItem(hMenu, IDM_CLIPRESIZE, MF_CHECKED);
							ClipResize_Flag = true;				
						}
						else
						{
							CheckMenuItem(hMenu, IDM_CLIPRESIZE, MF_UNCHECKED);
							ClipResize_Flag = false;
						}

						if (hClipResizeDlg!=NULL)
						{
							DestroyWindow(hClipResizeDlg);
							hClipResizeDlg = CreateDialog(hInst, (LPCTSTR)IDD_CLIPRESIZE, hWnd, (DLGPROC)ClipResize);
						}

						fscanf(D2VFile, "Field_Operation=%d\n", &FO_Flag);
						fscanf(D2VFile, "Frame_Rate=%d\n", &i);

						CheckFlag();

						if (File_Limit)
						{
							fscanf(D2VFile, "Location=%d,%X,%d,%X\n", &process.leftfile, 
								&process.leftlba, &process.rightfile, &process.rightlba);

							process.startfile = process.leftfile;
							process.startloc = process.leftlba * BUFFER_SIZE;
							process.endfile = process.rightfile;
							process.endloc = (process.rightlba - 1) * BUFFER_SIZE;

							process.run = 0;
							for (i=0; i<process.startfile; i++)
								process.run += process.length[i];
							process.start = process.run + process.startloc;

							process.end = 0;
							for (i=0; i<process.endfile; i++)
								process.end += process.length[i];
							process.end += process.endloc;

							process.trackleft = (int)(process.start*TRACK_PITCH/process.total);
							process.trackright = (int)(process.end*TRACK_PITCH/process.total);

							process.locate = LOCATE_INIT;

							if (!threadId || WaitForSingleObject(hThread, INFINITE)==WAIT_OBJECT_0)
								hThread = CreateThread(NULL, 0, MPEG2Dec, 0, 0, &threadId);
						}
					}
					break;

				case IDM_STOP:
					Stop_Flag = true;

					if (Pause_Flag)
						ResumeThread(hThread);
					break;

				case IDM_STORE_RGB24:
					if (Store_Flag!=STORE_RGB24)
					{
						if (DDOverlay_Flag)
							IDirectDrawSurface_UpdateOverlay(lpOverlay, NULL, lpPrimary, NULL, DDOVER_HIDE, NULL);

						Store_Flag = STORE_RGB24;
						CheckMenuItem(hMenu, IDM_STORE_RGB24, MF_CHECKED);
						CheckMenuItem(hMenu, IDM_STORE_YUY2, MF_UNCHECKED);
						EnableMenuItem(hMenu, IDM_BMP, MF_ENABLED);

						RefreshWindow();
					}
					break;

				case IDM_STORE_YUY2:
					if (Store_Flag!=STORE_YUY2)
					{
						Store_Flag = STORE_YUY2;
						CheckMenuItem(hMenu, IDM_STORE_RGB24, MF_UNCHECKED);
						CheckMenuItem(hMenu, IDM_STORE_YUY2, MF_CHECKED);
						EnableMenuItem(hMenu, IDM_BMP, MF_GRAYED);

						if (DDOverlay_Flag)
						{
							GetClientRect(hWnd, &crect);
							crect.bottom -= TRACK_HEIGHT;
							FillRect(hDC, &crect, hBrush);
							crect.bottom += TRACK_HEIGHT;
						}

						RefreshWindow();
					}
					break;

				case IDM_TRACK_NONE:
					ClearTrack();
					Track_Flag = TRACK_NONE;
					CheckMenuItem(hMenu, IDM_TRACK_NONE, MF_CHECKED);
					break;

				case IDM_TRACK_1:
					ClearTrack();
					Track_Flag = TRACK_1;
					CheckMenuItem(hMenu, IDM_TRACK_1, MF_CHECKED);
					break;

				case IDM_TRACK_2:
					ClearTrack();
					Track_Flag = TRACK_2;
					CheckMenuItem(hMenu, IDM_TRACK_2, MF_CHECKED);
					break;

				case IDM_TRACK_3:
					ClearTrack();
					Track_Flag = TRACK_3;
					CheckMenuItem(hMenu, IDM_TRACK_3, MF_CHECKED);
					break;

				case IDM_TRACK_4:
					ClearTrack();
					Track_Flag = TRACK_4;
					CheckMenuItem(hMenu, IDM_TRACK_4, MF_CHECKED);
					break;

				case IDM_TRACK_5:
					ClearTrack();
					Track_Flag = TRACK_5;
					CheckMenuItem(hMenu, IDM_TRACK_5, MF_CHECKED);
					break;

				case IDM_TRACK_6:
					ClearTrack();
					Track_Flag = TRACK_6;
					CheckMenuItem(hMenu, IDM_TRACK_6, MF_CHECKED);
					break;

				case IDM_TRACK_7:
					ClearTrack();
					Track_Flag = TRACK_7;
					CheckMenuItem(hMenu, IDM_TRACK_7, MF_CHECKED);
					break;

				case IDM_TRACK_8:
					ClearTrack();
					Track_Flag = TRACK_8;
					CheckMenuItem(hMenu, IDM_TRACK_8, MF_CHECKED);
					break;

				case IDM_AC3:
					Format_Flag = FORMAT_AC3;
					CheckMenuItem(hMenu, IDM_AC3, MF_CHECKED);
					CheckMenuItem(hMenu, IDM_MPA, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_LPCM, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_SELECT, MF_UNCHECKED);
					break;

				case IDM_MPA:
					Format_Flag = FORMAT_MPA;
					CheckMenuItem(hMenu, IDM_AC3, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_MPA, MF_CHECKED);
					CheckMenuItem(hMenu, IDM_LPCM, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_SELECT, MF_UNCHECKED);
					break;

				case IDM_LPCM:
					Format_Flag = FORMAT_LPCM;
					CheckMenuItem(hMenu, IDM_AC3, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_MPA, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_LPCM, MF_CHECKED);
					CheckMenuItem(hMenu, IDM_SELECT, MF_UNCHECKED);
					break;

				case IDM_AC3_DEMUXALL:
					AC3_Flag = AUDIO_DEMUXALL;
					CheckMenuItem(hMenu, IDM_AC3_DEMUXALL, MF_CHECKED);
					CheckMenuItem(hMenu, IDM_AC3_DEMUXONE, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_AC3_DECODE, MF_UNCHECKED);
					break;

				case IDM_AC3_DEMUXONE:
					AC3_Flag = AUDIO_DEMUXONE;
					CheckMenuItem(hMenu, IDM_AC3_DEMUXALL, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_AC3_DEMUXONE, MF_CHECKED);
					CheckMenuItem(hMenu, IDM_AC3_DECODE, MF_UNCHECKED);
					break;

				case IDM_AC3_DECODE:
					AC3_Flag = AUDIO_DECODE;
					CheckMenuItem(hMenu, IDM_AC3_DEMUXALL, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_AC3_DEMUXONE, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_AC3_DECODE, MF_CHECKED);
					break;

				case IDM_DRC_NONE:
					DRC_Flag = DRC_NONE;
					CheckMenuItem(hMenu, IDM_DRC_NONE, MF_CHECKED);
					CheckMenuItem(hMenu, IDM_DRC_LIGHT, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_DRC_NORMAL, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_DRC_HEAVY, MF_UNCHECKED);
					break;

				case IDM_DRC_LIGHT:
					DRC_Flag = DRC_LIGHT;
					CheckMenuItem(hMenu, IDM_DRC_NONE, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_DRC_LIGHT, MF_CHECKED);
					CheckMenuItem(hMenu, IDM_DRC_NORMAL, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_DRC_HEAVY, MF_UNCHECKED);
					break;

				case IDM_DRC_NORMAL:
					DRC_Flag = DRC_NORMAL;
					CheckMenuItem(hMenu, IDM_DRC_NONE, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_DRC_LIGHT, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_DRC_NORMAL, MF_CHECKED);
					CheckMenuItem(hMenu, IDM_DRC_HEAVY, MF_UNCHECKED);
					break;

				case IDM_DRC_HEAVY:
					DRC_Flag = DRC_HEAVY;
					CheckMenuItem(hMenu, IDM_DRC_NONE, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_DRC_LIGHT, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_DRC_NORMAL, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_DRC_HEAVY, MF_CHECKED);
					break;

				case IDM_DSDOWN:
					if (DSDown_Flag)
						CheckMenuItem(hMenu, IDM_DSDOWN, MF_UNCHECKED);
					else
						CheckMenuItem(hMenu, IDM_DSDOWN, MF_CHECKED);

					DSDown_Flag = !DSDown_Flag;
					break;

				case IDM_PRESCALE:
					if (PreScale_Ratio!=1.0 || !Check_Flag || !IsWindowEnabled(hTrack))
					{
						CheckMenuItem(hMenu, IDM_PRESCALE, MF_UNCHECKED);
						PreScale_Ratio = 1.0;
					}
					else
					{
						Decision_Flag = true;
						Display_Flag = false;

						EnableMenuItem(hMenu, IDM_OPEN, MF_GRAYED);
						EnableMenuItem(hMenu, IDM_SAVE, MF_GRAYED);
						EnableMenuItem(hMenu, IDM_LOAD_D2V, MF_GRAYED);
						EnableMenuItem(hMenu, IDM_SAVE_D2V, MF_GRAYED);
						EnableMenuItem(hMenu, IDM_STOP, MF_ENABLED);

						DisableControl(true);
						ShowStatistics(true);

						process.locate = LOCATE_RIP;
						PreScale_Ratio = 1.0;

						if (WaitForSingleObject(hThread, INFINITE)==WAIT_OBJECT_0)
							hThread = CreateThread(NULL, 0, MPEG2Dec, 0, 0, &threadId);
					}
					break;

				case IDM_MPA_DEMUXALL:
					MPA_Flag = AUDIO_DEMUXALL;
					CheckMenuItem(hMenu, IDM_MPA_DEMUXALL, MF_CHECKED);
					CheckMenuItem(hMenu, IDM_MPA_DEMUXONE, MF_UNCHECKED);
					break;

				case IDM_MPA_DEMUXONE:
					MPA_Flag = AUDIO_DEMUXONE;
					CheckMenuItem(hMenu, IDM_MPA_DEMUXALL, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_MPA_DEMUXONE, MF_CHECKED);
					break;

				case IDM_SELECT:
					Format_Flag = FORMAT_AUTO;
					CheckMenuItem(hMenu, IDM_AC3, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_MPA, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_LPCM, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_SELECT, MF_CHECKED);
					break;

				case IDM_IDCT_MMX:
					iDCT_Flag = IDCT_MMX;
					CheckMenuItem(hMenu, IDM_IDCT_MMX, MF_CHECKED);
					CheckMenuItem(hMenu, IDM_IDCT_SSEMMX, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_IDCT_FPU, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_IDCT_REF, MF_UNCHECKED);
					break;

				case IDM_IDCT_SSEMMX:
					iDCT_Flag = IDCT_SSEMMX;
					CheckMenuItem(hMenu, IDM_IDCT_MMX, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_IDCT_SSEMMX, MF_CHECKED);
					CheckMenuItem(hMenu, IDM_IDCT_FPU, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_IDCT_REF, MF_UNCHECKED);
					break;

				case IDM_IDCT_FPU:
					iDCT_Flag = IDCT_FPU;
					CheckMenuItem(hMenu, IDM_IDCT_MMX, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_IDCT_SSEMMX, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_IDCT_FPU, MF_CHECKED);
					CheckMenuItem(hMenu, IDM_IDCT_REF, MF_UNCHECKED);
					break;

				case IDM_IDCT_REF:
					iDCT_Flag = IDCT_REF;
					CheckMenuItem(hMenu, IDM_IDCT_MMX, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_IDCT_SSEMMX, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_IDCT_FPU, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_IDCT_REF, MF_CHECKED);
					break;

				case IDM_FO_NONE:
					FO_Flag = FO_NONE;
					CheckMenuItem(hMenu, IDM_FO_NONE, MF_CHECKED);
					CheckMenuItem(hMenu, IDM_FO_FILM, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_FO_SWAP, MF_UNCHECKED);
					SetDlgItemText(hDlg, IDC_INFO, "");
					break;

				case IDM_FO_FILM:
					FO_Flag = FO_FILM;
					CheckMenuItem(hMenu, IDM_FO_NONE, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_FO_FILM, MF_CHECKED);
					CheckMenuItem(hMenu, IDM_FO_SWAP, MF_UNCHECKED);
					SetDlgItemText(hDlg, IDC_INFO, "");
					break;

				case IDM_FO_SWAP:
					FO_Flag = FO_SWAP;
					CheckMenuItem(hMenu, IDM_FO_NONE, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_FO_FILM, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_FO_SWAP, MF_CHECKED);
					SetDlgItemText(hDlg, IDC_INFO, "");
					break;

				case IDM_TVSCALE:
					RGB_Scale = 0x1000200010002000;
					RGB_Offset = 0x0000000000000000;
					RGB_CBU = 0x000038B4000038B4;
					RGB_CGX = 0xF4FDE926F4FDE926;
					RGB_CRV = 0x00002CDD00002CDD;

					Scale_Flag = false;
					CheckMenuItem(hMenu, IDM_TVSCALE, MF_CHECKED);
					CheckMenuItem(hMenu, IDM_PCSCALE, MF_UNCHECKED);

					RefreshWindow();
					break;

				case IDM_PCSCALE:
					RGB_Scale = 0x1000254310002543;
					RGB_Offset = 0x0010001000100010;
					RGB_CBU = 0x0000408D0000408D;
					RGB_CGX = 0xF377E5FCF377E5FC;
					RGB_CRV = 0x0000331300003313;

					Scale_Flag = true;
					CheckMenuItem(hMenu, IDM_TVSCALE, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_PCSCALE, MF_CHECKED);

					RefreshWindow();
					break;

				case IDM_CLIPRESIZE:
					if (hClipResizeDlg!=NULL)
					{
						DestroyWindow(hClipResizeDlg);
						hClipResizeDlg = NULL;
					}
					else
						hClipResizeDlg = CreateDialog(hInst, (LPCTSTR)IDD_CLIPRESIZE, hWnd, (DLGPROC)ClipResize);
					break;

				case IDM_LUMINANCE:
					if (hLumDlg!=NULL)
					{
						DestroyWindow(hLumDlg);
						hLumDlg = NULL;
					}
					else
						hLumDlg = CreateDialog(hInst, (LPCTSTR)IDD_LUMINANCE, hWnd, (DLGPROC)Luminance);
					break;

				case IDM_NORM:
					if (hNormDlg!=NULL)
					{
						DestroyWindow(hNormDlg);
						hNormDlg = NULL;
					}
					else
						hNormDlg = CreateDialog(hInst, (LPCTSTR)IDD_NORM, hWnd, (DLGPROC)Normalization);
					break;

				case IDM_PP_HIGH:
					Priority_Flag = PRIORITY_HIGH;
					SetPriorityClass(hProcess, HIGH_PRIORITY_CLASS);
					CheckMenuItem(hMenu, IDM_PP_HIGH, MF_CHECKED);
					CheckMenuItem(hMenu, IDM_PP_NORMAL, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_PP_LOW, MF_UNCHECKED);
					break;

				case IDM_PP_NORMAL:
					Priority_Flag = PRIORITY_NORMAL;
					SetPriorityClass(hProcess, NORMAL_PRIORITY_CLASS);
					CheckMenuItem(hMenu, IDM_PP_HIGH, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_PP_NORMAL, MF_CHECKED);
					CheckMenuItem(hMenu, IDM_PP_LOW, MF_UNCHECKED);
					break;

				case IDM_PP_LOW:
					Priority_Flag = PRIORITY_LOW;
					SetPriorityClass(hProcess, IDLE_PRIORITY_CLASS);
					CheckMenuItem(hMenu, IDM_PP_HIGH, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_PP_NORMAL, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_PP_LOW, MF_CHECKED);
					break;

				case IDM_PAUSE:
					if (Pause_Flag)
						ResumeThread(hThread);
					else
						SuspendThread(hThread);

					Pause_Flag = !Pause_Flag;
					break;

				case IDM_STATISTICS:
					if (Statistics_Flag)
					{
						DestroyWindow(hDlg);
						Statistics_Flag = false;
						hDlg = NULL;
					}
					else
						ShowStatistics(true);
					break;

				case IDM_DISPLAY:
					Display_Flag = !Display_Flag;
					break;

				case IDM_BMP:
					int width, height;

					width = Coded_Picture_Width;
					height = Coded_Picture_Height;

					if (ClipResize_Flag)
					{
						width -= Clip_Left+Clip_Right;
						height -= Clip_Top+Clip_Bottom;
					}

					SYSTEMTIME st;
					GetLocalTime(&st);

					sprintf(szTemp, "%sDVD2AVI %02d%02d%02d%02d%02d.bmp", szPath, st.wMonth, st.wDay, st.wHour, st.wMinute, st.wSecond);
					BMPFile = fopen(szTemp, "wb");

					i = fwrite(BMPHeader, 1, sizeof(BMPHeader), BMPFile);
					i += fwrite(rgb24, 1, width * height * 3, BMPFile);

					j = i & 3;

					while (j>0)
					{
						i += fputc(0, BMPFile);
						j--;
					}

					fseek(BMPFile, 2, SEEK_SET);
					fwrite(&i, sizeof(int), 1, BMPFile);
					fseek(BMPFile, 18, SEEK_SET);
					fwrite(&width, sizeof(int), 1, BMPFile);
					fwrite(&height, sizeof(int), 1, BMPFile);
					
					fclose(BMPFile);
					break;

				case IDM_SRC_NONE:
					SRC_Flag = SRC_NONE;
					CheckMenuItem(hMenu, IDM_SRC_NONE, MF_CHECKED);
					CheckMenuItem(hMenu, IDM_SRC_LOW, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_SRC_MID, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_SRC_HIGH, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_SRC_UHIGH, MF_UNCHECKED);
					break;

				case IDM_SRC_LOW:
					SRC_Flag = SRC_LOW;
					CheckMenuItem(hMenu, IDM_SRC_NONE, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_SRC_LOW, MF_CHECKED);
					CheckMenuItem(hMenu, IDM_SRC_MID, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_SRC_HIGH, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_SRC_UHIGH, MF_UNCHECKED);
					break;

				case IDM_SRC_MID:
					SRC_Flag = SRC_MID;
					CheckMenuItem(hMenu, IDM_SRC_NONE, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_SRC_LOW, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_SRC_MID, MF_CHECKED);
					CheckMenuItem(hMenu, IDM_SRC_HIGH, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_SRC_UHIGH, MF_UNCHECKED);
					break;

				case IDM_SRC_HIGH:
					SRC_Flag = SRC_HIGH;
					CheckMenuItem(hMenu, IDM_SRC_NONE, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_SRC_LOW, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_SRC_MID, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_SRC_HIGH, MF_CHECKED);
					CheckMenuItem(hMenu, IDM_SRC_UHIGH, MF_UNCHECKED);
					break;

				case IDM_SRC_UHIGH:
					SRC_Flag = SRC_UHIGH;
					CheckMenuItem(hMenu, IDM_SRC_NONE, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_SRC_LOW, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_SRC_MID, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_SRC_HIGH, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_SRC_UHIGH, MF_CHECKED);
					break;

				case IDM_KEY_OFF:
					KeyOp_Flag = KEY_OFF;
					CheckMenuItem(hMenu, IDM_KEY_OFF, MF_CHECKED);
					CheckMenuItem(hMenu, IDM_KEY_INPUT, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_KEY_OP, MF_UNCHECKED);
					break;

				case IDM_KEY_INPUT:
					KeyOp_Flag = KEY_INPUT;
					CheckMenuItem(hMenu, IDM_KEY_OFF, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_KEY_INPUT, MF_CHECKED);
					CheckMenuItem(hMenu, IDM_KEY_OP, MF_UNCHECKED);
					break;

				case IDM_KEY_OP:
					KeyOp_Flag = KEY_OP;
					CheckMenuItem(hMenu, IDM_KEY_OFF, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_KEY_INPUT, MF_UNCHECKED);
					CheckMenuItem(hMenu, IDM_KEY_OP, MF_CHECKED);
					break;

				case IDM_ABOUT:
					DialogBox(hInst, (LPCTSTR)IDD_ABOUT, hWnd, (DLGPROC)About);
					break;

				case IDM_HOMEPAGE:
					ShellExecute(NULL, "open", "http://hiroko.ee.ntu.edu.tw/", NULL, NULL, SW_SHOWNORMAL);
					break;

				case IDM_EXIT:
					DestroyWindow(hWnd);
					break;

				case ID_LEFT_BUTTON:
					if (IsWindowEnabled(hTrack))
					{
						SetFocus(hWnd);

						if ((process.file < process.rightfile) || (process.file==process.rightfile && process.lba<process.rightlba))
						{
							process.leftfile = process.file;
							process.leftlba = process.lba;

							process.run = 0;
							for (i=0; i<process.leftfile; i++)
								process.run += process.length[i];
							process.trackleft = (int)((process.run + process.leftlba * BUFFER_SIZE) * TRACK_PITCH / process.total);

							SendMessage(hTrack, TBM_SETPOS, (WPARAM) true, process.trackleft);
							SendMessage(hTrack, TBM_SETSEL, (WPARAM) true, (LPARAM) MAKELONG(process.trackleft, process.trackright));
						}
					}
					break;

				case ID_LEFT_ARROW:
					SetFocus(hWnd);

					if (WaitForSingleObject(hThread, 0)==WAIT_OBJECT_0)
					{
						Display_Flag = true;

						process.locate = LOCATE_BACKWARD;
						hThread = CreateThread(NULL, 0, MPEG2Dec, 0, 0, &threadId);
					}
					break;

				case ID_RIGHT_ARROW:
					SetFocus(hWnd);

					if (WaitForSingleObject(hThread, 0)==WAIT_OBJECT_0)
					{
						Display_Flag = true;

						process.locate = LOCATE_FORWARD;
						hThread = CreateThread(NULL, 0, MPEG2Dec, 0, 0, &threadId);
					}
					break;

				case ID_RIGHT_BUTTON:
					if (IsWindowEnabled(hTrack))
					{
						SetFocus(hWnd);

						if ((process.file>process.leftfile) || (process.file==process.leftfile && process.lba>process.leftlba))
						{
							process.rightfile = process.file;
							process.rightlba = process.lba;

							process.run = 0;
							for (i=0; i<process.rightfile; i++)
								process.run += process.length[i];
							process.trackright = (int)((process.run + (__int64)process.rightlba*BUFFER_SIZE)*TRACK_PITCH/process.total);

							SendMessage(hTrack, TBM_SETPOS, (WPARAM) true, process.trackright);
							SendMessage(hTrack, TBM_SETSEL, (WPARAM) true, (LPARAM) MAKELONG(process.trackleft, process.trackright));
						}
					}
					break;

				default:
					return DefWindowProc(hWnd, message, wParam, lParam);
			}
			break;

		case WM_HSCROLL:
			SetFocus(hWnd);

			if (WaitForSingleObject(hThread, 0)==WAIT_OBJECT_0)
			{
				Display_Flag = true;

				process.startloc = process.start = process.total*SendMessage(hTrack, TBM_GETPOS, 0, 0)/TRACK_PITCH;

				process.startfile = 0; process.run = 0;
				while (process.startloc > process.length[process.startfile])
				{
					process.startloc -= process.length[process.startfile];
					process.run += process.length[process.startfile];
					process.startfile++;
				}

				process.end = process.total - BUFFER_SIZE;
				process.endfile = File_Limit - 1;
				process.endloc = (process.length[File_Limit-1]/BUFFER_SIZE-1)*BUFFER_SIZE;

				process.locate = LOCATE_SCROLL;

				hThread = CreateThread(NULL, 0, MPEG2Dec, 0, 0, &threadId);
			}
			break;

		case WM_KEYDOWN:
			if (IsWindowEnabled(hTrack))
			{
				switch (wParam)
				{
					case VK_LEFT:
						if (WaitForSingleObject(hThread, 0)==WAIT_OBJECT_0)
						{
							Display_Flag = true;

							process.locate = LOCATE_BACKWARD;
							hThread = CreateThread(NULL, 0, MPEG2Dec, 0, 0, &threadId);
						}
						break;

					case VK_RIGHT:
						if (WaitForSingleObject(hThread, 0)==WAIT_OBJECT_0)
						{
							Display_Flag = true;

							process.locate = LOCATE_FORWARD;
							hThread = CreateThread(NULL, 0, MPEG2Dec, 0, 0, &threadId);
						}
						break;
				}
			}
			break;

		case WM_SIZE:
			if (!IsIconic(hWnd))
				ShowStatistics(false);
			else if (DDOverlay_Flag && Check_Flag)
				IDirectDrawSurface_UpdateOverlay(lpOverlay, NULL, lpPrimary, NULL, DDOVER_HIDE, NULL);
			break;

		case WM_MOVE:
			if (!IsIconic(hWnd))
			{
				ShowStatistics(false);

				if (Check_Flag)
				{
					if (DDOverlay_Flag && Store_Flag==STORE_YUY2)
						RenderYUY2();
					else
						RenderRGB24();
				}
			}
			break;

		case WM_PAINT:
			hdc = BeginPaint(hWnd, &ps);
			EndPaint(hWnd, &ps);
			ReleaseDC(hWnd, hdc);

			if (Check_Flag && Store_Flag==STORE_RGB24)
				RenderRGB24();
			break;

		case WM_DROPFILES:
			char seq, *ext;

			DragQueryFile((HDROP)wParam, 0, szInput, sizeof(szInput));
			DragFinish((HDROP)wParam);
			SetForegroundWindow(hWnd);

			ext = strrchr(szInput, '.');
			if (ext!=NULL)
			{
				if (!_strnicmp(ext, ".d2v", 4))
					goto D2V_PROCESS;

				if (!_strnicmp(ext, ".wav", 4))
				{
					if (Check_Flag)
						break;

					if (!CheckWAV())
					{
						DialogBox(hInst, (LPCTSTR)IDD_ERROR, hWnd, (DLGPROC)About);
						break;
					}

					if (PopFileDlg(szOutput, hWnd, SAVE_WAV))
					{
						DialogBox(hInst, (LPCTSTR)IDD_DELAY, hWnd, (DLGPROC)Delay);
						ShowStatistics(true);

						if (!threadId || WaitForSingleObject(hThread, INFINITE)==WAIT_OBJECT_0)
							hThread = CreateThread(NULL, 0, ProcessWAV, 0, 0, &threadId);
					}
					break;
				}
			}

			while (File_Limit)
			{
				File_Limit--;
				_close(Infile[File_Limit]);
			}

			while (_findfirst(szInput, &seqfile) != -1L)
			{
				strcpy(Infilename[File_Limit], szInput);
				Infile[File_Limit] = _open(szInput, _O_RDONLY | _O_BINARY);
				File_Limit++;

				strncpy(&seq, szInput+strlen(szInput)-5, 1);
				if (seq>='0' && seq<'9')
					seq++;
				else
					break;
				strncpy(szInput+strlen(szInput)-5, &seq, 1);
			}

			Recovery();

			if (File_Limit)
			{
				process.rightfile = File_Limit-1;
				process.rightlba = (int)(process.length[File_Limit-1]/BUFFER_SIZE);

				process.end = process.total - BUFFER_SIZE;
				process.endfile = File_Limit - 1;
				process.endloc = (process.length[File_Limit-1]/BUFFER_SIZE - 1)*BUFFER_SIZE;

				process.locate = LOCATE_INIT;

				if (!threadId || WaitForSingleObject(hThread, INFINITE)==WAIT_OBJECT_0)
					hThread = CreateThread(NULL, 0, MPEG2Dec, 0, 0, &threadId);
			}
			break;

		case WM_DESTROY:
			strcat(szPath, "DVD2AVI.ini");
			GetWindowRect(hWnd, &wrect);

			if ((INIFile = fopen(szPath, "w")) != NULL)
			{
				fprintf(INIFile, "INI_Version=%d\n", INI_Version);
				fprintf(INIFile, "Window_Position=%d,%d\n", wrect.left, wrect.top);
				fprintf(INIFile, "iDCT_Algorithm=%d\n", iDCT_Flag);
				fprintf(INIFile, "YUVRGB_Scale=%d\n", Scale_Flag);
				fprintf(INIFile, "Field_Operation=%d\n", FO_Flag);
				fprintf(INIFile, "Track_Number=%d\n", Track_Flag);
				fprintf(INIFile, "Channel_Format=%d\n", Format_Flag);
				fprintf(INIFile, "AC3=%d\n", AC3_Flag);
				fprintf(INIFile, "DR_Control=%d\n", DRC_Flag);
				fprintf(INIFile, "DS_Downmix=%d\n", DSDown_Flag);
				fprintf(INIFile, "MPA=%d\n", MPA_Flag);
				fprintf(INIFile, "SRC_Precision=%d\n", SRC_Flag);
				fprintf(INIFile, "Norm_Ratio=%d\n", 100 * Normalization_Flag + Norm_Ratio);
				fprintf(INIFile, "Process_Priority=%d\n", Priority_Flag);

				fclose(INIFile);
			}

			while (File_Limit)
			{
				File_Limit--;
				_close(Infile[File_Limit]);
			}

			Recovery();

			for (i=0; i<8; i++)
				free(p_block[i]);
			free(p_fTempArray);

			for (i=0; i<MAX_FILE_NUMBER; i++)
				free(Infilename[i]);

			if (hLibrary)
				FreeLibrary(hLibrary);

			ReleaseDC(hWnd, hDC);
			DeleteObject(hBrush);

			PostQuitMessage(0);
			break;

		default:
			return DefWindowProc(hWnd, message, wParam, lParam);
	}
	return false;
}

LRESULT CALLBACK VideoList(HWND hVideoListDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	int i, j;

	switch (message)
	{
		case WM_INITDIALOG:
			if (File_Limit)
				for (i=0; i<File_Limit; i++)
					SendDlgItemMessage(hVideoListDlg, IDC_LIST, LB_ADDSTRING, 0, (LPARAM)Infilename[i]);
			else
				OpenVideoFile(hVideoListDlg);

			if (File_Limit)
				SendDlgItemMessage(hVideoListDlg, IDC_LIST, LB_SETCURSEL, File_Limit-1, 0);
			return true;

		case WM_COMMAND:
			switch (LOWORD(wParam))
			{
				case ID_ADD:
					OpenVideoFile(hVideoListDlg);

					if (File_Limit)
						SendDlgItemMessage(hVideoListDlg, IDC_LIST, LB_SETCURSEL, File_Limit-1, 0);
					break;

				case ID_DEL:
					if (File_Limit)
					{
						i= SendDlgItemMessage(hVideoListDlg, IDC_LIST, LB_GETCURSEL, 0, 0);
						SendDlgItemMessage(hVideoListDlg, IDC_LIST, LB_DELETESTRING, i, 0);

						File_Limit--;
						_close(Infile[i]);

						for (j=i; j<File_Limit; j++)
						{
							Infile[j] = Infile[j+1];
							strcpy(Infilename[j], Infilename[j+1]);
						}

						SendDlgItemMessage(hVideoListDlg, IDC_LIST, LB_SETCURSEL, i>=File_Limit ? File_Limit-1 : i, 0);
					}
					break;

				case IDOK:
				case IDCANCEL:
					EndDialog(hVideoListDlg, 0);
					Recovery();

					if (File_Limit)
					{
						process.rightfile = File_Limit-1;
						process.rightlba = (int)(process.length[File_Limit-1]/BUFFER_SIZE);

						process.end = process.total - BUFFER_SIZE;
						process.endfile = File_Limit - 1;
						process.endloc = (process.length[File_Limit-1]/BUFFER_SIZE - 1)*BUFFER_SIZE;

						process.locate = LOCATE_INIT;

						if (!threadId || WaitForSingleObject(hThread, INFINITE)==WAIT_OBJECT_0)
							hThread = CreateThread(NULL, 0, MPEG2Dec, 0, 0, &threadId);
					}
					return true;
			}
			break;
	}
    return false;
}

static void OpenVideoFile(HWND hVideoListDlg)
{
	char seq;

	if (PopFileDlg(szInput, hVideoListDlg, OPEN_VOB))
	{
		while (_findfirst(szInput, &seqfile) != -1L)
		{
			SendDlgItemMessage(hVideoListDlg, IDC_LIST, LB_ADDSTRING, 0, (LPARAM)szInput);
			strcpy(Infilename[File_Limit], szInput);
			Infile[File_Limit] = _open(szInput, _O_RDONLY | _O_BINARY);
			File_Limit++;

			strncpy(&seq, szInput+strlen(szInput)-5, 1);
			if (seq>='0' && seq<'9')
				seq++;
			else
				break;
			strncpy(szInput+strlen(szInput)-5, &seq, 1);
		}
	}
}

LRESULT CALLBACK AudioList(HWND hAudioListDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	int i, j;

	switch (message)
	{
		case WM_INITDIALOG:
			File_Limit = 0;
			OpenAudioFile(hAudioListDlg);
			return true;

		case WM_COMMAND:
			switch (LOWORD(wParam))
			{
				case ID_ADD:
					OpenAudioFile(hAudioListDlg);
					break;

				case ID_DEL:
					if (File_Limit)
					{
						i= SendDlgItemMessage(hAudioListDlg, IDC_LIST, LB_GETCURSEL, 0, 0);
						SendDlgItemMessage(hAudioListDlg, IDC_LIST, LB_DELETESTRING, i, 0);

						File_Limit--;

						for (j=i; j<File_Limit; j++)
						{
							strcpy(Infilename[j], Infilename[j+1]);
							strcpy(Outfilename[j], Outfilename[j+1]);
							SoundDelay[j] = SoundDelay[j+1];
						}

						SendDlgItemMessage(hAudioListDlg, IDC_LIST, LB_SETCURSEL, i>=File_Limit ? File_Limit-1 : i, 0);
					}
					break;

				case IDOK:
				case IDCANCEL:
					EndDialog(hAudioListDlg, 0);

					if (File_Limit)
					{
						ShowStatistics(true);

						if (!threadId || WaitForSingleObject(hThread, INFINITE)==WAIT_OBJECT_0)
							hThread = CreateThread(NULL, 0, ProcessWAV, (void *)File_Limit, 0, &threadId);
					}
					return true;
			}
			break;
	}
    return false;
}

static void OpenAudioFile(HWND hAudioListDlg)
{
	if (PopFileDlg(szInput, hAudioListDlg, OPEN_WAV))
	{
		if (!CheckWAV())
		{
			DialogBox(hInst, (LPCTSTR)IDD_ERROR, hAudioListDlg, (DLGPROC)About);
			return;
		}

		if (PopFileDlg(szOutput, hAudioListDlg, SAVE_WAV))
		{
			strcpy(Infilename[File_Limit], szInput);
			strcpy(Outfilename[File_Limit], szOutput);
			DialogBox(hInst, (LPCTSTR)IDD_DELAY, hWnd, (DLGPROC)Delay);
			sprintf(szBuffer, "%s %dms", szInput, SoundDelay[File_Limit]);
			SendDlgItemMessage(hAudioListDlg, IDC_LIST, LB_ADDSTRING, 0, (LPARAM)szBuffer);
			File_Limit++;
		}
	}

	if (File_Limit)
		SendDlgItemMessage(hAudioListDlg, IDC_LIST, LB_SETCURSEL, File_Limit-1, 0);
}

DWORD WINAPI ProcessWAV(LPVOID n)
{
	int i;

	Stop_Flag = Pause_Flag = false;

	EnableMenuItem(hMenu, IDM_OPEN, MF_GRAYED);
	EnableMenuItem(hMenu, IDM_PROCESS_WAV, MF_GRAYED);
	EnableMenuItem(hMenu, IDM_LOAD_D2V, MF_GRAYED);
	EnableMenuItem(hMenu, IDM_STOP, MF_ENABLED);
	EnableMenuItem(hMenu, 2, MF_BYPOSITION | MF_GRAYED);

	DragAcceptFiles(hWnd, false);

	if (!n)
		Wavefs44File(SoundDelay[0], 1, 1);
	else
		for (i=0; i<File_Limit && !Stop_Flag; i++)
		{
			strcpy(szInput, Infilename[i]);
			strcpy(szOutput, Outfilename[i]);
			Wavefs44File(SoundDelay[i], i+1, File_Limit);
		}

	File_Limit = 0;

	EnableMenuItem(hMenu, IDM_OPEN, MF_ENABLED);
	EnableMenuItem(hMenu, IDM_PROCESS_WAV, MF_ENABLED);
	EnableMenuItem(hMenu, IDM_LOAD_D2V, MF_ENABLED);
	EnableMenuItem(hMenu, IDM_STOP, MF_GRAYED);
	EnableMenuItem(hMenu, 2, MF_BYPOSITION | MF_ENABLED);

	if (!Stop_Flag)
	{
		MessageBeep(MB_OK);
		SetDlgItemText(hDlg, IDC_REMAIN, "FINISH");
	}

	SetForegroundWindow(hWnd);
	DragAcceptFiles(hWnd, true);
	return 0;
}

void ThreadKill()
{
	if ((AVI_Flag || D2V_Flag) && Track_Flag!=TRACK_NONE && 
		((ac3[Track_Flag].rip && AC3_Flag==AUDIO_DECODE) || pcm.rip))
	{
		if (SRC_Flag)
		{
			EndSRC(pcm.file);
			pcm.size = ((int)(0.91875*pcm.size)>>2)<<2;
		}

		Normalize(NULL, 44, pcm.filename, pcm.file, 44, pcm.size);
		EndWAV(pcm.file, pcm.size);
	}

	if (process.locate==LOCATE_INIT || process.locate==LOCATE_RIP)
	{
		if (D2V_Flag)
		{
			if (Stop_Flag)
				fprintf(D2VFile, " 9\n\nINTERRUPTED");
			else
				fprintf(D2VFile, " 9\n\nFINISHED");
		}

		_fcloseall();

		if (Decision_Flag)
		{
			if (Sound_Max > 1)
			{
				PreScale_Ratio = 327.68 * Norm_Ratio / Sound_Max;

				if (PreScale_Ratio > 1.0 && PreScale_Ratio < 1.01)
					PreScale_Ratio = 1.0;

				sprintf(szBuffer, "%.2f", PreScale_Ratio);
				SetDlgItemText(hDlg, IDC_INFO, szBuffer);

				CheckMenuItem(hMenu, IDM_PRESCALE, MF_CHECKED);
				CheckMenuItem(hMenu, IDM_NORM, MF_UNCHECKED);
				Normalization_Flag = false;
			}
			else
			{
				SetDlgItemText(hDlg, IDC_INFO, "N.A.");
				CheckMenuItem(hMenu, IDM_PRESCALE, MF_UNCHECKED);
			}
		}

		AVI_Flag = false;
		D2V_Flag = false;
		Decision_Flag = false;
		Display_Flag = false;

		EnableMenuItem(hMenu, IDM_OPEN, MF_ENABLED);
		EnableMenuItem(hMenu, IDM_SAVE, MF_ENABLED);
		EnableMenuItem(hMenu, IDM_LOAD_D2V, MF_ENABLED);
		EnableMenuItem(hMenu, IDM_SAVE_D2V, MF_ENABLED);
		EnableMenuItem(hMenu, IDM_STOP, MF_GRAYED);

		EnableMenuItem(hMenu, 1, MF_BYPOSITION | MF_ENABLED);
		EnableMenuItem(hMenu, 2, MF_BYPOSITION | MF_ENABLED);

		EnableWindow(hTrack, true);
		EnableWindow(hLeftButton, true);
		EnableWindow(hLeftArrow, true);
		EnableWindow(hRightArrow, true);
		EnableWindow(hRightButton, true);
		SendMessage(hTrack, TBM_SETSEL, (WPARAM) true, (LPARAM) MAKELONG(process.trackleft, process.trackright));
		DragAcceptFiles(hWnd, true);
	}

	if (process.locate==LOCATE_RIP)
	{
		SetForegroundWindow(hWnd);

		if (!Stop_Flag)
		{
			MessageBeep(MB_OK);	
			SetDlgItemText(hDlg, IDC_REMAIN, "FINISH");
		}
	}

	ExitThread(0);
}

LRESULT CALLBACK Statistics(HWND hStatisticsDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message)
	{
		case WM_INITDIALOG:
			SendDlgItemMessage(hStatisticsDlg, IDC_PROGRESS, PBM_SETRANGE, 0, MAKELPARAM(0, 10000));
			return true;

		case WM_COMMAND:
			if (LOWORD(wParam) == IDCANCEL)
			{
				DestroyWindow(hStatisticsDlg);
				Statistics_Flag = false;
				return true;
			}
	}
    return false;
}

LRESULT CALLBACK Delay(HWND hDelayDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message)
	{
		case WM_INITDIALOG:
			SetDlgItemText(hDelayDlg, IDC_DELAY, "0");
			return true;

		case WM_COMMAND:
			if (LOWORD(wParam) == IDOK || LOWORD(wParam) == IDCANCEL) 
			{
				SoundDelay[File_Limit] = GetDlgItemInt(hDelayDlg, IDC_DELAY, NULL, true);
				if (abs(SoundDelay[File_Limit]) > 10000000)
					SoundDelay[File_Limit] = 0;

				EndDialog(hDelayDlg, 0);
				return true;
			}
	}
    return false;
}

LRESULT CALLBACK About(HWND hAboutDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message)
	{
		case WM_INITDIALOG:
			return true;

		case WM_COMMAND:
			if (LOWORD(wParam) == IDOK || LOWORD(wParam) == IDCANCEL) 
			{
				EndDialog(hAboutDlg, 0);
				return true;
			}
	}
    return false;
}

LRESULT CALLBACK ClipResize(HWND hDialog, UINT message, WPARAM wParam, LPARAM lParam)
{
	int i;

	switch (message)
	{
		case WM_INITDIALOG:
			SendDlgItemMessage(hDialog, IDC_LEFT_SLIDER, TBM_SETRANGE, 0, MAKELPARAM(0, 128));
			SendDlgItemMessage(hDialog, IDC_LEFT_SLIDER, TBM_SETPOS, 1, Clip_Left>>1);
			sprintf(szTemp, "%d", Clip_Left);
			SetDlgItemText(hDialog, IDC_LEFT, szTemp);

			SendDlgItemMessage(hDialog, IDC_RIGHT_SLIDER, TBM_SETRANGE, 0, MAKELPARAM(0, 128));
			SendDlgItemMessage(hDialog, IDC_RIGHT_SLIDER, TBM_SETPOS, 1, Clip_Right>>1);
			sprintf(szTemp, "%d", Clip_Right);
			SetDlgItemText(hDialog, IDC_RIGHT, szTemp);

			SendDlgItemMessage(hDialog, IDC_TOP_SLIDER, TBM_SETRANGE, 0, MAKELPARAM(0, 128));
			SendDlgItemMessage(hDialog, IDC_TOP_SLIDER, TBM_SETPOS, 1, Clip_Top>>1);
			sprintf(szTemp, "%d", Clip_Top);
			SetDlgItemText(hDialog, IDC_TOP, szTemp);

			SendDlgItemMessage(hDialog, IDC_BOTTOM_SLIDER, TBM_SETRANGE, 0, MAKELPARAM(0, 128));
			SendDlgItemMessage(hDialog, IDC_BOTTOM_SLIDER, TBM_SETPOS, 1, Clip_Bottom>>1);
			sprintf(szTemp, "%d", Clip_Bottom);
			SetDlgItemText(hDialog, IDC_BOTTOM, szTemp);

			SendDlgItemMessage(hDialog, IDC_WIDTH_SLIDER, TBM_SETRANGE, 0, MAKELPARAM(0, (Coded_Picture_Width-Clip_Left-Clip_Right)>>3));
			SendDlgItemMessage(hDialog, IDC_WIDTH_SLIDER, TBM_SETPOS, 1, (Coded_Picture_Width-Clip_Left-Clip_Right-Squeeze_Width)>>3);
			sprintf(szTemp, "%d", Coded_Picture_Width-Clip_Left-Clip_Right-Squeeze_Width);
			SetDlgItemText(hDialog, IDC_WIDTH, szTemp);

			SendDlgItemMessage(hDialog, IDC_HEIGHT_SLIDER, TBM_SETRANGE, 0, MAKELPARAM(0, (Coded_Picture_Height-Clip_Top-Clip_Bottom)>>3));
			SendDlgItemMessage(hDialog, IDC_HEIGHT_SLIDER, TBM_SETPOS, 1, (Coded_Picture_Height-Clip_Top-Clip_Bottom-Squeeze_Height)>>3);
			sprintf(szTemp, "%d", Coded_Picture_Height-Clip_Top-Clip_Bottom-Squeeze_Height);
			SetDlgItemText(hDialog, IDC_HEIGHT, szTemp);

			ShowWindow(hDialog, SW_SHOW);

			if (ClipResize_Flag)
				SendDlgItemMessage(hDialog, IDC_CLIPRESIZE_CHECK, BM_SETCHECK, BST_CHECKED, 0);
			return true;

		case WM_COMMAND:
			switch (LOWORD(wParam))
			{
				case IDC_CLIPRESIZE_CHECK:
					if (SendDlgItemMessage(hDialog, IDC_CLIPRESIZE_CHECK, BM_GETCHECK, 1, 0)==BST_CHECKED)
					{
						CheckMenuItem(hMenu, IDM_CLIPRESIZE, MF_CHECKED);
						ClipResize_Flag = true;
					}
					else
					{
						CheckMenuItem(hMenu, IDM_CLIPRESIZE, MF_UNCHECKED);
						ClipResize_Flag = false;
					}

					RefreshWindow();
					ShowStatistics(false);
					break;

				case IDCANCEL:
					DestroyWindow(hDialog);
					hClipResizeDlg = NULL;
					return true;
			}
			break;

		case WM_HSCROLL:
			switch (GetWindowLong((HWND)lParam, GWL_ID))
			{
				case IDC_LEFT_SLIDER:
					i = SendDlgItemMessage(hDialog, IDC_LEFT_SLIDER, TBM_GETPOS, 0, 0)<<1;
					if (i + Clip_Right + MIN_WIDTH <= Coded_Picture_Width)
					{
						Clip_Left = i;
						sprintf(szTemp, "%d", Clip_Left);
						SetDlgItemText(hDialog, IDC_LEFT, szTemp);

						Clip_Right = 8 - Clip_Left%8;
						sprintf(szTemp, "%d", Clip_Right);
						SetDlgItemText(hDialog, IDC_RIGHT, szTemp);
						SendDlgItemMessage(hDialog, IDC_RIGHT_SLIDER, TBM_SETPOS, 1, Clip_Right>>1);

						Squeeze_Width = 0;
						SendDlgItemMessage(hDialog, IDC_WIDTH_SLIDER, TBM_SETRANGE, 0, MAKELPARAM(0, (Coded_Picture_Width-Clip_Left-Clip_Right)>>3));
						SendDlgItemMessage(hDialog, IDC_WIDTH_SLIDER, TBM_SETPOS, 1, (Coded_Picture_Width-Clip_Left-Clip_Right-Squeeze_Width)>>3);
						sprintf(szTemp, "%d", Coded_Picture_Width-Clip_Left-Clip_Right-Squeeze_Width);
						SetDlgItemText(hDialog, IDC_WIDTH, szTemp);
					}
					break;

				case IDC_RIGHT_SLIDER:
					i = SendDlgItemMessage(hDialog, IDC_RIGHT_SLIDER, TBM_GETPOS, 0, 0)<<1;
					if (i + Clip_Left + MIN_WIDTH <= Coded_Picture_Width && (i + Clip_Left)%8 == 0)
					{
						Clip_Right = i;
						sprintf(szTemp, "%d", Clip_Right);
						SetDlgItemText(hDialog, IDC_RIGHT, szTemp);

						Squeeze_Width = 0;
						SendDlgItemMessage(hDialog, IDC_WIDTH_SLIDER, TBM_SETRANGE, 0, MAKELPARAM(0, (Coded_Picture_Width-Clip_Left-Clip_Right)>>3));
						SendDlgItemMessage(hDialog, IDC_WIDTH_SLIDER, TBM_SETPOS, 1, (Coded_Picture_Width-Clip_Left-Clip_Right-Squeeze_Width)>>3);
						sprintf(szTemp, "%d", Coded_Picture_Width-Clip_Left-Clip_Right-Squeeze_Width);
						SetDlgItemText(hDialog, IDC_WIDTH, szTemp);
					}
					break;

				case IDC_TOP_SLIDER:
					i = SendDlgItemMessage(hDialog, IDC_TOP_SLIDER, TBM_GETPOS, 0, 0)<<1;
					if (i + Clip_Bottom + MIN_HEIGHT <= Coded_Picture_Height)
					{
						Clip_Top = i;
						sprintf(szTemp, "%d", Clip_Top);
						SetDlgItemText(hDialog, IDC_TOP, szTemp);

						Clip_Bottom = 8 - Clip_Top%8;
						sprintf(szTemp, "%d", Clip_Bottom);
						SetDlgItemText(hDialog, IDC_BOTTOM, szTemp);
						SendDlgItemMessage(hDialog, IDC_BOTTOM_SLIDER, TBM_SETPOS, 1, Clip_Bottom>>1);

						Squeeze_Height = 0;
						SendDlgItemMessage(hDialog, IDC_HEIGHT_SLIDER, TBM_SETRANGE, 0, MAKELPARAM(0, (Coded_Picture_Height-Clip_Top-Clip_Bottom)>>3));
						SendDlgItemMessage(hDialog, IDC_HEIGHT_SLIDER, TBM_SETPOS, 1, (Coded_Picture_Height-Clip_Top-Clip_Bottom-Squeeze_Height)>>3);
						sprintf(szTemp, "%d", Coded_Picture_Height-Clip_Top-Clip_Bottom-Squeeze_Height);
						SetDlgItemText(hDialog, IDC_HEIGHT, szTemp);
					}
					break;

				case IDC_BOTTOM_SLIDER:
					i = SendDlgItemMessage(hDialog, IDC_BOTTOM_SLIDER, TBM_GETPOS, 0, 0)<<1;
					if (i + Clip_Top + MIN_HEIGHT <= Coded_Picture_Height && (i + Clip_Top)%8 == 0)
					{
						Clip_Bottom = i;
						sprintf(szTemp, "%d", Clip_Bottom);
						SetDlgItemText(hDialog, IDC_BOTTOM, szTemp);

						Squeeze_Height = 0;
						SendDlgItemMessage(hDialog, IDC_HEIGHT_SLIDER, TBM_SETRANGE, 0, MAKELPARAM(0, (Coded_Picture_Height-Clip_Top-Clip_Bottom)>>3));
						SendDlgItemMessage(hDialog, IDC_HEIGHT_SLIDER, TBM_SETPOS, 1, (Coded_Picture_Height-Clip_Top-Clip_Bottom-Squeeze_Height)>>3);
						sprintf(szTemp, "%d", Coded_Picture_Height-Clip_Top-Clip_Bottom-Squeeze_Height);
						SetDlgItemText(hDialog, IDC_HEIGHT, szTemp);
					}
					break;

				case IDC_WIDTH_SLIDER:
					i = Coded_Picture_Width - Clip_Left - Clip_Right - (SendDlgItemMessage(hDialog, IDC_WIDTH_SLIDER, TBM_GETPOS, 0, 0)<<3);
					if (MIN_WIDTH + i + Clip_Left + Clip_Right <= Coded_Picture_Width)
					{
						Squeeze_Width = i;
						sprintf(szTemp, "%d", Coded_Picture_Width - Clip_Left - Clip_Right - Squeeze_Width);
						SetDlgItemText(hDialog, IDC_WIDTH, szTemp);
					}
					break;

				case IDC_HEIGHT_SLIDER:
					i = Coded_Picture_Height - Clip_Top - Clip_Bottom - (SendDlgItemMessage(hDialog, IDC_HEIGHT_SLIDER, TBM_GETPOS, 0, 0)<<3);
					if (MIN_HEIGHT + i + Clip_Top + Clip_Bottom <= Coded_Picture_Height)
					{
						Squeeze_Height = i;
						sprintf(szTemp, "%d", Coded_Picture_Height - Clip_Top - Clip_Bottom - Squeeze_Height);
						SetDlgItemText(hDialog, IDC_HEIGHT, szTemp);
					}
					break;
			}

			RefreshWindow();
			ShowStatistics(false);
	}
    return false;
}

LRESULT CALLBACK Luminance(HWND hDialog, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message)
	{
		case WM_INITDIALOG:
			SendDlgItemMessage(hDialog, IDC_GAIN_SLIDER, TBM_SETRANGE, 0, MAKELPARAM(0, 256));
			SendDlgItemMessage(hDialog, IDC_GAIN_SLIDER, TBM_SETTICFREQ, 128, 0);
			SendDlgItemMessage(hDialog, IDC_GAIN_SLIDER, TBM_SETPOS, 1, LumGain);
			sprintf(szTemp, "%d", LumGain-128);
			SetDlgItemText(hDialog, IDC_GAIN, szTemp);

			SendDlgItemMessage(hDialog, IDC_OFFSET_SLIDER, TBM_SETRANGE, 0, MAKELPARAM(0, 512));
			SendDlgItemMessage(hDialog, IDC_OFFSET_SLIDER, TBM_SETTICFREQ, 256, 0);
			SendDlgItemMessage(hDialog, IDC_OFFSET_SLIDER, TBM_SETPOS, 1, LumOffset+256);
			sprintf(szTemp, "%d", LumOffset);
			SetDlgItemText(hDialog, IDC_OFFSET, szTemp);

			ShowWindow(hDialog, SW_SHOW);

			if (Luminance_Flag)
				SendDlgItemMessage(hDialog, IDC_LUM_CHECK, BM_SETCHECK, BST_CHECKED, 0);
			return true;

		case WM_COMMAND:
			switch (LOWORD(wParam))
			{
				case IDC_LUM_CHECK:
					if (SendDlgItemMessage(hDialog, IDC_LUM_CHECK, BM_GETCHECK, 1, 0)==BST_CHECKED)
					{
						CheckMenuItem(hMenu, IDM_LUMINANCE, MF_CHECKED);
						Luminance_Flag = true;
					}
					else
					{
						CheckMenuItem(hMenu, IDM_LUMINANCE, MF_UNCHECKED);
						Luminance_Flag = false;
					}
					
					RefreshWindow();
					break;

				case IDCANCEL:
					DestroyWindow(hDialog);
					hLumDlg = NULL;
					return true;
			}
			break;

		case WM_HSCROLL:
			switch (GetWindowLong((HWND)lParam, GWL_ID))
			{
				case IDC_GAIN_SLIDER:
					LumGain = SendDlgItemMessage(hDialog, IDC_GAIN_SLIDER, TBM_GETPOS, 0, 0);
					sprintf(szTemp, "%d", LumGain-128);
					SetDlgItemText(hDialog, IDC_GAIN, szTemp);	
					break;

				case IDC_OFFSET_SLIDER:
					LumOffset = SendDlgItemMessage(hDialog, IDC_OFFSET_SLIDER, TBM_GETPOS, 0, 0) - 256;
					sprintf(szTemp, "%d", LumOffset);
					SetDlgItemText(hDialog, IDC_OFFSET, szTemp);
					break;
			}

			RefreshWindow();
			break;
	}
    return false;
}

LRESULT CALLBACK Normalization(HWND hDialog, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message)
	{
		case WM_INITDIALOG:
			SendDlgItemMessage(hDialog, IDC_NORM_SLIDER, TBM_SETRANGE, 0, MAKELPARAM(0, 100));
			SendDlgItemMessage(hDialog, IDC_NORM_SLIDER, TBM_SETTICFREQ, 50, 0);
			SendDlgItemMessage(hDialog, IDC_NORM_SLIDER, TBM_SETPOS, 1, Norm_Ratio);
			sprintf(szTemp, "%d", Norm_Ratio);
			SetDlgItemText(hDialog, IDC_NORM, szTemp);

			ShowWindow(hDialog, SW_SHOW);

			if (Normalization_Flag)
				SendDlgItemMessage(hDialog, IDC_NORM_CHECK, BM_SETCHECK, BST_CHECKED, 0);
			return true;

		case WM_COMMAND:
			switch (LOWORD(wParam))
			{
				case IDC_NORM_CHECK:
					if (SendDlgItemMessage(hDialog, IDC_NORM_CHECK, BM_GETCHECK, 1, 0)==BST_CHECKED)
					{
						CheckMenuItem(hMenu, IDM_NORM, MF_CHECKED);
						Normalization_Flag = true;
					}
					else
					{
						CheckMenuItem(hMenu, IDM_NORM, MF_UNCHECKED);
						Normalization_Flag = false;
					}
					break;

				case IDCANCEL:
					DestroyWindow(hDialog);
					hNormDlg = NULL;
					return true;
			}
			break;

		case WM_HSCROLL:
			if (GetWindowLong((HWND)lParam, GWL_ID)==IDC_NORM_SLIDER)
			{
				Norm_Ratio = SendDlgItemMessage(hDialog, IDC_NORM_SLIDER, TBM_GETPOS, 0, 0);
				sprintf(szTemp, "%d", Norm_Ratio);
				SetDlgItemText(hDialog, IDC_NORM, szTemp);
			}
			break;
	}
    return false;
}

/* register the window class */
ATOM MyRegisterClass(HINSTANCE hInstance)
{
	WNDCLASSEX wcex;

	wcex.cbSize			= sizeof(WNDCLASSEX);
	wcex.style			= CS_HREDRAW | CS_VREDRAW;
	wcex.lpfnWndProc	= (WNDPROC)WndProc;
	wcex.cbClsExtra		= false;
	wcex.cbWndExtra		= false;
	wcex.hInstance		= hInstance;
	wcex.hIcon			= LoadIcon(hInstance, (LPCTSTR)IDI_MOVIE);
	wcex.hCursor		= LoadCursor(NULL, IDC_ARROW);
	wcex.hbrBackground	= CreateSolidBrush(MASKCOLOR);
	wcex.lpszMenuName	= (LPCSTR)IDC_GUI;
	wcex.lpszClassName	= szWindowClass;
	wcex.hIconSm		= LoadIcon(wcex.hInstance, (LPCTSTR)IDI_SMALL);

	return RegisterClassEx(&wcex);
}

char *ExtFilter[3] = 
{
	".avi", ".d2v", ".wav"
};

BOOL PopFileDlg(PTSTR pstrFileName, HWND hOwner, int Status)
{
	static char *szFilter, *ext;	
	int count = 0;

	switch (Status)
	{
		case OPEN_VOB:
			szFilter = TEXT ("MPEG-2 Stream (*.vob; *.m2p; *.m2v; *.mpv)\0*.vob;*.m2p;*.m2v;*.mpv\0")  \
				TEXT ("All Files (*.*)\0*.*\0");
			break;

		case SAVE_AVI:
			szFilter = TEXT ("AVI File (*.avi)\0*.avi; *.ac3; *.wav; *.mpa\0")  \
				TEXT ("All Files (*.*)\0*.*\0");
			break;

		case OPEN_D2V:
			szFilter = TEXT ("DVD2AVI Project File (*.d2v)\0*.d2v\0")  \
				TEXT ("All Files (*.*)\0*.*\0");
			break;

		case SAVE_D2V:
			szFilter = TEXT ("DVD2AVI Project File (*.d2v)\0*.d2v; *.ac3; *.wav; *.mpa\0")  \
				TEXT ("All Files (*.*)\0*.*\0");
			break;

		case OPEN_WAV:
		case SAVE_WAV:
			szFilter = TEXT ("WAV File (*.wav)\0*.wav\0")  \
				TEXT ("All Files (*.*)\0*.*\0");
			break;
	}

	ofn.lStructSize       = sizeof (OPENFILENAME) ;
	ofn.hwndOwner         = hOwner ;
	ofn.hInstance         = hInst ;
	ofn.lpstrFilter       = szFilter ;
	ofn.nMaxFile          = _MAX_PATH ;
	ofn.nMaxFileTitle     = _MAX_PATH ;
	ofn.lpstrFile         = pstrFileName ;

	switch (Status)
	{
		case OPEN_VOB:
		case OPEN_D2V:
		case OPEN_WAV:
			ofn.Flags = OFN_HIDEREADONLY | OFN_FILEMUSTEXIST;
			return GetOpenFileName(&ofn);

		case SAVE_WAV:
			count++;
		case SAVE_D2V:
			count++;
		case SAVE_AVI:
			ofn.Flags = OFN_HIDEREADONLY | OFN_ALLOWMULTISELECT | OFN_EXPLORER;
			if (GetSaveFileName(&ofn))
			{
				ext = strrchr(pstrFileName, '.');

				if (ext!=NULL && !_strnicmp(ext, ExtFilter[count], 4))
					strncpy(ext, ext+4, 1);

				return true;
			}
	}

	return false;
}

static void ShowStatistics(bool refresh)
{
	if (refresh)
	{
		if (Statistics_Flag)
			DestroyWindow(hDlg);

		Statistics_Flag = true;
		hDlg = CreateDialog(hInst, (LPCTSTR)IDD_STATISTICS, hWnd, (DLGPROC)Statistics);
	}

	if (Statistics_Flag)
	{
		GetWindowRect(hDlg, &crect);
		GetWindowRect(hWnd, &wrect);
		MoveWindow(hDlg, wrect.right, wrect.top+Edge_Height-Edge_Width/2, crect.right-crect.left, crect.bottom-crect.top, true);
		ShowWindow(hDlg, SW_SHOW);
	}
}

static void DisableControl(int audio)
{
	DragAcceptFiles(hWnd, false);

	EnableWindow(hTrack, false);
	EnableWindow(hLeftButton, false);
	EnableWindow(hLeftArrow, false);
	EnableWindow(hRightArrow, false);
	EnableWindow(hRightButton, false);

	EnableMenuItem(hMenu, 1, MF_BYPOSITION | MF_GRAYED);

	if (audio)
		EnableMenuItem(hMenu, 2, MF_BYPOSITION | MF_GRAYED);

	if (hClipResizeDlg!=NULL)
	{
		DestroyWindow(hClipResizeDlg);
		hClipResizeDlg = NULL;
	}

	if (hLumDlg!=NULL)
	{
		DestroyWindow(hLumDlg);
		hLumDlg = NULL;
	}

	if (hNormDlg!=NULL)
	{
		DestroyWindow(hNormDlg);
		hNormDlg = NULL;
	}
}

static void ClearTrack()
{
	CheckMenuItem(hMenu, IDM_TRACK_NONE, MF_UNCHECKED);
	CheckMenuItem(hMenu, IDM_TRACK_1, MF_UNCHECKED);
	CheckMenuItem(hMenu, IDM_TRACK_2, MF_UNCHECKED);
	CheckMenuItem(hMenu, IDM_TRACK_3, MF_UNCHECKED);
	CheckMenuItem(hMenu, IDM_TRACK_4, MF_UNCHECKED);
	CheckMenuItem(hMenu, IDM_TRACK_5, MF_UNCHECKED);
	CheckMenuItem(hMenu, IDM_TRACK_6, MF_UNCHECKED);
	CheckMenuItem(hMenu, IDM_TRACK_7, MF_UNCHECKED);
	CheckMenuItem(hMenu, IDM_TRACK_8, MF_UNCHECKED);
}

static void CheckINI()
{
	__asm
	{
		mov			eax, 1
		cpuid
		test		edx, 0x00800000		// STD MMX
		jz			TEST_SSE
		mov			[cpu.mmx], 1
TEST_SSE:
		test		edx, 0x02000000		// STD SSE
		jz			TEST_3DNOW
		mov			[cpu.ssemmx], 1
		mov			[cpu.ssefpu], 1
TEST_3DNOW:
		mov			eax, 0x80000001
		cpuid
		test		edx, 0x80000000		// 3D NOW
		jz			TEST_SSEMMX
		mov			[cpu._3dnow], 1
TEST_SSEMMX:
		test		edx, 0x00400000		// SSE MMX
		jz			TEST_END
		mov			[cpu.ssemmx], 1
TEST_END:
	}

	if (cpu.mmx)
		CheckMenuItem(hMenu, IDM_MMX, MF_CHECKED);

	if (cpu.ssemmx)
	{
		CheckMenuItem(hMenu, IDM_SSEMMX, MF_CHECKED);
		DeleteMenu(hMenu, IDM_IDCT_MMX, 0);
	}
	else
		DeleteMenu(hMenu, IDM_IDCT_SSEMMX, 0);

	if (cpu._3dnow)
		CheckMenuItem(hMenu, IDM_3DNOW, MF_CHECKED);

	if (cpu.ssefpu)
		CheckMenuItem(hMenu, IDM_SSEFPU, MF_CHECKED);

	if ((INIFile = fopen("DVD2AVI.ini", "r")) == NULL)
	{
NEW_VERSION:
		INIT_X = INIT_Y = 100;
		INI_Version = INI_VERSION;

		if (cpu.ssemmx)
			iDCT_Flag = IDCT_SSEMMX;
		else
			iDCT_Flag = IDCT_MMX;

		Scale_Flag = true;
		FO_Flag = FO_NONE;
		Track_Flag = TRACK_1;
		Format_Flag = FORMAT_AUTO;
		AC3_Flag = AUDIO_DECODE;
		DRC_Flag = DRC_NORMAL;
		DSDown_Flag = false;
		MPA_Flag = AUDIO_DEMUXONE;
		SRC_Flag = SRC_NONE;
		Norm_Ratio = 100;
		Priority_Flag = PRIORITY_NORMAL;
	}
	else
	{
		fscanf(INIFile, "INI_Version=%d\n", &INI_Version);
		if (INI_Version != INI_VERSION)
		{
			fclose(INIFile);
			goto NEW_VERSION;
		}

		fscanf(INIFile, "Window_Position=%d,%d\n", &INIT_X, &INIT_Y);
		if (INIT_X < 0 || INIT_X + 100 > GetSystemMetrics(SM_CXSCREEN) || 
			INIT_Y < 0 || INIT_Y + 100 > GetSystemMetrics(SM_CYSCREEN))
			INIT_X = INIT_Y = 100;

		fscanf(INIFile, "iDCT_Algorithm=%d\n", &iDCT_Flag);
		fscanf(INIFile, "YUVRGB_Scale=%d\n", &Scale_Flag);
		fscanf(INIFile, "Field_Operation=%d\n", &FO_Flag);
		fscanf(INIFile, "Track_Number=%d\n", &Track_Flag);
		fscanf(INIFile, "Channel_Format=%d\n", &Format_Flag);
		fscanf(INIFile, "AC3=%d\n", &AC3_Flag);
		fscanf(INIFile, "DR_Control=%d\n", &DRC_Flag);
		fscanf(INIFile, "DS_Downmix=%d\n", &DSDown_Flag);
		fscanf(INIFile, "MPA=%d\n", &MPA_Flag);
		fscanf(INIFile, "SRC_Precision=%d\n", &SRC_Flag);
		fscanf(INIFile, "Norm_Ratio=%d\n", &Norm_Ratio);
		fscanf(INIFile, "Process_Priority=%d\n", &Priority_Flag);

		fclose(INIFile);
	}
}

static void CheckFlag()
{
	CheckMenuItem(hMenu, IDM_KEY_OFF, MF_UNCHECKED);
	CheckMenuItem(hMenu, IDM_KEY_INPUT, MF_UNCHECKED);
	CheckMenuItem(hMenu, IDM_KEY_OP, MF_UNCHECKED);

	if (!PlugIn_Flag)
		KeyOp_Flag = 0;

	switch (KeyOp_Flag)
	{
		case KEY_OFF:
			CheckMenuItem(hMenu, IDM_KEY_OFF, MF_CHECKED);
			break;

		case KEY_INPUT:
			CheckMenuItem(hMenu, IDM_KEY_INPUT, MF_CHECKED);
			break;

		case KEY_OP:
			CheckMenuItem(hMenu, IDM_KEY_OP, MF_CHECKED);
			break;
	}

	CheckMenuItem(hMenu, IDM_IDCT_MMX, MF_UNCHECKED);
	CheckMenuItem(hMenu, IDM_IDCT_SSEMMX, MF_UNCHECKED);
	CheckMenuItem(hMenu, IDM_IDCT_FPU, MF_UNCHECKED);
	CheckMenuItem(hMenu, IDM_IDCT_REF, MF_UNCHECKED);
	CheckMenuItem(hMenu, IDM_PCSCALE, MF_UNCHECKED);
	CheckMenuItem(hMenu, IDM_TVSCALE, MF_UNCHECKED);
	CheckMenuItem(hMenu, IDM_FO_FILM, MF_UNCHECKED);

	if (cpu.ssemmx && iDCT_Flag==IDCT_MMX)
		iDCT_Flag = IDCT_SSEMMX;

	if (!cpu.ssemmx && iDCT_Flag==IDCT_SSEMMX)
		iDCT_Flag = IDCT_MMX;

	switch (iDCT_Flag)
	{
		case IDCT_MMX:
			CheckMenuItem(hMenu, IDM_IDCT_MMX, MF_CHECKED);
			break;

		case IDCT_SSEMMX:
			CheckMenuItem(hMenu, IDM_IDCT_SSEMMX, MF_CHECKED);
			break;

		case IDCT_FPU:
			CheckMenuItem(hMenu, IDM_IDCT_FPU, MF_CHECKED);
			break;

		case IDCT_REF:
			CheckMenuItem(hMenu, IDM_IDCT_REF, MF_CHECKED);
			break;
	}

	if (Scale_Flag)
	{
		RGB_Scale = 0x1000254310002543;
		RGB_Offset = 0x0010001000100010;
		RGB_CBU = 0x0000408D0000408D;
		RGB_CGX = 0xF377E5FCF377E5FC;
		RGB_CRV = 0x0000331300003313;

		CheckMenuItem(hMenu, IDM_PCSCALE, MF_CHECKED);
	}
	else
	{
		RGB_Scale = 0x1000200010002000;
		RGB_Offset = 0x0000000000000000;
		RGB_CBU = 0x000038B4000038B4;
		RGB_CGX = 0xF4FDE926F4FDE926;
		RGB_CRV = 0x00002CDD00002CDD;

		CheckMenuItem(hMenu, IDM_TVSCALE, MF_CHECKED);
	}

	switch (FO_Flag)
	{
		case FO_NONE:
			CheckMenuItem(hMenu, IDM_FO_NONE, MF_CHECKED);
			CheckMenuItem(hMenu, IDM_FO_FILM, MF_UNCHECKED);
			CheckMenuItem(hMenu, IDM_FO_SWAP, MF_UNCHECKED);
			break;

		case FO_FILM:
			CheckMenuItem(hMenu, IDM_FO_NONE, MF_UNCHECKED);
			CheckMenuItem(hMenu, IDM_FO_FILM, MF_CHECKED);
			CheckMenuItem(hMenu, IDM_FO_SWAP, MF_UNCHECKED);
			break;

		case FO_SWAP:
			CheckMenuItem(hMenu, IDM_FO_NONE, MF_UNCHECKED);
			CheckMenuItem(hMenu, IDM_FO_FILM, MF_UNCHECKED);
			CheckMenuItem(hMenu, IDM_FO_SWAP, MF_CHECKED);
			break;
	}

	switch (Track_Flag)
	{
		case TRACK_NONE:
			CheckMenuItem(hMenu, IDM_TRACK_NONE, MF_CHECKED);
			break;

		case TRACK_1:
			CheckMenuItem(hMenu, IDM_TRACK_1, MF_CHECKED);
			break;

		case TRACK_2:
			CheckMenuItem(hMenu, IDM_TRACK_2, MF_CHECKED);
			break;

		case TRACK_3:
			CheckMenuItem(hMenu, IDM_TRACK_3, MF_CHECKED);
			break;

		case TRACK_4:
			CheckMenuItem(hMenu, IDM_TRACK_4, MF_CHECKED);
			break;

		case TRACK_5:
			CheckMenuItem(hMenu, IDM_TRACK_5, MF_CHECKED);
			break;

		case TRACK_6:
			CheckMenuItem(hMenu, IDM_TRACK_6, MF_CHECKED);
			break;

		case TRACK_7:
			CheckMenuItem(hMenu, IDM_TRACK_7, MF_CHECKED);
			break;

		case TRACK_8:
			CheckMenuItem(hMenu, IDM_TRACK_8, MF_CHECKED);
			break;
	}

	switch (Format_Flag)
	{
		case FORMAT_AC3:
			CheckMenuItem(hMenu, IDM_AC3, MF_CHECKED);
			break;

		case FORMAT_MPA:
			CheckMenuItem(hMenu, IDM_MPA, MF_CHECKED);
			break;

		case FORMAT_LPCM:
			CheckMenuItem(hMenu, IDM_LPCM, MF_CHECKED);
			break;

		case FORMAT_AUTO:
			CheckMenuItem(hMenu, IDM_SELECT, MF_CHECKED);
			break;
	}

	switch (AC3_Flag)
	{
		case AUDIO_DEMUXALL:
			CheckMenuItem(hMenu, IDM_AC3_DEMUXALL, MF_CHECKED);
			break;

		case AUDIO_DEMUXONE:
			CheckMenuItem(hMenu, IDM_AC3_DEMUXONE, MF_CHECKED);
			break;

		case AUDIO_DECODE:
			CheckMenuItem(hMenu, IDM_AC3_DECODE, MF_CHECKED);
			break;
	}

	switch (DRC_Flag)
	{
		case DRC_NONE:
			CheckMenuItem(hMenu, IDM_DRC_NONE, MF_CHECKED);
			break;

		case DRC_LIGHT:
			CheckMenuItem(hMenu, IDM_DRC_LIGHT, MF_CHECKED);
			break;

		case DRC_NORMAL:
			CheckMenuItem(hMenu, IDM_DRC_NORMAL, MF_CHECKED);
			break;

		case DRC_HEAVY:
			CheckMenuItem(hMenu, IDM_DRC_HEAVY, MF_CHECKED);
			break;
	}

	if (DSDown_Flag)
		CheckMenuItem(hMenu, IDM_DSDOWN, MF_CHECKED);

	switch (MPA_Flag)
	{
		case AUDIO_DEMUXALL:
			CheckMenuItem(hMenu, IDM_MPA_DEMUXALL, MF_CHECKED);
			break;

		case AUDIO_DEMUXONE:
			CheckMenuItem(hMenu, IDM_MPA_DEMUXONE, MF_CHECKED);
			break;
	}

	switch (SRC_Flag)
	{
		case SRC_NONE:
			CheckMenuItem(hMenu, IDM_SRC_NONE, MF_CHECKED);
			break;

		case SRC_LOW:
			CheckMenuItem(hMenu, IDM_SRC_LOW, MF_CHECKED);
			break;

		case SRC_MID:
			CheckMenuItem(hMenu, IDM_SRC_MID, MF_CHECKED);
			break;

		case SRC_HIGH:
			CheckMenuItem(hMenu, IDM_SRC_HIGH, MF_CHECKED);
			break;

		case SRC_UHIGH:
			CheckMenuItem(hMenu, IDM_SRC_UHIGH, MF_CHECKED);
			break;
	}

	if (Norm_Ratio > 100)
	{
		CheckMenuItem(hMenu, IDM_NORM, MF_CHECKED);
		Normalization_Flag = true;
		Norm_Ratio -= 100;
	}

	switch (Priority_Flag)
	{
		case PRIORITY_HIGH:
			CheckMenuItem(hMenu, IDM_PP_HIGH, MF_CHECKED);
			SetPriorityClass(hProcess, HIGH_PRIORITY_CLASS);
			break;

		case PRIORITY_NORMAL:
			CheckMenuItem(hMenu, IDM_PP_NORMAL, MF_CHECKED);
			break;

		case PRIORITY_LOW:
			CheckMenuItem(hMenu, IDM_PP_LOW, MF_CHECKED);
			SetPriorityClass(hProcess, IDLE_PRIORITY_CLASS);
			break;
	}
}

static void Recovery()
{
	int i;

	if (Check_Flag)
	{
		if (DDOverlay_Flag)
		{
			DDOverlay_Flag = false;
			CheckMenuItem(hMenu, IDM_DIRECTDRAW, MF_UNCHECKED);
			IDirectDrawSurface_UpdateOverlay(lpOverlay, NULL, lpPrimary, NULL, DDOVER_HIDE, NULL);
		}

		if (lpDD2)
			lpDD2->Release();
		if (lpDD)
			lpDD->Release();

		for (i=0; i<3; i++)
		{
			free(backward_reference_frame[i]);
			free(forward_reference_frame[i]);
			free(auxframe[i]);
		}

		free(u422);
		free(v422);
		free(u444);
		free(v444);
		free(rgb24);
		free(yuy2);
		free(lum);
	}

	Check_Flag = false;

	SendMessage(hTrack, TBM_SETPOS, (WPARAM) true, 0);
	SendMessage(hTrack, TBM_SETSEL, (WPARAM) true, (LPARAM) MAKELONG(0, 0));

	DisableControl(false);

	EnableMenuItem(hMenu, IDM_SAVE, MF_GRAYED);
	EnableMenuItem(hMenu, IDM_SAVE_D2V, MF_GRAYED);
	EnableMenuItem(hMenu, IDM_BMP, MF_GRAYED);
	EnableMenuItem(hMenu, IDM_PROCESS_WAV, MF_ENABLED);

	LumGain = 128; LumOffset = 0;
	Luminance_Flag = false;
	CheckMenuItem(hMenu, IDM_LUMINANCE, MF_UNCHECKED);

	Clip_Left = Clip_Right = Clip_Top = Clip_Bottom = Squeeze_Width = Squeeze_Height = 0;
	ClipResize_Flag = false;
	CheckMenuItem(hMenu, IDM_CLIPRESIZE, MF_UNCHECKED);

	PreScale_Ratio = 1.0;
	CheckMenuItem(hMenu, IDM_PRESCALE, MF_UNCHECKED);

	SetWindowText(hWnd, "DVD2AVI");

	if (File_Limit)
	{
		EnableMenuItem(hMenu, IDM_PROCESS_WAV, MF_GRAYED);

		ZeroMemory(&process, sizeof(PROCESS));
		process.trackright = TRACK_PITCH;

		SystemStream_Flag = false;
		Display_Flag = true;

		for (i=0; i<File_Limit; i++)
		{
			process.length[i] = _filelengthi64(Infile[i]);
			process.total += process.length[i];
		}
	}
	else
		DragAcceptFiles(hWnd, true);

	ResizeWindow(INIT_WIDTH, INIT_HEIGHT);
}

void ResizeWindow(int width, int height)
{
	MoveWindow(hTrack, 0, height, width-4*TRACK_HEIGHT, TRACK_HEIGHT, true);
	MoveWindow(hLeftButton, width-4*TRACK_HEIGHT, height, TRACK_HEIGHT, TRACK_HEIGHT, true);
	MoveWindow(hLeftArrow, width-3*TRACK_HEIGHT, height, TRACK_HEIGHT, TRACK_HEIGHT, true);
	MoveWindow(hRightArrow, width-2*TRACK_HEIGHT, height, TRACK_HEIGHT, TRACK_HEIGHT, true);
	MoveWindow(hRightButton, width-TRACK_HEIGHT, height, TRACK_HEIGHT, TRACK_HEIGHT, true);

	GetWindowRect(hWnd, &wrect);
	GetClientRect(hWnd, &crect);
	Edge_Width = wrect.right - wrect.left - crect.right;
	Edge_Height = wrect.bottom - wrect.top - crect.bottom;

	MoveWindow(hWnd, wrect.left, wrect.top, width+Edge_Width, height+Edge_Height+TRACK_HEIGHT, true);
}

void CheckDirectDraw()
{
	if (DirectDrawCreate(NULL, &lpDD, NULL)==DD_OK)
	{
		if (lpDD->QueryInterface(IID_IDirectDraw2, (LPVOID*)&lpDD2)==DD_OK)
		{
			if (lpDD2->SetCooperativeLevel(hWnd, DDSCL_NORMAL)==DD_OK)
			{
				ZeroMemory(&ddsd, sizeof(DDSURFACEDESC));
				ddsd.dwSize = sizeof(DDSURFACEDESC);
				ddsd.dwFlags = DDSD_CAPS;
				ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE | DDSCAPS_VIDEOMEMORY;

				if (lpDD2->CreateSurface(&ddsd, &lpPrimary, NULL)==DD_OK)
				{
					ZeroMemory(&halcaps, sizeof(DDCAPS));
					halcaps.dwSize = sizeof(DDCAPS);

					if (lpDD2->GetCaps(&halcaps, NULL)==DD_OK)
					{
						if (halcaps.dwCaps & DDCAPS_OVERLAY)
						{
							DDPIXELFORMAT ddPixelFormat;
							ddPixelFormat.dwFlags = DDPF_FOURCC;
							ddPixelFormat.dwFourCC = mmioFOURCC('Y','U','Y','2');
							ddPixelFormat.dwYUVBitCount = 16;

							ZeroMemory(&ddsd, sizeof(DDSURFACEDESC));
							ddsd.dwSize = sizeof(DDSURFACEDESC);
							ddsd.dwFlags = DDSD_CAPS | DDSD_WIDTH | DDSD_HEIGHT | DDSD_PIXELFORMAT;
							ddsd.ddsCaps.dwCaps = DDSCAPS_OVERLAY | DDSCAPS_VIDEOMEMORY;
							ddsd.dwWidth = Coded_Picture_Width;
							ddsd.dwHeight = Coded_Picture_Height;

							memcpy(&(ddsd.ddpfPixelFormat), &ddPixelFormat, sizeof(DDPIXELFORMAT));

							if (lpDD2->CreateSurface(&ddsd, &lpOverlay, NULL)==DD_OK)
							{
								DDOverlay_Flag = true;

								ZeroMemory(&ddofx, sizeof(DDOVERLAYFX));
								ddofx.dwSize = sizeof(DDOVERLAYFX);

								ddofx.dckDestColorkey.dwColorSpaceLowValue = DDColorMatch(lpPrimary, MASKCOLOR);
								ddofx.dckDestColorkey.dwColorSpaceHighValue = ddofx.dckDestColorkey.dwColorSpaceLowValue;
							}
						}
					}
				}
			}
		}
	}

	if (DDOverlay_Flag)
	{
		Store_Flag = STORE_YUY2;
		CheckMenuItem(hMenu, IDM_STORE_RGB24, MF_UNCHECKED);
		CheckMenuItem(hMenu, IDM_STORE_YUY2, MF_CHECKED);
		EnableMenuItem(hMenu, IDM_BMP, MF_GRAYED);
		CheckMenuItem(hMenu, IDM_DIRECTDRAW, MF_CHECKED);
	}
	else
	{
		Store_Flag = STORE_RGB24;
		CheckMenuItem(hMenu, IDM_STORE_RGB24, MF_CHECKED);
		CheckMenuItem(hMenu, IDM_STORE_YUY2, MF_UNCHECKED);
		EnableMenuItem(hMenu, IDM_BMP, MF_ENABLED);
	}
}

static DWORD DDColorMatch(LPDIRECTDRAWSURFACE pdds, COLORREF rgb)
{
	COLORREF	rgbT;
	DWORD		dw = CLR_INVALID;
	HRESULT		hres;
	HDC			hdc;

	if (IDirectDrawSurface_GetDC(pdds, &hdc)==DD_OK)
	{
		rgbT = GetPixel(hdc, 0, 0);
		SetPixel(hdc, 0, 0, rgb);
		IDirectDrawSurface_ReleaseDC(pdds, hdc);
	}

	ZeroMemory(&ddsd, sizeof(DDSURFACEDESC));
	ddsd.dwSize = sizeof(DDSURFACEDESC);

	while ((hres = IDirectDrawSurface_Lock(pdds, NULL, &ddsd, 0, NULL)) == DDERR_WASSTILLDRAWING)
		;

	if (hres==DD_OK)
	{
		dw = *(DWORD *) ddsd.lpSurface;
		if (ddsd.ddpfPixelFormat.dwRGBBitCount < 32)
			dw &= (1 << ddsd.ddpfPixelFormat.dwRGBBitCount) - 1;
		IDirectDrawSurface_Unlock(pdds, NULL);
	}

	if (IDirectDrawSurface_GetDC(pdds, &hdc)==DD_OK)
	{
		SetPixel(hdc, 0, 0, rgbT);
		IDirectDrawSurface_ReleaseDC(pdds, hdc);
	}

	return dw;
}

static void RefreshWindow()
{
	if (WaitForSingleObject(hThread, 0)==WAIT_OBJECT_0)
	{
		Fault_Flag = false; Display_Flag = true;
		Write_Frame(backward_reference_frame, d2v_backward, 0);
	}
}
