//-----------------------------------------------------------------------------
//
// AttackAudio.cpp - An instance of this type holds the
//             the audio used to insert or mix into a 
//             conversation. 
//
//    Copyright (C) 2004  Mark D. Collier
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//   Author: Mark D. Collier   - 12/01/2006   v1.1
//                   Mark D. Collier   -  04/26/2004  v1.0
//         www.securelogix.com - mark.collier@securelogix.com
//         www.hackingexposedvoip.com
//
//-----------------------------------------------------------------------------

#include "AttackAudio.h"

AttackAudio::AttackAudio( AttackAudioMethod attackAudioMethod )
{
    
    mNumG711PacketEquivalents   = 0;
//    mNumPacketsRetrieved        = 0;

    if ( attackAudioMethod == ATTACK_AUDIO_INSERT )
    {        
        mAttackAudioMethod      = attackAudioMethod;
        mAttackAudioBuf         = (unsigned char *) new pcmuSamplesToInsert;
        return;
    }
    
    if ( attackAudioMethod == ATTACK_AUDIO_MIX )
    {        
        mAttackAudioMethod      = attackAudioMethod;
        mAttackAudioBuf         = (unsigned char *) new pcmSamplesToMix;
        return;
    }
    
    mAttackAudioMethod = ATTACK_AUDIO_NONE;
    mAttackAudioBuf    = NULL;
    return;
}

AttackAudio::~AttackAudio( void )
{
    mAttackAudioMethod          = ATTACK_AUDIO_NONE;
    mNumG711PacketEquivalents   = 0;
//    mNumPacketsRetrieved        = 0;
    
    if ( mAttackAudioBuf )
    {
        delete mAttackAudioBuf;
        mAttackAudioBuf = NULL;
    }
}

/*
const char * AttackAudio::GetNextAttackAudioPacket( void )
{
    unsigned char *pPacket = NULL;
    
    if ( mNumPacketsRetrieved == mNumG711PacketEquivalents )
    {
        return NULL;
    }
    
    if ( mAttackAudioMethod == ATTACK_AUDIO_INSERT ) {
        
        pPacket = mAttackAudioBuf +
                    ( mNumPacketsRetrieved *
                      __ATTACKAUDIO_G711_PAYLOAD_LEN *
                      sizeof(pcmu_value) );
        
        mNumPacketsRetrieved++;
        return (const char *) pPacket;
    }

        
    if ( mAttackAudioMethod == ATTACK_AUDIO_MIX ) {
     
        pPacket = mAttackAudioBuf +
                    ( mNumPacketsRetrieved *
                      __ATTACKAUDIO_G711_PAYLOAD_LEN *
                      sizeof(pcm_value) );
    
        mNumPacketsRetrieved++;
        return (const char *) pPacket;
    }
    
    return NULL;

}
*/

unsigned int AttackAudio::GetNumberOfAttackAudioPackets( void )
{
    return mNumG711PacketEquivalents;
}

//
//  The first packet AttackAudio packet is number 1
//  There are mNumG711PacketEquivalents number of packets in the AttackAudio object
//
const char * AttackAudio::GetAttackAudioPacket( unsigned int packetNum )
{
    unsigned char *pPacket = NULL;
    
    if ( packetNum == 0 ||
         packetNum > mNumG711PacketEquivalents )
    {
        return NULL;
    }
    
    if ( mAttackAudioMethod == ATTACK_AUDIO_INSERT ) {
        
        pPacket = mAttackAudioBuf +
                    ( ( packetNum - 1 ) * sizeof(pcmu_value) );
        
        return (const char *) pPacket;
    }

        
    if ( mAttackAudioMethod == ATTACK_AUDIO_MIX ) {
     
        pPacket = mAttackAudioBuf +
                    ( ( packetNum - 1 ) * sizeof(pcm_value) );

        return (const char *) pPacket;
    }
    
    return NULL;

}

AttackAudioMethod AttackAudio::GetAttackAudioMethod( void )
{    
    return mAttackAudioMethod;
}

bool  AttackAudio::PrepAttackAudio( char *psInputAudioFile )
{
    unsigned int idxExtension = 0;
    
    float expectedAudioPlaybackTime = 0;
        
    bool bSuccess = false;
    
    if ( !mAttackAudioBuf ) {
        fprintf ( stderr,
                  "\nError in AttackAudio::PrepAttackAudio:\n"
                  "\n  Audio Buffer in AttackAudio object was NULL.\n"
                  "\n  The type of audio attack might have been erronesously\n"
                  "\n  specified when the object was instantiated.\n" );
        return false;
    }
    
    mNumG711PacketEquivalents = 0;
    
//
//  Vet the specified pre-recorded audio file to confirm it complies
//  with various restrictions imposed by this tool and load it into
//  memory. 
//
    
    idxExtension = strlen ( psInputAudioFile ) - 4;
    
    if ( idxExtension > 0 ) {
        // filename could have .wav extension
        if ( psInputAudioFile[ idxExtension     ] == '.' &&
             psInputAudioFile[ idxExtension + 1 ] == 'w' &&
             psInputAudioFile[ idxExtension + 2 ] == 'a' &&
             psInputAudioFile[ idxExtension + 3 ] == 'v'    ) {
            bSuccess = preloadWavAudio ( psInputAudioFile );
        } else {
            bSuccess = preloadTCPdumpAudio ( psInputAudioFile );
        }
    } else {
        bSuccess = preloadTCPdumpAudio ( psInputAudioFile );
    }
    
    if ( !bSuccess || mNumG711PacketEquivalents == 0 ) {

        fprintf ( stderr, "\nError: Audio input file is empty or doesn't hold"
                          "\nat least one G.711 packet's worth of audio.\n" );
        
        return false;
    }
    
    fprintf ( stderr,
              "\nAudio read from input file equates to %u G711 packets.\n",
              mNumG711PacketEquivalents );

    expectedAudioPlaybackTime = (float) mNumG711PacketEquivalents /
                                (float) __ATTACKAUDIO_G711_CODEC_RATE_HZ;
    
    fprintf ( stderr, "At an ideal playback rate of %u Hz, this represents\n"
                      "%4.2f seconds of audio.\n",
                      __ATTACKAUDIO_G711_CODEC_RATE_HZ,
                      expectedAudioPlaybackTime );
    
    return true;
}

//-----------------------------------------------------------------------------
//
//  preloadWavAudio ( char *psInputAudioFile  )
//
//  This routine expects psInputAudioFile to point to a  
//  string with the name of a file containing pre-recorded
//  audio to load into memory. The file is expected to be a
//  standard Microsoft multimedia RIFF formatted file
//  containing WAVE formated audio.
//
//  This routine shall confirm that the file content meets the
//  the following restrictions:
//
//  File type:                      Microsoft RIFF
//  File Resource Type:  WAVE
//  Compression Code:   PCM/uncompressed
//  Sample Frequency:   8000 Hz
//  Sample size:                unsigned 8-bit or signed 16-bit.
//
//  The file must have "chunks" in this order:
//
//     RIFF chunk
//     format chunk
//     fact chunk
//     data chunk
//
//   or
//
//     RIFF chunk
//     format chunk
//     data chunk
//
//  If the file does not comply with the required
//  configuration, the Linux sox command is
//  suggested as a means to convert the file
//  to the required configuration.
//
//  How the pre-recoded audio is treated as it is loaded
//  into memory is a function of whether the audio attack
//  is one of insertion (i.e. replacement), or mixing with
//  the targeted audio stream. The type of audio attack
//  must have already been set in member variable:
//  mAttackAudioMethod.
//
//  The mixing attack requires the memory resident
//  pre-recorded audio to be signed, 16-bit, linear PCM.
//  Already having the audio in the format ready to 
//  be mixed saves computation time once the attack
//  is launched. PCMU bytes cannot be added directly.
//
//  The insertion attack requires the memory resident
//  audio to be unsigned, 8-bit, non-linear PCMU.
//  Already having the audio in a format ready to 
//  be inserted saves computation time once the attack
//  is launched.
//
//  The number of memory-resident G711 packets is
//  recorded in member variable: mNumG711PacketEquivalents
//
//  FIXME: Declare that some data should remain
//                   memory-resident at all times.
//
//-----------------------------------------------------------------------------

bool AttackAudio::preloadWavAudio ( char *psInputAudioFile ) {
                                        
    FILE *fpAudio = NULL;
    
    int i, j, rc;
                           
    bool bVerbose = false;
 
    struct {
        char           c_chunkID[4];        // Should be 4 ASCII characters        
        unsigned int   ui_chunkSize;        // i.e. does not include hdrCHUNK        
    } hdrCHUNK;
        
    struct {
        unsigned char c_chunkID[4];          // Should be the ASCII characters: RIFF        
        unsigned int  ui_audioFileSize;      // i.e. file size - 8        
        unsigned char c_typeRIFF[4];         // Should be the ASCII characters: WAVE
    } hdrRIFF;    
    
    //
    //  hdrFMT is the structure of the fmt chunk when there are no extra format bytes
    //  (i.e. when ui_fmtChunkSize = 16).
    //
    struct {
        char           c_chunkID[4];                   // Should be the ASCII characters: fmt<space>        
        unsigned int   ui_fmtChunkSize;                // i.e. 16 + extra format bytes        
        unsigned short us_compressionCode;             // 1 to 65535
        unsigned short us_numChannels;                 // 1 to 65535
        unsigned int   ui_sampleRate;                  // 1 to ( 2**32   - 1 )
        unsigned int   ui_avgBytesPerSec;              // 1 to ( 2**32   - 1 )
        unsigned short us_blockAlign;                  // 1 to 65535
        unsigned short us_numSignificantBitsPerSample; // 2 to 65535
    } hdrFMT;
    
    short *pPcmVal = NULL;
        
//
//  Check that the audio input file has a standard RIFF header and that the
//  format type is: WAVE
//

    fpAudio = fopen ( psInputAudioFile, "r" );
    
    if ( !fpAudio ) {
        fprintf ( stderr, "\nCouldn't open pre-recorded audio file: %s\n",
                          psInputAudioFile );
        
        return false;
    }

    rc = fread ( &hdrRIFF, sizeof ( hdrRIFF ), 1, fpAudio );
    
    if ( rc != 1 ) {
        fprintf ( stderr, "\nEOF on input file attempting to read"
                          "\nthe standard RIFF header\n" );
        fclose ( fpAudio );
        fpAudio = NULL;

        return false;
    }

    if ( bVerbose ) {
        fprintf ( stderr,
                  "\nAudio file format: %c%c%c%c  0x%0x 0x%0x 0x%0x 0x%0x \n",
                  hdrRIFF.c_chunkID[0],
                  hdrRIFF.c_chunkID[1],
                  hdrRIFF.c_chunkID[2],
                  hdrRIFF.c_chunkID[3],
                  hdrRIFF.c_chunkID[0],
                  hdrRIFF.c_chunkID[1],
                  hdrRIFF.c_chunkID[2],
                  hdrRIFF.c_chunkID[3]  );
    }
        
    if ( hdrRIFF.c_chunkID[0] != 'R' ||
         hdrRIFF.c_chunkID[1] != 'I' ||
         hdrRIFF.c_chunkID[2] != 'F' ||
         hdrRIFF.c_chunkID[3] != 'F' ) {
        fprintf ( stderr, "\nPre-recorded audio input does not have "
                          "a standard RIFF header\n" );
        fclose ( fpAudio );
        fpAudio = NULL;
        
        return false;
    }

    if ( bVerbose ) {
        fprintf ( stderr,
                  "\nTotal Audio File Size: %u\n", hdrRIFF.ui_audioFileSize + 8 );
        
        fprintf ( stderr,
                  "\nPre-recorded audio content format: %c%c%c%c  "
                  "0x%0x 0x%0x 0x%0x 0x%0x\n",
                  hdrRIFF.c_typeRIFF[0],
                  hdrRIFF.c_typeRIFF[1],
                  hdrRIFF.c_typeRIFF[2],
                  hdrRIFF.c_typeRIFF[3],
                  hdrRIFF.c_typeRIFF[0],
                  hdrRIFF.c_typeRIFF[1],
                  hdrRIFF.c_typeRIFF[2],
                  hdrRIFF.c_typeRIFF[3] );
    }
    
    if ( hdrRIFF.c_typeRIFF[0] != 'W' ||
         hdrRIFF.c_typeRIFF[1] != 'A' ||
         hdrRIFF.c_typeRIFF[2] != 'V' ||
         hdrRIFF.c_typeRIFF[3] != 'E' ) {
        fprintf ( stderr, "\nPre-recorded audio input is not in WAVE format\n" );
        fclose ( fpAudio );
        fpAudio = NULL;

        return false;
    }
    
//
//  Check that the audio input file has a standard "fmt " chunk following the 
//  RIFF chunk. The "fmt " chunk contains the parameters of the audio
//  data.
//
    
    rc = fread ( &hdrFMT, sizeof ( hdrFMT ), 1, fpAudio );
    
    if ( rc != 1 ) {
        fprintf ( stderr, "\nEOF on input file attempting to read"
                          "\nthe standard RIFF fmt header\n" );
        fclose ( fpAudio );
        fpAudio = NULL;

        return false;
    }

    if ( bVerbose ) {
        fprintf ( stderr,
                  "\nNext \"chunk\" header type: %c%c%c%c  "
                  "0x%0x 0x%0x 0x%0x 0x%0x\n",
                  hdrFMT.c_chunkID[0],
                  hdrFMT.c_chunkID[1],
                  hdrFMT.c_chunkID[2],
                  hdrFMT.c_chunkID[3],
                  hdrFMT.c_chunkID[0],
                  hdrFMT.c_chunkID[1],
                  hdrFMT.c_chunkID[2],
                  hdrFMT.c_chunkID[3] );
    }
    
    if ( hdrFMT.c_chunkID[0] != 'f' ||
         hdrFMT.c_chunkID[1] != 'm' ||
         hdrFMT.c_chunkID[2] != 't' ||
         hdrFMT.c_chunkID[3] != ' ' ) {
        fprintf ( stderr, "\nRIFF header not followed by fmt header\n" );
        fclose ( fpAudio );
        fpAudio = NULL;

        return false;
    }

    if ( bVerbose ) {
        fprintf ( stderr, "\nCompression Code:        %u",   hdrFMT.us_compressionCode );
        fprintf ( stderr, "\nChannels:                %u",   hdrFMT.us_numChannels );
        fprintf ( stderr, "\nSample Rate (Hz):        %u",   hdrFMT.ui_sampleRate );
        fprintf ( stderr, "\nAvg. Bytes/sec:          %u",   hdrFMT.ui_avgBytesPerSec );
        fprintf ( stderr, "\nBlock Align:             %u",   hdrFMT.us_blockAlign );
        fprintf ( stderr, "\nSignificant Bits/sample: %u",   hdrFMT.us_numSignificantBitsPerSample );
    }
    
    if ( hdrFMT.ui_fmtChunkSize > 16 ) {
        fprintf ( stderr,
                  "\n\nThe fmt chunk has %u Extra Format Bytes."
                  "\nExtra format bytes are not supported.\n",
                  hdrFMT.ui_fmtChunkSize - 16 );
        
        fprintf ( stderr,
                  "\nYou might be able to use the Linux 'sox' command"
                  "\nto convert your WAVE file from its current"
                  "\nformat to the format required by this tool"
                  "\nFor a comprehensive assessment of your file's"
                  "\nparameters use the sox command:"
                  "\n   sox -V %s -e stat\n",
                  psInputAudioFile );
        fclose ( fpAudio );
        fpAudio = NULL;

        return false;
    }

//
//  Determine the chunk id of the chunk following the "fmt " chunk
//  when there are no extra format bytes
//
    
    rc = fread ( &hdrCHUNK, sizeof ( hdrCHUNK ), 1, fpAudio );
    
    if ( rc != 1 ) {
        fprintf ( stderr,
                  "\nEOF on input file attempting to read"
                  "\nthe chunk header following the fmt chunk\n" );
        fclose ( fpAudio );
        fpAudio = NULL;

        return false;
    }

    if ( bVerbose ) {
        fprintf ( stderr,
                  "\n\nNext \"chunk\" header type: %c%c%c%c  "
                  "0x%0x 0x%0x 0x%0x 0x%0x\n",
                  hdrCHUNK.c_chunkID[0],
                  hdrCHUNK.c_chunkID[1],
                  hdrCHUNK.c_chunkID[2],
                  hdrCHUNK.c_chunkID[3],
                  hdrCHUNK.c_chunkID[0],
                  hdrCHUNK.c_chunkID[1],
                  hdrCHUNK.c_chunkID[2],
                  hdrCHUNK.c_chunkID[3] );
    
        fprintf ( stderr, "chunk data size: %u\n", hdrCHUNK.ui_chunkSize );
    }
    
//
//  The tool requires the chunk header just read to be either a 'fact' chunk or a 'data' chunk
//
    
    if ( hdrCHUNK.c_chunkID[0] == 'f' &&
         hdrCHUNK.c_chunkID[1] == 'a' &&
         hdrCHUNK.c_chunkID[2] == 'c' &&
         hdrCHUNK.c_chunkID[3] == 't' ) {
        
        unsigned int ui_numSamplesInWaveformChunk = 0;

        rc = fread ( &ui_numSamplesInWaveformChunk,
                     sizeof ( ui_numSamplesInWaveformChunk ),
                     1,
                     fpAudio );
        
        if ( rc != 1 ) {
            fprintf ( stderr,
                      "\nEOF on input file attempting to read"
                      "\nthe content of a \"fact\" chunk\n" );
            fclose ( fpAudio );
            fpAudio = NULL;

            return false;
        }                
             
        if ( bVerbose ) {
            fprintf ( stderr, "Number of samples in waveform data chunk: %u\n",
                              ui_numSamplesInWaveformChunk );
        }
        
        //
        //  This tool now requires the next chunk to be the 'data' chunk.
        //
        
        rc = fread ( &hdrCHUNK, sizeof ( hdrCHUNK ), 1, fpAudio );
        
        if ( rc != 1 ) {
            fprintf ( stderr,
                      "\nEOF on input file attempting to read"
                      "\nthe chunk header following the fmt chunk\n" );
            fclose ( fpAudio );
            fpAudio = NULL;

            return false;
        }

        if ( bVerbose ) {
            fprintf ( stderr,
                      "\nNext \"chunk\" header type: %c%c%c%c  "
                      "0x%0x 0x%0x 0x%0x 0x%0x\n",
                      hdrCHUNK.c_chunkID[0],
                      hdrCHUNK.c_chunkID[1],
                      hdrCHUNK.c_chunkID[2],
                      hdrCHUNK.c_chunkID[3],
                      hdrCHUNK.c_chunkID[0],
                      hdrCHUNK.c_chunkID[1],
                      hdrCHUNK.c_chunkID[2],
                      hdrCHUNK.c_chunkID[3] );
        
            fprintf ( stderr, "chunk data size: %u\n", hdrCHUNK.ui_chunkSize );
        }
    }

//
//  The chunk just read must be the 'data' chunk
//

    if ( hdrCHUNK.c_chunkID[0] != 'd' ||
         hdrCHUNK.c_chunkID[1] != 'a' ||
         hdrCHUNK.c_chunkID[2] != 't' ||
         hdrCHUNK.c_chunkID[3] != 'a' ) {
        fprintf ( stderr,
                  "\nThis tool supports only two sequences of RIFF WAVE file"
                  "\nchunks:\n"
                  "\n   RIFF, fmt, fact, data"
                  "\n       or"
                  "\n   RIFF, fmt, data\n"
                  "\nAn unsupported chunk header was encountered.\n" );

        fprintf ( stderr,
                  "\nYou might be able to use the Linux 'sox' command"
                  "\nto convert your WAVE file from its current"
                  "\nformat to the format required by this tool"
                  "\nFor a comprehensive assessment of your file's"
                  "\nparameters use the sox command:"
                  "\n   sox -V %s -e stat\n",
                  psInputAudioFile );
             
        fclose ( fpAudio );
        fpAudio = NULL;
  
        return false;
    }
    
//
//  The file appears to okay with respect to the sequence of chunk headers.
//  Now make sure that the fmt chunk information describes audio 
//  content that complies with the constraints of this tool.
//
    
    if ( hdrFMT.us_compressionCode !=
         __ATTACKAUDIO_PCM_UNCOMPRESSED_COMPRESSION_CODE ) {
        fprintf ( stderr,
                  "\nThe WAVE file compression format of your audio"
                  "\nfile is not supported. Only code = 1 is supported"
                  "\n(i.e. PCM/Uncompressed). Compression code"
                  "\nspecified by audio file is: %u\n",
                  hdrFMT.us_compressionCode );
        
        fprintf ( stderr,
                  "\nYou might be able to use the Linux 'sox' command"
                  "\nto convert your WAVE file from its current"
                  "\ncompression to the PCM/Uncompressed format"
                  "\nrequired by this tool. For a comprehensive assessment"
                  "\nof your file's parameters, use the sox command:"
                  "\n   sox -V %s -e stat\n",
                  psInputAudioFile );

        fclose ( fpAudio );
        fpAudio = NULL;
  
        return false;
    }

    if ( hdrFMT.us_numChannels != 1 ) {
        fprintf ( stderr,
                  "\nThe number of audio channels in your audio file"
                  "\nis not supported. Only mono (i.e. 1 channel) is"
                  "\nsupported. The number of audio channels in your"
                  "\naudio is: %u\n",
                  hdrFMT.us_numChannels );
        
        fprintf ( stderr,
                  "\nYou might be able to use the Linux 'sox' command"
                  "\nto convert your WAVE file from its current"
                  "\nnumber of audio channels to the mono format"
                  "\nrequired by this tool. For a comprehensive assessment"
                  "\nof your file's parameters, use the sox command:"
                  "\n   sox -V %s -e stat\n",
                  psInputAudioFile );
        fclose ( fpAudio );
        fpAudio = NULL;

        return false;
    }        

    if ( hdrFMT.ui_sampleRate != 8000 ) {
        fprintf ( stderr,
                  "\nThe sample rate of your audio file is not supported."
                  "\nOnly 8000 Hz is supported. The sample rate of"
                  "\nyour audio is: %u Hz\n",
                  hdrFMT.ui_sampleRate );
        
        fprintf ( stderr, 
                  "\nYou might be able to use the Linux 'sox' command"
                  "\nto convert your WAVE file from its current"
                  "\nsample rate to the 8000 Hz required by this tool."
                  "\nFor a comprehensive assessment of your file's"
                  "\nparameters, use the sox command:"
                  "\n   sox -V %s -e stat\n",
                 psInputAudioFile );
        fclose ( fpAudio );
        fpAudio = NULL;

        return false;
    }
    
    if ( hdrFMT.us_numSignificantBitsPerSample != 8  &&
         hdrFMT.us_numSignificantBitsPerSample != 16    ) {
        fprintf ( stderr,
                  "\nThe number of significant bits per audio sample"
                  "\nof your audio file is not supported. Only 8 or"
                  "\n16 bits per sample is supported. The number of"
                  "\nsignificant bits of your audio is: %u\n",
                  hdrFMT.us_numSignificantBitsPerSample );
        
        fprintf ( stderr,
                  "\nYou might be able to use the Linux 'sox' command"
                  "\nto convert your WAVE file from its current number"
                  "\nof significant bits/sample to the 8 or 16 required"
                  "\nby this tool. For a comprehensive assessment"
                  "\nof your file's parameters, use the sox command:"
                  "\n   sox -V %s -e stat\n",
                  psInputAudioFile );
        fclose ( fpAudio );
        fpAudio = NULL;

        return false;
    }
    
//
//  Read, convert (if necessary), and load audio data into the
//  memory buffer pointed to by member variable: mAttackAudioBuf.
//  Limit the magnitude of the audio to the maximum capacity of 
//  the buffer.
//
    
    if ( mAttackAudioMethod == ATTACK_AUDIO_MIX ) {
    
        if ( hdrFMT.us_numSignificantBitsPerSample == 16 ) {
            
            //
            //  The format of the audio is exactly as desired. It can be loaded
            //  directly into the reserved memory area
            //
            
            unsigned char *pAudioBuf;
            pAudioBuf = mAttackAudioBuf;
            
           for ( i = 0; i < __ATTACKAUDIO_G711_MAX_NUMBER_RTP_MSGS_IN_ATTACK; i++ ) {
    
                rc = fread ( pAudioBuf,
                             sizeof ( short ),
                             __ATTACKAUDIO_G711_PAYLOAD_LEN,
                             fpAudio );
                
                if ( rc != __ATTACKAUDIO_G711_PAYLOAD_LEN ) {
                    
                    //
                    //  Less than a complete G.711 packet equivalent audio remained in
                    //  the audio file. So, return only the number of complete G.711
                    //  sample equivalents to the caller.
                    //
                
                    break; // out of the for loop            
                }
                pAudioBuf += sizeof ( pcm_value );
            }
            
            mNumG711PacketEquivalents = i;
            fclose ( fpAudio );
            fpAudio = NULL;
            
            return true;
        }
        
        //
        //  Because of the test above, the only other possibility is that the
        //  number of significant bits/sample must be 8, but a test is made
        //  here - just in case the above logic is modified at a future date
        //
        
        if ( hdrFMT.us_numSignificantBitsPerSample == 8 ) {
            
            //
            //  The audio needs to be converted from unsigned, 8-bit to signed, 16-bit
            //  Even though, by convention, 8-bit PCM is referred to as unsigned, it
            //  is scaled
            //
            
            signed char wavePCM[__ATTACKAUDIO_G711_PAYLOAD_LEN];
                    
            pPcmVal = (short *) mAttackAudioBuf;
            
            for ( i = 0; i < __ATTACKAUDIO_G711_MAX_NUMBER_RTP_MSGS_IN_ATTACK; i++ ) {
                
                rc = fread ( &wavePCM,
                             sizeof ( wavePCM ),
                             1,
                             fpAudio );
                
                if ( rc != 1 ) {
                    
                    //
                    //  Less than a complete G.711 packet equivalent audio remained in
                    //  the audio file. So, return only the number of complete G.711
                    //  packet equivalents to the caller.
                    //
                
                    mNumG711PacketEquivalents = i;
                    fclose ( fpAudio );
                    fpAudio = NULL;
    
                    return true;                
                }
                
                for ( j = 0; j < __ATTACKAUDIO_G711_PAYLOAD_LEN; j++ ) {
                    *pPcmVal = (( (int) wavePCM[j] ) << 8 ) - 32768;
                    pPcmVal++;
                }
    
            }
        
            mNumG711PacketEquivalents = i;
            fclose ( fpAudio );
            fpAudio = NULL;
     
            return true;
        }
    } else if ( mAttackAudioMethod == ATTACK_AUDIO_INSERT ) {
        if ( hdrFMT.us_numSignificantBitsPerSample == 16 ) {

            //
            //  Read each signed, linear 16-bit PCM value, convert
            //  it to PCMU, and load it into memory
            //
            
            short wavePCM;
            unsigned char *pAttackPcm;
            
            pAttackPcm = mAttackAudioBuf;
            
            for ( i = 0;
                  i < __ATTACKAUDIO_G711_MAX_NUMBER_RTP_MSGS_IN_ATTACK;
                  i++ ) {
                      
                for ( j = 0;
                      j < __ATTACKAUDIO_G711_PAYLOAD_LEN;
                      j++ ) {
                    
                    rc = fread ( &wavePCM,
                                 sizeof ( wavePCM ),
                                 1,
                                 fpAudio );
                
                    if ( rc != 1 ) {
                        
                        //
                        //  Less than a complete G.711 packet equivalent audio remained in
                        //  the audio file. So, return only the number of complete G.711
                        //  packet equivalents to the caller.
                        //
                    
                        mNumG711PacketEquivalents = i;
                        fclose ( fpAudio );
                        fpAudio = NULL;
        
                        return true;                
                    }
                    
                    *pAttackPcm = linear2ulaw ( wavePCM );
                    pAttackPcm++;
                }
            }
    
            mNumG711PacketEquivalents = i;        
            fclose ( fpAudio );
            fpAudio = NULL;
            
            return true;
        }

        //
        //  Because of the test above, the only other possibility is that the
        //  number of significant bits/sample must be 8, but a test is made
        //  here - just in case the above logic is modified at a future date
        //
        
        if ( hdrFMT.us_numSignificantBitsPerSample == 8 ) {
            //
            //  Even though, by convention, 8-bit PCM is referred to as unsigned, it
            //  is scaled. So it must be converted to signed, 16-bit, then unscaled,
            //  then converted to 8-bit PCMU.
            //
            
            signed char wavePCM;
            unsigned char *pAttackPcm;
            
            pAttackPcm = mAttackAudioBuf;
            
            for ( i = 0;
                  i < __ATTACKAUDIO_G711_MAX_NUMBER_RTP_MSGS_IN_ATTACK;
                  i++ ) {
                      
                for ( j = 0;
                      j < __ATTACKAUDIO_G711_PAYLOAD_LEN;
                      j++ ) {
                 
                    rc = fread ( &wavePCM,
                                 sizeof ( wavePCM ),
                                 1,
                                 fpAudio );
                    
                    if ( rc != 1 ) {
                        
                        //
                        //  Less than a complete G.711 packet equivalent audio remained in
                        //  the audio file. So, return only the number of complete G.711
                        //  packet equivalents to the caller.
                        //
                    
                        mNumG711PacketEquivalents = i;
                        fclose ( fpAudio );
                        fpAudio = NULL;
        
                        return true;                
                    }
                
                    *pAttackPcm =
                        linear2ulaw ( (( (int) wavePCM ) << 8 ) - 32768 );
                    pAttackPcm++;
                }
            }

            mNumG711PacketEquivalents = i;        
            fclose ( fpAudio );
            fpAudio = NULL;
            
            return true;
        }
    }
    
    return false;

} // end preloadWavAudio


//-----------------------------------------------------------------------------
//
//  preloadTCPdumpAudio ( char *psInputAudioFile )
//
//  This routine expects psInputAudioFile to point to a  
//  string with the name of file containing the pre-recorded
//  audio to load into memory. The file is expected to be a
//  standard libpcap tcpdump formatted file containing
//  G.711 PCMU RTP/UDP/IP/ETHERNET messages.
//  A memory buffer of adequate size must have been
//  previously allocated and pointed to by member variable:
//  mAttackAudioBuf.
//
//  How the pre-recoded audio is treated as it is loaded
//  into memory is a function of whether the audio attack
//  is one of insertion (i.e. replacement), or mixing with
//  the targeted audio stream. The type of audio attack
//  must have already been set in member variable:
//  mAttackAudioMethod.
//
//  Since the audio source is a tcpdump file with
//  captured G.711 audio packets, the content is already
//  unsigned, non-linear, 8-bit, PCMU. In the case of 
//  an audio insertion attack, the audio payload is already
//  in the exact form required for insertion in a G.711
//  audio stream. The payload of each packet in the
//  audio file is simply loaded into memory.
//
//  If the case of a audio mixing attack, the audio payload
//  of each pre-recorded G.711 PCMU packet must
//  be converted to signed, 16-bit, PCM as it is loaded into
//  memory. That is the form required to mix audio.
//  PCMU audio bytes cannot be mixed directly. They must
//  first be converted to signed, 16-bit PCM, mixed, and
//  then converted back to PCMU.
//
//  The number of memory-resident G711 packets is
//  recorded in member variable: mNumG711PacketEquivalents
//
//  The return value of this function is:
//    false - when a failure to load pre-recorded audio occurs
//    true  - when pre-recorded audio is loaded succsssfully
//  
//  FIXME: Declare that some data should remain
//                   memory-resident at all times.
//
//-----------------------------------------------------------------------------

bool  AttackAudio::preloadTCPdumpAudio ( char *psInputAudioFile ) {

    int rc = 0;

    unsigned int i;
    unsigned int numPackets = 0;

    unsigned char *pUlawByte  = NULL;

    const u_char *packet = NULL;

    char pcap_errbuf[ PCAP_ERRBUF_SIZE ];

    pcap_t *h_pcap_tcpdump_rtp = NULL;  //  libpcap "handle"
  
    struct pcap_pkthdr *ppcap_pkthdr = NULL;
        
    unsigned int offset_to_rtp_msg = LIBNET_ETH_H + LIBNET_IPV4_H + LIBNET_UDP_H;
        
    unsigned int offset_to_rtp_payload = 
                    LIBNET_ETH_H + LIBNET_IPV4_H + LIBNET_UDP_H +
                    sizeof ( struct rfc1889_rtp_hdr );
                        
    short *pPcmVal;
    unsigned char *pPcmuVal;

    bool bVerbose       = false;
    bool bPacketsRemain = true;
                    
    pPcmVal = (short *) mAttackAudioBuf;            // used for audio mixing attack
    pPcmuVal = (unsigned char *) mAttackAudioBuf;   // used for audio insertion attack

//
//  Read pre-recorded audio from the RTP packets stored in
//  the specified tcpdump file. 
//
    
    h_pcap_tcpdump_rtp =
        pcap_open_offline ( psInputAudioFile, pcap_errbuf );
    
    if ( h_pcap_tcpdump_rtp == NULL ) {
        fprintf ( stderr,
                  "\nCouldn't open pre-recorded RTP audio file %s: %s\n",
                  psInputAudioFile, pcap_errbuf );
        return false;
    }
    
    fprintf ( stderr,
              "\n\nReading pre-recorded G.711 PCMU audio from input audio"
              "\nfile, converting it into 16-bit linear PCM, and loading"
              "\nit into memory. This is the audio to mix into the"
              "\ntarget live audio stream.\n" );
    
    do {

        //
        //  FIXME:  One outstanding question is whether repeated calls to
        //                   pcap_next_ex( ) result in additional memory being
        //                   consumed. Or, is only - at most - one packet's worth of
        //                   memory consumed despite repeated calls to pcap_next_ex( ).
        //                   After all, the objective of this part of the code is to load
        //                   into memory all of the audio  you'd like to mix into a
        //                   targeted conversion.
        //
        //                   The working presumption until this question can be
        //                   answered is that repeated calls to pcap_next_ex( ) result
        //                   in only one packet's worth of memory being consumed.
        //                   (i.e. the pcap_next_ex( ) routine releases memory for
        //                   the packet "returned" in a prior call and allocates
        //                   memory as needed for the next packet).
        //
        
        rc = pcap_next_ex ( h_pcap_tcpdump_rtp, &ppcap_pkthdr, &packet );
         
        switch ( rc ) {
            case -2: {
                //  EOF
                bPacketsRemain = false;
                break;
            }            
            case -1: {
                //  error occurred reading file
                pcap_perror ( h_pcap_tcpdump_rtp,
                              "\nError reading pre-recorded audio "
                              "capture into memory! " );
                pcap_close ( h_pcap_tcpdump_rtp );
                h_pcap_tcpdump_rtp = NULL;

                return false;
            }
            case 1: {
                //  no error reading packet
                
                numPackets++;
                
                if ( numPackets == 1 && bVerbose ) {
                    fprintf( stderr,
                             "\n\nRTP Header of 1st packet in pre-recorded "
                             "audio:" );
                    decodeAndPrintRTPMsg ( packet );
                }
                
                //
                //  FIXME: tool only supports G.711 audio at this time
                //
                
                rtp_hdr = ( struct rfc1889_rtp_hdr * )
                    ( packet + offset_to_rtp_msg );
                
                if ( rtp_hdr->payloadType !=
                     __ATTACKAUDIO_G711_PAYLOAD_TYPE ) {
                    fprintf ( stderr,
                              "\nPacket #%u of the pre-recorded audio file"
                              "\nis not bearing G.711 u-law encoded audio"
                              "\nPayload type = %u\n",
                              numPackets, rtp_hdr->payloadType );                         
                    pcap_close ( h_pcap_tcpdump_rtp );
                    h_pcap_tcpdump_rtp = NULL;
 
                    return false;
                }
                
                pUlawByte = ( (unsigned char *) packet ) + offset_to_rtp_payload;
                
                if ( mAttackAudioMethod == ATTACK_AUDIO_MIX )
                {                    
                    for ( i = 0; i < __ATTACKAUDIO_G711_PAYLOAD_LEN; i++ ) {
                        *pPcmVal = ulaw2linear ( *pUlawByte );
                        pPcmVal++;
                        pUlawByte++;
                    }
                } else {  
                    memcpy ( pPcmuVal, pUlawByte, 
                             __ATTACKAUDIO_G711_PAYLOAD_LEN );
                    pPcmuVal += __ATTACKAUDIO_G711_PAYLOAD_LEN;
                }
                    
                if ( numPackets ==
                     __ATTACKAUDIO_G711_MAX_NUMBER_RTP_MSGS_IN_ATTACK ) {
                    bPacketsRemain = false;
                }
                break;
            }            
            default: {
                //  no other error code should be returned when using pcap_next_ex( ) to
                //  read a "saved" file.
                fprintf ( stderr, "\nReceived an unexpected return code from "
                                  "pcap_next_ex( ): %d ", rc );
                pcap_perror ( h_pcap_tcpdump_rtp, NULL );
                pcap_close ( h_pcap_tcpdump_rtp );
                h_pcap_tcpdump_rtp = NULL;

                return false;
            }
        }  //  end switch ( rc )
        
    } while ( bPacketsRemain );
    
    mNumG711PacketEquivalents = numPackets;

    pcap_close ( h_pcap_tcpdump_rtp );
    h_pcap_tcpdump_rtp = NULL;
    
    return true;

} // end preloadTCPdumpAudio 


//-----------------------------------------------------------------------------
//
//  decodeAndPrintRTPMsg ( const u_char *packet )
//
//  This routine actually prints most of the content
//  of the headers leading up to the actual RTP payload,
//  but not the payload itself.
//
//  The input parameter, packet, is presumed to point
//  to the start of the Ethernet frame. That Ethernet
//  frame is presumed to contain an RTP/UDP/IP
//  datagram.
//
//  In order, the headers are:
//       Ethernet
//       IP
//       UDP
//       RTP
//
//
//  Portability Issues:
//
//  It is presumed this routine is operating on a
//  little endian machine, requiring the swapping of
//  some of the header content that is in network
//  (i.e. big-endian order).
//
//  It should be noted that the structure used to 
//  define bit fields within the rtp header is very
//  likely also implementation dependent. This
//  routine was developed to execute on an Intel
//  machine (i.e. Pentium) running Red Hat Linux. 
//-----------------------------------------------------------------------------

void AttackAudio::decodeAndPrintRTPMsg ( const u_char *packet ) {
    
//
//      Print some of the Ethernet header content
//

    struct libnet_ethernet_hdr *eth_hdr     = NULL;
    struct libnet_ipv4_hdr *ip_hdr          = NULL;
    struct libnet_udp_hdr *udp_hdr          = NULL;
    
    char macString[18] = "";   //  6 hex bytes * 2 char/byte + 5 colons  + end-of-string
    
    eth_hdr = ( struct libnet_ethernet_hdr * ) packet;

    printf ( "\n\n-----------------\n\n");
        
    sprintf ( macString, "%02x:%02x:%02x:%02x:%02x:%02x", 
              eth_hdr->ether_shost[0],
              eth_hdr->ether_shost[1],
              eth_hdr->ether_shost[2],
              eth_hdr->ether_shost[3],
              eth_hdr->ether_shost[4],
              eth_hdr->ether_shost[5] );
    
    printf ( "source      MAC: %s\n", macString );
    
    macString[0] = '\0';       //  re-initialize workspace string to NUL string
    
    sprintf ( macString, "%02x:%02x:%02x:%02x:%02x:%02x", 
              eth_hdr->ether_dhost[0],
              eth_hdr->ether_dhost[1],
              eth_hdr->ether_dhost[2],
              eth_hdr->ether_dhost[3],
              eth_hdr->ether_dhost[4],
              eth_hdr->ether_dhost[5] );
    
    printf ( "destination MAC: %s\n\n", macString );

//
//      Print some of the IP header content
//

    ip_hdr = ( struct libnet_ipv4_hdr * )
                ( packet + LIBNET_ETH_H );

    //  This union is a workspace permitting an IPv4 address to be accessed as a 
    //   byte array.
    
    union {
        uint32_t ip_addr;
        char ip_bytes[4];
    } ip_addr_union;
    
    char ip_addr_dotted[16];    // workspace to synthesize a dotted IPv4 addr

    ip_addr_dotted[0] = '\0';   //  initialize workspace string to NUL string

    ip_addr_union.ip_addr = ip_hdr->ip_src.s_addr;    

    sprintf( ip_addr_dotted, "%hu.%hu.%hu.%hu",
                ip_addr_union.ip_bytes[0],
                ip_addr_union.ip_bytes[1],
                ip_addr_union.ip_bytes[2],
                ip_addr_union.ip_bytes[3] ); 

    printf ( "source      IP: %s\n", ip_addr_dotted );

    ip_addr_dotted[0] = '\0';   //  initialize workspace string to NUL string

    ip_addr_union.ip_addr = ip_hdr->ip_dst.s_addr;    

    sprintf( ip_addr_dotted, "%hu.%hu.%hu.%hu",
                ip_addr_union.ip_bytes[0],
                ip_addr_union.ip_bytes[1],
                ip_addr_union.ip_bytes[2],
                ip_addr_union.ip_bytes[3] ); 

    printf ( "destination IP: %s\n\n", ip_addr_dotted );

//
//      Print some of the UDP header content
//

    udp_hdr = ( struct libnet_udp_hdr * )
                ( packet + LIBNET_ETH_H + LIBNET_IPV4_H );
    
    printf ( "source      port: %u\n",
                ntohs ( udp_hdr->uh_sport ) );
    
    printf ( "destination port: %u\n\n",
                ntohs ( udp_hdr->uh_dport ) );
    
    printf ( "UDP packet  length: %u\n\n",
                ntohs ( udp_hdr->uh_ulen ) );
            
    rtp_hdr = ( struct rfc1889_rtp_hdr * )
                ( packet + LIBNET_ETH_H + LIBNET_IPV4_H + LIBNET_UDP_H );
    
    printf ( "RTP message length: %u\n",
                ntohs ( udp_hdr->uh_ulen ) - LIBNET_UDP_H );
    
    printf ( "Size of RTP Header: %u\n", sizeof( struct rfc1889_rtp_hdr ) );
        
//
//      Print RTP header content
//

/*    
    printf ( "RTP Header Dump:\n");
    
    const u_char *rtp_hdr_bytes;
    
    rtp_hdr_bytes = packet + LIBNET_ETH_H + LIBNET_IPV4_H + LIBNET_UDP_H;
    
    printf ( "%02x %02x %02x %02x\n"
             "%02x %02x %02x %02x\n"
             "%02x %02x %02x %02x\n",
             rtp_hdr_bytes[0],
             rtp_hdr_bytes[1],
             rtp_hdr_bytes[2],
             rtp_hdr_bytes[3],
             rtp_hdr_bytes[4],
             rtp_hdr_bytes[5],
             rtp_hdr_bytes[6],
             rtp_hdr_bytes[7],
             rtp_hdr_bytes[8],
             rtp_hdr_bytes[9],
             rtp_hdr_bytes[10],
             rtp_hdr_bytes[11] );
*/
        
    printf ( "RTP Version: %u\n", rtp_hdr->version );
    
    printf ( "RTP Packet Padded?: %s\n",
                ( rtp_hdr->bPaddingIncluded == 0 )? "no":"yes" );
    
    printf ( "RTP Packet Fixed Hdr Followed by Extension Hdr?: %s\n",
                ( rtp_hdr->bExtensionIncluded == 0 )? "no":"yes" );
    
    printf ( "RTP Packet CSRC Count: %u\n", rtp_hdr->cc );

    printf ( "RTP Packet Marked?: %s\n",
                ( rtp_hdr->bMarker == 0 )? "no":"yes" );
    
    printf ( "RTP Packet Payload Type: %u\n", rtp_hdr->payloadType );
    
    printf ( "RTP Packet Sequence #: %u\n",
                ntohs ( rtp_hdr->sequenceNumber ) );
    
    printf ( "RTP Packet Timestamp: %u\n",
                ntohl ( rtp_hdr->timestamp ) );
    
    printf ( "RTP Packet SSRC: %u\n",
                ntohl ( rtp_hdr->ssrc ) );

    printf ( "\n-----------------\n\n");

    
} //  end decodeAndPrintRTPMsg

