//-----------------------------------------------------------------------------
//
// SdpMessage.h - Encapsulates SDP messages.
//
//    Copyright (C) 2004  Mark D. Collier
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//   Author: Mark D. Collier   - 12/01/2006   v1.1
//                   Mark D. Collier   -  04/26/2004  v1.0
//         www.securelogix.com - mark.collier@securelogix.com
//         www.hackingexposedvoip.com
//
//-----------------------------------------------------------------------------

#ifndef SDP_MESSAGE_H
#define SDP_MESSAGE_H

class  SdpMessage
{
    public:

        class Origin
        {
            public:
                Origin( void );
                // aSdpLine contents will be duplicated within Origin.
                Origin( char *  aSdpLine );
                // Ensure aUserName, aNetworkType, aAddressType, and aAddress
                // are all malloced or NULL, as they will be freed.
                Origin( char *  aUserName,
                        int     aSessionId,
                        int     aVersion,
                        char *  aNetworkType,
                        char *  aAddressType,
                        char *  aAddress );
                ~Origin( void );

                // Do not free return value.
                char *  GetUserName( void );
                int     GetSessionId( void );
                int     GetVersion( void );
                // Do not free return value.
                char *  GetNetworkType( void );
                // Do not free return value.
                char *  GetAddressType( void );
                // Do not free return value.
                char *  GetAddress( void );
                // Caller must free return value. Contains trailing newline.
                char *  GetSdpLine( void );

            private:
                char *  mUserName;
                int     mSessionId;
                int     mVersion;
                char *  mNetworkType;
                char *  mAddressType;
                char *  mAddress;
        };

        class ConnectionData
        {
            public:
                ConnectionData( void );
                // aSdpLine contents will be duplicated within ConnectionData.
                ConnectionData( char *  aSdpLine );
                ConnectionData( char *  aNetworkType,
                                char *  aAddressType,
                                char *  aAddress );
                ~ConnectionData( void );

                // Do not free return values.
                char *  GetNetworkType( void );
                char *  GetAddressType( void );
                char *  GetAddress( void );
                // Caller must free return value. Contains trailing newline.
                char *  GetSdpLine( void );

            private:
                char *  mNetworkType;
                char *  mAddressType;
                char *  mAddress;
        };

        class MediaDescription
        {
            public:
                MediaDescription( void );
                // aSdpLines contents will be duplicated within MediaDescription
                MediaDescription( char *  aSdpLines );
                // Ensure aType, aProtocol, and aFormats are malloced or NULL,
                // as they will be freed. Ensure aConnectionData is newed or
                // NULL, as it will be deleted.
                MediaDescription( char *            aType,
                                  char *            aPortSpec,
                                  char *            aProtocol,
                                  int               aFormatsCount,
                                  char **           aFormats,
                                  ConnectionData *  aConnectionData );
                ~MediaDescription( void );

                MediaDescription *  GetNext( void );
                void                SetNext( MediaDescription *  aValue );

                // Do not free return values.
                char *            GetType( void );
                // Do not free return values.
                char *            GetPortSpec( void );
                int               GetPortIP4Value( void );
                int               GetPortIP4Count( void );
                // Do not delete return value.
                char *            GetProtocol( void );
                int               GetFormatsCount( void );
                // Do not free return value.
                char **           GetFormats( void );
                // Do not delete return value.
                ConnectionData *  GetConnectionData( void );
                // Caller must free return value. Contains trailing newline.
                char *            GetSdpLines( void );

            private:
                MediaDescription *  mNext;
                char *              mType;
                char *              mPortSpec;
                char *              mProtocol;
                int                 mFormatsCount;
                char **             mFormats;
                ConnectionData *    mConnectionData;
        };

        SdpMessage( void );
        // aPayload contents will be duplicated within SdpMessage.
        SdpMessage( char *  aPayload );
        // Ensure aName is malloced as it will be freed. Ensure aOrigin,
        // aConnectionData, and aMediaDescriptions are newed or NULL, as they
        // will be deleted.
        SdpMessage( int                 aVersion,
                    Origin *            aOrigin,
                    char *              aName,
                    ConnectionData *    aConnectionData,
                    MediaDescription *  aMediaDescriptions );
        ~SdpMessage( void );

        int                 GetVersion( void );

        // Do not delete return value.
        Origin *            GetOrigin( void );

        // Do not free return value.
        char *              GetName( void );

        // Do not delete return value.
        ConnectionData *    GetConnectionData( void );

        // Do not delete return value(s).
        MediaDescription *  GetMediaDescriptions( void );

        // Caller must free return value.
        char *              GetPacket( void );


    private:

        int                 mVersion;
        Origin *            mOrigin;
        char *              mName;
        ConnectionData *    mConnectionData;
        MediaDescription *  mMediaDescriptions;
};

#endif

