//-----------------------------------------------------------------------------
//
// SipCall.h - Handles a single call. A single SipCall
//             object should not be used to place or receive
//             more than one single call.
//
//    Copyright (C) 2004  Mark D. Collier
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//   Author: Mark D. Collier   - 12/01/2006   v1.1
//                   Mark D. Collier   -  04/26/2004  v1.0
//         www.securelogix.com - mark.collier@securelogix.com
//         www.hackingexposedvoip.com
//
//-----------------------------------------------------------------------------

#ifndef SIP_CALL_H
#define SIP_CALL_H

#include <arpa/inet.h>
#include <netinet/in.h>
#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netdb.h>

#include "jrtplib/rtpsession.h"
#include "jrtplib/rtppacket.h"

#include "util.h"
#include "SipUri.h"
#include "SipMessage.h"
#include "SdpMessage.h"
#include "SipUdpPort.h"
#include "RtpHandler.h"
#include "SipDispatcher.h"
#include "AttackAudio.h"
#include "SipEndPoint.h"

//
//  Need to wrap the include since this is a C++ application
//  accessing a C library.
//

extern "C" {
#include "g711conversions.h"
};

class SipUri;
class SdpMessage;
class SipUdpPort;
class SipEndPoint;
class SipCallTimer;

class  SipCall : public SipUdpPort::UdpSource, public RtpHandler::RtpEndPoint
{
    public:
        typedef enum
        {
            STATE_INVALID = -1,
            STATE_TIMER_FIRE,
            STATE_CREATED,
            STATE_GOT_MESSAGE,
            STATE_SEND_INVITE,
            STATE_RESEND_INVITE,
            STATE_SENT_INVITE,
            STATE_INVITE_GOT_PROVISIONAL,
            STATE_INVITE_GOT_PROVISIONAL_WAIT,
            STATE_GOT_OK_FOR_INVITE,
            STATE_SEND_ACK_FOR_INVITE,
            STATE_SEND_TRYING_FOR_INVITE,
            STATE_SENT_TRYING_FOR_INVITE,
            STATE_SEND_RINGING,
            STATE_SENT_RINGING,
            STATE_SEND_OK_FOR_INVITE,
            STATE_RESEND_OK_FOR_INVITE,
            STATE_SENT_OK_FOR_INVITE,
            STATE_ESTABLISHED,
            STATE_SEND_BYE,
            STATE_RESEND_BYE,
            STATE_SENT_BYE,
            STATE_GOT_OK_FOR_BYE,
            STATE_GOT_BYE,
            STATE_SEND_OK_FOR_BYE,
            STATE_TERMINATE,
            STATE_TERMINATED
        }  State;

        static char *  GetNameForState( State  aState );

        SipCall( void );
        ~SipCall( void );

        // If true is returned, aMessage was handled and deleted.
        bool           UpdateStateMachine( SipMessage *  aMessage,
                                           State         aDesiredState );
        void           Connect( void );
        void           Disconnect( void );
        // If true is returned, aMessage was handled and deleted.
        bool           Incoming( SipMessage *  aMessage );
        State          GetState( void );
        SipEndPoint *  GetEndPoint( void );
        void           SetEndPoint( SipEndPoint *  aEndPoint );
        char *         GetCallId( void );
        char *         GetLocalTag( void );
        char *         GetRemoteTag( void );
        // aIdentifier should be newed, as it will be deleted.
        void           SetRemoteIdentifier( SipIdentifier *  aIdentifier );
        // aIdentifier should be newed, as it will be deleted.
        void           SetContactIdentifier( SipIdentifier *  aIdentifier );
        void           SendShell( char *  aCommandLine );
        void           SendRtpBurst( int aSeconds );
        void           SendRtpJitter( int aSeconds );
        void           SendRtpUuTest( void );
        void           SetRelayParty( SipCall *  aCall );
        void           SetFarEndRoleToCallee( void );
        void           SetFarEndRoleToCaller( void );
        void           SetTappedParty( SipCall *  aCall );
        void           SetTappedRelayParty( SipCall *  aCall );
        bool           FarEndRoleIsCaller( void );
        bool           FarEndRoleIsCallee( void );

        // UdpSource Functions
        //
        in_addr_t  GetUdpAddress( void );
        in_port_t  GetUdpPort( void );
        // Caller must free return value.
        char *     GetUdpPacket( void );
        //
        // End of UdpSource Functions

        // RtpEndPoint Functions
        //
        RTPSession *  GetSession( void );
        int           GetRtpSpacing( void ); // in milliseconds
        timeval       GetBurstUntil( void );
        timeval       GetJitterUntil( void );
        RTPPacket *   GetNextRtpPacket( void );
        bool          GetUuTest( void );
        void          IncomingRtpPacket( RTPPacket *  aPacket );
        //
        // End of RtpEndPoint Functions


    private:
        
        RTPPacket *     TappedPartyAudioProcessing( void );
    
        class  stringListNode
        {
            public:
                struct stringListNode *  Next;
                char *                   Value;
        };

        // If true is returned, aMessage was handled and deleted.
        bool          setup( SipMessage *  aMessage );
        SipMessage *  getInvite( void );
        SipMessage *  getAckForInvite( void );
        SipMessage *  getBye( void );
        SipMessage *  getTryingForInvite( void );
        SipMessage *  getRinging( void );
        SipMessage *  getOkForInvite( void );
        SipMessage *  getOkForBye( void );
        void          setupLocalMediaPort( void );
        void          recordRouting( SipMessage *  aMessage );
        void          addRouting( SipMessage *  aMessage );

        bool                mIsCaller;
        SipEndPoint *       mEndPoint;
        State               mState;
        pthread_mutex_t     mStateMutex;
        SipIdentifier *     mLocalIdentifier;
        SipIdentifier *     mRemoteIdentifier;
        SipIdentifier *     mContactIdentifier;
        char *              mLocalTag;
        char *              mRemoteTag;
        in_addr_t           mLocalAddress;
        in_addr_t           mRemoteAddress;
        in_port_t           mLocalPort;
        in_port_t           mRemotePort;
        char *              mCallId;
        char *              mLastBranch;
        unsigned int        mCseq;
        char *              mRemoteCseq;
        stringListNode *    mRoutes;
        stringListNode *    mViaHeadersHead;
        stringListNode *    mViaHeadersTail;
        SipCallTimer *      mTimer;
        int                 mTimerWaitTime;
        // Timeout time is in milliseconds. It is formulated with:
        // ( timeval.tv_sec & 0x003fffff ) * 1000 + timeval.tv_usec / 1000;
        unsigned long       mTimerTimeout;
        in_addr_t           mRemoteMediaAddress;
        in_port_t           mRemoteMediaPort;
        in_port_t           mLocalMediaPort;
        RTPSession *        mRtpSession;
        int                 mRtpSpacing; // in milliseconds
        timeval             mRtpBurstUntil;
        timeval             mRtpJitterUntil;
        bool                mRtpUuTest;
        SipCall *           mRelayParty;
        SipCall *           mTappedParty;
        SipCall *           mTappedRelayParty;
        bool                mbFarEndRoleIsCaller;
        RTPPacket *         mLastRtpPacket;
        timeval             mLastRtpPacketExpiration;
        char *              mSendShellCommand;
        char *              mSendPrint;

        unsigned int        mSipEndpointPollingDelay;
        unsigned int        mNextAttackAudioPacketNumber;
        AttackAudio *       mAttackAudio;
};


class  SipCallTimer : public SipDispatcher::Timer
{
    public:
        SipCallTimer( SipCall *  aValue );
        ~SipCallTimer( void );

        void           SetWhen( unsigned long  aValue );
        unsigned long  GetWhen( void );
        void           Fire( void );

    private:
        unsigned long  mWhen;
        SipCall *      mCall;
};

#endif

