//-----------------------------------------------------------------------------
//
// SipDispatcher.cpp - Pulls messages from a SipUdpPort
//                     and delivers them to registered SipCalls,
//                     SipEndPoints, and special undispatched
//                     message handlers. As a last resort, the
//                     dispatcher will respond with an error
//                     message to completely undispatched
//                     messages. The SipDispatcher also provides
//                     a timer facility allowing objects such as
//                     SipCalls to update their state at a particular
//                     time.
//
//    Copyright (C) 2004  Mark D. Collier
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//   Author: Mark D. Collier   - 12/01/2006   v1.1
//                   Mark D. Collier   -  04/26/2004  v1.0
//         www.securelogix.com - mark.collier@securelogix.com
//         www.hackingexposedvoip.com
//
//-----------------------------------------------------------------------------

#include <arpa/inet.h>
#include <netinet/in.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <pthread.h>
#include <sys/time.h>
#include <netdb.h>

#include "util.h"
#include "SipMessage.h"
#include "SipUri.h"
#include "SipUdpPort.h"
#include "RtpHandler.h"
#include "SipDispatcher.h"
#include "SipEndPoint.h"
#include "SipProxyEndPoint.h"
#include "SipRegistrar.h"
#include "SipCall.h"

#define MAX_TAG_LENGTH  127
#define MAX_USER_LENGTH 127

#define malloc  mymalloc
#define free    myfree
#define strdup  mystrdup
#define strndup mystrndup


SipDispatcher::SipDispatcher( void )
{
    mUdpPort                 = NULL;
    pthread_mutex_init( &mCallListMutex, NULL );
    mCallList                = NULL;
    pthread_mutex_init( &mEndPointListMutex, NULL );
    mEndPointList            = NULL;
    pthread_mutex_init( &mProxyEndPointListMutex, NULL );
    mProxyEndPointList            = NULL;
    pthread_mutex_init( &mRegistrarListMutex, NULL );
    mRegistrarList            = NULL;
    pthread_mutex_init( &mUndispatchedListMutex, NULL );
    mUndispatchedList        = NULL;
    pthread_mutex_init( &mTimerListMutex, NULL );
    mTimerList               = NULL;
    mTimerListSize           = 0;
    mStopFlag                = 0;
    mSipRfcT1                = 500;
    mSipRfcT2                = 4000;
    mRetransmissionsDesired  = true;
    mTimeoutsDesired         = true;
    mProvisionalsDesired     = true;
}


SipDispatcher::~SipDispatcher( void )
{
}


static char *  fallbackResponseCopyHeaders[7] =
{
    "To", "t", "From", "f", "Call-Id", "i", "CSeq"
};

static bool  fallbackUndispatchedHandler( SipDispatcher *  aDispatcher,
                                          SipMessage    *  aMessage )
{
    SipMessage *  response;
    unsigned int  ux;
    int  ix;
    SipHeader *   header;
    char *        requestMethod;
    SipIdentifier *  headerIdentifier;
    SipUri *  headerUri;
    char * cp;
    struct hostent *  hent;
    in_addr_t    address;

    // As per RFC 3261:
    // We are ignoring responses to messages we don't think we've sent.
    // We are ignoring messages we can't even translate.
    // We will now respond as best as possible to requests we can't service.
    if ( aMessage->IsRequest() )
    {
        response = new SipMessage( false );

        header = aMessage->GetHeader( "via" );
        headerIdentifier =  header
                          ? header->GetIdentifier( 0 )
                          : NULL;
        headerUri = headerIdentifier ? headerIdentifier->GetUri() : NULL;
        cp = headerUri ? headerUri->GetHost() : NULL;
        hent = cp ? gethostbyname( cp ) : NULL;
        if ( hent && *hent->h_addr_list )
        {
            memcpy( ( char * )&address, *hent->h_addr_list, sizeof( address ) );
            response->SetDestinationAddress( address );
            if ( headerUri->GetPort() > 0 )
            {
                response->SetDestinationPort( htons( headerUri->GetPort() ) );
            }
            else
            {
                response->SetDestinationPort( htons( 5060 ) );
            }
        }

        for ( ix = 0; ix < aMessage->GetHeaderCount(); ix++ )
        {
            header = aMessage->GetHeader( ix );
            if ( header && strcasecmp( header->GetName(), "via" ) == 0 )
            {
                response->AddHeader( new SipHeader( header) );
            }
        }

        response->SetSourceAddress( aMessage->GetDestinationAddress() );
        response->SetSourcePort( aMessage->GetDestinationPort() );

        response->SetResponseCode( 501 );
        response->SetResponseText( strdup( "Not Implemented" ) );

        for ( ux = 0;
              ux < sizeof( fallbackResponseCopyHeaders ) / sizeof( char * );
              ux++ )
        {
            header = aMessage->GetHeader( fallbackResponseCopyHeaders[ux] );
            if ( header )
            {
                response->AddHeader( new SipHeader( header ) );
            }
        }

        requestMethod = aMessage->GetRequestMethod();
        if ( strcmp( "OPTIONS", requestMethod ) == 0 )
        {
            // FIXME: Respond
            response->SetResponseCode( 405 );
            response->SetResponseText( strdup( "Method Not Allowed" ) );
        }
        else if ( strcmp( "INVITE", requestMethod ) == 0 )
        {
            response->SetResponseCode( 404 );
            response->SetResponseText( strdup( "Not Found" ) );
        }
        else if ( strcmp( "REGISTER", requestMethod ) == 0 )
        {
            response->SetResponseCode( 405 );
            response->SetResponseText( strdup( "Method Not Allowed" ) );
        }
        else if (   strcmp( "ACK",    requestMethod ) == 0
                 || strcmp( "BYE",    requestMethod ) == 0
                 || strcmp( "CANCEL", requestMethod ) == 0 )
        {
            response->SetResponseCode( 481 );
            response->SetResponseText(
                                  strdup( "Call/Transaction Does Not Exist" ) );
        }
        else
        {
            response->SetResponseCode( 501 );
            response->SetResponseText( strdup( "Not Implemented" ) );
        }

        aDispatcher->GetUdpPort()->AddOutgoing( response );
    }

    delete aMessage;
    return true;
}


void  SipDispatcher::Run( void )
{
    bool                isInvite;
    bool                isInviteResponse;
    bool                skipSleep;
    callNode *          callNodeI;
    callNode *          callNodesHead;
    callNode *          callNodesTail;
    char *              callId;
    char *              cCallId;
    char *              cLocalTag;
    char *              cRemoteTag;
    char *              epUser;
    SipHeader *         header;
    char *              localTag;
    char *              remoteTag;
    char *              user1;
    char *              user2;
    char *              domain;
    char *              rDomain;
    endPointNode *      endPointNodeI;
    endPointNode *      endPointNodesHead;
    endPointNode *      endPointNodesTail;
    proxyEndPointNode *      proxyEndPointNodeI;
    proxyEndPointNode *      proxyEndPointNodesHead;
    proxyEndPointNode *      proxyEndPointNodesTail;
    registrarNode *     registrarNodeI;
    registrarNode *     registrarNodesHead;
    registrarNode *     registrarNodesTail;
    SipMessage *        message;
    struct timeval      tv;
    timerNode *         timerNodeI;
    Timer *             timer;
    undispatchedNode *  undispatchedNodeI;
    undispatchedNode *  undispatchedNodesHead;
    undispatchedNode *  undispatchedNodesTail;
    unsigned long       now;
    SipIdentifier *     headerIdentifier;


    mStopFlag = 0;
    while ( !mStopFlag )
    {
        skipSleep = true;
        while ( skipSleep )
        {
            skipSleep = false;
            pthread_mutex_lock( &mTimerListMutex );
            gettimeofday( &tv, NULL );
            now        = ( tv.tv_sec & 0x003fffff ) * 1000 + tv.tv_usec / 1000;
            timerNodeI = mTimerList;
            if ( timerNodeI )
            {
                if ( timerNodeI->Value->GetWhen() > now )
                {
                    timerNodeI = NULL;
                }
            }
            pthread_mutex_unlock( &mTimerListMutex );
            if ( timerNodeI )
            {
                skipSleep = true;
                timer     = timerNodeI->Value;
                RemoveTimer( timer );
                timer->Fire();
            }

            message = mUdpPort->GetNextIncoming();

            if ( message && !message->GetError() )
            {
                skipSleep        = true;
                isInvite         = false;
                isInviteResponse = false;
                if ( message->IsRequest() )
                {
                    isInvite =
                           strcmp( "INVITE", message->GetRequestMethod() ) == 0;
                }
                else
                {
                    header = message->GetHeader( "cseq" );
                    if ( header )
                    {
                        isInviteResponse =
                                 strstr( header->GetValue(), "INVITE" ) != NULL;
                    }
                }

                localTag  = NULL;
                remoteTag = NULL;
                header = message->GetHeader( "from" );
                if ( !header )
                {
                    header = message->GetHeader( "f" );
                }
                headerIdentifier = header ? header->GetIdentifier( 0 ) : NULL;
                if ( headerIdentifier )
                {
                    if ( message->IsResponse() )
                    {
                        localTag = headerIdentifier->GetParameterValue( "tag" );
                    }
                    else
                    {
                        remoteTag =
                                   headerIdentifier->GetParameterValue( "tag" );
                    }
                }
                header = message->GetHeader( "to" );
                if ( !header )
                {
                    header = message->GetHeader( "t" );
                }
                headerIdentifier = header ? header->GetIdentifier( 0 ) : NULL;
                if ( headerIdentifier )
                {
                    if ( message->IsResponse() )
                    {
                        remoteTag =
                                   headerIdentifier->GetParameterValue( "tag" );
                    }
                    else
                    {
                        localTag = headerIdentifier->GetParameterValue( "tag" );
                    }
                }

                header = message->GetHeader( "call-id" );
                if ( header )
                {
                    callId = header->GetValue();
                }
                else
                {
                    callId = NULL;
                }
                if ( !callId )
                {
                    header = message->GetHeader( "i" );
                    if ( header )
                    {
                        callId = header->GetValue();
                    }
                }

                if ( callId )
                {
                    callNodesHead = NULL;
                    callNodesTail = NULL;

                    pthread_mutex_lock( &mCallListMutex );
                    for ( callNodeI = mCallList;
                          callNodeI;
                          callNodeI = callNodeI->Next )
                    {
                        cCallId = callNodeI->Call->GetCallId();
                        if ( !cCallId )
                        {
                            continue;
                        }
                        cLocalTag = callNodeI->Call->GetLocalTag();
                        cRemoteTag = callNodeI->Call->GetRemoteTag();
                        if (   strcmp( callId, cCallId ) == 0
                            && (   isInvite
                                || (   (   localTag
                                        && cLocalTag
                                        && strcmp( localTag, cLocalTag ) == 0 )
                                    && (   (   !cRemoteTag
                                            && (   isInviteResponse
                                                || !remoteTag ) )
                                        || (   cRemoteTag
                                            && remoteTag
                                            &&    strcmp( cRemoteTag,
                                                          remoteTag )
                                               == 0 ) ) ) ) )
                        {
                            if ( callNodesTail )
                            {
                                callNodesTail->Next = new callNode();
                                callNodesTail       = callNodesTail->Next;
                            }
                            else
                            {
                                callNodesHead = callNodesTail = new callNode();
                            }
                            callNodesTail->Next = NULL;
                            callNodesTail->Call = callNodeI->Call;
                        }
                        sched_yield();
                    }
                    pthread_mutex_unlock( &mCallListMutex );

                    callNodeI = callNodesHead;
                    while ( callNodeI )
                    {
                        if ( callNodeI->Call->Incoming( message ) )
                        {
                            message = NULL;
                            break;
                        }
                        callNodeI = callNodeI->Next;
                    }
                    while ( callNodesHead )
                    {
                        callNodeI     = callNodesHead;
                        callNodesHead = callNodeI->Next;
                        delete callNodeI;
                    }
                    callNodesTail = NULL;
                }

                if ( message )
                {
                    if ( message->IsRequest() )
                    {
                        user1  = message->GetRequestUri()->GetUser();
                        user2 = NULL;
                    }
                    else
                    {
                        header = message->GetHeader( "from" );
                        if ( !header )
                        {
                            header = message->GetHeader( "f" );
                        }
                        headerIdentifier =  header
                                          ? header->GetIdentifier( 0 )
                                          : NULL;
                        if ( headerIdentifier )
                        {
                            user1 = headerIdentifier->GetUri()->GetUser();
                        }
                        else
                        {
                            user1 = NULL;
                        }
                        user2  = NULL;
                    }
                    endPointNodesHead = NULL;
                    endPointNodesTail = NULL;

                    pthread_mutex_lock( &mEndPointListMutex );
                    for ( endPointNodeI = mEndPointList;
                          endPointNodeI;
                          endPointNodeI = endPointNodeI->Next )
                    {
                        epUser = endPointNodeI->EndPoint->GetUser();
                        if (   ( user1 && strcmp( user1, epUser ) == 0 )
                            || ( user2 && strcmp( user2, epUser ) == 0 ) )
                        {
                            if ( endPointNodesTail )
                            {
                                endPointNodesTail->Next = new endPointNode();
                                endPointNodesTail = endPointNodesTail->Next;
                            }
                            else
                            {
                                endPointNodesHead = new endPointNode();
                                endPointNodesTail = endPointNodesHead;
                            }
                            endPointNodesTail->Next     = NULL;
                            endPointNodesTail->EndPoint =
                                                        endPointNodeI->EndPoint;
                        }
                        sched_yield();
                    }
                    pthread_mutex_unlock( &mEndPointListMutex );

                    endPointNodeI = endPointNodesHead;
                    while ( endPointNodeI )
                    {
                        if ( endPointNodeI->EndPoint->Incoming( message ) )
                        {
                            message = NULL;
                            break;
                        }
                        endPointNodeI = endPointNodeI->Next;
                    }
                    while ( endPointNodesHead )
                    {
                        endPointNodeI      = endPointNodesHead;
                        endPointNodesHead = endPointNodeI->Next;
                        delete endPointNodeI;
                    }
                    endPointNodesTail = NULL;

                }

                if ( message )
                {
                    user1 =  message->IsRequest()
                           ? message->GetRequestUri()->GetUser()
                           : NULL;
                    header = message->GetHeader( "via" );
                    headerIdentifier =  header
                                      ? header->GetIdentifier( 0 )
                                      : NULL;
                    user2 =  headerIdentifier
                           ? headerIdentifier->GetParameterValue( "user" )
                           : NULL;
                    proxyEndPointNodesHead = NULL;
                    proxyEndPointNodesTail = NULL;

                    pthread_mutex_lock( &mProxyEndPointListMutex );
                    for ( proxyEndPointNodeI = mProxyEndPointList;
                          proxyEndPointNodeI;
                          proxyEndPointNodeI = proxyEndPointNodeI->Next )
                    {
                        epUser = proxyEndPointNodeI->ProxyEndPoint->GetUser();
                        if (   ( user1 && strcmp( user1, epUser ) == 0 )
                            || ( user2 && strcmp( user2, epUser ) == 0 ) )
                        {
                            if ( proxyEndPointNodesTail )
                            {
                                proxyEndPointNodesTail->Next = new proxyEndPointNode();
                                proxyEndPointNodesTail = proxyEndPointNodesTail->Next;
                            }
                            else
                            {
                                proxyEndPointNodesHead = new proxyEndPointNode();
                                proxyEndPointNodesTail = proxyEndPointNodesHead;
                            }
                            proxyEndPointNodesTail->Next     = NULL;
                            proxyEndPointNodesTail->ProxyEndPoint =
                                                        proxyEndPointNodeI->ProxyEndPoint;
                        }
                        sched_yield();
                    }
                    pthread_mutex_unlock( &mProxyEndPointListMutex );

                    proxyEndPointNodeI = proxyEndPointNodesHead;
                    while ( proxyEndPointNodeI )
                    {
                        if ( proxyEndPointNodeI->ProxyEndPoint->Incoming( message ) )
                        {
                            message = NULL;
                            break;
                        }
                        proxyEndPointNodeI = proxyEndPointNodeI->Next;
                    }
                    while ( proxyEndPointNodesHead )
                    {
                        proxyEndPointNodeI      = proxyEndPointNodesHead;
                        proxyEndPointNodesHead = proxyEndPointNodeI->Next;
                        delete proxyEndPointNodeI;
                    }
                    proxyEndPointNodesTail = NULL;

                }

                if ( message )
                {

                    if (   message->IsRequest()
                        && strcmp( "REGISTER",
                                   message->GetRequestMethod() ) == 0 )
                    {
                        domain = message->GetRequestUri()->GetHost();

                        registrarNodesHead = NULL;
                        registrarNodesTail = NULL;

                        pthread_mutex_lock( &mRegistrarListMutex );
                        for ( registrarNodeI = mRegistrarList;
                              registrarNodeI;
                              registrarNodeI = registrarNodeI->Next )
                        {
                            rDomain = registrarNodeI->Registrar->GetDomain();
                            if (   domain && rDomain
                                && strcmp( domain, rDomain ) == 0 )
                            {
                                if ( registrarNodesTail )
                                {
                                    registrarNodesTail->Next =
                                                            new registrarNode();
                                    registrarNodesTail =
                                                       registrarNodesTail->Next;
                                }
                                else
                                {
                                    registrarNodesHead = new registrarNode();
                                    registrarNodesTail = registrarNodesHead;
                                }
                                registrarNodesTail->Next      = NULL;
                                registrarNodesTail->Registrar =
                                                      registrarNodeI->Registrar;
                            }
                            sched_yield();
                        }
                        pthread_mutex_unlock( &mRegistrarListMutex );

                        registrarNodeI = registrarNodesHead;
                        while ( registrarNodeI )
                        {
                            if ( registrarNodeI->Registrar->Incoming(
                                                                     message ) )
                            {
                                message = NULL;
                                break;
                            }
                            registrarNodeI = registrarNodeI->Next;
                        }
                        while ( registrarNodesHead )
                        {
                            registrarNodeI      = registrarNodesHead;
                            registrarNodesHead = registrarNodeI->Next;
                            delete registrarNodeI;
                        }
                        registrarNodesTail = NULL;
                    }

                }

                if ( message )
                {
                    undispatchedNodesHead = NULL;
                    undispatchedNodesTail = NULL;

                    pthread_mutex_lock( &mUndispatchedListMutex );
                    for ( undispatchedNodeI = mUndispatchedList;
                          undispatchedNodeI;
                          undispatchedNodeI = undispatchedNodeI->Next )
                    {
                        if ( undispatchedNodesHead )
                        {
                            undispatchedNodesTail->Next =
                                                         new undispatchedNode();
                            undispatchedNodesTail = undispatchedNodesTail->Next;
                        }
                        else
                        {
                            undispatchedNodesHead = undispatchedNodesTail =
                                                         new undispatchedNode();
                        }
                        undispatchedNodesTail->Next     = NULL;
                        undispatchedNodesTail->Handler =
                                                     undispatchedNodeI->Handler;
                        sched_yield();
                    }
                    pthread_mutex_unlock( &mUndispatchedListMutex );

                    undispatchedNodeI = undispatchedNodesHead;
                    while ( undispatchedNodeI )
                    {
                        if ( undispatchedNodeI->Handler->Incoming( message ) )
                        {
                            message = NULL;
                            break;
                        }
                        undispatchedNodeI = undispatchedNodeI->Next;
                    }
                    while ( undispatchedNodesHead )
                    {
                        undispatchedNodeI      = undispatchedNodesHead;
                        undispatchedNodesHead = undispatchedNodeI->Next;
                        delete undispatchedNodeI;
                    }
                    undispatchedNodesTail = NULL;
                }

                if ( message )
                {
                    if ( !fallbackUndispatchedHandler( this, message ) )
                    {
                        delete message;
                    }
                    message = NULL;
                }

            }
            
            sched_yield();
        }

        // NOTE: The "real" resolution here is about 20000us on standard Linux.
        //       But, I'd love to keep within a half ms on my timer firings.
        sched_yield();
        usleep( 500 );
    }
}


void  SipDispatcher::Stop( void )
{
    mStopFlag = -1;
}


void  SipDispatcher::AddCall( SipCall *  aCall )
{
    callNode *  newNode;

    newNode       = new callNode();
    newNode->Call = aCall;

    pthread_mutex_lock( &mCallListMutex );
    newNode->Next = mCallList;
    mCallList      = newNode;
    pthread_mutex_unlock( &mCallListMutex );
}


void  SipDispatcher::AddEndPoint( SipEndPoint *  aEndPoint )
{
    endPointNode *  newNode;

    newNode           = new endPointNode();
    newNode->EndPoint = aEndPoint;

    pthread_mutex_lock( &mEndPointListMutex );
    newNode->Next     = mEndPointList;
    mEndPointList      = newNode;
    pthread_mutex_unlock( &mEndPointListMutex );
}


void  SipDispatcher::AddProxyEndPoint( SipProxyEndPoint *  aProxyEndPoint )
{
    proxyEndPointNode *  newNode;

    newNode           = new proxyEndPointNode();
    newNode->ProxyEndPoint = aProxyEndPoint;

    pthread_mutex_lock( &mProxyEndPointListMutex );
    newNode->Next     = mProxyEndPointList;
    mProxyEndPointList      = newNode;
    pthread_mutex_unlock( &mProxyEndPointListMutex );
}


void  SipDispatcher::AddRegistrar( SipRegistrar *  aRegistrar )
{
    registrarNode *  newNode;

    newNode           = new registrarNode();
    newNode->Registrar = aRegistrar;

    pthread_mutex_lock( &mRegistrarListMutex );
    newNode->Next     = mRegistrarList;
    mRegistrarList      = newNode;
    pthread_mutex_unlock( &mRegistrarListMutex );
}


void  SipDispatcher::AddUndispatchedHandler( UndispatchedHandler *  aValue )
{
    undispatchedNode *  newNode;

    newNode = new undispatchedNode();
    newNode->Handler = aValue;

    pthread_mutex_lock( &mUndispatchedListMutex );
    newNode->Next     = mUndispatchedList;
    mUndispatchedList  = newNode;
    pthread_mutex_unlock( &mUndispatchedListMutex );
}


void  SipDispatcher::AddTimer( SipDispatcher::Timer *  aTimer )
{
    timerNode *    newNode;
    timerNode *    node;
    timerNode *    previousNode;
    unsigned long  when;

    newNode        = new timerNode();
    newNode->Value = aTimer;
    previousNode   = NULL;
    when           = aTimer->GetWhen();

    pthread_mutex_lock( &mTimerListMutex );
    // Remove timer first...
    node = mTimerList;
    while ( node )
    {
        if ( node->Value == aTimer )
        {
            if ( previousNode )
            {
                previousNode->Next = node->Next;
            }
            else
            {
                mTimerList = node->Next;
            }
            delete node;
            mTimerListSize--;
            break;
        }
        previousNode = node;
        node         = node->Next;
        sched_yield();
    }
    // Now add timer...
    previousNode = NULL;
    node         = mTimerList;
    while ( node )
    {
        if ( node->Value->GetWhen() > when )
        {
            newNode->Next = node;
            if ( previousNode )
            {
                previousNode->Next = newNode;
            }
            else
            {
                mTimerList = newNode;
            }
            mTimerListSize++;
            break;
        }
        previousNode = node;
        node         = node->Next;
        sched_yield();
    }
    if ( !node )
    {
        newNode->Next = NULL;
        if ( previousNode )
        {
            previousNode->Next = newNode;
        }
        else
        {
            mTimerList = newNode;
        }
        mTimerListSize++;
    }
    pthread_mutex_unlock( &mTimerListMutex );
}


void  SipDispatcher::RemoveCall( SipCall *  aCall )
{
    callNode *  node;
    callNode *  previousNode;

    previousNode = NULL;

    pthread_mutex_lock( &mCallListMutex );
    node = mCallList;
    while ( node )
    {
        if ( node->Call == aCall )
        {
            if ( previousNode )
            {
                previousNode->Next = node->Next;
            }
            else
            {
                mCallList = node->Next;
            }
            delete node;
            break;
        }
        previousNode = node;
        node         = node->Next;
        sched_yield();
    }
    pthread_mutex_unlock( &mCallListMutex );
}


void  SipDispatcher::RemoveEndPoint( SipEndPoint *  aEndPoint )
{
    endPointNode *  node;
    endPointNode *  previousNode;

    previousNode = NULL;

    pthread_mutex_lock( &mEndPointListMutex );
    node = mEndPointList;
    while ( node )
    {
        if ( node->EndPoint == aEndPoint )
        {
            if ( previousNode )
            {
                previousNode->Next = node->Next;
            }
            else
            {
                mEndPointList = node->Next;
            }
            delete node;
            break;
        }
        previousNode = node;
        node         = node->Next;
        sched_yield();
    }
    pthread_mutex_unlock( &mEndPointListMutex );
}


void  SipDispatcher::RemoveProxyEndPoint( SipProxyEndPoint *  aProxyEndPoint )
{
    proxyEndPointNode *  node;
    proxyEndPointNode *  previousNode;

    previousNode = NULL;

    pthread_mutex_lock( &mProxyEndPointListMutex );
    node = mProxyEndPointList;
    while ( node )
    {
        if ( node->ProxyEndPoint == aProxyEndPoint )
        {
            if ( previousNode )
            {
                previousNode->Next = node->Next;
            }
            else
            {
                mProxyEndPointList = node->Next;
            }
            delete node;
            break;
        }
        previousNode = node;
        node         = node->Next;
        sched_yield();
    }
    pthread_mutex_unlock( &mProxyEndPointListMutex );
}


void  SipDispatcher::RemoveRegistrar( SipRegistrar *  aRegistrar )
{
    registrarNode *  node;
    registrarNode *  previousNode;

    previousNode = NULL;

    pthread_mutex_lock( &mRegistrarListMutex );
    node = mRegistrarList;
    while ( node )
    {
        if ( node->Registrar == aRegistrar )
        {
            if ( previousNode )
            {
                previousNode->Next = node->Next;
            }
            else
            {
                mRegistrarList = node->Next;
            }
            delete node;
            break;
        }
        previousNode = node;
        node         = node->Next;
        sched_yield();
    }
    pthread_mutex_unlock( &mRegistrarListMutex );
}


void  SipDispatcher::RemoveUndispatchedHandler( UndispatchedHandler *  aValue )
{
    undispatchedNode *  node;
    undispatchedNode *  previousNode;

    previousNode = NULL;

    pthread_mutex_lock( &mUndispatchedListMutex );
    node = mUndispatchedList;
    while ( node )
    {
        if ( node->Handler == aValue )
        {
            if ( previousNode )
            {
                previousNode->Next = node->Next;
            }
            else
            {
                mUndispatchedList = node->Next;
            }
            delete node;
            break;
        }
        previousNode = node;
        node         = node->Next;
        sched_yield();
    }
    pthread_mutex_unlock( &mUndispatchedListMutex );
}


void  SipDispatcher::RemoveTimer( SipDispatcher::Timer *  aTimer )
{
    timerNode *  node;
    timerNode *  previousNode;

    previousNode = NULL;

    pthread_mutex_lock( &mTimerListMutex );
    node = mTimerList;
    while ( node )
    {
        if ( node->Value == aTimer )
        {
            if ( previousNode )
            {
                previousNode->Next = node->Next;
            }
            else
            {
                mTimerList = node->Next;
            }
            mTimerListSize--;
            delete node;
            break;
        }
        previousNode = node;
        node         = node->Next;
        sched_yield();
    }
    pthread_mutex_unlock( &mTimerListMutex );
}


SipEndPoint *  SipDispatcher::GetEndPointByUser( char *  aUser )
{
    endPointNode *  node;
    SipEndPoint *   endPoint;

    endPoint = NULL;

    pthread_mutex_lock( &mEndPointListMutex );
    node = mEndPointList;
    while ( node )
    {
        if ( strcmp( node->EndPoint->GetUser(), aUser ) == 0 )
        {
            endPoint = node->EndPoint;
            break;
        }
        node = node->Next;
    }
    pthread_mutex_unlock( &mEndPointListMutex );

    return endPoint;
}


SipProxyEndPoint *  SipDispatcher::GetProxyEndPointByUser( char *  aUser )
{
    proxyEndPointNode *  node;
    SipProxyEndPoint *   proxyEndPoint;

    proxyEndPoint = NULL;

    pthread_mutex_lock( &mProxyEndPointListMutex );
    node = mProxyEndPointList;
    while ( node )
    {
        if ( strcmp( node->ProxyEndPoint->GetUser(), aUser ) == 0 )
        {
            proxyEndPoint = node->ProxyEndPoint;
            break;
        }
        node = node->Next;
    }
    pthread_mutex_unlock( &mProxyEndPointListMutex );

    return proxyEndPoint;
}


SipUdpPort *  SipDispatcher::GetUdpPort( void )
{
    return mUdpPort;
}


void  SipDispatcher::SetUdpPort( SipUdpPort *  aPort )
{
    mUdpPort = aPort;
}


int  SipDispatcher::GetT1( void )
{
    return mSipRfcT1;
}


void  SipDispatcher::SetT1( int  aMilliseconds )
{
    mSipRfcT1 = aMilliseconds;
}


int  SipDispatcher::GetT2( void )
{
    return mSipRfcT2;
}


void  SipDispatcher::SetT2( int  aMilliseconds )
{
    mSipRfcT2 = aMilliseconds;
}


bool  SipDispatcher::GetRetransmissionsDesired( void )
{
    return mRetransmissionsDesired;
}


void  SipDispatcher::SetRetransmissionsDesired( bool  aValue )
{
    mRetransmissionsDesired = aValue;
}


bool  SipDispatcher::GetTimeoutsDesired( void )
{
    return mTimeoutsDesired;
}


void  SipDispatcher::SetTimeoutsDesired( bool  aValue )
{
    mTimeoutsDesired = aValue;
}


bool  SipDispatcher::GetProvisionalsDesired( void )
{
    return mProvisionalsDesired;
}


void  SipDispatcher::SetProvisionalsDesired( bool  aValue )
{
    mProvisionalsDesired = aValue;
}

