//-----------------------------------------------------------------------------
//
// SipIdentifier.cpp - Class for a single SIP identifier,
//                     which could be a SIP URI, SIPS URI,
//                     Absolute URI, or a display name + any of
//                     the previous. Also, SIP identifier can have
//                     parameters of their own.
//
//    Copyright (C) 2004  Mark D. Collier
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//   Author: Mark D. Collier   - 12/01/2006   v1.1
//                   Mark D. Collier   -  04/26/2004  v1.0
//         www.securelogix.com - mark.collier@securelogix.com
//         www.hackingexposedvoip.com
//
//-----------------------------------------------------------------------------

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "util.h"
#include "SipIdentifier.h"
#include "SipUri.h"

#define malloc  mymalloc
#define free    myfree
#define strdup  mystrdup
#define strndup mystrndup

SipIdentifier::SipIdentifier( SipIdentifier *  aCopyFrom )
{
    parameterNode *  node;
    parameterNode *  newNode;
    parameterNode *  lastNewNode;

    mFullText        = strdup( aCopyFrom->mFullText );
    mName            = aCopyFrom->mName ? strdup( aCopyFrom->mName ) : NULL;
    mUri             = aCopyFrom->mUri ? new SipUri( aCopyFrom->mUri ) : NULL;
    if ( aCopyFrom->mStartOfParameters )
    {
        mStartOfParameters =  mFullText
                            + (  aCopyFrom->mStartOfParameters
                               - aCopyFrom->mFullText );
    }
    else
    {
        mStartOfParameters = NULL;
    }
    mParameterCount = aCopyFrom->mParameterCount;
    mParameters     = NULL;
    node            = aCopyFrom->mParameters;
    lastNewNode     = NULL;
    while ( node )
    {
        newNode = new parameterNode();
        newNode->Name = strdup( node->Name );
        newNode->Value = strdup( node->Value );
        if ( lastNewNode )
        {
            lastNewNode->Next = newNode;
        }
        else
        {
            mParameters = newNode;
        }
        lastNewNode = newNode;
        node = node->Next;
    }
}


// Ensure aFullText is malloced, as it will be freed.
SipIdentifier::SipIdentifier( char *  aFullText )
{
    mFullText          = aFullText;
    mName              = NULL;
    mUri               = NULL;
    mStartOfParameters = NULL;
    mParameterCount    = -1;
    mParameters        = NULL;
}


SipIdentifier::~SipIdentifier( void )
{
    parameterNode *  node;

    if ( mFullText )
    {
        free( mFullText );
        mFullText = NULL;
    }
    if ( mName )
    {
        free( mName );
        mName = NULL;
    }
    if ( mUri )
    {
        delete mUri;
        mUri = NULL;
    }
    mStartOfParameters = NULL;
    mParameterCount    = -1;
    while ( mParameters )
    {
        node = mParameters;
        mParameters = node->Next;
        node->Next = NULL;
        free( node->Name );
        free( node->Value );
        delete node;
    }
}


char *  SipIdentifier::GetFullText( void )
{
    return mFullText;
}


char *  SipIdentifier::GetName( void )
{
    char *  startOfName;
    char *  endOfName;

    if ( !mName )
    {
        startOfName = mFullText;
        while (   *startOfName
               && ( *startOfName == ' ' || *startOfName == '\t' ) )
        {
            startOfName++;
        }
        if ( *startOfName == '"' )
        {
            startOfName++;
            endOfName = startOfName;
            while ( *endOfName && *endOfName != '"' )
            {
                endOfName++;
            }
            if ( *endOfName )
            {
                mName = strndup( startOfName, endOfName - startOfName );
            }
            else
            {
                mName = strdup( startOfName );
            }
        }
        else if ( *startOfName != '<' )
        {
            endOfName = startOfName;
            while (   *endOfName && *endOfName != ' ' && *endOfName != '\t'
                   && *endOfName != '<' && *endOfName != ':'
                   && *endOfName != '@' )
            {
                endOfName++;
            }
            if ( *endOfName )
            {
                if ( *endOfName != ':' && *endOfName != '@' )
                {
                    mName = strndup( startOfName, endOfName - startOfName );
                }
            }
            else
            {
                mName = strdup( startOfName );
            }
        }
    }

    if ( !mName )
    {
        mName = strdup( "" );
    }

    return mName;
}


SipUri *  SipIdentifier::GetUri( void )
{
    char *  startOfUri;
    char *  endOfUri;

    if ( !mUri )
    {
        startOfUri = mFullText;
        while (   *startOfUri
               && ( *startOfUri == ' ' || *startOfUri == '\t' ) )
        {
            startOfUri++;
        }
        if ( *startOfUri == '"' )
        {
            startOfUri++;
            while ( *startOfUri && *startOfUri != '"' )
            {
                startOfUri++;
            }
            if ( *startOfUri )
            {
                startOfUri++;
                while (   *startOfUri
                       && ( *startOfUri == ' ' || *startOfUri == '\t' ) )
                {
                    startOfUri++;
                }
            }
        }
        endOfUri = startOfUri;
        while (   *endOfUri
               && ( *startOfUri != '<' || *endOfUri != '>' )
               && ( *startOfUri == '<' || *endOfUri != ';' ) )
        {
            if ( *endOfUri == '<' )
            {
                startOfUri = endOfUri;
            }
            endOfUri++;
        }
        if ( *endOfUri != ':' && *endOfUri != '>' )
        {
            while (   endOfUri > startOfUri
                   && ( endOfUri[-1] == ' ' || endOfUri[-1] == '\t' ) )
            {
                endOfUri--;
            }
        }
        if ( *startOfUri == '<' )
        {
            startOfUri++;
        }
        if ( *endOfUri )
        {
            mUri = new SipUri( strndup( startOfUri, endOfUri - startOfUri ) );
        }
        else
        {
            mUri = new SipUri( strdup( startOfUri ) );
        }
        mStartOfParameters = endOfUri;
        if ( *mStartOfParameters == '>' )
        {
            mStartOfParameters++;
        }
    }

    if ( !mUri )
    {
        mUri = new SipUri( strdup( "" ) );
    }

    return mUri;
}


int  SipIdentifier::GetParameterCount( void )
{
    char *           startOfName;
    char *           endOfName;
    char *           endOfValue;
    parameterNode *  node;
    parameterNode *  lastNode;

    if ( mParameterCount < 0 )
    {
        if ( !mStartOfParameters )
        {
            GetUri();
        }

        mParameterCount = 0;
        lastNode        = NULL;
        startOfName     = mStartOfParameters;
        while ( *startOfName == ';' )
        {
            startOfName++;
            if ( !*startOfName )
            {
                break;
            }
            endOfName = startOfName + 1;
            while ( *endOfName && *endOfName != '=' && *endOfName != ';' )
            {
                endOfName++;
            }
            node = new parameterNode();
            node->Next = NULL;
            if ( *endOfName )
            {
                node->Name = strndup( startOfName, endOfName - startOfName );
            }
            else
            {
                node->Name = strdup( startOfName );
            }
            if ( *endOfName == '=' )
            {
                endOfValue = endOfName + 1;
                while ( *endOfValue && *endOfValue != ';' )
                {
                    endOfValue++;
                }
                if ( *endOfValue )
                {
                    node->Value = strndup( endOfName + 1,
                                           endOfValue - endOfName - 1 );
                }
                else
                {
                    node->Value = strdup( endOfName + 1 );
                }
                startOfName = endOfValue;
            }
            else if ( *endOfName == ';' )
            {
                node->Value = strdup( "" );
                startOfName = endOfName;
            }
            else
            {
                node->Value = strdup( "" );
                startOfName = endOfName;
            }
            if ( lastNode )
            {
                lastNode->Next = node;
                lastNode = node;
            }
            else
            {
                mParameters = lastNode = node;
            }
            mParameterCount++;
        }
    }

    return mParameterCount;
}


char *  SipIdentifier::GetParameterName( int  aIndex )
{
    parameterNode *  node;
    
    if ( mParameterCount < 0 )
    {
        GetParameterCount();
    }

    node = mParameters;
    while ( node && aIndex > 0 )
    {
        node = node->Next;
        aIndex--;
    }

    return node ? node->Name : ( char * )"";
}


char *  SipIdentifier::GetParameterValue( int  aIndex )
{
    parameterNode *  node;
    
    if ( mParameterCount < 0 )
    {
        GetParameterCount();
    }

    node = mParameters;
    while ( node && aIndex > 0 )
    {
        node = node->Next;
        aIndex--;
    }

    return node ? node->Value : ( char * )"";
}


char *  SipIdentifier::GetParameterValue( char *  aName )
{
    parameterNode *  node;

    if ( mParameterCount < 0 )
    {
        GetParameterCount();
    }

    node = mParameters;
    while ( node )
    {
        if ( strcasecmp( node->Name, aName ) == 0 )
        {
            return node->Value;
        }
        node = node->Next;
    }

    return "";
}

