//-----------------------------------------------------------------------------
//
// SipProxyEndPoint.cpp
//
//    Copyright (C) 2004  Mark D. Collier
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//   Author: Mark D. Collier   - 12/01/2006   v1.1
//                   Mark D. Collier   -  04/26/2004  v1.0
//         www.securelogix.com - mark.collier@securelogix.com
//         www.hackingexposedvoip.com
//
//-----------------------------------------------------------------------------

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <netinet/in.h>
#include <netdb.h>

#include "util.h"
#include "SipMessage.h"
#include "SipUdpPort.h"
#include "SipDispatcher.h"
#include "SipRegistrarConnector.h"
#include "SipRegistrar.h"
#include "SipProxyEndPoint.h"
#include "SipCall.h"
#include "ControlPort.h"

#define malloc  mymalloc
#define free    myfree
#define strdup  mystrdup
#define strndup mystrndup

SipProxyEndPoint::SipProxyEndPoint( void )
{
    mUser                    = NULL;
    mTextName                = NULL;
    mTo                      = NULL;
    mToAddress               = INADDR_ANY;
    mToPort                  = htons( 5060 );
    mDispatcher              = NULL;
    mRegistrarConnector      = NULL;
    mControlPort             = NULL;
    mControlPortConnectionId = -1;
    mParentRegistrar         = NULL;
}

SipProxyEndPoint::~SipProxyEndPoint( void )
{
    if ( mUser )
    {
        myfree( mUser );
        mUser = NULL;
    }
    if ( mTextName )
    {
        myfree( mTextName );
        mTextName = NULL;
    }
    if ( mTo )
    {
        delete mTo;
        mTo = NULL;
    }
}


char *  SipProxyEndPoint::GetUser( void )
{
    return mUser;
}


void  SipProxyEndPoint::SetUser( char *  aValue )
{
    if ( mUser )
    {
        myfree( mUser );
    }
    mUser = aValue;
}


char *  SipProxyEndPoint::GetTextName( void )
{
    return mTextName;
}


void  SipProxyEndPoint::SetTextName( char *  aValue )
{
    if ( mTextName )
    {
        myfree( mTextName );
    }
    mTextName = aValue;
}


SipIdentifier *  SipProxyEndPoint::GetTo( void )
{
    return mTo;
}


void  SipProxyEndPoint::SetTo( SipIdentifier *  aValue )
{
    SipUri *          uri;
    struct hostent *  hent;

    uri = aValue->GetUri();
    if ( !uri )
    {
        fprintf( stderr, "%s:%d: Error: No URI associated with %s.\n",
                 __FILE__, __LINE__, aValue->GetFullText() );
        return;
    }

    hent = gethostbyname( uri->GetHost() );
    if ( !hent )
    {
        perror( "gethostbyname" );
    }
    else if ( !*hent->h_addr_list )
    {
        fprintf( stderr, "%s:%d: Error: No IP associated with %s.\n",
                 __FILE__, __LINE__, aValue->GetFullText() );
    }
    else
    {
        memcpy( ( char * )&mToAddress,
                *hent->h_addr_list,
                sizeof( mToAddress ) );
    }

    if ( uri->GetPort() < 0 )
    {
        mToPort = htons( 5060 );
    }
    else
    {
        mToPort = htons( uri->GetPort() );
    }

    if ( mTo )
    {
        delete mTo;
    }
    mTo = aValue;
}


SipDispatcher *  SipProxyEndPoint::GetDispatcher( void )
{
    return mDispatcher;
}


void  SipProxyEndPoint::SetDispatcher( SipDispatcher *  aValue )
{
    mDispatcher = aValue;
}


SipRegistrarConnector *  SipProxyEndPoint::GetRegistrarConnector( void )
{
    return mRegistrarConnector;
}


void  SipProxyEndPoint::SetRegistrarConnector( SipRegistrarConnector *  aValue )
{
    if ( mRegistrarConnector )
    {
        mRegistrarConnector->RemoveProxyEndPoint( this );
    }
    mRegistrarConnector = aValue;
    if ( mRegistrarConnector )
    {
        mRegistrarConnector->AddProxyEndPoint( this );
    }
}


ControlPort *  SipProxyEndPoint::GetControlPort( void )
{
    return mControlPort;
}


void  SipProxyEndPoint::SetControlPort( ControlPort *  aValue )
{
    mControlPort = aValue;
}


int  SipProxyEndPoint::GetControlPortConnectionId( void )
{
    return mControlPortConnectionId;
}


void  SipProxyEndPoint::SetControlPortConnectionId( int aId )
{
    mControlPortConnectionId = aId;
}


bool  SipProxyEndPoint::Incoming( struct SipMessage *  aMessage )
{
    int          ix;
    char *       cp;
    SipHeader *  header;
    SipIdentifier *  headerIdentifier;
    SipUri *  headerUri;
    in_addr_t    localAddress;
    struct hostent *  hent;

    if ( aMessage->IsRequest() )
    {
        header = aMessage->GetHeader( "via" );
        if ( !header )
        {
            header = aMessage->GetHeader( "v" );
        }
        headerIdentifier =  header
                          ? header->GetIdentifier( 0 )
                          : NULL;
        cp =  headerIdentifier
            ? headerIdentifier->GetParameterValue( "user" )
            : NULL;
        if ( cp && strcmp( cp, GetUser() ) == 0 )
        {
            return false;
        }
        if (   mParentRegistrar
            && mParentRegistrar->GetRandomization()
            && strcmp( aMessage->GetRequestMethod(), "INVITE" ) == 0 )
        {
            header = aMessage->GetHeader( "from" );
            if ( !header )
            {
                header = aMessage->GetHeader( "f" );
            }
            if ( header )
            {
                headerIdentifier =  header
                                  ? header->GetIdentifier( 0 )
                                  : NULL;
                cp =  headerIdentifier
                    ? headerIdentifier->GetParameterValue( "user" )
                    : NULL;
                cp = mParentRegistrar->GetRandomResolution( cp, GetUser() );
                if ( cp )
                {
                    SetTo( new SipIdentifier( strdup( cp ) ) );
                }
            }
        }
        localAddress = DetermineLocalAddressFor( mToAddress );
        aMessage->SetDestinationAddress( mToAddress );
        aMessage->SetDestinationPort( mToPort );
        aMessage->SetRequestUri( new SipUri( mTo->GetUri() ) );

        cp = ( char * )malloc( 1301 );
        if ( !cp )
        {
            ReportError( "out of memory", __FILE__, __LINE__ );
            return false;
        }

        ix = snprintf( cp,
                       1301,
                       "SIP/2.0/UDP %d.%d.%d.%d:%d;user=%s",
                       localAddress       & 0xff,
                       localAddress >>  8 & 0xff,
                       localAddress >> 16 & 0xff,
                       localAddress >> 24 & 0xff,
                       ntohs( GetDispatcher()->GetUdpPort()->GetBoundPort() ),
                       GetUser() );
        aMessage->InsertHeader( new SipHeader( "Via", cp ) );

        GetDispatcher()->GetUdpPort()->AddOutgoing( aMessage );
        return true;
    }
    else
    {
        aMessage->RemoveHeader( "via" );
        header = aMessage->GetHeader( "via" );
        headerIdentifier =  header
                          ? header->GetIdentifier( 0 )
                          : NULL;
        headerUri = headerIdentifier ? headerIdentifier->GetUri() : NULL;
        cp = headerUri ? headerUri->GetHost() : NULL;
        hent = cp ? gethostbyname( cp ) : NULL;
        if ( hent && *hent->h_addr_list )
        {
            memcpy( ( char * )&localAddress,
                    *hent->h_addr_list,
                    sizeof( localAddress ) );
            aMessage->SetDestinationAddress( localAddress );
            if ( headerUri->GetPort() > 0 )
            {
                aMessage->SetDestinationPort( htons( headerUri->GetPort() ) );
            }
            else
            {
                aMessage->SetDestinationPort( htons( 5060 ) );
            }
            GetDispatcher()->GetUdpPort()->AddOutgoing( aMessage );
            return true;
        }
    }

    return false;
}


void  SipProxyEndPoint::SetParentRegistrar( SipRegistrar *  aRegistrar )
{
    mParentRegistrar = aRegistrar;
}

