//-----------------------------------------------------------------------------
//
// SipRegistrar.cpp
//
//    Copyright (C) 2004  Mark D. Collier
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//   Author: Mark D. Collier   - 12/01/2006   v1.1
//                   Mark D. Collier   -  04/26/2004  v1.0
//         www.securelogix.com - mark.collier@securelogix.com
//         www.hackingexposedvoip.com
//
//-----------------------------------------------------------------------------

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "util.h"
#include "hash.h"
#include "SipMessage.h"
#include "SipIdentifier.h"
#include "SipUri.h"
#include "SipUdpPort.h"
#include "SipDispatcher.h"
#include "SipRegistrar.h"
#include "SipProxyEndPoint.h"
#include "ControlPort.h"

#define malloc  mymalloc
#define free    myfree
#define strdup  mystrdup
#define strndup mystrndup

SipRegistrar::SipRegistrar( void )
{
    int ix;

    mDispatcher = NULL;
    mDomain     = NULL;
    for ( ix = 0; ix < SIP_REGISTRAR_TABLE_SIZE; ix++ )
    {
        mEntryTable[ix] = NULL;
    }
    mControlPort             = NULL;
    mControlPortConnectionId = -1;
    mRandomization           = false;
}

SipRegistrar::~SipRegistrar( void )
{
    int                  ix;
    registrationEntry *  entry;

    for ( ix = 0; ix < SIP_REGISTRAR_TABLE_SIZE; ix++ )
    {
        while ( mEntryTable[ix] )
        {
            entry = mEntryTable[ix];
            free( entry->ToUser );
            entry->ToUser = NULL;
            free( entry->ContactLine );
            entry->ContactLine = NULL;
            // Trigger the proxy end point's deletion
            entry->ProxyEndPoint->SetControlPort( NULL );
            entry->ProxyEndPoint = NULL;
            mEntryTable[ix] = entry->Next;
            entry->Next = NULL;
            delete entry;
        }
    }
}


char *  SipRegistrar::GetDomain( void )
{
    return mDomain;
}


void  SipRegistrar::SetDomain( char *  aValue )
{
    if ( mDomain )
    {
        myfree( mDomain );
    }
    mDomain = aValue;
}


SipDispatcher *  SipRegistrar::GetDispatcher( void )
{
    return mDispatcher;
}


void  SipRegistrar::SetDispatcher( SipDispatcher *  aValue )
{
    mDispatcher = aValue;
}


bool  SipRegistrar::Incoming( struct SipMessage *  aMessage )
{
    char *               contactLine;
    char *               cp;
    char *               toUser;
    in_addr_t            remoteAddress;
    int                  ix;
    int                  len;
    registrationEntry *  entry;
    registrationEntry *  next;
    registrationEntry *  previous;
    SipHeader *          contactHeader;
    SipHeader *          header;
    SipHeader *          toHeader;
    SipIdentifier *      contactIdentifier;
    SipIdentifier *      headerIdentifier;
    SipIdentifier *      toIdentifier;
    SipMessage *         response;
    SipUri *             headerUri;
    SipUri *             toUri;
    struct hostent *     hent;
    unsigned long int    entryHash;

    toHeader = aMessage->GetHeader( "t" );
    if ( !toHeader )
    {
        toHeader = aMessage->GetHeader( "to" );
    }
    if ( toHeader )
    {
        toIdentifier = NULL;
        toUri        = NULL;
        toUser       = NULL;
        if ( toHeader->GetIdentifierCount() )
        {
            toIdentifier = toHeader->GetIdentifier( 0 );
            toUri = toIdentifier->GetUri();
            if ( toUri )
            {
                toUser = toUri->GetUser();
            }
        }
        if ( toIdentifier && toUri && toUser )
        {
            entryHash =  hash( ( unsigned char * )toUser, strlen( toUser ), 0 )
                       & SIP_REGISTRAR_TABLE_MASK;
            contactHeader     = aMessage->GetHeader( "contact" );
            contactLine       = NULL;
            contactIdentifier = NULL;
            if ( contactHeader )
            {
                contactLine = contactHeader->GetValue();
                if ( contactHeader->GetIdentifierCount() )
                {
                    contactIdentifier = contactHeader->GetIdentifier( 0 );
                }
            }
            if ( contactLine && contactIdentifier )
            {
                previous = NULL;
                entry    = mEntryTable[entryHash];
                while ( entry )
                {
                    if ( strcmp( entry->ToUser, toUser ) == 0 )
                    {
                        if ( previous )
                        {
                            previous->Next = entry->Next;
                        }
                        else
                        {
                            mEntryTable[entryHash] = entry->Next;
                        }
                        next     = entry->Next;
                        free( entry->ToUser );
                        entry->ToUser = NULL;
                        free( entry->ContactLine );
                        entry->ContactLine = NULL;
                        // Trigger the proxy end point's deletion
                        entry->ProxyEndPoint->SetControlPort( NULL );
                        entry->ProxyEndPoint = NULL;
                        entry->Next = NULL;
                        delete entry;
                        entry = next;
                    }
                    else
                    {
                        previous = entry;
                        entry    = entry->Next;
                    }
                }

                entry                = new registrationEntry();
		entry->Next          = NULL;
                entry->ToUser        = strdup( toUser );
                entry->ContactLine   = strdup( contactLine );
                entry->ProxyEndPoint =
                              GetDispatcher()->GetProxyEndPointByUser( toUser );
                if ( !entry->ProxyEndPoint )
                {
                    entry->ProxyEndPoint = new SipProxyEndPoint();
                    entry->ProxyEndPoint->SetParentRegistrar( this );
                    entry->ProxyEndPoint->SetUser( strdup( toUser ) );
                    entry->ProxyEndPoint->SetTo(
                                       new SipIdentifier( contactIdentifier ) );
                    if ( GetControlPort() )
                    {
                        entry->ProxyEndPoint->SetControlPort(
                                                             GetControlPort() );
                        entry->ProxyEndPoint->SetControlPortConnectionId(
                                                 GetControlPortConnectionId() );
                        GetControlPort()->AddSipProxyEndPoint(
                                                   GetControlPortConnectionId(),
                                                   entry->ProxyEndPoint );
                    }
                    entry->ProxyEndPoint->SetDispatcher( GetDispatcher() );
                    entry->ProxyEndPoint->GetDispatcher()->AddProxyEndPoint(
                                                         entry->ProxyEndPoint );
                }
                entry->Next            = mEntryTable[entryHash];
                mEntryTable[entryHash] = entry;

                response = new SipMessage( false );
                response->SetResponseCode( 200 );
                response->SetResponseText( strdup( "Ok" ) );
                response->SetSourceAddress( 
                                        aMessage->GetDestinationAddress() );
                response->SetDestinationAddress( 
                                        aMessage->GetSourceAddress() );
                response->SetSourcePort( 
                                        aMessage->GetDestinationPort() );
                response->SetDestinationPort( 
                                        aMessage->GetSourcePort() );

                len = aMessage->GetHeaderCount();
                for ( ix = len - 1; ix >= 0; ix-- )
                {
                    header = aMessage->GetHeader( ix );
                    if ( strcasecmp( "record-route",
                                     header->GetName() ) == 0 )
                    {
                        response->AddHeader( new SipHeader( header ) );
                    }
                }

                len = aMessage->GetHeaderCount();
                for ( ix = len - 1; ix >= 0; ix-- )
                {
                    cp = aMessage->GetHeader( ix )->GetName();
                    if (   strcasecmp( "v"  , cp ) == 0
                        || strcasecmp( "via", cp ) == 0 )
                    {
                        response->AddHeader(
                               new SipHeader( aMessage->GetHeader( ix ) ) );
                    }
                }

                response->AddHeader( new SipHeader( toHeader ) );
                header = aMessage->GetHeader( "f" );
                if ( !header )
                {
                    header = aMessage->GetHeader( "from" );
                }
                if ( header )
                {
                    response->AddHeader( new SipHeader( header ) );
                }
                header = aMessage->GetHeader( "i" );
                if ( !header )
                {
                    header = aMessage->GetHeader( "call-id" );
                }
                if ( header )
                {
                    response->AddHeader( new SipHeader( header ) );
                }
                header = aMessage->GetHeader( "cseq" );
                if ( header )
                {
                    response->AddHeader( new SipHeader( header ) );
                }

                header = aMessage->GetHeader( "via" );
                if ( !header )
                {
                    header = aMessage->GetHeader( "v" );
                }
                headerIdentifier =  header
                                  ? header->GetIdentifier( 0 )
                                  : NULL;
                headerUri =  headerIdentifier
                           ? headerIdentifier->GetUri()
                           : NULL;
                cp = headerUri ? headerUri->GetHost() : NULL;
                hent = cp ? gethostbyname( cp ) : NULL;
                if ( hent && *hent->h_addr_list )
                {
                    memcpy( ( char * )&remoteAddress,
                            *hent->h_addr_list,
                            sizeof( remoteAddress ) );
                    response->SetDestinationAddress( remoteAddress );
                    if ( headerUri->GetPort() > 0 )
                    {
                        response->SetDestinationPort(
                                                htons( headerUri->GetPort() ) );
                    }
                    else
                    {
                        response->SetDestinationPort( htons( 5060 ) );
                    }
                }

                GetDispatcher()->GetUdpPort()->AddOutgoing( response );

                delete aMessage;
                return true;
            }
        }
    }

    return false;
}


char *  SipRegistrar::GetFirstResolution( char *  aName )
{
    registrationEntry *  entry;

    entry = mEntryTable[  hash( ( unsigned char * )aName, strlen( aName ), 0 )
                        & SIP_REGISTRAR_TABLE_MASK ];
    while ( entry && strcmp( entry->ToUser, aName ) != 0 )
    {
        entry = entry->Next;
    }
    if ( entry )
    {
        return entry->ContactLine;
    }

    return NULL;
}


char *  SipRegistrar::GetRandomResolution( char *  aFromName, char *  aToName )
{
    registrationEntry *  entry;
    registrationEntry *  tmpEntry;
    int                  ix;
    int                  iy;

    // Try to resolve to a user name other than the aFromName, up to 10 times
    for ( iy = 0; iy < 10; iy++ )
    {
        entry = NULL;

        // Try to find an occupied bucket randomly, but give up after a 1000
        // tries.
        for ( ix = 0; ix < 1000 && !entry; ix++ )
        {
            entry = mEntryTable[random() & SIP_REGISTRAR_TABLE_MASK];
        }

        // If we gave up, get out and just return the proper resolution, if
        // there is one.
        if ( !entry )
        {
            break;
        }

        ix = 0;
        tmpEntry = entry;
        while ( tmpEntry )
        {
            ix++;
            tmpEntry = tmpEntry->Next;
        }

        for ( ix = random() % ix; ix > 0; ix-- )
        {
            entry = entry->Next;
        }

        if ( !aFromName || strcasecmp( aFromName, entry->ToUser ) != 0 )
        {
            return entry->ContactLine;
        }
    }

    // If we gave up, just return the proper resolution, if there is one.
    return GetFirstResolution( aToName );
}


ControlPort *  SipRegistrar::GetControlPort( void )
{
    return mControlPort;
}


void  SipRegistrar::SetControlPort( ControlPort *  aValue )
{
    mControlPort = aValue;
}


int  SipRegistrar::GetControlPortConnectionId( void )
{
    return mControlPortConnectionId;
}


void  SipRegistrar::SetControlPortConnectionId( int  aId )
{
    mControlPortConnectionId = aId;
}


bool  SipRegistrar::GetRandomization( void )
{
    return mRandomization;
}


void  SipRegistrar::SetRandomization( bool  aValue )
{
    mRandomization = aValue;
}

