//-----------------------------------------------------------------------------
//
// SipUdpPort.h - Handles receiving and queuing
//                  incoming messages and dequeueing and
//                  sending outgoing messages over UDP.
//
//    Copyright (C) 2004  Mark D. Collier
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//   Author: Mark D. Collier   - 12/01/2006   v1.1
//                   Mark D. Collier   -  04/26/2004  v1.0
//         www.securelogix.com - mark.collier@securelogix.com
//         www.hackingexposedvoip.com
//
//-----------------------------------------------------------------------------

#ifndef SIP_UDP_PORT_H
#define SIP_UDP_PORT_H

#include <netinet/in.h>
#include <pthread.h>

#include "SipMessage.h"

class  SipUdpPort
{
    public:

        typedef enum
        {
            ERROR_NONE = 0,
            ERROR_MEMORY,
            ERROR_SOCKET,
            ERROR_BIND,
            ERROR_SELECT,
            ERROR_RECVFROM,
            ERROR_SENDTO,
            ERROR_SENDTO_ALL_BYTES_NOT_SENT,
            ERROR_BAD_OUTGOING_MESSAGE_TYPE,
            ERROR_OUTGOING_MESSAGE_TOO_LARGE,
            ERROR_UNKNOWN
        }  Error;

        class  UdpSource
        {
            public:
                virtual ~UdpSource( void ) {};

                virtual in_addr_t  GetUdpAddress( void ) = 0;
                virtual in_port_t  GetUdpPort( void ) = 0;
                // Caller must free return value.
                virtual char *     GetUdpPacket( void ) = 0;
        };

        SipUdpPort( void );
        ~SipUdpPort( void );

        Error               Run( void );
        void                Stop( void );
        void                SetHoldState( bool aState );
        // Caller must delete the return value.
        SipMessage *        GetNextIncoming( void );
        // message should be newed as it will be deleted.
        void                AddIncoming( SipMessage *  aMessage );
        // Caller must delete the return value.
        SipMessage *        GetNextOutgoing( void );
        // message should be newed as it will be deleted.
        void                AddOutgoing( SipMessage *  aMessage );
        UdpSource *         GetNextRespondingUdpSource( void );
        void                AddRespondingUdpSource( UdpSource *  aSource );
        void                RemoveRespondingUdpSource( UdpSource *  aSource );
        UdpSource *         GetNextRequestingUdpSource( void );
        void                AddRequestingUdpSource( UdpSource *  aSource );
        void                RemoveRequestingUdpSource( UdpSource *  aSource );
        UdpSource *         GetNextRetransmittingUdpSource( void );
        void                AddRetransmittingUdpSource( UdpSource *  aSource );
        void                RemoveRetransmittingUdpSource(
                                                         UdpSource *  aSource );
        time_t              GetCreationTime( void );
        in_addr_t           GetBoundAddress( void );
        void                SetBoundAddress( in_addr_t  aAddress );
        in_port_t           GetBoundPort( void );
        void                SetBoundPort( in_port_t  aPort );
        unsigned long       GetMessagesSent( void );
        unsigned long       GetMessagesReceived( void );
        unsigned int        GetOutgoingQueueQueued( void );
        unsigned int        GetIncomingQueueQueued( void );
        unsigned int        GetOutgoingQueueMostQueued( void );
        unsigned int        GetIncomingQueueMostQueued( void );
        unsigned int        GetRespondingUdpSourceQueueQueued( void );
        unsigned int        GetRespondingUdpSourceQueueMostQueued( void );
        unsigned int        GetRequestingUdpSourceQueueQueued( void );
        unsigned int        GetRequestingUdpSourceQueueMostQueued( void );
        unsigned int        GetRetransmittingUdpSourceQueueQueued( void );
        unsigned int        GetRetransmittingUdpSourceQueueMostQueued( void );
        unsigned long long  GetBytesSent( void );
        unsigned long long  GetBytesReceived( void );

    private:

        class  messageNode
        {
            public:
                messageNode *  Next;
                SipMessage *   Message;
        };


        class  sourceNode
        {
            public:
                sourceNode *  Next;
                UdpSource *   Source;
        };


        class messageQueue
        {
            public:
                messageQueue( void );
                ~messageQueue( void );

                // Caller must delete the return value.
                SipMessage *  GetNext( void );
                // Note that aMessage will be deleted.
                void          Add( SipMessage *  aMessage );

                unsigned int  GetQueuedCount( void );
                unsigned int  GetMostQueuedCount( void );

            private:
                pthread_mutex_t       mMutex;
                struct messageNode *  mHead;
                struct messageNode *  mTail;
                unsigned int          mQueued;
                unsigned int          mMostQueued;
        };

        class sourceQueue
        {
            public:
                sourceQueue( void );
                ~sourceQueue( void );

                UdpSource *   GetNext( void );
                void          Add( UdpSource *  aSource );
                void          Remove( UdpSource *  aSource );

                unsigned int  GetQueuedCount( void );
                unsigned int  GetMostQueuedCount( void );

            private:
                pthread_mutex_t      mMutex;
                struct sourceNode *  mHead;
                struct sourceNode *  mTail;
                unsigned int         mQueued;
                unsigned int         mMostQueued;
        };

        in_addr_t           mAddress;
        in_port_t           mPort;
        messageQueue        mIncomingQueue;
        messageQueue        mOutgoingQueue;
        sourceQueue         mRespondingUdpSourceQueue;
        sourceQueue         mRequestingUdpSourceQueue;
        sourceQueue         mRetransmittingUdpSourceQueue;
        bool                mStopFlag;
        time_t              mCreationTime;
        unsigned long       mMessagesSent;
        unsigned long       mMessagesReceived;
        unsigned long long  mBytesSent;
        unsigned long long  mBytesReceived;
        bool                mHoldState;
};

#endif

