//-----------------------------------------------------------------------------
//
// main.cpp - Allocates the ControlPort object.
//
//                       Translates the command line inputs used
//                       to invoke sip_rogue, and records the control
//                       socket IP address/port for later use by 
//                       the NetworkRun control I/O thread.
//
//                       Spawn a companion NetworkRun (i.e. 
//                       control I/O) thread to the Process thread
//                       run under main.
//
//    Copyright (C) 2004  Mark D. Collier
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//   Author: Mark D. Collier   - 12/01/2006   v1.1
//                   Mark D. Collier   -  04/26/2004  v1.0
//         www.securelogix.com - mark.collier@securelogix.com
//         www.hackingexposedvoip.com
//
//-----------------------------------------------------------------------------

#include <arpa/inet.h>
#include <errno.h>
#include <time.h>
#include <netinet/in.h>
#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <unistd.h>

#include "ControlPort.h"

#define malloc  mymalloc
#define free    myfree
#define strdup  mystrdup
#define strndup mystrndup

void *  controlPortThreadFunction( void *  aParms )
{
    switch ( ( ( ControlPort * )aParms )->NetworkRun() )
    {
        case ControlPort::ERROR_NONE:
            break;
        case ControlPort::ERROR_SOCKET:
            perror( "Error creating control port socket" );
            break;
        case ControlPort::ERROR_FCNTL:
            perror( "Error calling FCNTL on control port socket" );
            break;
        case ControlPort::ERROR_BIND:
            perror( "Error binding control port socket" );
            break;
        case ControlPort::ERROR_LISTEN:
            perror( "Error performing listen binding on control port socket" );
            break;
        case ControlPort::ERROR_ACCEPT:
            perror( "Error accepting a control port socket connection" );
            break;
        case ControlPort::ERROR_SELECT:
            perror( "Error performing select on control port socket" );
            break;
        case ControlPort::ERROR_MSG_FOR_NONEXISTENT_CONNECTION:
            perror( "Error - attempting to add control msg to non-existent connection" );
            break;
        case ControlPort::ERROR_SOCKIDX_OUT_OF_RANGE:
            perror( "Error - socket idx out of range" );
            break;
        default:
            fprintf( stderr, "Unknown controlPort error\n" );
            break;
    }
    ( ( ControlPort * )aParms )->NetworkStop();
    ( ( ControlPort * )aParms )->ProcessStop();
    return NULL;
}

//
//  sip_rogue consists of two concurrently executing adminstrative
//  threads:
//    o   The "Process" thread is essentially the main thread
//    o   The "Network" thread is spawned from the Process thread.
//
//  Note: Other threads are spawned in accordance with 
//             control commands as needed (e.g. sipDispatcherThread)
//
//  Both threads access and share a a ControlPort object
//  named: controlPort. Thread mutexes are employed
//  where necessary to synchronize access to 
//  controlPort object members shared by the threads.
//
//  The control port is the IP address/port through
//  with the sip_rogue operator telnets into the
//  sip_rogue daemon to issue it commands. The 
//  operator has 3 options for invoking sip_rogue:
//
//  1) No command line arguments: The IP address
//      of sip_rogue defaults to the special IP address
//      INADDR_ANY (i.e. any and all IP addresses
//      of ethernet interfaces on the host computer) and
//      TCP port of 6060.
//
//  2) One command line argument: The IP address
//      of sip_rogue. Presumably this corresponds to 
//      an ethernet interface on the host running 
//      sip_rogue. The TCP port defaults to: 6060.
//
//  3) Two command line arguments which are,
//       respectively, the IP address and TCP port of
//       the command (i.e. telnet) interface.
//

int  main( int  aArgc, char **  aArgv )
{
    ControlPort  controlPort;
    pthread_t    controlPortThread;

    srandom( time( NULL ) );

    if ( aArgc == 2 )
    {
        controlPort.SetBoundAddress( inet_addr( ( ( char ** )aArgv )[1] ) );
        controlPort.SetBoundPort( htons( 6060 ) );
    }
    else if ( aArgc == 3 )
    {
        controlPort.SetBoundAddress( inet_addr( ( ( char ** )aArgv )[1] ) );
        controlPort.SetBoundPort( htons( ( short )strtol(
                                                        ( ( char ** )aArgv )[2],
                                                        NULL, 10 ) ) );
    }
    else
    {
        //
        //  INADDR_ANY is a special IP address. If the operator
        //  has not specified an explicit IP address on the command line
        //  when sip_rogue is executed, then the default is to bind
        //  the sip_rogue control interface to the IP address of any and
        //  all ethernet interfaces for TCP port 6060. Any TCP packets
        //  arriving on an IP address of the sip_rogue host and
        //  specifying port  6060 are submitted to sip_rogue.
        //
        
        controlPort.SetBoundAddress( INADDR_ANY );
        controlPort.SetBoundPort( htons( 6060 ) );
    }
    
    //
    //  Spawn the companion "Network" thread.
    //

    switch ( pthread_create( &controlPortThread,
                             NULL,
                             &controlPortThreadFunction,
                             &controlPort ) )
    {
        case 0:
            break;
        case EAGAIN:
            fprintf( stderr,
                     "Too many threads to start controlPort thread.\n" );
            return 1;
        case EINVAL:
            fprintf( stderr,
                     "Invalid thread attributes when attempting to start "
                     "controlPort thread.\n" );
            return 1;
        case EPERM:
            fprintf( stderr,
                     "Insufficient permissions when attempting to start "
                     "controlPort thread.\n" );
            return 1;
        default:
            fprintf( stderr,
                     "Unknown error when attempting to start controlPort "
                     "thread.\n" );
            return 1;
    }

    controlPort.ProcessRun();

    controlPort.ProcessStop();
    controlPort.NetworkStop();
    pthread_join( controlPortThread, NULL );

    return 0;
}

