//-----------------------------------------------------------------------------
//
// test.cpp - Unit tests
//
//    Copyright (C) 2004  Mark D. Collier
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//   Author: Mark D. Collier   - 12/01/2006   v1.1
//                   Mark D. Collier   -  04/26/2004  v1.0
//         www.securelogix.com - mark.collier@securelogix.com
//         www.hackingexposedvoip.com
//
//-----------------------------------------------------------------------------

#include <arpa/inet.h>
#include <netinet/in.h>
#include <stdio.h>
#include <string.h>
#include <sys/socket.h>

#include "SipUri.h"
#include "SdpMessage.h"
#include "SipHeader.h"
#include "SipIdentifier.h"
#include "util.h"

#define malloc  mymalloc
#define free    myfree
#define strdup  mystrdup
#define strndup mystrndup

int  determineLocalAddressForTest( void )
{
    /* This test only works on my (glh) local box.
    int       failedTests;
    in_addr_t localAddress;
    in_addr_t remoteAddress;

    failedTests = 0;

    remoteAddress = inet_addr( "127.0.0.1" );
    localAddress  = DetermineLocalAddressFor( remoteAddress );
    if ( localAddress != htonl( 0x7F000001 ) )
    {
        failedTests++;
        fprintf( stderr,
                "%s:%d: Test Failed: local address resolved to %d.%d.%d.%d "
                "when remote address was %d.%d.%d.%d\n",
                __FILE__, __LINE__,
                localAddress        & 0xFF,
                localAddress  >>  8 & 0xFF,
                localAddress  >> 16 & 0xFF,
                localAddress  >> 24 & 0xFF,
                remoteAddress       & 0xFF,
                remoteAddress >>  8 & 0xFF,
                remoteAddress >> 16 & 0xFF,
                remoteAddress >> 24 & 0xFF );
    }

    remoteAddress = inet_addr( "127.1.2.3" );
    localAddress  = DetermineLocalAddressFor( remoteAddress );
    if ( localAddress != htonl( 0x7F010203 ) )
    {
        failedTests++;
        fprintf( stderr,
                "%s:%d: Test Failed: local address resolved to %d.%d.%d.%d "
                "when remote address was %d.%d.%d.%d\n",
                __FILE__, __LINE__,
                localAddress        & 0xFF,
                localAddress  >>  8 & 0xFF,
                localAddress  >> 16 & 0xFF,
                localAddress  >> 24 & 0xFF,
                remoteAddress       & 0xFF,
                remoteAddress >>  8 & 0xFF,
                remoteAddress >> 16 & 0xFF,
                remoteAddress >> 24 & 0xFF );
    }

    remoteAddress = inet_addr( "192.168.1.99" );
    localAddress  = DetermineLocalAddressFor( remoteAddress );
    if ( !localAddress )
    {
        failedTests++;
        fprintf( stderr,
                "%s:%d: Test Failed: Could not resolve local address when "
                "remote address was %d.%d.%d.%d\n",
                __FILE__, __LINE__,
                remoteAddress       & 0xFF,
                remoteAddress >>  8 & 0xFF,
                remoteAddress >> 16 & 0xFF,
                remoteAddress >> 24 & 0xFF );
    }

    remoteAddress = inet_addr( "10.1.1.1" );
    localAddress  = DetermineLocalAddressFor( remoteAddress );
    if ( !localAddress )
    {
        failedTests++;
        fprintf( stderr,
                "%s:%d: Test Failed: Could not resolve local address when "
                "remote address was %d.%d.%d.%d\n",
                __FILE__, __LINE__,
                remoteAddress       & 0xFF,
                remoteAddress >>  8 & 0xFF,
                remoteAddress >> 16 & 0xFF,
                remoteAddress >> 24 & 0xFF );
    }

    remoteAddress = inet_addr( "207.218.239.210" );
    localAddress  = DetermineLocalAddressFor( remoteAddress );
    if ( !localAddress )
    {
        failedTests++;
        fprintf( stderr,
                "%s:%d: Test Failed: Could not resolve local address when "
                "remote address was %d.%d.%d.%d\n",
                __FILE__, __LINE__,
                remoteAddress       & 0xFF,
                remoteAddress >>  8 & 0xFF,
                remoteAddress >> 16 & 0xFF,
                remoteAddress >> 24 & 0xFF );
    }

    return failedTests;
    */
    return 0;
}

int  determineLocalAddressAndHardwareAddressForTest( void )
{
    /* This test only works on my (glh) local box.
    int            failedTests;
    in_addr_t      localAddress;
    in_addr_t      remoteAddress;
    unsigned char  hardwareAddress[6];

    failedTests = 0;

    remoteAddress = inet_addr( "127.0.0.1" );
    DetermineLocalAddressAndHardwareAddressFor( &localAddress,
                                                 hardwareAddress,
                                                 remoteAddress );
    if ( localAddress != htonl( 0x7F000001 ) )
    {
        failedTests++;
        fprintf( stderr,
                "%s:%d: Test Failed: local address resolved to %d.%d.%d.%d "
                "when remote address was %d.%d.%d.%d\n",
                __FILE__, __LINE__,
                localAddress        & 0xFF,
                localAddress  >>  8 & 0xFF,
                localAddress  >> 16 & 0xFF,
                localAddress  >> 24 & 0xFF,
                remoteAddress       & 0xFF,
                remoteAddress >>  8 & 0xFF,
                remoteAddress >> 16 & 0xFF,
                remoteAddress >> 24 & 0xFF );
    }
    if (   hardwareAddress[0] != 0 || hardwareAddress[1] != 0
        || hardwareAddress[2] != 0 || hardwareAddress[3] != 0
        || hardwareAddress[4] != 0 || hardwareAddress[5] != 0 )
    {
        failedTests++;
        fprintf( stderr,
                "%s:%d: Test Failed: hardware address resolved to "
                "%02X:%02X:%02X:%02X:%02X:%02X when remote address was "
                "%d.%d.%d.%d\n",
                __FILE__, __LINE__,
                hardwareAddress[0], hardwareAddress[1], hardwareAddress[2],
                hardwareAddress[3], hardwareAddress[4], hardwareAddress[5],
                remoteAddress       & 0xFF,
                remoteAddress >>  8 & 0xFF,
                remoteAddress >> 16 & 0xFF,
                remoteAddress >> 24 & 0xFF );
    }

    remoteAddress = inet_addr( "127.1.2.3" );
    DetermineLocalAddressAndHardwareAddressFor( &localAddress,
                                                hardwareAddress,
                                                remoteAddress );
    if ( localAddress != htonl( 0x7F010203 ) )
    {
        failedTests++;
        fprintf( stderr,
                "%s:%d: Test Failed: local address resolved to %d.%d.%d.%d "
                "when remote address was %d.%d.%d.%d\n",
                __FILE__, __LINE__,
                localAddress        & 0xFF,
                localAddress  >>  8 & 0xFF,
                localAddress  >> 16 & 0xFF,
                localAddress  >> 24 & 0xFF,
                remoteAddress       & 0xFF,
                remoteAddress >>  8 & 0xFF,
                remoteAddress >> 16 & 0xFF,
                remoteAddress >> 24 & 0xFF );
    }
    if (   hardwareAddress[0] != 0 || hardwareAddress[1] != 0
        || hardwareAddress[2] != 0 || hardwareAddress[3] != 0
        || hardwareAddress[4] != 0 || hardwareAddress[5] != 0 )
    {
        failedTests++;
        fprintf( stderr,
                "%s:%d: Test Failed: hardware address resolved to "
                "%02X:%02X:%02X:%02X:%02X:%02X when remote address was "
                "%d.%d.%d.%d\n",
                __FILE__, __LINE__,
                hardwareAddress[0], hardwareAddress[1], hardwareAddress[2],
                hardwareAddress[3], hardwareAddress[4], hardwareAddress[5],
                remoteAddress       & 0xFF,
                remoteAddress >>  8 & 0xFF,
                remoteAddress >> 16 & 0xFF,
                remoteAddress >> 24 & 0xFF );
    }

    remoteAddress = inet_addr( "192.168.1.99" );
    DetermineLocalAddressAndHardwareAddressFor( &localAddress,
                                                hardwareAddress,
                                                remoteAddress );
    if ( !localAddress )
    {
        failedTests++;
        fprintf( stderr,
                "%s:%d: Test Failed: Could not resolve local address when "
                "remote address was %d.%d.%d.%d\n",
                __FILE__, __LINE__,
                remoteAddress       & 0xFF,
                remoteAddress >>  8 & 0xFF,
                remoteAddress >> 16 & 0xFF,
                remoteAddress >> 24 & 0xFF );
    }

    remoteAddress = inet_addr( "10.1.1.1" );
    DetermineLocalAddressAndHardwareAddressFor( &localAddress,
                                                hardwareAddress,
                                                remoteAddress );
    if ( !localAddress )
    {
        failedTests++;
        fprintf( stderr,
                "%s:%d: Test Failed: Could not resolve local address when "
                "remote address was %d.%d.%d.%d\n",
                __FILE__, __LINE__,
                remoteAddress       & 0xFF,
                remoteAddress >>  8 & 0xFF,
                remoteAddress >> 16 & 0xFF,
                remoteAddress >> 24 & 0xFF );
    }

    remoteAddress = inet_addr( "207.218.239.210" );
    DetermineLocalAddressAndHardwareAddressFor( &localAddress,
                                                hardwareAddress,
                                                remoteAddress );
    if ( !localAddress )
    {
        failedTests++;
        fprintf( stderr,
                "%s:%d: Test Failed: Could not resolve local address when "
                "remote address was %d.%d.%d.%d\n",
                __FILE__, __LINE__,
                remoteAddress       & 0xFF,
                remoteAddress >>  8 & 0xFF,
                remoteAddress >> 16 & 0xFF,
                remoteAddress >> 24 & 0xFF );
    }

    return failedTests;
    */
    return 0;
}


int  sdpMessageTest( void )
{
    SdpMessage *  sdpMessage;
    char *        in;
    char *        out;
    int           failedTests;

    failedTests = 0;

    in = "v=0\r\n"
         "o=- 1 2 IN IP4 127.0.0.1\r\n"
         "s=call\r\n"
         "c=IN IP4 127.0.0.1\r\n"
         "m=audio 12340 RTP/AVP 0 96\r\n";
    sdpMessage = new SdpMessage( in );
    out = sdpMessage->GetPacket();
    if ( strcmp( in, out ) != 0 )
    {
        failedTests++;
        fprintf( stderr,
                "%s:%d: Test Failed: SDP out didn't match in:\r\n::: IN :::\r\n"
                "%s\r\n::: OUT :::\r\n%s",
                __FILE__, __LINE__, in, out );
    }
    free( out );
    if ( sdpMessage->GetMediaDescriptions()->GetPortIP4Value() != 12340 )
    {
        failedTests++;
        fprintf( stderr,
                "%s:%d: Test Failed: PortIP4Value is %d, expected %d",
                __FILE__, __LINE__,
                sdpMessage->GetMediaDescriptions()->GetPortIP4Value(), 12340 );
    }
    if ( sdpMessage->GetMediaDescriptions()->GetPortIP4Count() != 1 )
    {
        failedTests++;
        fprintf( stderr,
                "%s:%d: Test Failed: PortIP4Count is %d, expected %d",
                __FILE__, __LINE__,
                sdpMessage->GetMediaDescriptions()->GetPortIP4Count(), 1 );
    }
    delete sdpMessage;

    return failedTests;
}


int  sipHeaderTestHelper( char *   aLine,
                          char *   aName,
                          char *   aValue,
                          int      aIdentifierCount,
                          char **  aIdentifiers )
{
    int          failedTests;
    SipHeader *  header;
    int          index;

    failedTests = 0;
    header = new SipHeader( aLine );

    if ( strcmp( header->GetFullText(), aLine ) )
    {
        failedTests++;
        fprintf( stderr, "%s:%d: Test Failed: %s != %s\n", __FILE__, __LINE__,
                 header->GetFullText(), aLine );
    }

    if ( strcmp( header->GetName(), aName ) )
    {
        failedTests++;
        fprintf( stderr, "%s:%d: Test Failed: %s != %s\n", __FILE__, __LINE__,
                 header->GetName(), aName );
    }

    if ( strcmp( header->GetValue(), aValue ) )
    {
        failedTests++;
        fprintf( stderr, "%s:%d: Test Failed: %s != %s\n", __FILE__, __LINE__,
                 header->GetValue(), aValue );
    }

    if ( header->GetIdentifierCount() != aIdentifierCount )
    {
        failedTests++;
        fprintf( stderr, "%s:%d: Test Failed: %d != %d\n", __FILE__, __LINE__,
                 header->GetIdentifierCount(), aIdentifierCount );
    }

    for ( index = 0; index < aIdentifierCount; index++ )
    {
        if ( strcmp( header->GetIdentifier( index )->GetFullText(),
                     aIdentifiers[index] ) )
        {
            failedTests++;
            fprintf( stderr, "%s:%d: Test Failed: %s != %s\n",
                     __FILE__, __LINE__,
                     header->GetIdentifier( index )->GetFullText(),
                     aIdentifiers[index] );
        }
    }

    return failedTests;
}


int  sipHeaderTest( void )
{
    int     failedTests;
    char *  identifiers[128];

    failedTests = 0;

    identifiers[0] = "sip:user@test.com";
    failedTests += sipHeaderTestHelper( "Contact: sip:user@test.com",
                                        "Contact",
                                        "sip:user@test.com",
                                        1,
                                        identifiers );

    identifiers[0] = "sip:via@10.1.1.131";
    failedTests += sipHeaderTestHelper( "Via: SIP/UDP/2.0 10.1.1.131",
                                        "Via",
                                        "SIP/UDP/2.0 10.1.1.131",
                                        1,
                                        identifiers );

    identifiers[0] = "sip:user@test.com";
    identifiers[1] = "sip:user2@test.com";
    failedTests += sipHeaderTestHelper( "Contact: sip:user@test.com, "
                                        "sip:user2@test.com",
                                        "Contact",
                                        "sip:user@test.com, "
                                        "sip:user2@test.com",
                                        2,
                                        identifiers );

    identifiers[0] = "sip:user@test.com";
    identifiers[1] = "sip:user2@test.com";
    identifiers[2] = "sip:user3@test.com";
    failedTests += sipHeaderTestHelper( "Contact: sip:user@test.com, "
                                        "sip:user2@test.com, "
                                        "sip:user3@test.com",
                                        "Contact",
                                        "sip:user@test.com, "
                                        "sip:user2@test.com, "
                                        "sip:user3@test.com",
                                        3,
                                        identifiers );

    identifiers[0] = "sip:user@test.com";
    identifiers[1] = "sip:user2@test.com";
    identifiers[2] = "sip:user3@test.com";
    failedTests += sipHeaderTestHelper( "Contact\t: \t\t sip:user@test.com,    "
                                        "sip:user2@test.com, \t\t "
                                        "sip:user3@test.com",
                                        "Contact",
                                        "sip:user@test.com,    "
                                        "sip:user2@test.com, \t\t "
                                        "sip:user3@test.com",
                                        3,
                                        identifiers );

    identifiers[0] = "User <sip:user@test.com>";
    identifiers[1] = "User Two <sip:user2@test.com>";
    identifiers[2] = "\"User Three\" <sip:user3@test.com>";
    failedTests += sipHeaderTestHelper( "Contact: User <sip:user@test.com>, "
                                        "User Two <sip:user2@test.com>, "
                                        "\"User Three\" <sip:user3@test.com>",
                                        "Contact",
                                        "User <sip:user@test.com>, "
                                        "User Two <sip:user2@test.com>, "
                                        "\"User Three\" <sip:user3@test.com>",
                                        3,
                                        identifiers );

    identifiers[0] = "User <sip:user@test.com>";
    identifiers[1] = "User Two <sip:user2@test.com>";
    identifiers[2] = "\"User, Three\" <sip:user3@test.com>";
    failedTests += sipHeaderTestHelper( "Contact: User <sip:user@test.com>, "
                                        "User Two <sip:user2@test.com>, "
                                        "\"User, Three\" <sip:user3@test.com>",
                                        "Contact",
                                        "User <sip:user@test.com>, "
                                        "User Two <sip:user2@test.com>, "
                                        "\"User, Three\" <sip:user3@test.com>",
                                        3,
                                        identifiers );

    identifiers[0] = "\"User One\"<sip:user@test.com>";
    identifiers[1] = "User Two <stp:us,er2@test.com>";
    identifiers[2] = "\"User, Three\" <sip:user3@test.com>";
    failedTests += sipHeaderTestHelper( "Contact: "
                                        "\"User One\"<sip:user@test.com>, "
                                        "User Two <stp:us,er2@test.com>, "
                                        "\"User, Three\" <sip:user3@test.com>",
                                        "Contact",
                                        "\"User One\"<sip:user@test.com>, "
                                        "User Two <stp:us,er2@test.com>, "
                                        "\"User, Three\" <sip:user3@test.com>",
                                        3,
                                        identifiers );

    return failedTests;
}


int  sipIdentifierTestHelper( char *   aLine,
                              char *   aName,
                              char *   aUri,
                              int      aParameterCount,
                              char **  aParameterNames,
                              char **  aParameterValues )
{
    int              failedTests;
    SipIdentifier *  identifier;
    int              index;

    failedTests = 0;
    identifier  = new SipIdentifier( strdup( aLine ) );

    if ( strcmp( identifier->GetFullText(), aLine ) )
    {
        failedTests++;
        fprintf( stderr, "%s:%d: Test Failed: %s != %s\n", __FILE__, __LINE__,
                 identifier->GetFullText(), aLine );
    }

    if ( strcmp( identifier->GetName(), aName ) )
    {
        failedTests++;
        fprintf( stderr, "%s:%d: Test Failed: %s != %s\n", __FILE__, __LINE__,
                 identifier->GetName(), aName );
    }

    if ( strcmp( identifier->GetUri()->GetFullText(), aUri ) )
    {
        failedTests++;
        fprintf( stderr, "%s:%d: Test Failed: %s != %s\n", __FILE__, __LINE__,
                 identifier->GetUri()->GetFullText(), aUri );
    }

    if ( identifier->GetParameterCount() != aParameterCount )
    {
        failedTests++;
        fprintf( stderr, "%s:%d: Test Failed: %d != %d\n", __FILE__, __LINE__,
                 identifier->GetParameterCount(), aParameterCount );
    }

    for ( index = 0; index < aParameterCount; index++ )
    {
        if ( strcmp( identifier->GetParameterName( index ),
                     aParameterNames[index] ) )
        {
            failedTests++;
            fprintf( stderr, "%s:%d: Test Failed: %s != %s\n",
                     __FILE__, __LINE__,
                     identifier->GetParameterName( index ),
                     aParameterNames[index] );
        }
        if ( strcmp( identifier->GetParameterValue( index ),
                     aParameterValues[index] ) )
        {
            failedTests++;
            fprintf( stderr, "%s:%d: Test Failed: %s != %s\n",
                     __FILE__, __LINE__,
                     identifier->GetParameterValue( index ),
                     aParameterValues[index] );
        }
    }

    delete identifier;

    return failedTests;
}


int  sipIdentifierTest( void )
{
    int              failedTests;
    char *           parameterNames[128];
    char *           parameterValues[128];
    SipIdentifier *  identifier;

    failedTests = 0;

    failedTests += sipIdentifierTestHelper( "sip:user@test.com",
                                            "",
                                            "sip:user@test.com",
                                            0,
                                            parameterNames,
                                            parameterValues );

    failedTests += sipIdentifierTestHelper( "<sip:user@test.com>",
                                            "",
                                            "sip:user@test.com",
                                            0,
                                            parameterNames,
                                            parameterValues );

    failedTests += sipIdentifierTestHelper( "  \t sip:user@test.com ",
                                            "",
                                            "sip:user@test.com",
                                            0,
                                            parameterNames,
                                            parameterValues );

    failedTests += sipIdentifierTestHelper( "User <sip:user@test.com>",
                                            "User",
                                            "sip:user@test.com",
                                            0,
                                            parameterNames,
                                            parameterValues );

    failedTests += sipIdentifierTestHelper( "User<sip:user@test.com>",
                                            "User",
                                            "sip:user@test.com",
                                            0,
                                            parameterNames,
                                            parameterValues );

    failedTests += sipIdentifierTestHelper( "User\t<sip:user@test.com>",
                                            "User",
                                            "sip:user@test.com",
                                            0,
                                            parameterNames,
                                            parameterValues );

    failedTests += sipIdentifierTestHelper( "\"User\" <sip:user@test.com>",
                                            "User",
                                            "sip:user@test.com",
                                            0,
                                            parameterNames,
                                            parameterValues );

    failedTests += sipIdentifierTestHelper( "\"User, Test\" "
                                            "<sip:user@test.com>",
                                            "User, Test",
                                            "sip:user@test.com",
                                            0,
                                            parameterNames,
                                            parameterValues );

    failedTests += sipIdentifierTestHelper( "\"User display name with "
                                            "<brackets> and @t signs "
                                            "embedded\" <sip:user@test.com>",
                                            "User display name with <brackets> "
                                            "and @t signs embedded",
                                            "sip:user@test.com",
                                            0,
                                            parameterNames,
                                            parameterValues );

    parameterNames[0]  = "name";
    parameterValues[0] = "value";
    failedTests += sipIdentifierTestHelper( "sip:user@test.com;name=value",
                                            "",
                                            "sip:user@test.com",
                                            1,
                                            parameterNames,
                                            parameterValues );

    parameterNames[0]  = "namex";
    parameterValues[0] = "valuex";
    failedTests += sipIdentifierTestHelper( "<sip:user@test.com;name=value>;"
                                            "namex=valuex",
                                            "",
                                            "sip:user@test.com;name=value",
                                            1,
                                            parameterNames,
                                            parameterValues );

    parameterNames[0]  = "name";
    parameterValues[0] = "value";
    parameterNames[1]  = "name2";
    parameterValues[1] = "value2";
    failedTests += sipIdentifierTestHelper( "sip:user@test.com;name=value;"
                                            "name2=value2",
                                            "",
                                            "sip:user@test.com",
                                            2,
                                            parameterNames,
                                            parameterValues );

    parameterNames[0]  = "name";
    parameterValues[0] = "value";
    parameterNames[1]  = "bool";
    parameterValues[1] = "";
    parameterNames[2]  = "name2";
    parameterValues[2] = "value2";
    parameterNames[3]  = "bool2";
    parameterValues[3] = "";
    failedTests += sipIdentifierTestHelper( "sip:user@test.com;name=value;bool;"
                                            "name2=value2;bool2",
                                            "",
                                            "sip:user@test.com",
                                            4,
                                            parameterNames,
                                            parameterValues );

    parameterNames[0]  = "name";
    parameterValues[0] = "value";
    parameterNames[1]  = "bool";
    parameterValues[1] = "";
    parameterNames[2]  = "name2";
    parameterValues[2] = "value2";
    parameterNames[3]  = "bool2";
    parameterValues[3] = "";
    failedTests += sipIdentifierTestHelper( "sip:user@test.com;name=value;bool;"
                                            "name2=value2;bool2;",
                                            "",
                                            "sip:user@test.com",
                                            4,
                                            parameterNames,
                                            parameterValues );

    identifier = new SipIdentifier(
                    strdup( "sip:user@test.com;one=two;three=four;five=six" ) );
    if ( strcmp( identifier->GetParameterValue( "one" ), "two" ) )
    {
        failedTests++;
        fprintf( stderr, "%s:%d: Test Failed: %s != %s\n",
                 __FILE__, __LINE__,
                 identifier->GetParameterValue( "one" ),
                 "two" );
    }
    if ( strcmp( identifier->GetParameterValue( "three" ), "four" ) )
    {
        failedTests++;
        fprintf( stderr, "%s:%d: Test Failed: %s != %s\n",
                 __FILE__, __LINE__,
                 identifier->GetParameterValue( "three" ),
                 "four" );
    }
    if ( strcmp( identifier->GetParameterValue( "five" ), "six" ) )
    {
        failedTests++;
        fprintf( stderr, "%s:%d: Test Failed: %s != %s\n",
                 __FILE__, __LINE__,
                 identifier->GetParameterValue( "five" ),
                 "six" );
    }
    delete identifier;

    return failedTests;
}


int  sipUriTestHelper( char *   aLine,
                       char *   aScheme,
                       char *   aSchemeValue,
                       char *   aAuthority,
                       char *   aUser,
                       char *   aPassword,
                       char *   aHost,
                       int      aPort,
                       char *   aPath,
                       char *   aQuery,
                       int      aParameterCount,
                       char **  aParameterNames,
                       char **  aParameterValues,
                       int      aHeaderCount,
                       char **  aHeaderNames,
                       char **  aHeaderValues )
{
    int       failedTests;
    SipUri *  uri;
    int       index;

    failedTests = 0;
    uri = new SipUri( strdup( aLine ) );

    if ( strcmp( uri->GetFullText(), aLine ) )
    {
        failedTests++;
        fprintf( stderr, "%s:%d: Test Failed: %s != %s\n", __FILE__, __LINE__,
                 uri->GetFullText(), aLine );
    }

    if ( strcmp( uri->GetScheme(), aScheme ) )
    {
        failedTests++;
        fprintf( stderr, "%s:%d: Test Failed: %s != %s\n", __FILE__, __LINE__,
                 uri->GetScheme(), aScheme );
    }

    if ( strcmp( uri->GetSchemeValue(), aSchemeValue ) )
    {
        failedTests++;
        fprintf( stderr, "%s:%d: Test Failed: %s != %s\n", __FILE__, __LINE__,
                 uri->GetSchemeValue(), aSchemeValue );
    }

    if ( strcmp( uri->GetAuthority(), aAuthority ) )
    {
        failedTests++;
        fprintf( stderr, "%s:%d: Test Failed: %s != %s\n", __FILE__, __LINE__,
                 uri->GetAuthority(), aAuthority );
    }

    if ( strcmp( uri->GetUser(), aUser ) )
    {
        failedTests++;
        fprintf( stderr, "%s:%d: Test Failed: %s != %s\n", __FILE__, __LINE__,
                 uri->GetUser(), aUser );
    }

    if ( strcmp( uri->GetPassword(), aPassword ) )
    {
        failedTests++;
        fprintf( stderr, "%s:%d: Test Failed: %s != %s\n", __FILE__, __LINE__,
                 uri->GetPassword(), aPassword );
    }

    if ( strcmp( uri->GetHost(), aHost ) )
    {
        failedTests++;
        fprintf( stderr, "%s:%d: Test Failed: %s != %s\n", __FILE__, __LINE__,
                 uri->GetHost(), aHost );
    }

    if ( uri->GetPort() != aPort )
    {
        failedTests++;
        fprintf( stderr, "%s:%d: Test Failed: %d != %d\n", __FILE__, __LINE__,
                 uri->GetPort(), aPort );
    }

    if ( strcmp( uri->GetPath(), aPath ) )
    {
        failedTests++;
        fprintf( stderr, "%s:%d: Test Failed: %s != %s\n", __FILE__, __LINE__,
                 uri->GetPath(), aPath );
    }

    if ( strcmp( uri->GetQuery(), aQuery ) )
    {
        failedTests++;
        fprintf( stderr, "%s:%d: Test Failed: %s != %s\n", __FILE__, __LINE__,
                 uri->GetQuery(), aQuery );
    }

    if ( uri->GetParameterCount() != aParameterCount )
    {
        failedTests++;
        fprintf( stderr, "%s:%d: Test Failed: %d != %d\n", __FILE__, __LINE__,
                 uri->GetParameterCount(), aParameterCount );
    }

    for ( index = 0; index < aParameterCount; index++ )
    {
        if ( strcmp( uri->GetParameterName( index ),
                     aParameterNames[index] ) )
        {
            failedTests++;
            fprintf( stderr, "%s:%d: Test Failed: %s != %s\n",
                     __FILE__, __LINE__,
                     uri->GetParameterName( index ),
                     aParameterNames[index] );
        }
        if ( strcmp( uri->GetParameterValue( index ),
                     aParameterValues[index] ) )
        {
            failedTests++;
            fprintf( stderr, "%s:%d: Test Failed: %s != %s\n",
                     __FILE__, __LINE__,
                     uri->GetParameterValue( index ),
                     aParameterValues[index] );
        }
    }

    if ( uri->GetHeaderCount() != aHeaderCount )
    {
        failedTests++;
        fprintf( stderr, "%s:%d: Test Failed: %d != %d\n", __FILE__, __LINE__,
                 uri->GetHeaderCount(), aHeaderCount );
    }

    for ( index = 0; index < aHeaderCount; index++ )
    {
        if ( strcmp( uri->GetHeaderName( index ),
                     aHeaderNames[index] ) )
        {
            failedTests++;
            fprintf( stderr, "%s:%d: Test Failed: %s != %s\n",
                     __FILE__, __LINE__,
                     uri->GetHeaderName( index ),
                     aHeaderNames[index] );
        }
        if ( strcmp( uri->GetHeaderValue( index ),
                     aHeaderValues[index] ) )
        {
            failedTests++;
            fprintf( stderr, "%s:%d: Test Failed: %s != %s\n",
                     __FILE__, __LINE__,
                     uri->GetHeaderValue( index ),
                     aHeaderValues[index] );
        }
    }

    delete uri;

    return failedTests;
}


int  sipUriTest( void )
{
    int       failedTests;
    char *    parameterNames[128];
    char *    parameterValues[128];
    char *    headerNames[128];
    char *    headerValues[128];

    failedTests = 0;

    failedTests += sipUriTestHelper( "10.1.1.131",
                                     "sip",
                                     "10.1.1.131",
                                     "",
                                     "",
                                     "",
                                     "10.1.1.131",
                                     -1,
                                     "",
                                     "",
                                     0,
                                     parameterNames,
                                     parameterValues,
                                     0,
                                     headerNames,
                                     headerValues );

    failedTests += sipUriTestHelper( "sip:test.com",
                                     "sip",
                                     "test.com",
                                     "",
                                     "",
                                     "",
                                     "test.com",
                                     -1,
                                     "",
                                     "",
                                     0,
                                     parameterNames,
                                     parameterValues,
                                     0,
                                     headerNames,
                                     headerValues );

    failedTests += sipUriTestHelper( "sip:user@test.com",
                                     "sip",
                                     "user@test.com",
                                     "",
                                     "user",
                                     "",
                                     "test.com",
                                     -1,
                                     "",
                                     "",
                                     0,
                                     parameterNames,
                                     parameterValues,
                                     0,
                                     headerNames,
                                     headerValues );

    failedTests += sipUriTestHelper( "sip:user:password@test1.com",
                                     "sip",
                                     "user:password@test1.com",
                                     "",
                                     "user",
                                     "password",
                                     "test1.com",
                                     -1,
                                     "",
                                     "",
                                     0,
                                     parameterNames,
                                     parameterValues,
                                     0,
                                     headerNames,
                                     headerValues );

    failedTests += sipUriTestHelper( "http://test2.com/path/file",
                                     "http",
                                     "//test2.com/path/file",
                                     "test2.com",
                                     "",
                                     "",
                                     "test2.com",
                                     -1,
                                     "/path/file",
                                     "",
                                     0,
                                     parameterNames,
                                     parameterValues,
                                     0,
                                     headerNames,
                                     headerValues );

    failedTests += sipUriTestHelper( "http://user@test3.com/path/file",
                                     "http",
                                     "//user@test3.com/path/file",
                                     "user@test3.com",
                                     "user",
                                     "",
                                     "test3.com",
                                     -1,
                                     "/path/file",
                                     "",
                                     0,
                                     parameterNames,
                                     parameterValues,
                                     0,
                                     headerNames,
                                     headerValues );

    failedTests += sipUriTestHelper( "http://user:password@test4.com/path/file",
                                     "http",
                                     "//user:password@test4.com/path/file",
                                     "user:password@test4.com",
                                     "user",
                                     "password",
                                     "test4.com",
                                     -1,
                                     "/path/file",
                                     "",
                                     0,
                                     parameterNames,
                                     parameterValues,
                                     0,
                                     headerNames,
                                     headerValues );

    failedTests += sipUriTestHelper( "sip:user@test.com:1234",
                                     "sip",
                                     "user@test.com:1234",
                                     "",
                                     "user",
                                     "",
                                     "test.com",
                                     1234,
                                     "",
                                     "",
                                     0,
                                     parameterNames,
                                     parameterValues,
                                     0,
                                     headerNames,
                                     headerValues );

    failedTests += sipUriTestHelper( "sip:user:password@test1.com:1234",
                                     "sip",
                                     "user:password@test1.com:1234",
                                     "",
                                     "user",
                                     "password",
                                     "test1.com",
                                     1234,
                                     "",
                                     "",
                                     0,
                                     parameterNames,
                                     parameterValues,
                                     0,
                                     headerNames,
                                     headerValues );

    failedTests += sipUriTestHelper( "http://test2.com:1234/path/file",
                                     "http",
                                     "//test2.com:1234/path/file",
                                     "test2.com:1234",
                                     "",
                                     "",
                                     "test2.com",
                                     1234,
                                     "/path/file",
                                     "",
                                     0,
                                     parameterNames,
                                     parameterValues,
                                     0,
                                     headerNames,
                                     headerValues );

    failedTests += sipUriTestHelper( "http://user@test3.com:1234/path/file",
                                     "http",
                                     "//user@test3.com:1234/path/file",
                                     "user@test3.com:1234",
                                     "user",
                                     "",
                                     "test3.com",
                                     1234,
                                     "/path/file",
                                     "",
                                     0,
                                     parameterNames,
                                     parameterValues,
                                     0,
                                     headerNames,
                                     headerValues );

    failedTests += sipUriTestHelper( "http://user:password@test4.com:1234"
                                     "/path/file",
                                     "http",
                                     "//user:password@test4.com:1234/path/file",
                                     "user:password@test4.com:1234",
                                     "user",
                                     "password",
                                     "test4.com",
                                     1234,
                                     "/path/file",
                                     "",
                                     0,
                                     parameterNames,
                                     parameterValues,
                                     0,
                                     headerNames,
                                     headerValues );

    failedTests += sipUriTestHelper( "http://user:password@test4.com:1234"
                                     "/path/file?query",
                                     "http",
                                     "//user:password@test4.com:1234/path/file"
                                     "?query",
                                     "user:password@test4.com:1234",
                                     "user",
                                     "password",
                                     "test4.com",
                                     1234,
                                     "/path/file",
                                     "query",
                                     0,
                                     parameterNames,
                                     parameterValues,
                                     0,
                                     headerNames,
                                     headerValues );

    failedTests += sipUriTestHelper( "http://user:password@test4.com:1234"
                                     "/?query",
                                     "http",
                                     "//user:password@test4.com:1234/?query",
                                     "user:password@test4.com:1234",
                                     "user",
                                     "password",
                                     "test4.com",
                                     1234,
                                     "/",
                                     "query",
                                     0,
                                     parameterNames,
                                     parameterValues,
                                     0,
                                     headerNames,
                                     headerValues );

    failedTests += sipUriTestHelper( "http://user:password@test4.com:1234/",
                                     "http",
                                     "//user:password@test4.com:1234/",
                                     "user:password@test4.com:1234",
                                     "user",
                                     "password",
                                     "test4.com",
                                     1234,
                                     "/",
                                     "",
                                     0,
                                     parameterNames,
                                     parameterValues,
                                     0,
                                     headerNames,
                                     headerValues );

    failedTests += sipUriTestHelper( "http://user:password@test4.com:1234",
                                     "http",
                                     "//user:password@test4.com:1234",
                                     "user:password@test4.com:1234",
                                     "user",
                                     "password",
                                     "test4.com",
                                     1234,
                                     "",
                                     "",
                                     0,
                                     parameterNames,
                                     parameterValues,
                                     0,
                                     headerNames,
                                     headerValues );

    failedTests += sipUriTestHelper( "squirrels:/a;lskjfa;lfdskjads;flkj//sdf/",
                                     "squirrels",
                                     "/a;lskjfa;lfdskjads;flkj//sdf/",
                                     "",
                                     "",
                                     "",
                                     "",
                                     -1,
                                     "/a;lskjfa;lfdskjads;flkj//sdf/",
                                     "",
                                     0,
                                     parameterNames,
                                     parameterValues,
                                     0,
                                     headerNames,
                                     headerValues );

    parameterNames[0]  = "parm";
    parameterValues[0] = "val";
    failedTests += sipUriTestHelper( "sip:user@1.2.3.4;parm=val",
                                     "sip",
                                     "user@1.2.3.4;parm=val",
                                     "",
                                     "user",
                                     "",
                                     "1.2.3.4",
                                     -1,
                                     "",
                                     "",
                                     1,
                                     parameterNames,
                                     parameterValues,
                                     0,
                                     headerNames,
                                     headerValues );

    return failedTests;
}


int  main( int  aArgc, char **  aArgv )
{
    int failedTests;

    failedTests  = 0;
    failedTests += determineLocalAddressForTest();
    failedTests += determineLocalAddressAndHardwareAddressForTest();
    failedTests += sdpMessageTest();
    failedTests += sipHeaderTest();
    failedTests += sipIdentifierTest();
    failedTests += sipUriTest();

    return failedTests;
}

