//-----------------------------------------------------------------------------
//
// util.cpp - Extra functions that are non-VoIP specific.
//
//    Copyright (C) 2004  Mark D. Collier
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//   Author: Mark D. Collier   - 12/01/2006   v1.1
//                   Mark D. Collier   -  04/26/2004  v1.0
//         www.securelogix.com - mark.collier@securelogix.com
//         www.hackingexposedvoip.com
//
//-----------------------------------------------------------------------------

#include <asm/types.h>
#include <errno.h>
#include <fcntl.h>
#include <netinet/in.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>

#include <linux/rtnetlink.h>

#include "util.h"

void  ReportError( char *  aMessage, char *  aFile,  int  aLine )
{
    static char  lineBuffer[16];

    fputs( aFile, stderr );
    fputc( ':', stderr );
    snprintf( lineBuffer, 16, "%d", aLine );
    fputs( lineBuffer, stderr );
    fputs( ": error: ", stderr );
    fputs( aMessage, stderr );
    fputc( '\n', stderr );
    fflush( stderr );
}


in_addr_t  DetermineLocalAddressFor( in_addr_t  aRemoteAddress )
{
    in_addr_t  localAddress;

    DetermineLocalAddressAndHardwareAddressFor( &localAddress,
                                                NULL,
                                                aRemoteAddress );
    return localAddress;
}

bool  DetermineLocalAddressAndHardwareAddressFor(
                                             in_addr_t *    aLocalAddress,
                                             unsigned char  aHardwareAddress[6],
                                             in_addr_t      aRemoteAddress )
{
    char                  buf[8192];
    int                   interfaceIndex;
    int                   len;
    int                   socketAddressLength;
    int                   status;
    struct ifinfomsg   *  im;
    struct iovec          iov;
    struct msghdr         mh;
    struct nlmsghdr    *  nh;
    struct rtattr      *  rta;
    struct rtmsg       *  rm;
    struct sockaddr_nl    nladdr;

    if ( aLocalAddress )
    {
        *aLocalAddress = 0;
    }
    if ( aHardwareAddress )
    {
        memset( aHardwareAddress, 0, 6 );
    }

    struct
    {
        unsigned char  Family;
        unsigned char  Bytelen;
        short          Bitlen;
        unsigned int   Data[4];
    }  addr;

    struct
    {
        struct nlmsghdr  Nh;
        struct rtmsg     Rm;
        char             Buf[1024];
    }  grRequest;

    struct
    {
        int                 Fd;
        struct sockaddr_nl  Local;
        struct sockaddr_nl  Peer;
        __u32               Seq;
        __u32               Dump;
    }  rh;

    struct
    {
        struct nlmsghdr  Nh;
        struct rtgenmsg  Rgm;
    }  glRequest;

    memset( &addr, 0, sizeof( addr ) );
    memset( &grRequest, 0, sizeof( grRequest ) );
    memset( &rh, 0, sizeof( rh ) );
    memset( &glRequest, 0, sizeof( glRequest ) );

    rh.Fd = socket( AF_NETLINK, SOCK_RAW, NETLINK_ROUTE );
    if ( rh.Fd < 0 )
    {
        ReportError( "netlink_route socket open error", __FILE__, __LINE__ );
        close( rh.Fd );
        return false;
    }

    rh.Local.nl_family = AF_NETLINK;
    if ( bind( rh.Fd, ( struct sockaddr * )&rh.Local, sizeof( rh.Local ) ) )
    {
        ReportError( "netlink_route socket bind error", __FILE__, __LINE__ );
        close( rh.Fd );
        return false;
    }

    socketAddressLength = sizeof( rh.Local );
    if ( getsockname( rh.Fd,
                      ( struct sockaddr * )&rh.Local,
                      ( socklen_t * )&socketAddressLength ) )
    {
        ReportError( "netlink_route socket getsockname error",
                      __FILE__,
                      __LINE__ );
        close( rh.Fd );
        return false;
    }

    rh.Seq = random();

    addr.Family  = AF_INET;
    addr.Bytelen = 4;
    addr.Bitlen  = 32;
    addr.Data[0] = aRemoteAddress;

    grRequest.Nh.nlmsg_len   = NLMSG_LENGTH( sizeof( struct rtmsg ) );
    grRequest.Nh.nlmsg_flags = NLM_F_REQUEST;
    grRequest.Nh.nlmsg_type  = RTM_GETROUTE;
    grRequest.Nh.nlmsg_seq   = ++rh.Seq;

    grRequest.Rm.rtm_family  = AF_INET;
    grRequest.Rm.rtm_dst_len = addr.Bitlen;
    grRequest.Rm.rtm_src_len = 0;

    rta = ( struct rtattr * )(  ( ( char * )&grRequest.Nh )
                              + NLMSG_ALIGN( grRequest.Nh.nlmsg_len ) );
    rta->rta_type = RTA_DST;
    rta->rta_len  = RTA_LENGTH( addr.Bytelen );
    memcpy( RTA_DATA( rta ), &addr.Data, addr.Bytelen );
    grRequest.Nh.nlmsg_len =  NLMSG_ALIGN( grRequest.Nh.nlmsg_len )
                            + RTA_LENGTH( addr.Bytelen );

    memset( &nladdr, 0, sizeof( nladdr ) );
    nladdr.nl_family = AF_NETLINK;
    sendto( rh.Fd,
            ( void* )&grRequest,
            sizeof( grRequest ),
            0,
            ( struct sockaddr * )&nladdr,
            sizeof( nladdr ) );

    status = 0;
    while ( status <= 0 )
    {
        iov.iov_base      = buf;
        iov.iov_len       = sizeof( buf );
        mh.msg_name       = &nladdr;
        mh.msg_namelen    = sizeof( nladdr );
        mh.msg_iov        = &iov;
        mh.msg_iovlen     = 1;
        mh.msg_control    = NULL;
        mh.msg_controllen = 0;
        mh.msg_flags      = 0;
        status = recvmsg( rh.Fd, &mh, 0 );
        if ( status <= 0 )
        {
            if ( errno == EINTR )
            {
                continue;
            }
            perror( "netlink_route recvmsg error" );
            ReportError( "netlink_route recvmsg error", __FILE__, __LINE__ );
            close( rh.Fd );
            return false;
        }
    }

    for ( nh = ( struct nlmsghdr * )buf;
          NLMSG_OK( nh, ( unsigned int )status );
          nh = NLMSG_NEXT( nh, status ) )
    {
        if (   nh->nlmsg_pid  == rh.Local.nl_pid
            && nh->nlmsg_seq  == rh.Seq
            && nh->nlmsg_type == RTM_NEWROUTE )
        {
            rm  = ( struct rtmsg * )NLMSG_DATA( nh );
            rta = RTM_RTA( rm );
            len = grRequest.Nh.nlmsg_len - sizeof( struct rtmsg );
            while ( RTA_OK( rta, len) )
            {
                if ( rta->rta_type == RTA_PREFSRC && aLocalAddress )
                {
                    *aLocalAddress = *( in_addr_t * )RTA_DATA( rta );
                }
                else if ( rta->rta_type == RTA_OIF )
                {
                    interfaceIndex = *( int * )RTA_DATA( rta );
                }
                rta = RTA_NEXT( rta, len );
            }
        }
    }

    if ( aHardwareAddress )
    {
        glRequest.Nh.nlmsg_len     = sizeof( glRequest );
        glRequest.Nh.nlmsg_type    = RTM_GETLINK;
        glRequest.Nh.nlmsg_flags   = NLM_F_ROOT | NLM_F_REQUEST;
        glRequest.Nh.nlmsg_pid     = rh.Local.nl_pid;
        glRequest.Nh.nlmsg_seq     = ++rh.Seq;
        glRequest.Rgm.rtgen_family = AF_PACKET;

        memset( &nladdr, 0, sizeof( nladdr ) );
        nladdr.nl_family = AF_NETLINK;
        sendto( rh.Fd,
                ( void* )&glRequest,
                sizeof( glRequest ),
                0,
                ( struct sockaddr * )&nladdr,
                sizeof( nladdr ) );

        status = 0;
        while ( status <= 0 )
        {
            iov.iov_base      = buf;
            iov.iov_len       = sizeof( buf );
            mh.msg_name       = &nladdr;
            mh.msg_namelen    = sizeof( nladdr );
            mh.msg_iov        = &iov;
            mh.msg_iovlen     = 1;
            mh.msg_control    = NULL;
            mh.msg_controllen = 0;
            mh.msg_flags      = 0;
            status = recvmsg( rh.Fd, &mh, 0 );
            if ( status <= 0 )
            {
                if ( errno == EINTR )
                {
                    continue;
                }
                perror( "netlink_route recvmsg error" );
                ReportError( "netlink_route recvmsg error",
                              __FILE__, __LINE__ );
                close( rh.Fd );
                return false;
            }
        }

        for ( nh = ( struct nlmsghdr * )buf;
              NLMSG_OK( nh, ( unsigned int )status );
              nh = NLMSG_NEXT( nh, status ) )
        {
            if (   nh->nlmsg_pid == rh.Local.nl_pid
                && nh->nlmsg_seq == rh.Seq
                && nh->nlmsg_type == RTM_NEWLINK )
            {
                im  = ( struct ifinfomsg * )NLMSG_DATA( nh );
                len = IFLA_PAYLOAD( nh );
                rta = IFLA_RTA( im );
                while ( RTA_OK( rta, len) )
                {
                    if (   im->ifi_index == interfaceIndex
                        && rta->rta_type == IFLA_ADDRESS )
                    {
                        memcpy( aHardwareAddress, RTA_DATA( rta ), 6 );
                    }
                    rta = RTA_NEXT( rta, len );
                }
            }
        }
    }

    if ( close( rh.Fd ) )
    {
        ReportError( "netlink_route socket close error", __FILE__, __LINE__ );
    }

    return true;
}


char *  GetNextGuid( void )
{
    char           *  guid;
    int               r1;
    int               r2;
    int               r3;
    int               ur;
    struct timeval    tv;

    ur = open( "/dev/urandom", O_RDONLY );
    if ( ur < 0 )
    {
        r1 = random();
        r2 = random();
        r3 = random();
    }
    else
    {
        if ( read( ur, &r1, sizeof( r1 ) ) < ( int )sizeof( r1 ) )
        {
            r1 = random();
        }
        if ( read( ur, &r2, sizeof( r2 ) ) < ( int )sizeof( r2 ) )
        {
            r2 = random();
        }
        if ( read( ur, &r3, sizeof( r3 ) ) < ( int )sizeof( r3 ) )
        {
            r3 = random();
        }
        close( ur );
    }

    guid = ( char * )mymalloc( 37 );
    if ( !guid )
    {
        ReportError( "out of memory", __FILE__, __LINE__ );
        return NULL;
    }

    gettimeofday( &tv, NULL );

    snprintf( guid,
              37,
              "%1x%05x%02x-%04x-%04x-%04x-%08x%04x",
              ( unsigned int )  tv.tv_sec       & 0x0000000f,
              ( unsigned int )  tv.tv_usec      & 0x000fffff,
              ( unsigned int )  r3        >> 16 & 0x000000ff,
              ( unsigned int )  tv.tv_sec >>  4 & 0x0000ffff,
              ( unsigned int )( tv.tv_sec >> 20 & 0x00000fff ) | 0x00004000,
              ( unsigned int )( r1              & 0x00003fff ) | 0x00008000,
              ( unsigned int )  r2              & 0xffffffff,
              ( unsigned int )  r3              & 0x0000ffff                 );

    return guid;
}


char *  GetNextWord( char **  aCommandLine )
{
    char *  cp;
    cp = *aCommandLine;
    while ( *cp && ( *cp == ' ' || *cp == '\t' ) )
    {
        cp++;
    }
    *aCommandLine = cp;
    while ( **aCommandLine && **aCommandLine != ' ' && **aCommandLine != '\t' )
    {
        ( *aCommandLine )++;
    }
    if ( **aCommandLine )
    {
        **aCommandLine = '\0';
        ( *aCommandLine )++;
    }
    return cp;
}


void *  mymalloc( size_t  aSize )
{
    char * rv;

    rv = ( char * )malloc( aSize + 8 );
    strcpy( rv, "Cookie\n" );
    rv += 8;
    return rv;
}


void  myfree( void *  aPtr )
{
    char *  cp;
    
    if ( !aPtr )
    {
        return;
    }

    cp = ( char * )aPtr;
        
    cp -= 8;
    if ( strcmp( cp, "Cookie\n" ) == 0 )
    {
        strcpy( cp, "iekooC\n" );
        free( cp );
    }
    else
    {
        fprintf( stderr, "Aaahh!\n" );
        fflush( stderr );
        return;
    }
}


char * mystrdup( const char *  aValue )
{
    char *  rv;

    rv = ( char * )mymalloc( strlen( aValue ) + 1 );
    strcpy( rv, aValue );

    return rv;
}


char *  mystrndup( const char *  aValue, size_t  aCount )
{
    size_t  size;
    char *  rv;

    size = strlen( aValue );
    if ( size >= aCount )
    {
        size = aCount;
    }
    rv = ( char * )mymalloc( size + 1 );
    strncpy( rv, aValue, size );
    rv[size] = '\0';

    return rv;
}

