unit in5obju;

// Copyright  2000 by Ziff Davis Media, Inc.
// Written by Neil J. Rubenking

interface
USES Windows, SysUtils, Classes, in5Share;

type
  TCompObjClass = class OF TCompObj;

  TCompObj = class(TObject)
    // Object used in comparing before/after data. Includes ability
    // to compare itself with another instance, and to read new
    // data from a TReader.
  private
    fLevl : Integer;
    fPath : TStringList;
    fDone : Boolean;
    fLeaf : Boolean; //"Leaf" = file, reg-value, ini-key, txt-line
    function GetFullPath: String; virtual;
    function GetPname: String;
    procedure ZeroData; virtual; abstract;
    procedure ReadData(Rdr : TReader); virtual; abstract;
    procedure LevLeafName(Rdr : TReader; VAR I : Int64;
      VAR S : String); virtual; abstract;
    function GetJustName: String;
    function GetPartPath: String;
  public
    constructor Create;
    destructor Destroy; override;
    function CompareWith(Othr : TCompObj) : Integer;
    procedure Read(Rdr : TReader);
    function Match(Other : TCompObj) : Boolean; virtual; abstract;
    property FullPath : String  read GetFullPath;
    property PartPath : String  read GetPartPath;
    property JustName : String  read GetJustName;
    property pLevl    : Integer read fLevl;
    property pDone    : Boolean read fDone;
    property pLeaf    : Boolean read fLeaf;
    property pName    : String  read GetPName;
  end;

  TDskCompObj = class(TCompObj)
  private
    fSize : Int64;
    fTime : Int64;
    procedure ZeroData; override;
    procedure ReadData(Rdr : TReader); override;
    procedure LevLeafName(Rdr : TReader; VAR I : Int64;
      VAR S : String); override;
  public
    constructor Create;
    function Match(Other : TCompObj) : Boolean; override;
  end;

  TRegCompObj = class(TCompObj)
  private
    fType : DWORD;
    fSize : DWORD;
    fData : ARRAY[0..255] OF Byte;
    fDefV : Boolean; //true if default value
    function GetFullPath: String; override;
    procedure ZeroData; override;
    procedure ReadData(Rdr : TReader); override;
    procedure LevLeafName(Rdr : TReader; VAR I : Int64;
      VAR S : String); override;
  public
    constructor Create;
    function Match(Other : TCompObj) : Boolean; override;
  end;

  TIniCompObj = class(TCompObj)
    // Object used in comparing before/after data. Includes ability
    // to compare itself with another instance, and to read new
    // data from a TReader.
  private
    fValu : String;
    function GetFullPath: String; override;
    procedure ZeroData; override;
    procedure ReadData(Rdr : TReader); override;
    procedure LevLeafName(Rdr : TReader; VAR I : Int64;
      VAR S : String); override;
  public
    constructor Create;
    function Match(Other : TCompObj) : Boolean; override;
  end;

  TTxtCompObj = class(TCompObj)
    // Object used in comparing before/after data. Includes ability
    // to compare itself with another instance, and to read new
    // data from a TReader.
  private
    fLocn : Integer;
    function GetFullPath: String; override;
    procedure ZeroData; override;
    procedure ReadData(Rdr : TReader); override;
    procedure LevLeafName(Rdr : TReader; VAR I : Int64;
      VAR S : String); override;
  public
    constructor Create;
    function Match(Other : TCompObj) : Boolean; override;
  end;

  TChangeObj = class(TObject)
  private
    fClass   : TCompObjClass;
    fPath    : TStringList;
    fBoth    : Boolean;
    fDataA_1 : Int64;
    fDataB_1 : Int64;
    fDataS_1 : String;
    fDataP_1 : PByteArray;
    fDataA_2 : Int64;
    fDataB_2 : Int64;
    fDataS_2 : String;
    fDataP_2 : PByteArray;
    function GetDskDate1: String;
    function GetDskDate2: String;
    function GetDskSize1: String;
    function GetDskSize2: String;
    function GetFullPath: String;
    function GetIniValu1: String;
    function GetIniValu2: String;
    function GetJustName: String;
    function GetPartPath: String;
    function GetRegData1: String;
    function GetRegData2: String;
    function GetRegType1: String;
    function GetRegType2: String;
    function GetTxtLocn1: String;
    function GetTxtLocn2: String;
    function StringForRegData(vType, fSize : Integer;
      fData : PByteArray) : String;
    function GetLocn1: Integer;
  public
    constructor Create(V : TCompObj); overload;
    constructor Create(VO, VN : TCompObj); overload;
    destructor Destroy; override;
    property DskDate1 : String read GetDskDate1;
    property DskSize1 : String read GetDskSize1;
    property RegType1 : String read GetRegType1;
    property RegData1 : String read GetRegData1;
    property IniValu1 : String read GetIniValu1;
    property TxtLocn1 : String read GetTxtLocn1;
    property Locn1    : Integer read GetLocn1;
    property DskDate2 : String read GetDskDate2;
    property DskSize2 : String read GetDskSize2;
    property RegType2 : String read GetRegType2;
    property RegData2 : String read GetRegData2;
    property IniValu2 : String read GetIniValu2;
    property TxtLocn2 : String read GetTxtLocn2;
    property FullPath : String read GetFullPath;
    property PartPath : String read GetPartPath;
    property JustName : String read GetJustName;
    property Path : TStringList read fPath;
    property pClass : TCompObjClass read fClass;
  end;

implementation
uses AllFuncs, md5u;

{ TDskCompObj }

constructor TDskCompObj.Create;
begin
  Inherited Create;
  fSize := 0;
  fTime := 0;
  fLeaf := False;
end;

procedure TDskCompObj.ZeroData;
begin
  fTime := 0; // remains 0 if folder
  fSize := 0; // remains 0 if folder
end;

procedure TDskCompObj.ReadData(Rdr : TReader);
begin
  fSize := Rdr.ReadInt64;
  fTime := Rdr.ReadInt64;
end;

procedure TDskCompObj.LevLeafName(Rdr : TReader; VAR I : Int64;
  VAR S : String);
begin
  fLeaf := I AND cFile > 0;
  I     := I AND $FFF;
  S     := Rdr.ReadString;
end;

function TDskCompObj.Match(Other : TCompObj) : Boolean;
VAR Othr : TDskCompObj;
begin
  Othr := Other AS TDskCompObj;
  Result := (Self.fTime = Othr.fTime) AND
    (Self.fSize = Othr.fSize);
end;

{ TRegCompObj }

constructor TRegCompObj.Create;
begin
  Inherited Create;
  fType := 0;
  fSize := 0;
  fLeaf := True;
  fDefV := False;
  FillChar(fData, SizeOf(fData), 0);
end;

function TRegCompObj.Match(Other: TCompObj): Boolean;
VAR Othr : TRegCompObj;
begin
  Othr := Other AS TRegCompObj;
  Result := False;
  IF Self.fType <> Othr.fType THEN Exit;
  IF Self.fSize <> Othr.fSize THEN Exit;
  IF Self.fSize <= 255 THEN
    Result := CompareMem(@Self.fData, @Othr.fData, Self.fSize)
  ELSE
    Result := CompareMem(@Self.fData, @Othr.fData, SizeOf(TDigest));
end;

procedure TRegCompObj.ZeroData;
begin
  fType := 0;
  fSize := 0;
  FillChar(fData, SizeOf(fData), 0);
end;

procedure TRegCompObj.ReadData(Rdr : TReader);
begin
//  fType := Rdr.ReadInteger;
// There was a problem with the above. Some value data types
// are larger than MaxInt, e.g. $80000001. These are stored
// as vaInt64, though they are just 4 bytes. Fixed by calling
// ReadInt64 instead!
  fType := Rdr.ReadInt64;
  fSize := Rdr.ReadInteger;
  IF fSize <= 255 THEN
    Rdr.Read(fData, fSize)
  ELSE Rdr.Read(fData, SizeOf(TDigest));
end;

procedure TRegCompObj.LevLeafName(Rdr : TReader; VAR I : Int64;
  VAR S : String);
begin
  IF I AND cfKey > 0 THEN
    begin
      fLeaf := False;
      fDefV := False;
    end
  ELSE IF I AND cfDef > 0 THEN
    begin
      fLeaf := True;
      fDefV := True;
    end
  ELSE
    begin
      fLeaf := True;
      fDefV := False;
    end;
  I := I AND $FFF;
  IF fLeaf AND fDefV THEN
    S := '(Default)'
  ELSE S := Rdr.ReadString;
end;

function TRegCompObj.GetFullPath: String;
VAR N : Integer;
begin
  IF fLeaf THEN
    begin
      IF fPath.Count > 0 THEN
        BEGIN
          Result := NoFinalSlash(fPath[0]);
          FOR N := 1 TO fPath.Count-2 DO
            Result := Result + '\' + fPath[N];
          Result := Result + '\\' + fPath[fPath.Count-1];
        END
      ELSE Result := '';
    end
  ELSE Result := Inherited GetFullPath;
end;

{ TCompObj }

constructor TCompObj.Create;
begin
  Inherited Create;
  fLevl := 0;
  fPath := TStringList.Create;
  fDone := False;
end;

destructor TCompObj.Destroy;
begin
  fPath.Free;
  inherited;
end;

function TCompObj.CompareWith(Othr: TCompObj): Integer;
// Return -1 if Self is less, 1 if Othr is less, 0 if equal
VAR N : Integer;
begin
  // If both streams are at the end, equal
  IF (Self.fLevl = cEndFlag) AND (Othr.fLevl = cEndFlag) THEN
    Result := 0
  // If self is at end, other is "less" (i.e. needs processing)
  ELSE IF Self.fLevl = cEndFlag THEN
    Result := 1
  // If other is at end, self is "less" (i.e. needs processing)
  ELSE IF Othr.fLevl = cEndFlag THEN
    Result := -1
  // If self's level is lower, other needs processing
  ELSE IF Self.fLevl < Othr.fLevl THEN
    Result := 1
  // If other's level is lower, self needs processing
  ELSE IF Self.fLevl > Othr.fLevl THEN
    Result := -1
  // Leaf before Node
  ELSE IF Self.fLeaf AND (NOT Othr.fLeaf) THEN
    Result := -1
  // Node after leaf
  ELSE IF Othr.fLeaf AND (NOT Self.fLeaf) THEN
    Result := 1
  ELSE // If otherwise equal, compare the two paths
    BEGIN
      N := 0;
      Result := 0;
      WHILE (N < Self.fPath.Count) AND (Result = 0) DO
        BEGIN
         // Use AnsiCompareText to be same as sorted list
          Result := AnsiCompareText(Self.fPath[N], Othr.fPath[N]);
          Inc(N);
        END;
      IF Result <> 0 THEN
        Result := Result DIV Abs(Result);
    END;
end;

procedure TCompObj.Read(Rdr : TReader);
VAR
  I : Int64;
  S : String;
begin
  ZeroData;
  I := Rdr.ReadInt64;
  IF I = cEndFlag THEN // End of data
    begin
      fPath.Clear;
      fLevl := cEndFlag;
      fDone := True;
    end
  ELSE
    begin
      LevLeafName(Rdr, I, S);
      IF I = 0 THEN // One of the startdirs
        begin
          fLevl := I;
          fPath.Clear;
          fPath.Add(S);
        end
      ELSE
        begin
          IF fLeaf THEN
            ReadData(Rdr);
          // Handle a "drop" in level
          IF I <= fLevl THEN fPath.Delete(fPath.Count-1);
          WHILE I < fLevl DO
            BEGIN
              fPath.Delete(fPath.Count-1);
              Dec(fLevl);
            END;
          fPath.Add(S);
          fLevl := I;
        end;
    end;
end;

function TCompObj.GetFullPath: String;
VAR N : Integer;
begin
  IF fPath.Count > 0 THEN
    BEGIN
      Result := NoFinalSlash(fPath[0]);
      FOR N := 1 TO fPath.Count-1 DO
        Result := Result + '\' + fPath[N];
    END
  ELSE Result := '';
end;

function TCompObj.GetpName: String;
begin
  IF fPath.Count = 0 THEN Result := ''
  ELSE Result := fPath[fPath.Count-1];
end;


function TCompObj.GetJustName: String;
begin
  IF fPath.Count > 0 THEN
    Result := fPath[fPath.Count-1]
  ELSE Result := '';
end;

function TCompObj.GetPartPath: String;
VAR N : Integer;
begin
  IF fPath.Count > 1 THEN
    BEGIN
      Result := NoFinalSlash(fPath[0]);
      FOR N := 1 TO fPath.Count-2 DO
        Result := Result + '\' + fPath[N];
    END
  ELSE Result := '';
end;

{ TIniCompObj }

constructor TIniCompObj.Create;
begin
  Inherited Create;
  fValu := '';
end;

function TIniCompObj.GetFullPath: String;
begin
  IF fPath.Count > 0 THEN
    BEGIN
      Result := NoFinalSlash(fPath[0]);       // filename
      IF fPath.Count > 1 THEN
        Result := Result + ' - ' + fPath[1];  // section
      IF fPath.Count > 2 THEN
        Result := Result + ' "' + fPath[2] + '"'; // key
    END
  ELSE Result := '';
end;

procedure TIniCompObj.LevLeafName(Rdr: TReader; var I: Int64;
  var S: String);
begin
  fLeaf := I > 1;
  S := Rdr.ReadString;
end;

function TIniCompObj.Match(Other: TCompObj): Boolean;
VAR Othr : TIniCompObj;
begin
  Othr := Other AS TIniCompObj;
  Result := AnsiCompareStr(Self.fValu, Othr.fValu) = 0;
end;

procedure TIniCompObj.ReadData(Rdr: TReader);
begin
  fValu := Rdr.ReadString;
end;

procedure TIniCompObj.ZeroData;
begin
  fValu := '';
end;

{ TTxtCompObj }

constructor TTxtCompObj.Create;
begin
  Inherited Create;
  fLocn := 0;
end;

function TTxtCompObj.GetFullPath: String;
VAR N : Integer;
begin
  IF fPath.Count > 0 THEN
    BEGIN
      Result := NoFinalSlash(fPath[0]);
      FOR N := 1 TO fPath.Count-1 DO
        Result := Result + '?' + fPath[N];
    END
  ELSE Result := '';
end;

procedure TTxtCompObj.LevLeafName(Rdr: TReader; var I: Int64;
  var S: String);
begin
  fLeaf := I > 0;
  S := Rdr.ReadString;
end;

function TTxtCompObj.Match(Other: TCompObj): Boolean;
VAR Othr : TTxtCompObj;
begin
  Othr := Other AS TTxtCompObj;
  Result := Self.fLocn = Othr.fLocn;
end;

procedure TTxtCompObj.ReadData(Rdr: TReader);
begin
  fLocn := Rdr.ReadInteger;
end;

procedure TTxtCompObj.ZeroData;
begin
  fLocn := 0;
end;

{ TChangeObj }

constructor TChangeObj.Create(V: TCompObj);
begin
  Inherited Create;
  fPath    := TStringList.Create;
  fPath.Assign(V.fPath);
  fBoth    := False;
  fDataA_1 := 0;
  fDataB_1 := 0;
  fDataS_1 := '';
  fDataP_1 := nil;
  fDataA_2 := 0;
  fDataB_2 := 0;
  fDataS_2 := '';
  fDataP_2 := nil;
  IF V IS TRegCompObj THEN
    begin
      fClass := TRegCompObj;
      WITH TRegCompObj(V) DO
        begin
          fDataA_1 := fType;
          fDataB_1 := fSize;
          GetMem(fDataP_1, fSize);
          Move(fData, fDataP_1^, fSize);
        end;
    end
  ELSE IF V IS TDskCompObj THEN
    begin
      fClass := TDskCompObj;
      WITH TDskCompObj(V) DO
        begin
          fDataA_1 := fSize;
          fDataB_1 := fTime;
        end;
    end
  ELSE IF V IS TIniCompObj THEN
    begin
      fClass := TIniCompObj;
      WITH TIniCompObj(V) DO
        fDataS_1 := fValu;
    end
  ELSE IF V IS TTxtCompObj THEN
    begin
      fClass := TTxtCompObj;
      WITH TTxtCompObj(V) DO
        fDataA_1 := fLocn;
    end;
end;

constructor TChangeObj.Create(VO, VN: TCompObj);
begin
  Create(VO);
  fBoth := True;
  IF VN IS TRegCompObj THEN
    begin
      WITH TRegCompObj(VN) DO
        begin
          fDataA_2 := fType;
          fDataB_2 := fSize;
          GetMem(fDataP_2, fSize);
          Move(fData, fDataP_2^, fSize);
        end;
    end
  ELSE IF VN IS TDskCompObj THEN
    begin
      WITH TDskCompObj(VN) DO
        begin
          fDataA_2 := fSize;
          fDataB_2 := fTime;
        end;
    end
  ELSE IF VN IS TIniCompObj THEN
    begin
      fClass := TIniCompObj;
      WITH TIniCompObj(VN) DO
        fDataS_2 := fValu;
    end
  ELSE IF VN IS TTxtCompObj THEN
    begin
      fClass := TTxtCompObj;
      WITH TTxtCompObj(VN) DO
        fDataA_2 := fLocn;
    end;
end;

destructor TChangeObj.Destroy;
begin
  fPath.Free;
  IF fDataP_1 <> nil THEN FreeMem(fDataP_1);
  IF fDataP_2 <> nil THEN FreeMem(fDataP_2);
  inherited;

end;

function TChangeObj.GetDskDate1: String;
begin
  IF fClass <> TDskCompObj THEN
    Raise(Exception.Create('Class type mismatch'));
  Result := FileTimeToStr(fDataB_1);
end;

function TChangeObj.GetDskDate2: String;
begin
  IF fClass <> TDskCompObj THEN
    Raise(Exception.Create('Class type mismatch'));
  IF NOT fBoth THEN
    Raise(Exception.Create('Error - just one dataset'));
  Result := FileTimeToStr(fDataB_2);
end;

function TChangeObj.GetDskSize1: String;
begin
  IF fClass <> TDskCompObj THEN
    Raise(Exception.Create('Class type mismatch'));
  Result := FormatFloat('0,', fDataA_1)+' bytes';
end;

function TChangeObj.GetDskSize2: String;
begin
  IF fClass <> TDskCompObj THEN
    Raise(Exception.Create('Class type mismatch'));
  IF NOT fBoth THEN
    Raise(Exception.Create('Error - just one dataset'));
  Result := FormatFloat('0,', fDataA_2)+' bytes';
end;

function TChangeObj.GetFullPath: String;
VAR N : Integer;
begin
  IF fPath.Count > 0 THEN
    begin
      Result := NoFinalSlash(fPath[0]);
      FOR N := 1 TO fPath.Count-1 DO
        Result := Result + '\' + fPath[N];
    end
  ELSE Result := '';
end;

function TChangeObj.GetIniValu1: String;
begin
  IF fClass <> TIniCompObj THEN
    Raise(Exception.Create('Class type mismatch'));
  Result := fDataS_1;
end;

function TChangeObj.GetIniValu2: String;
begin
  IF fClass <> TIniCompObj THEN
    Raise(Exception.Create('Class type mismatch'));
  IF NOT fBoth THEN
    Raise(Exception.Create('Error - just one dataset'));
  Result := fDataS_2;
end;

function TChangeObj.GetJustName: String;
begin
  IF fPath.Count > 0 THEN
    Result := fPath[fPath.Count-1]
  ELSE Result := '';
end;

function TChangeObj.GetLocn1: Integer;
begin
  IF fClass <> TTxtCompObj THEN
    Raise(Exception.Create('Class type mismatch'));
  Result := fDataA_1;
end;

function TChangeObj.GetPartPath: String;
VAR N : Integer;
begin
  IF fPath.Count > 1 THEN
    begin
      Result := NoFinalSlash(fPath[0]);
      FOR N := 1 TO fPath.Count-2 DO
        Result := Result + '\' + fPath[N];
    end
  ELSE Result := '';
end;

function TChangeObj.GetRegData1: String;
begin
  IF fClass <> TRegCompObj THEN
    Raise(Exception.Create('Class type mismatch'));
  Result := StringForRegData(fDataA_1, fDataB_1, fDataP_1);
end;

function TChangeObj.GetRegData2: String;
begin
  IF fClass <> TRegCompObj THEN
    Raise(Exception.Create('Class type mismatch'));
  IF NOT fBoth THEN
    Raise(Exception.Create('Error - just one dataset'));
  Result := StringForRegData(fDataA_2, fDataB_2, fDataP_2);
end;

function TChangeObj.GetRegType1: String;
begin
  IF fClass <> TRegCompObj THEN
    Raise(Exception.Create('Class type mismatch'));
  Result := NameForRegType(fDataA_1);
end;

function TChangeObj.GetRegType2: String;
begin
  IF fClass <> TRegCompObj THEN
    Raise(Exception.Create('Class type mismatch'));
  IF NOT fBoth THEN
    Raise(Exception.Create('Error - just one dataset'));
  Result := NameForRegType(fDataA_2);
end;

function TChangeObj.GetTxtLocn1: String;
begin
  IF fClass <> TTxtCompObj THEN
    Raise(Exception.Create('Class type mismatch'));
  Result := IntToStr(fDataA_1);
end;

function TChangeObj.GetTxtLocn2: String;
begin
  IF fClass <> TTxtCompObj THEN
    Raise(Exception.Create('Class type mismatch'));
  IF NOT fBoth THEN
    Raise(Exception.Create('Error - just one dataset'));
  Result := IntToStr(fDataA_2);
end;

function TChangeObj.StringForRegData(vType, fSize: Integer;
  fData: PByteArray): String;
VAR
  PLong : PDWord;
  PBuff : PChar;
  N     : Integer;
  fType : Integer;

  function AllText : Boolean;
  VAR N : Integer;
  begin
    Result := False;
    FOR N := 0 TO fSize-2 DO
      CASE fData^[N] OF
        9, 10, 13 : ; // OK
        32..255   : ; // OK
        ELSE Exit;
      end;
    Result := True;
  end;

  function StringOfBytes : String;
  VAR N : Integer;
  begin
    Result := '';
    IF fSize > 0 THEN
      Result := IntToHex(fData^[0], 2);
    FOR N := 1 TO fSize-1 DO
      Result := Result + ', ' + IntToHex(fData^[N], 2);
  end;

begin
  IF fSize > 255 THEN
    begin
      Result := Format('(data too large: %d bytes)', [fSize]);
      Exit;
    end;
  PLong := PDword(fData);
  PBuff := PChar(fData);
  fType := vType;
  CASE fType OF
    REG_DWORD               : Result := Format('0x%.08x (%0:d)',
                              [PLong^]);
    REG_DWORD_BIG_ENDIAN    : Result := Format('0x%.08x (%0:d)',
      [MakeLong(Swap(HiWord(PLong^)), Swap(LoWord(PLong^)))]);
    REG_EXPAND_SZ, REG_SZ   : ; // nothing yet
    ELSE IF AllText THEN  // If it's binary but contains only text,
      fType := Reg_SZ;    //    display it as text
  end;
  CASE fType OF
    REG_EXPAND_SZ,
    REG_SZ                  : begin
      IF fSize = 0 THEN Result := ''
      ELSE
        begin
          // Help for GetQueryValueEx *says* that dSize includes
          // the terminating NUL character; experience says it
          // sometimes does, sometimes does not
          IF PBuff[fSize-1] = #0 THEN
            SetString(Result, PBuff, fSize-1)
          ELSE SetString(Result, PBuff, fSize);
          // Get rid of any control characters
          FOR N := 1 TO Length(Result) DO
            IF Result[N] < ' ' THEN Result[N] := ' ';
        end;
    end;
    ELSE begin
      Result := StringOfBytes;
    end;
  END;
end;

end.
