unit in5share;

// Copyright  2000 by Ziff Davis Media, Inc.
// Written by Neil J. Rubenking

interface
USES Windows, SysUtils, Classes;
type
  UpStatFunc = procedure(const S : String; I :integer) of object;
const
  cFile = $1000;
  cfKey = $1000;
  cfDef = $2000;
  cEndFlag = MaxInt;
  cHidSys = FILE_ATTRIBUTE_SYSTEM OR FILE_ATTRIBUTE_HIDDEN;
  cCodeLen = 6;
  // Constants for AddToHierList
  ADL_OK      = 0;
  ADL_BELOW   = 1;
  ADL_ABOVE   = 2;
  ADL_DUPE    = 3;
  // Constants for ExeType
  ET_UNKNOWN  = -3;
  ET_NOTEXE   = -2;
  ET_NOEXIST  = -1;
  ET_BLANK    = 0;
  ET_DOSEXE   = 1;
  ET_COM      = 2;
  ET_BAT      = 3;
  ET_WINEXE   = 4;
  ET_PEXE     = 5;
  ET_INFFILE  = 6;
  // Constants for windows platform/version
  WV_95       = 1;
  WV_98       = 2;
  WV_ME       = 3;
  WV_NT4      = 4;
  WV_NT5      = 5;
  WV_NT3      = -1;
  WV_NTx      = -2;
  WV_9x       = -3;
  WV_UNK      = -4;

type
  codeType = ARRAY[0..cCodeLen-1] OF Char;
const
  codeReg : codeType = 'IN5REG';
  codeDsk : codeType = 'IN5DSK';
  codeINI : codeType = 'IN5INI';
  codeTxt : codeType = 'IN5TXT';
  sStatOK     = 0;
  sStatError  = 1;
  sStatIncomp = 2;
VAR
  IniName     : String;
  windir      : String;
  sysdir      : String;
  BootDrv     : String;
  Platform    : Integer;
  PlatVer     : Integer;
  PlatVerStr  : String;
  CurVerKey   : String;

function ProcMsgTerminated : Boolean;
function GetStreamStatus(Stream : TStream; code : CodeType;
  VAR Count : Integer) : Integer;
function FileTimeToStr(FT : Int64) : String;
function NameForRegType(DType : DWORD) : String;
FUNCTION ExeType(const TheName : String) : Integer;
FUNCTION ExeTypeName(typ : Integer) : String;
function AddToHierList(TS : TStrings; const S : String) : Integer;
function ValidReg(VAR S : String) : Integer;
function ValidINI(const S : String) : Boolean;
function ValidTXT(const S : String) : Integer;
procedure DefaultReg(TS : TStrings);
procedure DefaultDsk(TS : TStrings);
procedure DefaultIni(TS : TStrings);
procedure DefaultTxt(TS : TStrings);


implementation
uses Forms, shellapi, inifiles, allfuncs, ComObj, registry;

function ProcMsgTerminated : Boolean;
// call processMessages and return whether or not
// the app was terminated
BEGIN
  Application.ProcessMessages;
  Result := Application.Terminated OR (Application.Tag=-1);
  IF Result THEN Application.Tag := -1;
END;

function GetStreamStatus(Stream : TStream; code : CodeType;
  VAR Count : Integer) : Integer;
// Each stream begins with an identifying code followed by the
// count of items. The count is set LAST; until the stream is
// finished, the count is -1. This function returns sStatError
// if the code doesn't match, sStatIncomp if Count is still -1,
// or sStatOK if it's all OK
VAR getCode : CodeType;
begin
  FillChar(getCode, SizeOf(codeType), 0);
  Stream.Read(getCode, cCodeLen);
  IF StrLComp(getCode, code, cCodeLen) <> 0 THEN
    Result := sStatError
  ELSE
    begin
      Stream.Read(Count, SizeOf(Integer));
      IF Count = -1 THEN
        Result := sStatIncomp
      ELSE Result := sStatOK;
    end;
end;

function FileTimeToStr(FT : Int64) : String;
VAR
  FLoc     : TFileTime;
  STmp     : TSystemTime;
  DT       : TDateTime;
begin
  Result := '*ERR*';
  IF FT = 0 THEN Exit;
  IF NOT FileTimeToLocalFileTime(TFileTime(FT), FLoc) THEN Exit;
  IF NOT FileTimeToSystemTime(FLoc, STmp) THEN Exit;
  try
    DT := SystemTimeToDateTime(STmp);
  except
    ON EConvertError DO Exit;
  end;
  Result := FormatDateTime('m/d/yyyy h:nn AM/PM', DT);
end;

function NameForRegType(DType : DWORD) : String;
begin
  CASE DType OF
    REG_BINARY              : Result := 'REG_BINARY';
    REG_DWORD               : Result := 'REG_DWORD';
    REG_DWORD_BIG_ENDIAN    : Result := 'REG_DWORD_BIG_ENDIAN';
    REG_EXPAND_SZ           : Result := 'REG_EXPAND_SZ';
    REG_LINK                : Result := 'REG_LINK';
    REG_MULTI_SZ            : Result := 'REG_MULTI_SZ';
    REG_NONE                : Result := 'REG_NONE';
    REG_RESOURCE_LIST       : Result := 'REG_RESOURCE_LIST';
    REG_SZ                  : Result := 'REG_SZ';
    ELSE Result := Format('Hex(%x)', [DType]);
  END;
end;

FUNCTION ExeType(const TheName : String) : Integer;
VAR
  TSFI : TShFileInfo;
  I    : DWORD;
  Ext  : String;
BEGIN
  Result := ET_BLANK;
  IF TheName = '' THEN Exit;
  Result := ET_UNKNOWN;
  IF Pos('.', theName) = 0 THEN Exit;
  IF Length(theName) - Pos('.', theName) < 3 THEN Exit;
  Result := ET_NOEXIST;
  IF NOT FileExists(theName) THEN Exit;
  Result := ET_NOTEXE;
  IF PlatForm = VER_PLATFORM_WIN32_WINDOWS THEN
    begin
      I := SHGetFileInfo(PChar(theName), 0, TSFI, SizeOf(TSFI),
        SHGFI_EXETYPE);
      CASE LoWord(I) OF
        $4550 : Result := ET_PEXE;   //PE signature
        $454E : Result := ET_WINEXE; //NE signature
        $5A4D : Result := ET_DOSEXE; //MZ signature
      END;
    end
  ELSE
    begin
      IF GetBinaryType(PChar(theName), I) THEN
        BEGIN
          CASE I OF
            SCS_32BIT_BINARY : Result := ET_PEXE;
            SCS_WOW_BINARY   : Result := ET_WINEXE;
            SCS_DOS_BINARY   : Result := ET_DOSEXE;
          END;
        END;
    END;
  IF Result <> ET_NOTEXE THEN Exit;
  Ext := Uppercase(ExtractFileExt(theName));
  IF Ext = '.EXE' THEN      Result := ET_DOSEXE
  ELSE IF Ext = '.COM' THEN Result := ET_COM
  ELSE IF Ext = '.BAT' THEN Result := ET_BAT
  ELSE IF Ext = '.INF' THEN Result := ET_INFFILE
  ELSE Result := ET_NOTEXE;
END;

FUNCTION ExeTypeName(typ : Integer) : String;
BEGIN
  CASE typ OF
    ET_BLANK    : Result := '(two-phase mode)';
    ET_DOSEXE   : Result := 'DOS EXE file';
    ET_WINEXE   : Result := '16-bit Windows Executable';
    ET_PEXE     : Result := '32-bit Windows Executable';
    ET_COM      : Result := 'DOS COM file';
    ET_BAT      : Result := 'DOS batch file';
    ET_NOTEXE   : Result := 'Not an executable file';
    ET_NOEXIST  : Result := 'File does not exist';
    ET_INFFILE  : Result := 'Setup .INF file';
    ELSE          Result := '';
  END;
END;

function AddToHierList(TS : TStrings; const S : String) : Integer;
// TS is a sorted list of lowercase hierarchical paths (disk or reg).
// If S is a sub-item ofan item in the list, don't add it, and
// return ADL_BELOW. If an item in the list is a sub-item of S,
// replace it with S and return ADL_ABOVE. If S is present in the
// list, don't add it, and return ADL_DUPE. Otherwise, add S and
// return ADL_OK.
var idx : Integer;
begin
  Result := ADL_OK;
  idx    := TS.Add(lowercase(S));
  IF (idx > 0) THEN
    begin
      // If the item preceding S contains S, then
      //   don't add S
      IF Pos(FinalSlash(TS[idx-1]), lowercase(S)) = 1 THEN
        begin
          TS.Delete(idx);
          Result := ADL_BELOW;
        end;
    end;
  IF (Result = ADL_OK) AND (idx < TS.Count-1) THEN
    begin
      // If the item following S equals S, don't add S;
      //   it's a duplicate.
      IF FinalSlash(TS[idx+1]) = FinalSlash(lowercase(S)) THEN
        begin
          TS.Delete(idx+1);
          Result := ADL_DUPE
        end
      // If S is a prefix of the item following S, remove
      //   the item following
      ELSE
        WHILE (idx < TS.Count-1) AND
              (Pos(FinalSlash(lowercase(S)), TS[idx+1]) = 1) DO
          begin
            TS.Delete(idx+1);
            Result := ADL_ABOVE;
          end;
    end;
end;

function ValidTXT(const S : String) : Integer;
// Analyze a file to see if it's a valid TXT file (from InCtrl5's
// point of view). If not, return a code indicating WHY.
VAR
  TS     : TStringList;
  N      : Integer;
  Siz    : Int64;
  Ctrls  : Integer;
  P      : PByteArray;
  F      : File;
begin
  Result := 1;
  IF NOT FileExists(S) THEN Exit; // Non-existent file
  Result := 2;
  Siz := SizeOfFile(S);
  IF Siz > 65535 THEN Exit;       // over 64KB
  Result := 3;
  GetMem(P, Siz);
  try
    AssignFile(F, S);
    Filemode := 0;
    Reset(F, 1);
    BlockRead(F, P^, Siz);
    CloseFile(F);
    Filemode := 2;
    Ctrls := 0;
    FOR N := 0 TO Siz-1 DO
      CASE P^[N] OF
        9, 10, 13 : ; // OK
        32..255   : ; // OK
        ELSE begin
          Inc(Ctrls);
          IF Ctrls > 2 THEN       // We'll allow 2; more is
            Exit;                 // too many control characters
        end;
      END;
  finally
    FreeMem(P);
  end;
  Result := 4;
  TS := TStringList.Create;
  try
    TS.LoadFromFile(S);
    IF TS.Count > 255 THEN Exit;  // more than 255 lines
    Result := 5;
    FOR N := 0 TO TS.Count-1 DO
      IF Length(TS[N]) > 255 THEN // lines over 255 chars
        Exit;
    Result := 0; // WHEW! It's OK!
  finally
    TS.Free;
  end;
end;

function ValidINI(const S : String) : Boolean;
// Returns TRUE if the file exists and contains at
//   least one section heading
VAR temps : TStringList;
begin
  WITH TIniFile.Create(S) DO
  try
    temps := TStringList.Create;
    try
      ReadSections(temps);
      Result := temps.Count > 0;
    finally
      temps.Free;
    end;
  finally
    Free;
  end;
end;

function ValidReg(VAR S : String) : Integer;
// Returns -1 if the key does not exist; otherwise returns
// length of string beyond rootkey
VAR
  hRoot, HK : hKey;
  Key : String;

  function NameFor(HK : HKey) : String;
  begin
    IF HK = HKEY_USERS
      THEN Result := 'HKEY_USERS'
    ELSE IF HK = HKEY_CURRENT_USER
      THEN Result := 'HKEY_CURRENT_USER'
    ELSE IF HK = HKEY_CURRENT_CONFIG
      THEN Result := 'HKEY_CURRENT_CONFIG'
    ELSE IF HK = HKEY_CLASSES_ROOT
      THEN Result := 'HKEY_CLASSES_ROOT'
    ELSE IF HK = HKEY_LOCAL_MACHINE
      THEN Result := 'HKEY_LOCAL_MACHINE'
    ELSE Result := '';
  end;

  function MatchRoot(const RS : String; HK : HKey) : Boolean;
  VAR P : Integer;
  begin
    P := Pos(RS, Uppercase(S));
    IF P = 1 THEN
      begin
        hRoot := HK;
        Key   := S;
        Delete(Key, 1, Length(RS)+1);
        IF Key = '' THEN
          S := NameFor(hRoot)
        ELSE
          S     := NameFor(hRoot) + '\' + Key;
        Result := True;
      end
    ELSE Result := False;
  end;
begin
  Result := -1;
  hRoot  := 0;
  IF NOT MatchRoot('HKEY_USERS', HKEY_USERS) THEN
  IF NOT MatchRoot('HKEY_CURRENT_USER', HKEY_CURRENT_USER) THEN
  IF NOT MatchRoot('HKEY_CURRENT_CONFIG', HKEY_CURRENT_CONFIG) THEN
  IF NOT MatchRoot('HKEY_CLASSES_ROOT', HKEY_CLASSES_ROOT) THEN
  IF NOT MatchRoot('HKEY_LOCAL_MACHINE', HKEY_LOCAL_MACHINE) THEN
  IF NOT MatchRoot('HKU', HKEY_USERS) THEN
  IF NOT MatchRoot('HKCU', HKEY_CURRENT_USER) THEN
  IF NOT MatchRoot('HKCC', HKEY_CURRENT_CONFIG) THEN
  IF NOT MatchRoot('HKCR', HKEY_CLASSES_ROOT) THEN
  IF NOT MatchRoot('HKLM', HKEY_LOCAL_MACHINE) THEN
  Exit;
  IF RegOpenKey(hRoot, PChar(Key), HK) = ERROR_SUCCESS THEN
    begin
      Result := Length(Key);
      RegCloseKey(HK)
    end
  ELSE Result := -1;
end;

procedure DefaultReg(TS : TStrings);
begin
  TS.Clear;
end;

procedure DefaultDsk(TS : TStrings);
begin
  TS.Text := lowercase(LocalDrives(':\', #13#10, False));
end;

procedure DefaultIni(TS : TStrings);
VAR S : String;
begin
  TS.Clear;
  S := windir+'win.ini';
  IF FileExists(S) THEN TS.Add(S);
  S := windir+'system.ini';
  IF FileExists(S) THEN TS.Add(S);
  S := windir+'control.ini';
  IF FileExists(S) THEN TS.Add(S);
  IF IsWinNT THEN
    begin
      S := BootDrv + 'boot.ini';
      IF FileExists(S) THEN TS.Add(S);
    end
  ELSE
    begin
      S := BootDrv + 'msdos.sys';
      IF FileExists(S) THEN TS.Add(S);
    end;
end;

procedure DefaultTxt(TS : TStrings);
VAR S : String;
begin
  TS.Clear;
  IF IsWinNT THEN
    begin
      S := SysDir + 'autoexec.nt';
      IF FileExists(S) THEN TS.Add(S);
      S := SysDir + 'config.nt';
      IF FileExists(S) THEN TS.Add(S);
    end
  ELSE
    begin
      S := BootDrv + 'autoexec.bat';
      IF FileExists(S) THEN TS.Add(S);
      S := BootDrv + 'config.sys';
      IF FileExists(S) THEN TS.Add(S);
      S := WinDir + 'winstart.bat';
      IF FileExists(S) THEN TS.Add(S);
    end;
end;

initialization
  IniName := ChangeFileExt(Application.Exename, '.INI');
  SetLength(windir, MAX_PATH);
  SetLength(windir, GetWindowsDirectory(PChar(windir), MAX_PATH));
  windir := lowercase(FinalSlash(windir));
  SetLength(sysdir, MAX_PATH);
  SetLength(sysdir, GetSystemDirectory(PChar(sysdir), MAX_PATH));
  sysdir := lowercase(FinalSlash(sysdir));
  Platform := Win32Platform;
  CASE Win32Platform OF
    VER_PLATFORM_WIN32_WINDOWS : begin
      CurVerKey := '\SOFTWARE\Microsoft\Windows\CurrentVersion';
      IF Win32MajorVersion = 4 THEN
        begin
          CASE Win32MinorVersion OF
            0    : PlatVer := WV_95;
            10   : PlatVer := WV_98;
            90   : PlatVer := WV_ME;
            ELSE   PlatVer := WV_9x;
          END;
        end
      ELSE PlatVer := WV_9x;
    end;
    VER_PLATFORM_WIN32_NT : begin
      CurVerKey := '\SOFTWARE\Microsoft\Windows NT\CurrentVersion';
      CASE Win32MajorVersion OF
        3    : PlatVer := WV_NT3;
        4    : PlatVer := WV_NT4;
        5    : PlatVer := WV_NT5;
        ELSE   PlatVer := WV_NTx;
      END;
    end;
    ELSE PlatVer := WV_UNK;
  END;
  CASE PlatVer OF
    WV_95  : PlatVerStr := 'Windows 95';
    WV_98  : PlatVerStr := 'Windows 98';
    WV_ME  : PlatVerStr := 'Windows ME';
    WV_NT4 : PlatVerStr := 'Windows NT4';
    WV_NT5 : PlatVerStr := 'Windows 2000';
    WV_NT3 : PlatVerStr := 'Windows NT3';
    WV_NTx : PlatVerStr := 'Windows NT?';
    WV_9x  : PlatVerStr := 'Windows 9?';
    WV_UNK : PlatVerStr := 'Windows ???';
  end;
end.
