{$Q-,T-} {needed in 16-bit Delphi; not in 32?}
{$WARNINGS OFF}
unit md5u;
(* Copyright (C) 1991-2, RSA Data Security, Inc. Created 1991. All
rights reserved.

License to copy and use this software is granted provided that it
is identified as the "RSA Data Security, Inc. MD5 Message-Digest
Algorithm" in all material mentioning or referencing this software
or this function.

License is also granted to make and use derivative works provided
that such works are identified as "derived from the RSA Data
Security, Inc. MD5 Message-Digest Algorithm" in all material
mentioning or referencing the derived work.

RSA Data Security, Inc. makes no representations concerning either
the merchantability of this software or the suitability of this
software for any particular purpose. It is provided "as is"
without express or implied warranty of any kind.

These notices must be retained in any copies of any part of this
documentation and/or software.
 *)

(* Translated into Delphi / Pascal by Neil J. Rubenking *)

interface
uses SysUtils;

TYPE
  PLongArray = ^TLongArray;
  TLongArray = ARRAY[0..(65520 DIV SizeOf(LongInt))] OF LongInt;
  TMD5State = ARRAY[0..3] OF LongInt;
  PMD5Buffer = ^TMD5Buffer;
  TMD5Buffer = ARRAY[0..63] OF Char;
  MD5_CTX = record
    state  : TMD5State;
    count  : ARRAY[0..1] OF LongInt;
    buffer : TMD5Buffer;
  end;
  TDigest = array[0..15] OF Char;

procedure MD5Digest(
  S          : PByteArray;  {String of bytes to be "digested"}
  SLen       : SmallInt;    {Length of S}
  VAR digest : TDigest); {Result}

(**********************)
(**) implementation (**)
(**********************)

procedure MD5Init(VAR context: MD5_CTX);
begin
  FillChar(context, SizeOf(context), 0);
  WITH context DO
    begin
      state[0] := $67452301;
      state[1] := $efcdab89;
      state[2] := $98badcfe;
      state[3] := $10325476;
    end;
end;

procedure MD5Update(VAR context: MD5_CTX; input : PByteArray;
  inputLen : LongInt);
{--- MD5 block update operation. Continues an MD5 message-digest
  operation, processing another message block, and updating
  the context. ---}
VAR
  i, index, partLen : Word;
  inputBuffer : TMD5Buffer;

  procedure MD5Transform(VAR state: TMD5State; block: TMD5Buffer);
  {--- MD5 basic transformation. Transforms state based on block
    (which is 64 chars long). ---}
  const
    S11 = 7;  S12 = 12; S13 = 17; S14 = 22;
    S21 = 5;  S22 = 9;  S23 = 14; S24 = 20;
    S31 = 4;  S32 = 11; S33 = 16; S34 = 23;
    S41 = 6;  S42 = 10; S43 = 15; S44 = 21;
  VAR
    A, B, C, D : LongInt;
    X : ARRAY[0..15] OF LongInt;

    procedure Decode(output : PLongArray; input: TMD5Buffer;
      Len : SmallInt);
    {--- Decodes input (byte array) into output (LongInt array).
      Assumes Len (length of byte array)is a multiple of 4
      and <= 64. ---}
    VAR i, j : Word;
    begin
      i := 0; j := 0;
      WHILE j < len DO
        begin
          output^[i] := LongInt(ord(input[j])) OR
                       (LongInt(ord(input[j+1])) SHL 8) OR
                       (LongInt(ord(input[j+2])) SHL 16) OR
                       (LongInt(ord(input[j+3])) SHL 24);
          Inc(i);
          Inc(J, 4);
        end;
    end;

    {--- F, G, H and I are basic MD5 functions. ---}
    function  F(x, y, z: LongInt):LongInt;
    begin Result := (x AND y) OR ((NOT x) AND z); end;

    function  G(x, y, z: LongInt):LongInt;
    begin Result := (x AND z) OR (y AND (NOT z)) ; end;

    function  H(x, y, z: LongInt):LongInt;
    begin Result := x XOR y XOR z ; end;

    function  I(x, y, z: LongInt):LongInt;
    begin Result := y XOR (x OR (NOT z)) ; end;

    function rotate_left(X, N : LongInt) : LongInt;
    begin Result := (X SHL N) + (X SHR (32-N)); end;

    {--- FF, GG, HH, and II transformations for rounds 1, 2, 3,
      and 4. Rotation is separate from addition to prevent
      recomputation. ---}
    procedure FF(VAR a: LongInt; b, c, d, x, s, ac: LongInt);
    begin
      Inc(a, F(b,c,d) + x + ac);
      a := rotate_left(a, s);
      Inc(a, b);
    end;
    procedure GG(VAR a: LongInt; b, c, d, x, s, ac: LongInt);
    begin
      Inc(a, G(b,c,d) + x + ac);
      a := rotate_left(a, s);
      Inc(a, b);
    end;
    procedure HH(VAR a: LongInt; b, c, d, x, s, ac: LongInt);
    begin
      Inc(a, H(b,c,d) + x + ac);
      a := rotate_left(a, s);
      Inc(a, b);
    end;
    procedure II(VAR a: LongInt; b, c, d, x, s, ac: LongInt);
    begin
      Inc(a, I(b,c,d) + x + ac);
      a := rotate_left(a, s);
      Inc(a, b);
    end;

  begin {MD5Transform}
    A := state[0];
    B := state[1];
    C := state[2];
    D := state[3];
    Decode (@x, block, 64);
    (* Round 1 *)
    FF (a, b, c, d, x[ 0], S11, $d76aa478); (* 1 *)
    FF (d, a, b, c, x[ 1], S12, $e8c7b756); (* 2 *)
    FF (c, d, a, b, x[ 2], S13, $242070db); (* 3 *)
    FF (b, c, d, a, x[ 3], S14, $c1bdceee); (* 4 *)
    FF (a, b, c, d, x[ 4], S11, $f57c0faf); (* 5 *)
    FF (d, a, b, c, x[ 5], S12, $4787c62a); (* 6 *)
    FF (c, d, a, b, x[ 6], S13, $a8304613); (* 7 *)
    FF (b, c, d, a, x[ 7], S14, $fd469501); (* 8 *)
    FF (a, b, c, d, x[ 8], S11, $698098d8); (* 9 *)
    FF (d, a, b, c, x[ 9], S12, $8b44f7af); (* 10 *)
    FF (c, d, a, b, x[10], S13, $ffff5bb1); (* 11 *)
    FF (b, c, d, a, x[11], S14, $895cd7be); (* 12 *)
    FF (a, b, c, d, x[12], S11, $6b901122); (* 13 *)
    FF (d, a, b, c, x[13], S12, $fd987193); (* 14 *)
    FF (c, d, a, b, x[14], S13, $a679438e); (* 15 *)
    FF (b, c, d, a, x[15], S14, $49b40821); (* 16 *)
   (* Round 2 *)
    GG (a, b, c, d, x[ 1], S21, $f61e2562); (* 17 *)
    GG (d, a, b, c, x[ 6], S22, $c040b340); (* 18 *)
    GG (c, d, a, b, x[11], S23, $265e5a51); (* 19 *)
    GG (b, c, d, a, x[ 0], S24, $e9b6c7aa); (* 20 *)
    GG (a, b, c, d, x[ 5], S21, $d62f105d); (* 21 *)
    GG (d, a, b, c, x[10], S22,  $2441453); (* 22 *)
    GG (c, d, a, b, x[15], S23, $d8a1e681); (* 23 *)
    GG (b, c, d, a, x[ 4], S24, $e7d3fbc8); (* 24 *)
    GG (a, b, c, d, x[ 9], S21, $21e1cde6); (* 25 *)
    GG (d, a, b, c, x[14], S22, $c33707d6); (* 26 *)
    GG (c, d, a, b, x[ 3], S23, $f4d50d87); (* 27 *)
    GG (b, c, d, a, x[ 8], S24, $455a14ed); (* 28 *)
    GG (a, b, c, d, x[13], S21, $a9e3e905); (* 29 *)
    GG (d, a, b, c, x[ 2], S22, $fcefa3f8); (* 30 *)
    GG (c, d, a, b, x[ 7], S23, $676f02d9); (* 31 *)
    GG (b, c, d, a, x[12], S24, $8d2a4c8a); (* 32 *)
    (* Round 3 *)
    HH (a, b, c, d, x[ 5], S31, $fffa3942); (* 33 *)
    HH (d, a, b, c, x[ 8], S32, $8771f681); (* 34 *)
    HH (c, d, a, b, x[11], S33, $6d9d6122); (* 35 *)
    HH (b, c, d, a, x[14], S34, $fde5380c); (* 36 *)
    HH (a, b, c, d, x[ 1], S31, $a4beea44); (* 37 *)
    HH (d, a, b, c, x[ 4], S32, $4bdecfa9); (* 38 *)
    HH (c, d, a, b, x[ 7], S33, $f6bb4b60); (* 39 *)
    HH (b, c, d, a, x[10], S34, $bebfbc70); (* 40 *)
    HH (a, b, c, d, x[13], S31, $289b7ec6); (* 41 *)
    HH (d, a, b, c, x[ 0], S32, $eaa127fa); (* 42 *)
    HH (c, d, a, b, x[ 3], S33, $d4ef3085); (* 43 *)
    HH (b, c, d, a, x[ 6], S34,  $4881d05); (* 44 *)
    HH (a, b, c, d, x[ 9], S31, $d9d4d039); (* 45 *)
    HH (d, a, b, c, x[12], S32, $e6db99e5); (* 46 *)
    HH (c, d, a, b, x[15], S33, $1fa27cf8); (* 47 *)
    HH (b, c, d, a, x[ 2], S34, $c4ac5665); (* 48 *)
    (* Round 4 *)
    II (a, b, c, d, x[ 0], S41, $f4292244); (* 49 *)
    II (d, a, b, c, x[ 7], S42, $432aff97); (* 50 *)
    II (c, d, a, b, x[14], S43, $ab9423a7); (* 51 *)
    II (b, c, d, a, x[ 5], S44, $fc93a039); (* 52 *)
    II (a, b, c, d, x[12], S41, $655b59c3); (* 53 *)
    II (d, a, b, c, x[ 3], S42, $8f0ccc92); (* 54 *)
    II (c, d, a, b, x[10], S43, $ffeff47d); (* 55 *)
    II (b, c, d, a, x[ 1], S44, $85845dd1); (* 56 *)
    II (a, b, c, d, x[ 8], S41, $6fa87e4f); (* 57 *)
    II (d, a, b, c, x[15], S42, $fe2ce6e0); (* 58 *)
    II (c, d, a, b, x[ 6], S43, $a3014314); (* 59 *)
    II (b, c, d, a, x[13], S44, $4e0811a1); (* 60 *)
    II (a, b, c, d, x[ 4], S41, $f7537e82); (* 61 *)
    II (d, a, b, c, x[11], S42, $bd3af235); (* 62 *)
    II (c, d, a, b, x[ 2], S43, $2ad7d2bb); (* 63 *)
    II (b, c, d, a, x[ 9], S44, $eb86d391); (* 64 *)
    Inc(state[0], a);
    Inc(state[1], b);
    Inc(state[2], c);
    Inc(state[3], d);
    (* Zeroize sensitive information.*)
     FillChar(X, SizeOf(X), 0);
  end; {MD5Transform}

begin {MD5Update}
  (*Compute number of bytes mod 64 *)
  index := (context.count[0] SHR 3) AND $3F;
  (* Update number of bits *)
  Inc(context.count[0], inputLen SHL 3);
  IF context.count[0] < inputLen SHL 3 THEN
    Inc(context.count[1]);
  Inc(context.count[1], inputLen SHR 29);
  partLen := 64 - index;
  (* Transform as many times as possible. *)
  if (inputLen >= partLen) THEN
    begin
      MOVE(input^, context.buffer[index], partLen);
      MD5Transform(context.state, context.buffer);
      i := partLen;
      WHILE i+63 < inputLen DO
        begin
          Move(input^[i], InputBuffer, SizeOf(InputBuffer));
          MD5Transform (context.state, InputBuffer);
          Inc(i, 64);
        end;
      index := 0;
    end
  else i := 0;
  MOVE(input^[i], context.buffer[index], inputLen-i);
end; {MD5Update}

procedure MD5Final(VAR digest: TDigest; VAR context: MD5_CTX);
CONST PADDING : ARRAY[0..63] OF Byte = (
  $80,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0);
VAR
  bits : ARRAY[0..7] OF Char;{unsigned char bits[8];}
  index, padLen : Word;

  procedure Encode (output : PMD5Buffer; input : PLongArray;
    len : SmallInt);
  {--- Encodes input (LongInt array) into output (byte array).
    Assumes len (length of byte array) is a multiple of 4
    and <= 64. ---}
  VAR i, j : Word;
  begin
    i := 0; j := 0;
    WHILE j < len DO
      begin
        output^[j]   := char(input^[i] AND $ff);
        output^[j+1] := char((input^[i] SHR 8) AND $ff);
        output^[j+2] := char((input^[i] SHR 16) AND $ff);
        output^[j+3] := char((input^[i] SHR 24) AND $ff);
        Inc(i);
        Inc(J, 4);
      end;
  end;

begin {MD5Final}
  (* Save number of bits *)
  Encode (@bits, @context.count, 8);
  (* Pad out to 56 mod 64.*)
  index := (context.count[0] SHR 3) AND $3f;
  IF Index < 56 THEN padLen := 56-index
  ELSE padLen := 120-index;
  MD5Update(context, @PADDING, padLen);
  (* Append length (before padding) *)
  MD5Update(context, @bits, 8);
  (* Store state in digest *)
  Encode (@digest, @context.state, 16);
  (* Zeroize sensitive information.*)
  FillChar(Context, SizeOf(Context), 0);
end; {MD5Final}

procedure MD5Digest(S : PByteArray; SLen : SmallInt;
  VAR digest : TDigest);
VAR
  context      : MD5_CTX ;
begin
  MD5Init(context);
  MD5Update(context, s, SLen);
  MD5Final(digest, context);
end;

end.
