unit regequiv;

interface
uses Windows, SysUtils, Classes;

function RootKeyEquivalents(TS : TStringList) : Integer;
const
  fail_HKCR = 1;
  fail_HKCU = 2;
  fail_HKCC = 4;

implementation
uses ComObj, Registry;

const
  HKU  = HKEY_USERS;
  HKLM = HKEY_LOCAL_MACHINE;
  HKCU = HKEY_CURRENT_USER;
  HKCR = HKEY_CLASSES_ROOT;
  HKCC = HKEY_CURRENT_CONFIG;

function CompareKeyInfo(test, root : HKey;
  const S : String) : Boolean;
// test is the pseudo-root key to be tested, root is the actual
// root key, and S is the subkey corresponding to test. We use
// RegQueryKeyInfo on each, and compare the results. This is only
// used when the key-creation test is not permitted.
VAR
  Class1,                Class2 : ARRAY[0..MAX_PATH] OF Char;
  cbClass1,              cbClass2,
  cSubKeys1,             cSubKeys2,
  cbMaxSubKeyLen1,       cbMaxSubKeyLen2,
  cbMaxClassLen1,        cbMaxClassLen2,
  cValues1,              cValues2,
  cbMaxValueNameLen1,    cbMaxValueNameLen2,
  cbMaxValueLen1,        cbMaxValueLen2,
  cbSecurityDescriptor1, cbSecurityDescriptor2 : DWORD;
  ftLastWriteTime1,      ftLastWriteTime2 : TFileTime;
  Rslt : LongWord;
begin
  Result   := False;
  cbClass1 := MAX_PATH;
  cbClass2 := MAX_PATH;
  Rslt := RegQueryInfoKey(test, @Class1, @cbClass1, nil, @cSubKeys1,
    @cbMaxSubKeyLen1, @cbMaxClassLen1, @cValues1, @cbMaxValueNameLen1,
    @cbMaxValueLen1, @cbSecurityDescriptor1, @ftLastWriteTime1);
  IF Rslt <> ERROR_SUCCESS THEN Exit;
  WITH TRegistry.Create DO
  try
    RootKey := root;
    Access := KEY_READ;
    IF NOT OpenKey(S, false) THEN Exit;
    Rslt := RegQueryInfoKey(CurrentKey, @Class2, @cbClass2, nil, @cSubKeys2,
      @cbMaxSubKeyLen2, @cbMaxClassLen2, @cValues2, @cbMaxValueNameLen2,
      @cbMaxValueLen2, @cbSecurityDescriptor2, @ftLastWriteTime2);
    IF Rslt <> ERROR_SUCCESS THEN Exit;
  finally
    Free;
  end;
  Result := (StrComp(Class1, Class2) = 0) AND
    (cSubKeys1          = cSubKeys2) AND
    (cbMaxSubKeyLen1    = cbMaxSubKeyLen2) AND
    (cbMaxClassLen1     = cbMaxClassLen2) AND
    (cValues1           = cValues2) AND
    (cbMaxValueNameLen1 = cbMaxValueNameLen2) AND
    (cbMaxValueLen1     = cbMaxValueLen2);
  // On systems without admin privilege, the security descriptor
  // length and date/time are apparently NOT the same for HKCR and
  // HKEY_LOCAL_MACHINE\SOFTWARE\Classes.
  IF Result AND (test <> HKCR) THEN
    Result := (cbSecurityDescriptor1 = cbSecurityDescriptor2) AND
    (ftLastWriteTime1.dwLowDateTime =
      ftLastWriteTime2.dwLowDateTime) AND
    (ftLastWriteTime1.dwHighDateTime =
      ftLastWriteTime2.dwHighDateTime);
end;

function TestKey(test, root : HKey; const S : String) : Integer;
// test is the pseudo-root key to be tested, root is the actual
// root key, and S is the subkey corresponding to test. We test
// by creating a unique key in one and checking for it in
// the other. Return 0 = tested and equal, 1 = tested and unequal,
// 2 = insufficient privilege, 3 = key did not exist.
VAR K, V : String;
begin
  Result    := 3;
  WITH TRegistry.Create DO
  try
    RootKey := root;
    Access := KEY_ALL_ACCESS;
    // Open the subkey of the base that we suspect points to the
    // desired pseudo-root key
    IF NOT OpenKey(S, False) THEN Exit;
    Result := 2;
    // If the user has the power to create keys, make a further test
    K := CreateClassID;
    V := CreateClassID;
    // Create a unique key within it
    IF NOT OpenKey(K, True) THEN Exit;
    try
      // Create a unique value
      WriteString(V, 'test');
      // Now open the pseudo-root key...
      Result := 1;
      RootKey := test;
      // ... and see if it contains our test key...
      IF NOT OpenKey(K, False) THEN Exit;
      // ... and value.
      IF ReadString(V) <> 'test' THEN Exit;
      Result := 0;
    finally
      RootKey := root;
      DeleteKey(S+'\'+K);
    end;
  finally
    Free;
  end;
end;

function TestAndAdd(TS : TStringList; const RootS, S : String;
  test, root : HKey) : Boolean;
begin
  Result := False;
  CASE TestKey(test, root, '\' + S) OF
    2 : begin // no privilege; use alternate test
      IF CompareKeyInfo(test, root, '\' + S) THEN
        begin
          TS.Add(Uppercase(RootS + '\' + S));
          Result := True;
        end
      ELSE Result := False;
    end;
    1 : ; // test failed; leave result False
    0 : begin
      TS.Add(Uppercase(RootS + '\' + S));
      Result := True;
    end;
  END;
end;

function GetHKCUString : String;
// HKEY_CURRENT_USER is a pointer to a particular subkey of
// HKEY_USERS. This function returns that subkey, represented
// as a string.
VAR
  S : String;
  R : TRegistry;

  function CurrentUserSID(VAR theSID : String) : Integer;
  // Calculate the Security ID for the current user, to
  //  determine which branch of HKEY_USERS is HKEY_CURRENT_USER
  VAR
    UserName : ARRAY[0..MAX_PATH] OF Char;
    UserLeng : DWORD;
    DomaName : ARRAY[0..MAX_PATH] OF Char;
    DomaLeng : DWORD;
    SID      : ARRAY[0..1000] OF Byte;
    SIDLeng  : DWORD;
    SIDuse   : DWORD;
    PSIA     : PSIDIdentifierAuthority;
    N        : Integer;
    NumSubs  : PUChar;
    OneSub   : PDWORD;
  begin
    UserLeng := MAX_PATH;
    Result   := 1;
    IF NOT GetUserName(UserName, UserLeng) THEN
      begin
        IF IsConsole THEN
          WriteLn('GetUserName() failed in CurrentUserSID(): '+
            SysErrorMessage(GetLastError));
        Exit;
      end;
    DomaLeng := MAX_PATH;
    Result   := 2;
    SidLeng  := 1000;
    IF NOT LookupAccountName(NIL, UserName, @SID, SIDLeng,
      DomaName, DomaLeng, SidUse) THEN
      begin
        IF IsConsole THEN
          WriteLn('LookupAccountName() failed in CurrentUserSID(): '+
            SysErrorMessage(GetLastError));
        Exit;
      end;
    TheSid := 'S-1-';
    Result := 3;
    SetLastError(0);
    PSIA := GetSidIdentifierAuthority(@SID);
    IF GetLastError <> 0 THEN
      begin
        IF IsConsole THEN
          WriteLn('GetSidIdentifierAuthority() failed in CurrentUserSID(): '+
            SysErrorMessage(GetLastError));
        Exit;
      end;
    WITH PSIA^ do
      IF (Value[0]<>0) OR (Value[1]<>0) THEN
        TheSid := Format('%s0x%.2x%.2x%.2x%.2x%.2x%.2x-',
          [TheSid, Value[0], Value[1], Value[2], Value[3],
          Value[4], Value[5]])
      else
        begin
          N := Value[2] SHL 8;
          N := (N + Value[3]) SHL 8;
          N := (N + Value[4]) SHL 8;
          N := N + Value[5];
          TheSid := Format('%s%d', [TheSid, N]);
        end;
    Result  := 4;
    NumSubs := GetSidSubAuthorityCount(@SID);
    IF GetLastError <> 0 THEN
      begin
        IF IsConsole THEN
          WriteLn('GetSidSubAuthorityCount() failed in CurrentUserSID(): '+
            SysErrorMessage(GetLastError));
        Exit;
      end;
    FOR N := 0 TO NumSubs^-1 DO
      begin
        Result := N + 5;
        OneSub := GetSidSubAuthority(@SID, N);
        IF GetLastError <> 0 THEN
          begin
            IF IsConsole THEN
              WriteLn('GetSidSubAuthority() failed in CurrentUserSID(): '+
                SysErrorMessage(GetLastError));
            Exit;
          end;
        TheSid := Format('%s-%d', [TheSid, OneSub^]);
      end;
    Result := 0;
  end;

begin
  Result := '';
  R      := TRegistry.Create;
  WITH R DO
    try
      IF Win32Platform = VER_PLATFORM_WIN32_NT THEN
        begin
          RootKey := HKEY_USERS;
          OpenKey('', FALSE);
          IF CurrentUserSID(S) = 0 THEN
            begin
              IF NOT KeyExists(S) THEN
                begin
                  IF IsConsole THEN
                    WriteLn('CurrentUserSID() returned non-existent key "',
                      S, '"');
                  S := '';
                end;
            end
          ELSE
            begin
              IF IsConsole THEN
                WriteLn('CurrentUserSID() failed: "', S, '"');
              S := '';
            end;
        end
      else
        begin
          RootKey := HKEY_LOCAL_MACHINE;
          OpenKey('\System\CurrentControlSet\control', False);
          S := ReadString('Current User');
        end;
      IF S = '' THEN S := '.Default';
      Result := S;
    finally
      Free;
    end;
end;

function FindHKCUString(TS : TStringList) : Boolean;
VAR
  R : TRegistry;
  S : String;
  N : Integer;
  TempS : TStringList;
begin
  Result := True;
  IF IsConsole THEN
    begin
      S := GetHKCUString;
      WriteLn(Format('(Trying HKEY_USERS\%s)',[S]));
      IF TestAndAdd(TS, 'HKEY_USERS', S, HKCU, HKU) THEN Exit;
    end
  ELSE IF TestAndAdd(TS, 'HKEY_USERS', GetHKCUString, HKCU, HKU) THEN Exit;
  R := TRegistry.Create;
  WITH R DO
  try
    RootKey := HKEY_USERS;
    OpenKey('', False);
    TempS := TStringList.Create;
    try
      GetKeyNames(TempS);
      FOR N := 0 TO TempS.Count-1 DO
        begin
          IF IsConsole THEN
            WriteLn(Format('(Trying HKEY_USERS\%s)',[TempS[N]]));
          IF TestAndAdd(TS, 'HKEY_USERS', TempS[N], HKCU, HKU) THEN Exit;
        end;
    finally
      TempS.Free;
    end;
  finally
    Free;
  end;
  Result := False;
end;

function AddHKCCStrings(TS : TStringList) : Boolean;
VAR
  R : TRegistry;
  Any : Boolean;
  Num : Integer;
  S   : String;
begin
  Result := False;
  R := TRegistry.Create;
  WITH R DO
    try
      RootKey := HKEY_LOCAL_MACHINE;
      IF NOT OpenKeyReadOnly('\System\CurrentControlSet\control\'+
        'IDConfigDB') THEN
          Exit;
      IF Win32Platform = VER_PLATFORM_WIN32_NT THEN
        begin
          Any := False;
          Num := ReadInteger('CurrentConfig');
          S := Format('SYSTEM\ControlSet%.03d\Hardware Profiles\Current',
            [Num]);
          IF TestAndAdd(TS, 'HKEY_LOCAL_MACHINE', S, HKCC, HKLM) THEN
            Any := True;
          S := Format('SYSTEM\ControlSet%.03d\Hardware Profiles\%0:.04d',
            [Num]);
          IF TestAndAdd(TS, 'HKEY_LOCAL_MACHINE', S, HKCC, HKLM) THEN
            Any := True;
          S := Format('SYSTEM\CurrentControlSet\Hardware Profiles\%.04d',
            [Num]);
          IF TestAndAdd(TS, 'HKEY_LOCAL_MACHINE', S, HKCC, HKLM) THEN
            Any := True;
          S := 'SYSTEM\CurrentControlSet\Hardware Profiles\Current';
          IF TestAndAdd(TS, 'HKEY_LOCAL_MACHINE', S, HKCC, HKLM) THEN
            Any := True;
          Result := Any;
        end
      ELSE
        begin
          S := 'CONFIG\'+ReadString('CurrentConfig');
          Result := TestAndAdd(TS, 'HKEY_LOCAL_MACHINE', S, HKCC, HKLM);
        end;  
    finally
      Free;
    end;
end;

function RootKeyEquivalents(TS : TStringList) : Integer;
begin
  Result := 0;
  IF NOT TestAndAdd(TS, 'HKEY_LOCAL_MACHINE', 'SOFTWARE\Classes',
    HKCR, HKLM) THEN
      Result := Result OR fail_HKCR;
  IF NOT FindHKCUString(TS) THEN
    Result := Result OR fail_HKCU
  ELSE IF IsConsole THEN
    WriteLn('Successfully found HKCU equivalent');
  IF NOT AddHKCCStrings(TS) THEN
    Result := Result OR fail_HKCC;  
end;

end.
