unit in5Strmu;

// Copyright  2000 by Ziff Davis Media, Inc.
// Written by Neil J. Rubenking

interface
USES Windows, SysUtils, Classes, in5Share, in5Obju, ComCtrls;

// The four "remember" functions write snapshot data to a stream
procedure RememberDisk(Stream : TStream; DirsWatched : TStringList;
  UpStatus : UpStatFunc);
procedure RememberInis(Stream : TStream; IniList : TStringList;
  UpStatus : UpStatFunc);
procedure RememberRegistry(Stream : TStream;
  Ignores : TStrings; UpStatus : UpStatFunc);
procedure RememberTxt(Stream : TStream; Files : TStringList;
  UpStatus : UpStatFunc);

// CompareStreams works on any of the four stream types.
procedure CompareStreams(theClass : TCompObjClass;
  theCode : CodeType; TFSO, TFSN : TStream; vAddNode, vDelNode,
  vAddLeaf, vDelLeaf, vChaLeaf : TStrings; UpStatus : UpStatFunc);

function GetStreamCount(const S : String) : Integer;
function BadTemp(const S : String; code : CodeType) : Boolean;

// StreamToTree is a temporary function, used only in testing
procedure StreamToTree(theClass : TCompObjClass; theCode : CodeType;
  Stream : TStream; TV : TTreeView; ShowLeaf : Boolean;
  UpStatus : UpStatFunc);

implementation
uses Registry, md5u, shellapi, shlobj, allfuncs;

type
  TFileRecObj = class(TObject)
    // Object to hold file size and time while files are collected,
    // and give back that info after files are sorted.
    fFSize : Int64;
    fTime  : Int64;
    Constructor Create(vSizeLo, vSizeHi : DWORD; vTime : TFileTime);
  end;

{ TFileRecObj }

constructor TFileRecObj.Create(vSizeLo, vSizeHi: DWORD;
  vTime: TFileTime);
VAR U : TULargeInteger;
begin
  Inherited Create;
  U.LowPart  := vSizeLo;
  U.HighPart := vSizeHi;
  fFSize     := U.QuadPart;
  fTime      := Int64(vTime);
end;

procedure RememberDisk(Stream : TStream; DirsWatched : TStringList;
  UpStatus : UpStatFunc);
VAR
  StrmPos  : Integer;
  Count    : Integer;
  CoMod    : Integer;
  Writer   : TWriter;
  N        : Integer;
  StartDir : String;

  procedure ClearObjects(TS : TStringList);
  VAR N : Integer;
  begin
    FOR N := 0 TO TS.Count-1 DO
      TS.Objects[N].Free;
    TS.Clear;
  end;

  procedure ProcessFiles(const S : String; Level : Integer);
  VAR
    TFils : TStringList;
    TDirs : TStringList;
    FH    : THandle;
    FD    : TWin32FindData;
    N     : Integer;
  begin
    IF ProcMsgTerminated THEN Exit;
    TFils := TStringList.Create;
    TDirs := TStringList.Create;
    try
      FH := FindFirstFile(PChar(S + '\*.*'), FD);
      IF FH = INVALID_HANDLE_VALUE THEN Exit;
      try
        REPEAT
          IF FD.dwFileAttributes AND FILE_ATTRIBUTE_DIRECTORY>0 THEN
            begin
              // Skip the "self" and "parent" entries
              IF StrComp(FD.cFilename, '.')  = 0 THEN Continue;
              IF StrComp(FD.cFilename, '..') = 0 THEN Continue;
              // Skip the Recycle bin
              try
                IF (Length(S) = 2) AND
                   ((StrIComp(FD.cFilename, 'RECYCLED') = 0) OR
                    (StrIComp(FD.cFilename, 'RECYCLER') = 0)) AND
                   (FD.dwfileAttributes AND cHidSys = cHidSys) THEN
                  Continue;
              except
                ON E:Exception DO
                  Continue;
              end;
              TDirs.Add(FD.cFilename);
            end
          ELSE TFils.AddObject(FD.cFilename, TFileRecObj.Create(
            FD.nFileSizeLow, FD.nFileSizeHigh, FD.ftLastWriteTime));
          CoMod := Succ(CoMod) MOD 200;
          IF CoMod = 0 THEN
            UpStatus(S+'\'+FD.cFilename, Count);
        UNTIL (NOT FindNextFile(FH, FD));
      finally
        Windows.FindClose(FH);
      end;
      TDirs.Sort;
      TFils.Sort;
      Inc(Count, TFils.Count);
      Inc(Count, TDirs.Count);
      FOR N := 0 TO TFils.Count-1 DO
        begin
          // Write level, with flag indicating this is a file
          Writer.WriteInteger(cFile OR Level);
          // Write filename
          Writer.WriteString(TFils[N]);
          // Write size and date/time
          WITH TFils.Objects[N] AS TFileRecObj DO
            begin
              Writer.WriteInteger(fFSize);
              Writer.WriteInteger(fTime);
            end;
        end;
      ClearObjects(TFils);
      FOR N := 0 TO TDirs.Count-1 DO
        begin
          // Write level
          Writer.WriteInteger(Level);
          // Write folder name
          Writer.WriteString(TDirs[N]);
          // Call recursively
          ProcessFiles(S+'\'+TDirs[N], Level+1);
        end;
    finally
      TFils.Free;
      TDirs.Free;
    end;
  end;

begin
  Stream.Write(codeDsk, cCodeLen);
  StrmPos := Stream.Position;
  Count := -1;
  Stream.Write(Count, SizeOf(Integer));
  Count := 0;
  CoMod := 0;
  Writer := TWriter.Create(Stream, 4096);
  try
    DirsWatched.Sort;
    FOR N := 0 TO DirsWatched.Count-1 DO
      begin
        StartDir := DirsWatched[N];
        Writer.WriteInteger(0);
        Writer.WriteString(StartDir);
        ProcessFiles(NoFinalSlash(StartDir), 1);
      end;
    Writer.WriteInteger(cEndFlag);
  finally
    Writer.Free;
  end;
  IF NOT ProcMsgTerminated THEN
    begin
      Stream.Seek(StrmPos, soFromBeginning);
      Stream.Write(Count, SizeOf(Integer));
    end;
  UpStatus('Done',-1);
end;

procedure RememberInis(Stream : TStream; IniList : TStringList;
  UpStatus : UpStatFunc);
VAR
  StrmPos  : Integer;
  Count, N : Integer;
  Writer   : TWriter;

  procedure ProcessFile(const S : String);
  VAR
    SecList : TStringList;
    KeyList : TStringList;
    N, M, P : Integer;
    T       : TextFile;
    Line    : String;

    function SameAsLast(S1, S2 : String) : Boolean;
    VAR
      k1, k2, v1, v2 : String;
      P              : Integer;
    begin
      Result := False;
      IF S1 = '' THEN Exit;
      P := Pos('=', S1);
      k1 := Copy(S1, 1, P-1);
      v1 := Copy(S1, P+1, Length(S1)-P);
      k2 := Copy(S2, 1, P-1);
      v2 := Copy(S2, P+1, Length(S2)-P);
      Result := (AnsiCompareText(k1, k2) = 0) AND
        (AnsiCompareStr(v1, v2) = 0);
    end;
    
  begin
    IF NOT FileExists(S) THEN Exit;
    Writer.WriteInteger(0);
    Writer.WriteString(S);
    Inc(Count);
    UpStatus('', Count);
    SecList := TStringList.Create;
    KeyList := nil;
    try
      Filemode := 0;
      AssignFile(T, S);
      Reset(T);
      try
        REPEAT
          ReadLn(T, Line);
          Line := Trim(Line);
          IF Line = '' THEN Continue;
          IF Line[1] = ';' THEN Continue;
          IF (Line[1] = '[') AND (Line[Length(Line)] = ']') THEN
            begin
              KeyList := TStringList.Create;
              SecList.AddObject(Line, KeyList);
            end
          ELSE IF KeyList = nil THEN Continue
          ELSE IF Pos('=', Line) = 0 THEN Continue
          ELSE KeyList.Add(Line);
        UNTIL EoF(T);
      finally
        CloseFile(T);
        Filemode := 2;
      end;
      SecList.Sort;
      Inc(Count, SecList.Count);
      FOR N := 0 TO SecList.Count-1 DO
        begin
          Writer.WriteInteger(1);
          Writer.WriteString(SecList[N]);
          KeyList := SecList.Objects[N] AS TStringList;
          KeyList.Sort;
          Inc(Count, KeyList.Count);
          Line := '';
          FOR M := 0 TO KeyList.Count-1 DO
            begin
              IF SameAsLast(Line,KeyList[M]) THEN
                begin
                  Dec(Count);
                  Continue;
                end;
              Line := KeyList[M];
              P := Pos('=', Line);
              Writer.WriteInteger(2);
              Writer.WriteString(Copy(Line, 1, P-1));
              Writer.WriteString(Copy(Line, P+1, Length(Line)-P));
            end;
          KeyList.Clear;
          KeyList.Free;
        end;
    finally
      SecList.Free;
    end;
  end;
begin
  Stream.Write(codeIni, cCodeLen);
  StrmPos := Stream.Position;
  Count := -1;
  Stream.Write(Count, SizeOf(Integer));
  Count := 0;
  Writer := TWriter.Create(Stream, 4096);
  try
    inilist.Sort;
    FOR N := 0 TO inilist.Count-1 DO
      ProcessFile(inilist[N]);
    Writer.WriteInteger(cEndFlag);
  finally
    Writer.Free;
  end;
  IF NOT ProcMsgTerminated THEN
    begin
      Stream.Seek(StrmPos, soFromBeginning);
      Stream.Write(Count, SizeOf(Integer));
    end;
  UpStatus('Done',-1);
end;

procedure RememberRegistry(Stream : TStream;
  Ignores : TStrings; UpStatus : UpStatFunc);
VAR
  StrmPos  : Integer;
  Count    : Integer;
  CoMod    : Integer;
  Writer   : TWriter;
  Reg      : TRegistry;
  RootName : String;

  procedure RememberKey(const Pth : String; level : Integer);
  VAR
    S     : String;
    TS    : TStringList;
    N     : Integer;
    DType : DWord;
    DSize : DWord;
    DBuff : PByteArray;
    Digest : TDigest;

  begin
    IF ProcMsgTerminated THEN Exit;
    S := RootName + Lowercase(Pth);
    // If this key is on the "ignores" list... ignore it!
    FOR N := 0 TO Ignores.Count-1 DO
      IF AnsiCompareText(S, Ignores[N]) = 0 THEN
        Exit;
    CoMod := Succ(CoMod) MOD 200;
    IF CoMod = 0 THEN
      UpStatus(S, Count);
    IF NOT Reg.OpenKeyReadOnly(Pth) THEN
      Exit;
    TS := TStringList.Create;
    try try
      Reg.GetValueNames(TS);
      Inc(Count, TS.Count);
      TS.Sort;
      FOR N := 0 TO TS.Count-1 DO
        begin
          DSize := 0; 
          IF RegQueryValueEx(Reg.CurrentKey, PChar(TS[N]), nil,
            @DType, nil, @Dsize) <> ERROR_SUCCESS THEN Continue;
          IF TS[N] = '' THEN
            Writer.WriteInteger(Level OR cfDef)
          ELSE
            begin
              Writer.WriteInteger(Level);
              Writer.WriteString(TS[N]);
            end;
          Writer.WriteInteger(DType);
          GetMem(DBuff, DSize);
          try
            IF RegQueryValueEx(Reg.CurrentKey, PChar(TS[N]), nil,
              @DType, @DBuff[0], @Dsize) <> ERROR_SUCCESS THEN
              Writer.WriteInteger(0)
            ELSE IF DSize <= 255 THEN
              begin
                Writer.WriteInteger(DSize);
                Writer.Write(DBuff^, DSize);
              end
            ELSE
              begin
                Writer.WriteInteger(DSize);
                md5Digest(DBuff, DSize, digest);
                Writer.Write(digest, SizeOf(digest));
              end;
          finally
            FreeMem(DBuff);
          end;
        end;
      Reg.GetKeyNames(TS);
      Inc(Count, TS.Count);
      TS.Sort;
      FOR N := 0 TO TS.Count-1 DO
        begin
          IF TS[N] = '' THEN
            Raise(Exception.Create('BAZOOKA!'));
          Writer.WriteInteger(Level OR cfKey);
          Writer.WriteString(TS[N]);
          RememberKey(Pth+'\'+TS[N], Level+1);
        end;
    except
      ON E:Exception DO
        S := E.Message;
    end;
    finally
      TS.Free;
    end;
  end;

  procedure DoKey(K : HKey; const Name : String);
  begin
    Reg.RootKey := K;
    RootName := Name;
    Writer.WriteInteger(0 OR cfKey);
    Writer.WriteString(Name);
    RememberKey('', 1);
  end;

begin
  Stream.Write(codeReg, cCodeLen);
  StrmPos := Stream.Position;
  Count   := -1;
  Stream.Write(Count, SizeOf(Integer));
  Count   := 0;
  CoMod   := 0;
  Writer := TWriter.Create(Stream, 4096);
  try
    Reg := TRegistry.Create;
    try
      DoKey(HKEY_USERS,          'HKEY_USERS');
      DoKey(HKEY_CURRENT_USER,   'HKEY_CURRENT_USER');
      DoKey(HKEY_CURRENT_CONFIG, 'HKEY_CURRENT_CONFIG');
      DoKey(HKEY_CLASSES_ROOT,   'HKEY_CLASSES_ROOT');
      DoKey(HKEY_LOCAL_MACHINE,  'HKEY_LOCAL_MACHINE');
    finally
      Reg.Free;
      Writer.WriteInteger(cEndFlag);
    end;
  finally
    Writer.Free;
  end;
  IF NOT ProcMsgTerminated THEN
    begin
      Stream.Seek(StrmPos, soFromBeginning);
      Stream.Write(Count, SizeOf(Integer));
    end;
  UpStatus('Done', -1);
end;

function CusSort(List: TStringList; Index1,
  Index2: Integer): Integer;
VAR L1, L2 : Integer;
begin
  Result := AnsiCompareStr(List[Index1], List[Index2]);
  IF Result <> 0 THEN Exit;
  L1 := Integer(List.Objects[Index1]);
  L2 := Integer(List.Objects[Index2]);
  IF L1 < L2 THEN Result := -1
  ELSE IF L1 > L2 THEN Result := 1
  ELSE Result := 0;
end;

procedure RememberTxt(Stream : TStream; Files : TStringList;
  UpStatus : UpStatFunc);
VAR
  StrmPos  : Integer;
  Count    : Integer;
  Writer   : TWriter;
  N        : Integer;

  procedure ProcessFile(const S : String);
  VAR
    TS : TStringList;
    N  : Integer;
  begin
    IF NOT FileExists(S) THEN Exit;
    Writer.WriteInteger(0);
    Writer.WriteString(S);
    Inc(Count);
    TS := TStringList.Create;
    try
      TS.LoadFromFile(S);
      FOR N := 0 TO TS.Count-1 DO
        begin
          TS.Objects[N] := Pointer(N+1);
          UpStatus('', Count+N);
        end;
      TS.CustomSort(CusSort);
      Inc(Count, TS.Count);
      FOR N := 0 TO TS.Count-1 DO
        begin
          Writer.WriteInteger(1);
          Writer.WriteString(TS[N]);
          Writer.WriteInteger(Integer(TS.Objects[N]));
        end;
    finally
      TS.Free;
    end;
  end;
begin
  Stream.Write(codeTxt, cCodeLen);
  StrmPos := Stream.Position;
  Count   := -1;
  Stream.Write(Count, SizeOf(Integer));
  Count   := 0;
  Writer := TWriter.Create(Stream, 4096);
  try
    FOR N := 0 TO Files.Count-1 DO
      ProcessFile(Files[N]);
    Writer.WriteInteger(cEndFlag);
  finally
    Writer.Free;
  end;
  IF NOT ProcMsgTerminated THEN
    begin
      Stream.Seek(StrmPos, soFromBeginning);
      Stream.Write(Count, SizeOf(Integer));
    end;
  UpStatus('Done', -1);

end;

procedure CompareStreams(theClass : TCompObjClass;
  theCode : CodeType; TFSO, TFSN : TStream; vAddNode, vDelNode,
  vAddLeaf, vDelLeaf, vChaLeaf : TStrings; UpStatus : UpStatFunc);
VAR
  RdrO, RdrN     : TReader;
  OldRec, NewRec : TCompObj;
  Count          : Integer;
  CoMod          : Integer;
begin
  CASE GetStreamStatus(TFSO, theCode, Count) OF
    sStatOK     : ; //OK
    sStatError  : Raise(Exception.Create('Defective stream'));
    sStatIncomp : Raise(Exception.Create('Stream incomplete'));
  end;
  CASE GetStreamStatus(TFSN, theCode, Count) OF
    sStatOK     : ; //OK
    sStatError  : Raise(Exception.Create('Defective stream'));
    sStatIncomp : Raise(Exception.Create('Stream incomplete'));
  end;
  Count := 0;
  CoMod := 0;
  vAddNode.Clear;
  vDelNode.Clear;
  vChaLeaf.Clear;
  vAddLeaf.Clear;
  vDelLeaf.Clear;
  RdrO   := TReader.Create(TFSO, 4096);
  RdrN   := TReader.Create(TFSN, 4096);
  OldRec := TheClass.Create;
  NewRec := TheClass.Create;
  try
    REPEAT
      IF ProcMsgTerminated THEN Exit;
      IF CoMod = 0 THEN UpStatus(OldRec.FullPath, Count);
      CASE OldRec.CompareWith(NewRec) OF
        -1 : BEGIN // old is less; was deleted
          Inc(Count);
          CoMod := Succ(CoMod) MOD 200;
          IF OldRec.pLeaf THEN
            vDelLeaf.AddObject(OldRec.FullPath,
              TChangeObj.Create(OldRec))
          ELSE
            vDelNode.Add(OldRec.FullPath);
          OldRec.Read(RdrO);
        END;
        0  : BEGIN
          Inc(Count);
          CoMod := Succ(CoMod) MOD 200;
          IF NOT OldRec.Match(NewRec) THEN
            vChaLeaf.AddObject(OldRec.FullPath,
              TChangeObj.Create(OldRec, NewRec));
          OldRec.Read(RdrO);
          NewRec.Read(RdrN);
        END;
        1  : BEGIN // new is less; was added
          IF NewRec.pLeaf THEN
            vAddLeaf.AddObject(NewRec.FullPath,
              TChangeObj.Create(NewRec))
          ELSE
            vAddNode.Add(NewRec.FullPath);
          NewRec.Read(RdrN);
        END;
      END;
    UNTIL OldRec.pDone AND NewRec.pDone;
  finally
    RdrO.Free;
    RdrN.Free;
    OldRec.Free;
    NewRec.Free;
  end;
  UpStatus('Done', -1);
end;

function GetStreamCount(const S : String) : Integer;
VAR
  Stream : TFileStream;
  code   : codeType;
begin
  Result := 0;
  Stream := TFileStream.Create(S, fmOpenRead OR fmShareExclusive);
  try
    Stream.Read(code, cCodeLen);
    Stream.Read(Result, SizeOf(Integer));
  finally
    Stream.Free;
  end;
end;

function BadTemp(const S : String; code : CodeType) : Boolean;
VAR
  Stream : TFileStream;
  Dummy  : Integer;
begin
  Stream := TFileStream.Create(S, fmOpenRead OR fmShareExclusive);
  try
    Result := GetStreamStatus(Stream, code, Dummy) <> sStatOK;
  finally
    Stream.Free;
  end;
end;

procedure StreamToTree(theClass : TCompObjClass; theCode : CodeType;
  Stream : TStream; TV : TTreeView;  ShowLeaf : Boolean;
  UpStatus : UpStatFunc);
VAR
  Reader     : TReader;
  WasLevel   : Integer;
  Count      : Integer;
  CoMod      : Integer;
  TN         : TTreeNode;
  TRec       : TCompObj;
  TChg       : TChangeObj;
begin
  CASE GetStreamStatus(Stream, theCode, Count) OF
    sStatOK     : ; //OK
    sStatError  : Raise(Exception.Create('Defective stream'));
    sStatIncomp : Raise(Exception.Create('Stream incomplete'));
  end;
  Count := 0;
  CoMod := 0;
  TV.Items.Clear;
  TN := nil;
  WasLevel := 0;
  TRec := theClass.Create;
  try
    Reader := TReader.Create(Stream, 4096);
    try
      REPEAT
        IF ProcMsgTerminated THEN Exit;
        TRec.Read(Reader);
        IF TRec.pDone THEN Exit;
        Inc(Count);
        CoMod := Succ(CoMod) MOD 200;
        IF CoMod = 0 THEN
          UpStatus(tRec.FullPath, Count);
        IF NOT ShowLeaf AND TRec.pLeaf THEN Continue;
        IF TRec.pLeaf THEN
          TChg := TChangeObj.Create(TRec)
        ELSE TChg := nil;
        IF TRec.pLevl > WasLevel+1 THEN
          Raise(Exception.Create('foo'))
        ELSE IF TRec.pLevl > WasLevel THEN
          TN := TV.Items.AddChildObject(TN, tRec.pName, TChg)
        ELSE IF TRec.pLevl = WasLevel THEN
          TN := TV.Items.AddObject(TN, tRec.pName, TChg)
        ELSE
          begin
            WHILE TRec.pLevl < WasLevel DO
              begin
                TN := TN.parent;
                Dec(WasLevel);
              end;
            TN := TV.Items.AddObject(TN, tRec.pName, TChg);
          end;
        IF TRec.pLeaf THEN
          TN.ImageIndex := 2
        ELSE TN.ImageIndex := 1;
        WasLevel := TRec.pLevl;
      UNTIL FALSE;
    finally
      Reader.Free;
    end;
  finally
    TRec.Free;
    UpStatus('Done',-1);
  end;
end;

end.
