%{
/*
     extract - A network log processor
     Copyright (C) 1993 Douglas Lee Schales, David K. Hess, David R. Safford

     Please see the file `COPYING' for the complete copyright notice.

grammar.y - 03/20/93

*/
#include <stdio.h>
#include <memory.h>
#include <netdb.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <malloc.h>
#include "lex.h"
#include "chario.h"
#include "parser.h"
#include "builder.h"
#include "timesub.h"
#include "stdunix.h"

extern int parse_error;
#define TCPDATA 0
#define UDPDATA 1
extern int udptcpflag;

%}

%union {
     unsigned int intval;
     char *strval;
}

%token DSTPORT SRCPORT DSTADDR SRCADDR SRCNET DSTNET DATE TIME HOST NET
%token TODAY YEST
%token OR AND MASKOPER NOT SINCE BEFORE
%token NEQ LEQ GEQ
%token INTEGER BYTEVAL STRING
%token PRINT NEXT
%type <intval> INTEGER BYTEVAL
%type <intval> OR AND
%type <intval> oper NEQ LEQ GEQ '=' '<' '>' '!' NOT
%type <intval> netop
%type <intval> DATE TIME TODAY YEST SINCE BEFORE
%type <intval> SRCADDR DSTADDR SRCPORT DSTPORT SRCNET DSTNET HOST NET
%type <intval> PRINT NEXT
%type <strval> STRING
%type <intval> ip_address netmask hostspec integer tcpport datespec timespec
%left OR
%left AND
%right '!'
%%
file
     : stmt file
     | stmt 
;

stmt
     : cond '{' actionset '}' {
	  queuestmt();
     }
;

cond
     : /* */
     | cond OR cond {
          pushoper($2);
     }
     | cond AND cond {
	  pushoper($2);
     }
     | term
;

actionset
     : action ';' actionset
     | action ';'
     | action
;

action
     : PRINT {
	  queueact($1);
     }
     | NEXT {
          queueact($1);
     }
     | error
;

term
     : SRCPORT oper tcpport {
	  pushkey(SRCPORT);
	  pushval($3);
	  pushoper($2);
     }
     | DSTPORT oper tcpport {
	  pushkey(DSTPORT);
	  pushval($3);
	  pushoper($2);
     }
     | HOST oper hostspec {
          pushkey(SRCADDR);
          pushval($3);
          pushoper($2);
          pushkey(DSTADDR);
          pushval($3);
          pushoper($2);
          pushoper(OR);
     }
     | SRCADDR oper hostspec {
	  pushkey(SRCADDR);
	  pushval($3);
	  pushoper($2);
     }
     | DSTADDR oper hostspec {
	  pushkey(DSTADDR);
	  pushval($3);
	  pushoper($2);
     }
     | NET netop hostspec {
          unsigned long mask = getmask($3);
          pushval($3);
          pushval(mask);
          pushoper(MASKOPER); /* net & mask */
          pushkey(SRCNET);
          pushval(mask);
          pushoper(MASKOPER); /* SRCADDR & mask */
          pushoper($2);
          pushval($3);
          pushval(mask);
          pushoper(MASKOPER); /* net & mask */
          pushkey(DSTNET);
          pushval(mask);      /* dstaddr & mask */
          pushoper(MASKOPER);
          pushoper($2);
          pushoper(OR);
     }
     | NET netop hostspec '/' netmask {
          pushval($3);
          pushval($5);
          pushoper(MASKOPER); /* net & mask */
          pushkey(SRCNET);
          pushval($5);
          pushoper(MASKOPER); /* srcaddr & mask */
          pushoper($2);
          pushval($3);
          pushval($5);
          pushoper(MASKOPER); /* net & mask */
          pushkey(DSTNET);
          pushval($5);
          pushoper(MASKOPER); /* dstaddr & mask */
          pushoper($2);
          pushoper(OR);
     }
     | SRCNET netop hostspec {
	  unsigned long mask = getmask($3);
	  pushval($3);
	  pushval(mask);
	  pushoper(MASKOPER);
	  pushkey(SRCNET);
	  pushval(mask);
	  pushoper(MASKOPER);
	  pushoper($2);
     }
     | DSTNET netop hostspec {
	  unsigned long mask = getmask($3);
	  pushval($3);
          pushval(mask);
	  pushoper(MASKOPER);
	  pushkey(DSTNET);
	  pushval(mask);
	  pushoper(MASKOPER);
	  pushoper($2);
     }
     | SRCNET netop hostspec '/' netmask {
	  pushval($3);
	  pushval($5);
	  pushoper(MASKOPER);
	  pushkey(SRCNET);
	  pushval($5);
	  pushoper(MASKOPER);
	  pushoper($2);
     }
     | DSTNET netop hostspec '/' netmask {
	  pushval($3);
          pushval($5);
	  pushoper(MASKOPER);
	  pushkey(DSTNET);
	  pushval($5);
	  pushoper(MASKOPER);
	  pushoper($2);
     }
     | DATE oper datespec {
          switch($2){
	  case '>':
	  case LEQ:
	       pushkey($1);
	       pushval($3 + (24*3600-1));
	       pushoper($2);
	       break;
	  case '=':
	       pushkey($1);
	       pushval($3);
	       pushoper(GEQ);
	       pushkey($1);
	       pushval($3 + (24*3600-1));
	       pushoper(LEQ);
	       pushoper(AND);
	       break;
	  case NEQ:
	       pushkey($1);
	       pushval($3);
	       pushoper('<');
	       pushkey($1);
	       pushval($3 + (24*3600-1));
	       pushoper('>');
	       pushoper(OR);
	       break;
	  default:
	       pushkey($1);
	       pushval($3);
	       pushoper($2);
	       break;
	  }
     }
     | TIME oper timespec {
	  pushkey($1);
	  pushval($3);
	  pushoper($2);
     }
     | SINCE datespec timespec {
          pushkey(DATE);
          pushval($2+$3);
          pushoper(GEQ);
     }
     | SINCE datespec {
          pushkey(DATE);
          pushval($2);
          pushoper(GEQ);
     }
     | BEFORE datespec timespec {
          pushkey(DATE);
          pushval($2+$3);
          pushoper('<');
     }
     | BEFORE datespec {
          pushkey(DATE);
          pushval($2);
          pushoper('<');
     }
     | '(' cond ')'
     | '!' term { pushuoper($1); }
     | NOT term { pushuoper('!'); }
     | error { pushval(0); }
;

oper
     : '=' {$$ = $1;}
     | '>' {$$ = $1;}
     | '<' {$$ = $1;}
     | NEQ {$$ = $1;}
     | LEQ {$$ = $1;}
     | GEQ {$$ = $1;}
     | error { pushoper(0); }
;

netop
     : '=' {$$ = $1;}
     | NEQ {$$ = $1;}
     | error { pushoper(0); }
;

datespec
     : BYTEVAL '/' BYTEVAL {
	  int year = getyear();
	  if(year < 100)
	       year += 1900;
	  if($1 < 1 || $1 > 12 ||
	     !validmday($1, $3, year)){
	       fprintf(stderr, "\"%s\", line %d: malformed date.\n",
		       getfilename(), getlinenum());
	       parse_error = 1;
	  }
	  else
	       $$ = makedate($1, $3, year);
     }
     | BYTEVAL '/' BYTEVAL '/' BYTEVAL {
	  int year = $5;
	  if(year < 100)
	       year += 1900;
	  if($1 < 1 || $1 > 12 ||
	     !validmday($1, $3, year)){
	       fprintf(stderr, "\"%s\", line %d: malformed date.\n",
		       getfilename(), getlinenum());
	       parse_error = 1;
	  }
	  else
	       $$ = makedate($1, $3, year);
     }
     | TODAY {
          $$ = today();
     }
     | YEST {
          $$ = today() - (24*3600);
     }
     | error { $$ = 0; }
;	       

timespec
     : BYTEVAL ':' BYTEVAL {
	  $$ = $1*3600 + $3*60;
     }
     | BYTEVAL ':' BYTEVAL ':' BYTEVAL {
	  $$ = $1*3600 + $3*60 + $5;
     }
     | error { $$ = 0; }
;



tcpport
     : integer { $$ = $1; }
     | STRING {
	  struct servent *se;
	  if((se = getservbyname($1,
				 udptcpflag == TCPDATA ? "tcp" : "udp"))){
	       $$ = htonl(se->s_port);
	  }
	  else {
	       fprintf(stderr, "\"%s\", line %d: unknown service: %s\n", 
		       getfilename(), getlinenum(),$1);
	       parse_error = 1;
	       $$ = 0;
	  }
	  free($1);
     }
     | error {$$ = 0;}
;

integer
     : INTEGER {
          $$ = $1;
     }
     | BYTEVAL {
          $$ = $1;
     }
;

hostspec
     : ip_address { $$ = $1;}
     | STRING {
          struct hostent *he;
	  struct {
	       int ipaddr:32;
	  } host;
	  if((he = gethostbyname($1))){
	       memcpy(&host, he->h_addr_list[0], 4);
	       $$ = host.ipaddr;
	  }
	  else {
	       fprintf(stderr, "\"%s\", line %d: unknown host: %s\n", 
		       getfilename(), getlinenum(), $1);
	       $$ = 0;
	       parse_error = 1;
	  }
	  free($1);
     }
     | error { $$ = 0; }
;

netmask
     : ip_address { $$=$1;}
     
;

ip_address
	: BYTEVAL '.' BYTEVAL '.' BYTEVAL '.' BYTEVAL {
	     $$ = (($1 << 24 | $3 << 16 | $5 << 8 | $7));
	}
;

%%

     struct keywords keywords[] = {
     { "dstport", DSTPORT },
     { "srcport", SRCPORT },
     { "srchost", SRCADDR },
     { "dsthost", DSTADDR },
     { "srcnet", SRCNET },
     { "dstnet", DSTNET },
     { "host", HOST },
     { "net", NET },
     { "time", TIME },
     { "date", DATE },
     { "since", SINCE },
     { "before", BEFORE },
     { "today", TODAY },
     { "yesterday", YEST },
     { "or", OR },
     { "and", AND },
     { "not", NOT }, 
     { "print", PRINT },
     { "next", NEXT },
     { (char *)0, 0 }
     };

void
yyerror()
{
     fprintf(stderr, "\"%s\", line %d: syntax error.\n",
	     getfilename(), getlinenum());
     parse_error = 1;
}

unsigned long
getmask(unsigned long addr)
{
     if(IN_CLASSA(addr))
	  return IN_CLASSA_NET;
     else if(IN_CLASSB(addr))
	  return IN_CLASSB_NET;
     else if(IN_CLASSC(addr))
	  return IN_CLASSC_NET;
     else
	  return 0xffffffff;
}

