//
// $COPYRIGHT$
//
//                                                                           
//
//===========================================================================

#ifndef MTL_DENSE_ITERATOR_H
#define MTL_DENSE_ITERATOR_H

#include "mtl/mtl_iterator.h"

#include "mtl/config.h"

namespace mtl {

using std::iterator_traits;

//: dense iterator
//
// An iterator for dense contiguous container that keeps track of the index.
//
//!category: iterators, adaptors
//!component: type
//!definition:dense_iterator.h
//!tparam: RandomAccessIterator - the base iterator
//!models: RandomAccessIterator? (with index())

template <class RandomAccessIterator, int IND_OFFSET=0>
class dense_iterator {
  typedef dense_iterator<RandomAccessIterator> self;
public:
  //: The value type
  typedef typename iterator_traits<RandomAccessIterator>::value_type value_type;
  //: This is a random access iterator
  typedef typename iterator_traits<RandomAccessIterator>::iterator_category iterator_category;
  //: The type for differences between iterators
  typedef typename iterator_traits<RandomAccessIterator>::difference_type difference_type;
  //: The type for pointers to the value type
  typedef typename iterator_traits<RandomAccessIterator>::pointer pointer;
  //: The type for references to the value type
  typedef typename iterator_traits<RandomAccessIterator>::reference reference;

  typedef difference_type Distance;
  
  /*
protected:
*/

  RandomAccessIterator start;
  int pos;
  int start_index;

public:
  //: Return the index of the current element
  //!wheredef: IndexedIterator
  inline int index() const { 
    return pos + start_index + IND_OFFSET; 
  }
  //: Default Constructor  
  inline dense_iterator() : pos(0), start_index(0) {}

  //: Constructor from underlying iterator
  inline dense_iterator(RandomAccessIterator s, int i, int first_index = 0)
    : start(s), pos(i), start_index(first_index) { }
  //: Copy Constructor  
  inline dense_iterator (const self& x) 
    : start(x.start), pos(x.pos), start_index(x.start_index) {}

  template <class SELF>
  inline dense_iterator (const SELF& x)
    : start(x.start), pos(x.pos), start_index(x.start_index) {}

  //: Assignment operator
  inline self& operator=(const self& x) {
    start = x.start;
    pos = x.pos;
    start_index = x.start_index;
    return *this;
  }
  //: Destructor
  inline ~dense_iterator () { }

  //: Access the underlying iterator  
  inline RandomAccessIterator base() const { return start + pos; }

  inline operator RandomAccessIterator() const { return start + pos; }
  //: Dereference operator  
  inline reference operator*() const { return *(start + pos);  }
  //: Member access operator
  inline pointer operator-> () const { return start + pos; }
  //: Pre-increment operator  
  inline self& operator++ () { ++pos; return *this; }
  //: Post-increment operator
  inline self operator++ (int) { self tmp = *this; ++pos; return tmp; }
  //: Pre-decrement operator
  inline self& operator-- () { --pos; return *this; }
  //: Post-decrement operator
  inline self operator-- (int) { self tmp = *this; --pos; return tmp; }
  //: Add iterator and distance n
  inline self operator+ (Distance n) const { return self(start, pos + n); }
  //: Add distance n to this iterator
  inline self& operator+= (Distance n) { pos += n; return *this; }
  //: Subtract iterator and distance n
  inline self operator- (Distance n) const { return self(start, pos - n); }
  //: Return the difference between two iterators
  inline difference_type operator- (const self& x) const { 
    return base() - x.base(); }
  //: Subtract distance n from this iterator
  inline self& operator-= (Distance n) { pos -= n; return *this; }
  //: Return whether this iterator is not equal to iterator x
  inline bool operator!= (const self& x) const { return pos != x.pos; }
  //: Return whether this iterator is less than iterator x
  inline bool operator < (const self& x) const { return pos < x.pos; }
  //: Return whether this iterator is greater than iterator x
  inline bool operator > (const self& x) const { return pos > x.pos; }
  //: Return whether this iterator is equal to iterator x
  inline bool operator== (const self& x) const { return pos == x.pos; }
  //: Return whether this iterator is less than or equal to iterator x
  inline bool operator<= (const self& x) const { return pos <= x.pos; }
  //: Return whether this iterator is greater than or equal to iterator x
  inline bool operator>= (const self& x) const { return pos >= x.pos; }
  //: Equivalent to *(i + n)
  inline reference operator[] (Distance n) const { 
    return *(start + pos + n); 
  }
};

template<class T, class Iterator>
inline Iterator&
operator>>(Iterator& iter, T& x) {
  x = *iter;
  ++iter;
  return iter;
}

template <class T>
inline
dense_iterator<T>
operator+ (int n, const dense_iterator<T> &x)
{
  return dense_iterator<T>(x.base(), n);
}


} /* namespace mtl */

#endif


