//
// $COPYRIGHT$
//

#ifndef MTL_LIGHT1D_H
#define MTL_LIGHT1D_H

#include "mtl/mtl_iterator.h"

#include "mtl/config.h"
#include "mtl/dense_iterator.h"
#include "mtl/reverse_iter.h"
#include "mtl/matrix_traits.h"
#include "mtl/scaled1D.h"

namespace mtl {

/**
  This is a {\em light} version of {\tt dense1D}.
  It does no memory management (or reference counting)
  and can only be used with pre-existing memory.
  The purpose of {\tt light1D} is to be used
  in the high performance kernels.

  @memo Light 1-D Container
 */
template <class T, int NN = 0, int IND_OFFSET = 0>
class light1D {
  typedef light1D<T> self;
public:
  enum { N = NN };

  typedef light1D<int> IndexArray; /* JGS */

  /**@name Type Definitions */
  //@{
  /// Numeric Precision Type
  typedef T PR;
  ///
  typedef dense_tag sparsity;
  ///
  typedef scaled1D< light1D<T> > scaled_type;
  ///
  typedef T value_type;
  ///
  typedef T& reference;
  ///
  typedef T* pointer;
  ///
  typedef const T& const_reference;
  ///
  typedef dense_iterator<T*, IND_OFFSET> iterator;
  ///
  typedef dense_iterator<const T*, IND_OFFSET> const_iterator;
  ///
  typedef reverse_iter<iterator> reverse_iterator;
  ///
  typedef reverse_iter<const_iterator> const_reverse_iterator;

  typedef size_t size_type;
  typedef ptrdiff_t difference_type;

  typedef self IndexArrayRef;
  
  typedef self subrange;

  typedef oned_tag dimension;

  //@}

  /**@name Constructors */
  //@{
  /// Default Constructor
  inline light1D() : rep(0), size_(0), first(0) { }

  /// Preallocated Memory Constructor with optional non-zero starting index
  inline light1D(T* data, size_type n, size_type start = 0)
    : rep(data), size_(n), first(start) { }

  /// Copy Constructor
  inline light1D(const self& x)
    : rep(x.rep), size_(x.size_), first(x.first) { }

  inline ~light1D() { }

  //@}

  /**@name Access Methods */
  //@{
  /**@name Iterator Access Methods */
  //@{
  ///
  inline iterator begin() { return iterator(rep, first); }
  ///
  inline iterator end() { return iterator(rep, first + size_); }
  ///
  inline const_iterator begin() const {
    return const_iterator(rep, first); 
  }
  ///
  inline const_iterator end() const{ 
    return const_iterator(rep, first + size_); 
  }
  ///
  inline reverse_iterator rbegin() {
    
    return reverse_iterator(end());
  }
  ///
  inline reverse_iterator rend() { return reverse_iterator(begin()); }
  ///
  inline const_reverse_iterator rbegin() const {
    return const_reverse_iterator(end()); 
  }
  ///
  inline const_reverse_iterator rend() const{ 
    return const_reverse_iterator(begin()); 
  }
  //@}
  /**@name Element Access Methods */
  //@{
  ///
  inline reference operator[](int n) { return rep[n - first]; }
  ///
  inline const_reference operator[](int n) const { return rep[n - first]; }

  inline subrange operator()(int s, int f) {
    return subrange(rep + s - first, f - s, s);
  }
  //@}
  /**@name Size Methods */
  //@{  
  ///
  inline int size() const { return size_; }
  ///
  inline int nnz() const { return size_; }
  ///
  inline void resize(int n) {
    if (rep) delete [] rep;
    size_ = n;
    rep = new T[size_];
  }
  // experimental and broken
  //  inline void reindex(int n) {
  //    cout << "reindex() first " << first << endl;
  //    cout << "n " << n << endl;
  //    rep += first - n;
  //    first = n;
  //  }
  //@}  
  /// Memory Access
  inline PR* data() const { return rep; }
  //@}  
protected:
  T* rep;
  int size_;
  int first;
};

} /* namespace mtl */

#endif // MTL_LIGHT1D_H

