//
// $COPYRIGHT$
//
//
//===========================================================================

#ifndef MTL_STRIDED_ITERATOR_H
#define MTL_STRIDED_ITERATOR_H

#include "mtl/mtl_iterator.h"

#include "mtl/config.h"

namespace mtl {

using std::iterator_traits;

//: strided iterator
//
// This iterator moves a constant stride for each increment or
// decrement operator invoked. The strided iterator is used
// to implement a row-view to column oriented matrices, or
// column-views to row oriented matrices.
//
//!category: iterators, adaptors
//!component: type
//!defined: strided_iterator.h
//!models: RandomAccessIterator
//!tparams: RandomAccessIterator - the base iterator type
template <class RandomAccessIterator>
class strided_iterator {
  typedef strided_iterator<RandomAccessIterator> self;
public:
  //: The type for the difference between two iterators
  typedef typename iterator_traits<RandomAccessIterator>::difference_type
                      difference_type;
  //: The value type pointed to by this iterator type
  typedef typename iterator_traits<RandomAccessIterator>::value_type
                      value_type;
  //: The type for references to the value type
  typedef typename iterator_traits<RandomAccessIterator>::reference
                      reference;
  //: The iterator category for this iterator
  typedef typename iterator_traits<RandomAccessIterator>::iterator_category
                      iterator_category;
  //: The type for pointers to the value type
  typedef typename iterator_traits<RandomAccessIterator>::pointer
                      pointer;
  typedef difference_type      Distance;
  //: The underlying iterator type
  typedef RandomAccessIterator iterator_type;

  //: Default Constructor
  inline strided_iterator() : stride(0) { }

  //: Construct from the underlying iterator
  inline strided_iterator(const RandomAccessIterator& x, int s) 
    : iter(x), stride(s) { }

  //: Copy Constructor
  inline strided_iterator(const self& x)
    : iter(x.iter), stride(x.stride) { }

  //: Assignment Operator
  inline self& operator=(const self& x) {
    iter = x.iter; stride = x.stride; return *this;
  }

  //: Return the index of the element this iterator points to
  //!wheredef: IndexedIterator
  inline int index() const { return iter.index() / stride; }
  
  //: Convert to the underlying iterator
  inline operator RandomAccessIterator () const { return iter; }

  inline RandomAccessIterator base() const { return iter; }

  //: Dereference, return the element currently pointed to
  inline reference operator*() const { return *iter; }

  //: Pre-increment operator
  inline self& operator++ (){
    iter += stride; return *this;
  }
  //: Post-increment operator
  inline self operator++ (int){ self tmp = *this; iter += stride; return tmp; }
  //: Pre-decrement operator
  inline self& operator-- (){ iter -= stride; return *this; }
  //: Post-decrement operator
  inline self operator-- (int){ self tmp = *this; iter -= stride; return tmp; }
  //: Add this iterator and n  
  inline self operator+ (Distance n) const {
    return self (iter + n*stride, stride);
  }
  //: Add distance n to this iterator
  inline self& operator+= (Distance n) { iter += n*stride; return *this; }
  //: Subtract this iterator and distance n
  inline self operator- (Distance n) const {
    return self (iter - n*stride, stride);
  }
  //: Subtract distance n from this iterator
  inline self& operator-= (Distance n) { iter -= n*stride; return *this; }

  //: Add this iterator and iterator x
  inline self operator+ (const self& x) const {
    return self(iter + x.iter, stride); 
  }
  //: Return this distance between this iterator and iterator x
  inline Distance operator- (const self& x) const { return iter - x.iter; }
  //: Return *(i + n)  
  inline reference operator[] (Distance n) const { return *(*this+n*stride); }
  //: Return whether this iterator is equal to iterator x
  inline bool operator==(const self& x) const { return iter == x.iter; }
  //: Return whether this iterator is not equal to iterator x
  inline bool operator!=(const self& x) const { return iter != x.iter; }
  //: Return whether this iterator is less than iterator x
  inline bool operator<(const self& x) const { return iter < x.iter; }  
protected:
  RandomAccessIterator iter;
  int stride;
};

template <class RandomAccessIterator>
inline strided_iterator<RandomAccessIterator>
str(const RandomAccessIterator& x, int stride = 1) {
  return strided_iterator<RandomAccessIterator>(x, stride);
}

} /* namespace mtl */

#endif







