/* qop.h

Several quantum gates. (one bit and multi-controlled)

This file is part of the OpenQubit project.

Copyright (C) 1998-1999 OpenQubit.org
Yan Pritzker <skwp@thepentagon.com>

Please see the CREDITS file for contributors.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

//! author="Rafal Podeszwa, Christopher Dawson, Peter Belkner, Yan Pritzker"
//! lib="Quantum Operators and Gates"

#ifndef _QOP_H_
#define _QOP_H_

#include <math.h>
#include "utility.h"
#include "qstate.h"

/*
NOTE: I have decided not to use MTL Matrix
because it complicates the process of applying
the matrix to an amplitude vector if you 
are to do it 2 qubits at a time.
*/

/*     
#include "mtl/matrix.h"
#include "mtl/mtl.h"
#include "mtl/utils.h"
#include "mtl/linalg_vec.h"
*/

class Q_2x2Gate
//: Base for one bit gates.
{
public:
	virtual void Transform(Q_BitSet &q, int bit);

protected:
	void _SetMatrix(const Complex& a, const Complex& b, 
                    const Complex& d, const Complex& t)
	{
		_a00 = a; _a01 = b;
		_a10 = d; _a11 = t;
	}

private:
	Complex _a00, _a01, _a10, _a11;
};

class opUnitary : public Q_2x2Gate
//: General Unitary operator.
{
public:
	void SetMatrix(double alpha, double beta, double delta, double theta)
	{                
	_SetMatrix (
		 Complex(cos(delta+alpha/2+beta/2)*cos(theta/2),
			sin(delta+alpha/2+beta/2)*cos(theta/2)),

		 Complex(cos(delta+alpha/2-beta/2)*sin(theta/2),
			sin(delta+alpha/2-beta/2)*sin(theta/2)),

		 Complex(-cos(delta-alpha/2+beta/2)*sin(theta/2),
			-sin(delta-alpha/2+beta/2)*sin(theta/2)),

		 Complex(cos(delta-alpha/2-beta/2)*cos(theta/2),
			sin(delta-alpha/2-beta/2)*cos(theta/2)));
 	}

	void Transform(Q_BitSet &q, int bit) { Q_2x2Gate::Transform(q,bit); }

	opUnitary(double a=0, double b=0, double d=0, double t=0)
		{ SetMatrix(a,b,d,t); }
};

/*
class opRotQubit : public Q_2x2Gate
//: Rotation on y-axis
{
public:
	//: Set the parameters needed for the gate to function.
	void Param(double theta) {
		BaseClassT::SetMatrix( 
			 cos(theta/2),  sin(theta/2),
			-sin(theta/2),  cos(theta/2)
		);
	}
	
	opRotQubit(double theta=0) { Param(theta); }
}; 

template <class BaseClassT>
class opRotPhase : public BaseClassT
//: Rotation on z-axis
{
public:
	void Param(double alpha) {
		BaseClassT::SetMatrix(
			Complex(cos(alpha/2), sin(alpha/2)), 0,
			0, Complex(cos(alpha/2), -sin(alpha/2))
		);
	}

	opRotPhase(double alpha=0) { Param(alpha); }
};

template <class BaseClassT>
class opPhaseShift : public BaseClassT
//: Scalar multiplication on z-axis
{
public:
	Param(double delta) {
		BaseClassT::SetMatrix(
			Complex(cos(delta), sin(delta)),0,
			0, Complex(cos(delta), sin(delta))
		);
	}

	opPhaseShift(double delta=0) { Param(delta); }
};

template <class BaseClassT>
class opHadamard : public BaseClassT
//: Hadamard operator [|0> -> |0> + |1> and |1> -> |0> - |1>]
{
public:
	opHadamard() : BaseClassT(M_SQRT1_2, M_SQRT1_2,
									  M_SQRT1_2, -M_SQRT1_2) {};
};

template <class BaseClassT>
class opNOT : public BaseClassT
//: SingleBit == Pauli negation, Controlled == CNot
{
public:
	opNOT() : BaseClassT(0,1,1,0) {};
};

class opFFT 
//: Fast Fourier Transform
{
public:
	opFFT() {};
	void operator() (Q_BitSet &q, int numbits=-1);
};

class opSPhaseShift 
//: Shor's phase shift operator (double-controlled phase shift);
{
// the matrix used for controlled single bit operation is
// (1             0             )
// (0   exp(i*Pi*delta/2^(k-j)) )

private:
	opUnitary<Q_GateMatrix::Controlled> CU;
public:
	opSPhaseShift() {};
	
	void operator() (Q_BitSet &q, int j, int k)
	{
		assert(j < k);    //see Shor's paper
		double delta = M_PI/(1 << (k-j));
		unsigned long mask;
		
		mask = (1 << j);
		
		//call general unitary controlled gate as suggested by Rafal
		CU.Param(delta,0,-delta/2,0);
		CU(q,mask,k);
	}
};

class ModExp
//: Modular Exponentiation
{
public:
	
	ModExp() {};
	void operator() (Q_BitSet &q, int a, int n, int b=-1) {
		//results go here first
		vector<Complex> _qArrayTmp(Q_StateVector::States()); 
		if(b==-1) b = q.Qubits();

		for(int i = 0; i < Q_StateVector::States(); i++)
			{
				if (Q_StateVector::Coef(i) != 0)
					_qArrayTmp[i + (modexp(a,i,n) << b)] = Q_StateVector::Coef(i);
			}

		for (int k = 0; k < Q_StateVector::States(); k++)
			Q_StateVector::Coef(k) = _qArrayTmp[k]; //copy results
	}
};

template <class OperatorType>
class DoAllBits
//: This class allows any operator to work on all bits of a state.
// This will only work for single bit operators requiring no parameters
// (i.e. this will not work with RotQubit which requires an angle theta)
{
private:
	OperatorType op;

public:
        void operator() (Q_BitSet &q) {
                for(int i=0; i < q.Qubits(); ++i) 
                        op(q,i);
	}
};

//: Unitary Operator
typedef opUnitary<Q_GateMatrix::SingleBit> 		Unitary;		

//: Controlled Unitary Operator
typedef opUnitary<Q_GateMatrix::Controlled>		CUnitary;

//: Qubit Rotation (Ry)
typedef opRotQubit<Q_GateMatrix::SingleBit>		RotQubit;

//: Controlled Qubit Rotation (Ry)
typedef opRotQubit<Q_GateMatrix::Controlled>	CRotQubit;

//: Phase Rotation (Rz)
typedef opRotPhase<Q_GateMatrix::SingleBit>		RotPhase;

//: Controlled Phase Rotation
typedef opRotPhase<Q_GateMatrix::Controlled>	CRotPhase;

//: Phase Shift (scalar multiplication of z axis)
typedef opPhaseShift<Q_GateMatrix::SingleBit>	PhaseShift;

//: Controlled Phase Shift
typedef opPhaseShift<Q_GateMatrix::Controlled>	CPhaseShift;

//: Shor Phase Shift
typedef opSPhaseShift				SPhaseShift;

//: Hadamard Operator
typedef opHadamard<Q_GateMatrix::SingleBit>		Hadamard;

//: Negation Operator
typedef opNOT<Q_GateMatrix::SingleBit>			Not;

//: Controlled Negation (CNot - a.k.a XOR)
typedef opNOT<Q_GateMatrix::Controlled>			CNot;

//: Fast Fourier Transform
typedef opFFT							FFT;

//: Hadamard on all bits == WalshHadamard
typedef DoAllBits<Hadamard> 		WalshHadamard;

*/
#endif
