/* qstate.cc

Implementation of Data Model of a Quantum State/Register
This file is part of the OpenQubit project.

Copyright (C) 1998-1999 OpenQubit.org
Yan Pritzker <skwp@thepentagon.com>

Please see the CREDITS file for contributors.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
#include "qstate.h"

extern RandGenerator<_RNGT_> *Q_StateVector::_RNG;

Q_StateVector::Q_StateVector(int nQubits, int seedval=-1)
{
	_nQubits = nQubits;
	_nStates = 1 << nQubits;
	_LastAllocated=-1;	//no qubits allocated yet
	_Reset();
	if(seedval==-1)
		_RNG = new _RNG_;	//use default (time)
	else
		_RNG = new _RNG_(seedval);	//seed with our value
}

void Q_StateVector::Save(char filename[])
{
	FILE *FH;
	if ((FH=fopen(filename,"w"))==NULL)
		cerr << "ERROR: could not open file " << filename << endl;

	fprintf(FH,"QSTATE SIZE %d\n", _nStates);

	for(amp_iter i=_AmplitudeVector.begin(); i!=_AmplitudeVector.end(); ++i) {
			fprintf(FH, "%+1.17f \t %+1.17f \t |0x%X>\n",
					 real(*i),
					 imag(*i),
					 i.index());
	}
	fclose(FH);
}

void Q_StateVector::Read(char filename[])
{
	FILE *FH;
	
	if((FH=fopen(filename,"r"))==NULL)
		cerr << "ERROR: could not open file " << filename << endl;

	double real,imag;
	int index;
	int size;

	fscanf(FH, "QSTATE SIZE %d\n", &size);
	
	while(FH)
	{
		int r=fscanf(FH,"%+1.17f \t %+1.17f \t |0x%X>\n", 
					 	 &real, &imag, &index);

		if(r==0) break;

		D("Scanned %f %f %d\n",real,imag,index);
		_AmplitudeVector[index] = Complex(real,imag);
	}

	fclose(FH);
}

Complex Q_StateVector::Amplitude(int i) const throw(Q_Exception::RangeErr)
{
	if(i<0 || i>_nStates) throw Q_Exception::RangeErr();
	return _AmplitudeVector[i];
}

//Main function: memory management
Q_BitSet* Q_StateVector::getBitSet(int length) throw (Q_Exception::MallocErr)
{
	int newLastAllocated  = _LastAllocated + length;

	if (newLastAllocated  > _nQubits)
		throw Q_Exception::MallocErr();	//unable to allocate more qubits
	else {
		//create new bit set
		Q_BitSet *newbitset = 
			new Q_BitSet( this, _LastAllocated + 1, newLastAllocated );

		_LastAllocated = newLastAllocated;
		return newbitset;
	}
}

Q_BitSet* Q_StateVector::getBitSet(const vector<int>& bitsToGet) 
	throw (Q_Exception::RangeErr)
{
	cerr << "ERROR: getBitSet by arbitrary vector not yet implemented\n";
}

int Q_StateVector::_Collapse(int bit) throw (Q_Exception::RangeErr)
//single qubit collapse
{
	//index is 0..nQubits-1, 0 being the most significant qubit,
	//e.g. in |0001>, the 1 is the least significant qubit and
	//its position is 0
	if (bit < 0 || bit > _nQubits-1) throw Q_Exception::RangeErr();

	double p0=0.0,p1=0.0;
	
	//calculate the aggregate probability of all states that have a 0
	//in this bit position, and those that have a 1 in this bit position
	for(amp_iter i=_AmplitudeVector.begin(); i!=_AmplitudeVector.end(); i++)
		if (IsBitSet(i.index(),bit))
			p1 += norm(*i);
		else
			p0 += norm(*i);

	D("Got probabilities...p0=%1.3f, p1=%1.3f\n", p0, p1);
   D("Sum of probabilities: %1.5f; Normalized amplitudes: %1.5f\n",
		 p0+p1, _NormalizedSum());
	
	//sum of normalized amplitudes and sum of added probabilities
	//should equal..if not we messed up somewhere in the addition/normalization
	D("Normalized Sum + ROUND_ERR = %1.25f\n", _NormalizedSum() + ROUND_ERR);
	D("norm(*this) - ROUND_ERR = %1.25f\n", _NormalizedSum() - ROUND_ERR);
	D("p0+p1 should be between those and it is %1.10f\n", p0+p1);
	
	double rnd=_RNG->GetRandBetween(0,_NormalizedSum());
	bool on  = p0 < rnd;
	
	for(amp_iter i=_AmplitudeVector.begin(); i!=_AmplitudeVector.end(); ++i)
		if (IsBitSet(i.index(),bit))
			if(!on) *i = 0.0;
			else *i /= sqrt(p1);
		else
			if(on) *i = 0.0;
			else *i /= sqrt(p0);

	D("Set bit state to %d\n", on);
	return on;
}

int Q_StateVector::_Collapse(int begin, int end) throw (Q_Exception::RangeErr)
{
	int total=0;
	if(begin < 0 || end > _nQubits-1) throw Q_Exception::RangeErr();
	
	for(int i=begin; i<=end; ++i) {
		total += _Collapse(i) * (int)pow(2,i);
	}
	return total;
}

double Q_StateVector::_NormalizedSum()
{
	double n=0.0;
	for(amp_iter i=_AmplitudeVector.begin(); i!=_AmplitudeVector.end(); i++) 
		n += norm(*i);
	return n;
}

Q_BitSet::Q_BitSet(Q_StateVector *Q, int s, int e) : QSV(Q)
{
	int totalbits = e-s+1;
	_nQubits = totalbits; 
	_Bitset.resize(totalbits);
	for(int i=0; i<e-s+1; ++i) _Bitset[i]=i+s;
}

Q_BitSet::Q_BitSet(Q_StateVector *Q, const vector<int>& bits) : QSV(Q)
{
	_nQubits = bits.size();
	_Bitset.resize(bits.size());
	for(int i=0; i<bits.size(); ++i) _Bitset[i]=bits[i];
}

int Q_BitSet::Measure(int i)
	{ return QSV->_Collapse(_Bitset[i]); }

int Q_BitSet::Measure()
{
	return Measure(0,_Bitset.size()-1);
}

int Q_BitSet::Measure(int start, int end)
{
	int total=0;
	for(int i=start; i<=end; ++i) total += (int)pow(2,i)*Measure(_Bitset[i]);
}
