//
// $COPYRIGHT$
//

#ifndef MTL_COMPRESSED_ITER_H
#define MTL_COMPRESSED_ITER_H

#include "mtl/mtl_iterator.h"
#include "mtl/config.h"

namespace mtl {

  //: Compressed Storage Iterator
  //
  // Iterates through a value array and index array.
  // This is used by the compressed1D and compressed2D classes.
  //
  //!category: iterators
  //!component: type
  //!tparam: Value - The type of the value array
  //!tparam: Index - The type of the index array
template <class Value, class Index, int IND_OFFSET>
class compressed_iter {
  typedef compressed_iter<Value,Index,IND_OFFSET> self;
  typedef typename Value::value_type PR;
  typedef Value values_t;
  typedef typename values_t::iterator        value_iterator;
  typedef typename values_t::const_iterator  value_const_iterator;
  typedef Index indices_t;
  typedef typename indices_t::iterator       index_iterator;
  typedef typename indices_t::const_iterator index_const_iterator;
public:
  typedef typename std::iterator_traits<value_iterator>::iterator_category iterator_category;
  typedef PR value_type;
  typedef int difference_type;
  typedef PR* pointer;
  typedef PR& reference;
  typedef const PR& const_reference;
  
  inline compressed_iter() : pos (-1) { }

  inline compressed_iter(const self& x)
    : index_iter_(x.index_iter_), value_iter_(x.value_iter_), pos(x.pos) { }
  
  inline self& operator=(const self& x)
    {
      index_iter_ = x.index_iter_;
      value_iter_ = x.value_iter_;
      pos         = x.pos; 
      return *this; 
    }

  inline compressed_iter(value_iterator vals, index_iterator inds, int p)
    : index_iter_(inds), value_iter_(vals), pos(p) { }
  

  inline int index() const { 
    return index_iter_[pos] + IND_OFFSET;  /* F to C */
  }
  inline void set_index(int i) { 
    index_iter_[pos] = i;
  }
  
  inline reference operator*() const { return value_iter_[pos]; }
  
  inline self operator++() { ++pos; return *this; }
  inline self operator++(int) {
    self tmp = *this; ++pos; return tmp;
  }
  inline self operator--() { --pos; return *this; }
  inline self operator--(int) {
    self tmp = *this; --pos; return tmp;
  }
  inline self& operator+=(int n) { pos += n; return *this; }
  inline self& operator-=(int n) { pos -= n; return *this; }

  inline self operator+(int n) {
    return self(value_iter_, index_iter_, pos + n);
  }
  inline self operator-(int n) const {
    return self(value_iter_, index_iter_, pos - n);
  }
  inline int operator-(const self& x) const {
    return pos - x.pos;
  }

  inline bool operator != (const self& y) const { 
    return pos != y.pos;
  }
  inline bool operator < (const self& y) const { 
    return pos != y.pos;
  }
  inline bool operator == (const self& y) const { 
    return pos != y.pos;
  }

  inline index_iterator index_iter() { return index_iter_ + pos; }
  inline value_iterator value_iter() { return value_iter_ + pos; }
  //protected:
  index_iterator index_iter_;
  value_iterator value_iter_;
  int pos;
};


template <class Value, class Index, int IND_OFFSET>
class const_compressed_iter {
  typedef const_compressed_iter<Value,Index, IND_OFFSET> self;
  typedef typename Value::value_type PR;
  typedef Value values_t;
  typedef typename values_t::iterator        value_iterator;
  typedef typename values_t::const_iterator  value_const_iterator;
  typedef Index indices_t;
  typedef typename indices_t::iterator       index_iterator;
  typedef typename indices_t::const_iterator index_const_iterator;
public:
  typedef typename std::iterator_traits<value_iterator>::iterator_category iterator_category;
  typedef PR value_type;
  typedef int difference_type;
  typedef PR* pointer;
  typedef const PR& reference; // JGS
  typedef const PR& const_reference; // JGS this is superfluous

  inline const_compressed_iter() : pos(0) { }
  
  inline const_compressed_iter(const self& x) 
    : index_iter_(x.index_iter_), value_iter_(x.value_iter_), pos(x.pos) { }

  inline self& operator=(const self& x) { 
    index_iter_ = x.index_iter_;
    value_iter_ = x.value_iter_; 
    pos         = x.pos;
    return *this; 
  }

  inline const_compressed_iter(const compressed_iter<Value,Index,IND_OFFSET>& x) 
    : index_iter_(x.index_iter_), value_iter_(x.value_iter_), pos(x.pos) { }
  
  inline self& operator=(const compressed_iter<Value,Index,IND_OFFSET>& x) 
    { 
      index_iter_ = x.index_iter_;
      value_iter_ = x.value_iter_;
      pos         = x.pos; 
      return *this; 
    }

  inline const_compressed_iter(value_const_iterator vals, 
			       index_const_iterator inds, int p)
    : index_iter_(inds), value_iter_(vals), pos(p) { }
  

  inline int index() const { 
    return index_iter_[pos] + IND_OFFSET;  /* F to C */
  }

  inline void set_index(int i) const { 
    index_iter_[pos] = i;
  }
  
  inline const PR& operator*() const { return value_iter_[pos]; }

  inline int operator-(const self& x) const {
    return pos - x.pos;
  }  
  inline self operator++() { ++pos; return *this; }
  inline self operator--() { --pos; return *this; }
  inline self& operator+=(int n) { pos += n; return *this; }
  inline self& operator-=(int n) { pos -= n; return *this; }
  inline self operator+(int n) {
    return self(value_iter_, index_iter_, pos + n);
  }
  inline self operator-(int n) const {
    return self(value_iter_, index_iter_, pos - n);
  }

  inline bool operator == (const self& y) const { 
    return pos != y.pos;
  }
  inline bool operator < (const self& y) const { 
    return pos != y.pos;
  }
  inline bool operator != (const self& y) const { 
    return pos != y.pos;
  }
  
  inline index_const_iterator index_iter() { return index_iter_ + pos; }
  inline value_const_iterator value_iter() { return value_iter_ + pos; }
  //protected:
  index_const_iterator index_iter_;
  value_const_iterator value_iter_;
  int pos;
};


} /* namespace mtl */

#endif
