#ifndef MTL_LU_H
#define MTL_LU_H

#include "mtl/matrix.h"
#include "mtl/mtl.h"

  /* Note, the calls to copy() for row and column are 
     because I can't use row and column directly in the rank_one_update
     due to their indexing being in terms of A, not subA 
     I need to create some way to do this in constant time :)
  */

namespace mtl {

//: LU Factorization of a general (dense) matrix
//
// This is the outer product (a level-2 operation) form of the LU
// Factorization with pivoting algorithm . This is equivalent to
// LAPACK's dgetf2.
// <p>
// The pivot indices in ipvt are indexed starting from 1
// so that this is compatible with LAPACK (Fortran).
//
//!tparam: Matrix - A dense MTL Matrix
//!tparam: Pvector - A Vector with integral element type
//!category: algorithms
//!component: function
//!example: lu_factorization.cc
template <class Matrix, class Pvector>
int
lu_factorize(Matrix& A, Pvector& ipvt)
{
  typedef rows_type<Matrix>::type RowMatrix;
  typedef columns_type<Matrix>::type ColumnMatrix;
  typedef triangle_view<ColumnMatrix, lower>::type Lower;
  typedef triangle_view<RowMatrix, unit_upper>::type Unit_Upper;
  typedef triangle_view<ColumnMatrix, unit_lower>::type Unit_Lower;
  typedef typename Matrix::value_type T;
  typedef typename Matrix::size_type sizet;
  int info = 0;
  sizet j, jp, M = A.nrows(), N = A.ncols();

  Lower D(columns(A));
  Unit_Upper U(rows(A));
  Unit_Lower L(columns(A));
  dense1D<T> c(M), r(N);
  typename Matrix::submatrix_type subA;
  
  typename Lower::iterator dcoli = D.begin();
  typename Unit_Upper::iterator rowi = U.begin();
  typename Unit_Lower::iterator columni = L.begin();
  
  for (j = 0; j < MTL_MIN(M, N); ++j, ++dcoli, ++rowi, ++columni) {

    jp = max_index(*dcoli);		   /* find pivot */
    ipvt[j] = jp + 1;

    if ( A(jp, j) != T(0) ) {		   /* make sure pivot isn't zero */
      if (jp != j)
	swap(rows(A)[j], rows(A)[jp]);	   /* swap the rows */
      if (j < M - 1)
	scale(*columni, T(1) / A(j,j));   /* update column under the pivot */
    } else
      info = j + 1;

    if (j < MTL_MIN(M - 1, N - 1)) {
      subA = A.sub_matrix(j+1, M, j+1, N);
      copy(*columni, c);  copy(*rowi, r);  /* translate to submatrix coords */
      rank_one_update(subA, scaled(c, T(-1)), r); /* update the submatrix */
    }
  }
  ipvt[j] = j + 1;

  return info;
}


} /* namespace mtl */

#endif
