// -*- c++ -*-
//
// $COPYRIGHT$
//
//===========================================================================

#ifndef _MTL_ONED_PART_
#define _MTL_ONED_PART_

/*
  A class template that allows for flexible indexing of a 1-D part of
  a matrix, depending on what shape and storage the matrix is in.
  The Indexer class is responsible for filling in how indexing is
  suppose to work.
*/

/* #include "mtl/iterator_adaptor.h" no longer used */
#include "mtl/reverse_iter.h"
#include "mtl/scaled1D.h"

namespace mtl {


template <class Vector, class VecRef, class Indexer>
class oned_part {
  typedef typename Vector::iterator base_iterator;
  typedef typename Vector::const_iterator const_base_iterator;
  typedef oned_part<Vector, VecRef, Indexer> self;
public:
  enum { N = Vector::N };

  /***** associated types ******/
  typedef typename Vector::value_type value_type;
  typedef typename Vector::reference reference;
  typedef typename Vector::const_reference const_reference;
  typedef typename Vector::pointer pointer;
  typedef typename Vector::size_type size_type;
  typedef typename Vector::difference_type difference_type;
  typedef typename Vector::subrange subrange;

  typedef typename Vector::sparsity sparsity;
  typedef typename Vector::IndexArrayRef IndexArrayRef;
  typedef typename Vector::IndexArray IndexArray;
  typedef typename Vector::dimension dimension;
  typedef scaled1D< self > scaled_type;

  /***** iterator classes *****/  

  /* Bummer, KCC doesn't light-weight object optimize this
  class iterator : public iterator_adaptor<iterator, base_iterator>
  { 
    typedef iterator_adaptor<iterator, base_iterator> super;
  public:
    inline iterator() : super(*this) { }
    inline iterator(base_iterator i, Indexer ind) 
      : indexer(ind), super(*this, i) { }
    inline iterator(const iterator& x)
      : indexer(x.indexer), super(*this, x.iter) { }
    inline iterator& operator=(const iterator& x) {
      indexer = x.indexer; me = this; super::operator=(x); return *this; }
    inline size_type row() const { return indexer.row(super(*this)); }
    inline size_type column() const { return indexer.column(super(*this)); }
    inline size_type index() const { return indexer.minor(super(*this)); }
  protected:
    Indexer indexer;
  };
  */
  class iterator { 
    typedef iterator self;
    typedef base_iterator Iterator;
  public:

    typedef typename std::iterator_traits<Iterator>::difference_type
                               difference_type;
    typedef typename std::iterator_traits<Iterator>::value_type value_type;
    typedef typename std::iterator_traits<Iterator>::iterator_category 
                               iterator_category;
    typedef typename std::iterator_traits<Iterator>::pointer pointer; 
    typedef typename std::iterator_traits<Iterator>::reference reference;

    typedef difference_type Distance;
    typedef Iterator iterator_type;

    inline iterator() { }

    inline iterator(const Iterator& x, Indexer ind)
      : iter(x), indexer(ind) { }

    inline iterator(const self& x)
      : iter(x.iter), indexer(x.indexer) { }

    inline self& operator=(const self& x) { 
      iter = x.iter; indexer = x.indexer; return *this; 
    }

    inline operator Iterator() { return iter; }

    inline Iterator base() const { return iter; }

    inline reference operator*() const { return *iter; }

    inline self& operator++() { ++iter; return *this; }

    inline self operator++(int) { 
      self tmp = (*this);
      ++(*this); 
      return tmp; 
    }
    
    inline self& operator--() { --iter; return *this; }

    inline self operator--(int) { 
      self tmp = (*this);
      --(*this); 
      return tmp; 
    }
    
    inline self operator+(Distance n) const {
      self tmp = (*this);
      tmp += n;
      return tmp;
    }

    inline self& operator+=(Distance n) {
      iter += n; return (*this); 
    }

    inline self operator-(Distance n) const {
      self tmp = (*this); 
      tmp -= n;
      return tmp;
    }

    inline self& operator-=(Distance n) { 
      iter -= n; return (*this); 
    }

    inline value_type operator[](Distance n) const {
      self tmp = (*this);
      return *(tmp += n);
    }
    
    inline Distance operator-(const self& y) const {
      return iter - y.iter;
    }
    
    inline bool operator==(const self& y) const {
      return iter == y.iter; 
    }
    
    inline bool operator!=(const self& y) const {
      return iter != y.iter; 
    }
    
    inline bool operator<(const self& y) const {
      return iter < y.iter; 
    }  
    
    inline size_type row() const { return indexer.row(iter); }
    inline size_type column() const { return indexer.column(iter); }
    inline size_type index() const { return indexer.minor(iter); }
  protected:
    Iterator iter;
    Indexer indexer;
  };

  /* Bummer, KCC doesn't light-weight object optimize this
  class const_iterator : public iterator_adaptor<const_iterator,
                                                 const_base_iterator> 
  {
    typedef iterator_adaptor<const_iterator, const_base_iterator> super;
  public:
    inline const_iterator() : super(*this) { }
    inline const_iterator(const_base_iterator i, Indexer ind) 
      : indexer(ind), super(*this, i) { }
    inline const_iterator(const const_iterator& x)
      : indexer(x.indexer), super(*this, x.iter) { }
    inline const_iterator& operator=(const const_iterator& x) {
      indexer = x.indexer; me = this; super::operator=(x); return *this; }
    inline size_type row() const { return indexer.row(super(*this)); }
    inline size_type column() const { return indexer.column(super(*this)); }
    inline size_type index() const { return indexer.minor(super(*this)); }
  protected:
    Indexer indexer;
  };
  */

  class const_iterator { 
    typedef const_iterator self;
    typedef const_base_iterator Iterator;
  public:

    typedef typename std::iterator_traits<Iterator>::difference_type
                               difference_type;
    typedef typename std::iterator_traits<Iterator>::value_type value_type;
    typedef typename std::iterator_traits<Iterator>::iterator_category 
                               iterator_category;
    typedef typename std::iterator_traits<Iterator>::pointer pointer; 
    typedef typename std::iterator_traits<Iterator>::reference reference;

    typedef difference_type Distance;
    typedef Iterator iterator_type;

    inline const_iterator() { }

    inline const_iterator(const Iterator& x, Indexer ind)
      : iter(x), indexer(ind) { }

    inline const_iterator(const self& x)
      : iter(x.iter), indexer(x.indexer) { }

    inline self& operator=(const self& x) { 
      iter = x.iter; indexer = x.indexer; return *this; 
    }

    inline operator Iterator() { return iter; }

    inline Iterator base() const { return iter; }

    inline reference operator*() const { return *iter; }

    inline self& operator++() { ++iter; return *this; }

    inline self operator++(int) { 
      self tmp = (*this);
      ++(*this); 
      return tmp; 
    }
    
    inline self& operator--() { --iter; return *this; }

    inline self operator--(int) { 
      self tmp = (*this);
      --(*this); 
      return tmp; 
    }
    
    inline self operator+(Distance n) const {
      self tmp = (*this);
      tmp += n;
      return tmp;
    }

    inline self& operator+=(Distance n) {
      iter += n;
      return (*this); 
    }

    inline self operator-(Distance n) const {
      self tmp = (*this); 
      tmp -= n;
      return tmp;
    }

    inline self& operator-=(Distance n) { 
      std::advance(iter, -n);
      return (*this); 
    }

    inline value_type operator[](Distance n) const {
      self tmp = (*this);
      return *(tmp += n);
    }

    inline Distance operator-(const self& y) const {
      return iter - y.iter;
    }

    inline bool operator==(const self& y) const {
      return iter == y.iter; 
    }
    
    inline bool operator!=(const self& y) const {
      return iter != y.iter; 
    }
    
    inline bool operator<(const self& y) const {
      return iter < y.iter; 
    }  
    
    inline size_type row() const { return indexer.row(iter); }
    inline size_type column() const { return indexer.column(iter); }
    inline size_type index() const { return indexer.minor(iter); }
  protected:
    Iterator iter;
    Indexer indexer;
  };

  typedef reverse_iter< iterator > reverse_iterator;
  typedef reverse_iter< const_iterator > const_reverse_iterator;

  /***** oned_part methods *****/

  inline oned_part(VecRef v, Indexer ind) : vec(v), indexer(ind) { }

  inline oned_part(const oned_part& x) : vec(x.vec), indexer(x.indexer) { }

  template <class OtherOneD>
  inline oned_part(const OtherOneD& x) : vec(x.vec), indexer(x.indexer) { }

  inline oned_part& operator=(const oned_part& x) {
    vec = x.vec; indexer = x.indexer; return *this;
  }

  template <class OtherOneD>
  inline oned_part& operator=(const OtherOneD& x) {
    vec = x.vec; indexer = x.indexer; return *this;
  }
  
  inline iterator begin() { /* JGS possibly don't need indexer::begin() */
    return iterator(indexer.begin(vec.begin()), indexer);
  }
  inline iterator end() {
    return iterator(vec.end(), indexer);
  }
  inline const_iterator begin() const {
    return const_iterator(indexer.begin(vec.begin()), indexer);
  }
  inline const_iterator end() const {
    return const_iterator(vec.end(), indexer);
  }

  inline reverse_iterator rbegin() { return reverse_iterator(end()); }
  inline reverse_iterator rend() { return reverse_iterator(begin()); }

  inline const_reverse_iterator rbegin() const {
    return const_reverse_iterator(end()); 
  }
  inline const_reverse_iterator rend() const {
    return const_reverse_iterator(begin()); 
  }
  
  inline reference operator[](size_type n) {
    return vec[indexer.at(n)]; 
  }
  inline const_reference operator[](size_type n) const { 
    return vec[indexer.at(n)]; 
  }

  inline subrange operator()(size_type i, size_type j) {
    return vec(indexer.at(i), indexer.at(j)); 
  }

  inline size_type size() const { return vec.size(); }

  inline size_type nnz() const { return vec.nnz(); }

  inline IndexArrayRef nz_struct() const { return vec.nz_struct(); }

  inline void adjust_index(size_type i) { vec.adjust_index(i); }
  
  /* protected:*/
  VecRef vec;
  Indexer indexer;
};

} /* namespace mtl */
#endif
