//
// $COPYRIGHT$
//                                                                           
//
//===========================================================================

#ifndef MTL_SPARSE1D_H
#define MTL_SPARSE1D_H

#include <algorithm>
#include <set>

#include "mtl/config.h"
#include "mtl/entry.h"
#include "mtl/sparse_iterator.h"
#include "mtl/reverse_iter.h"
#include "mtl/not_at.h"
#include "mtl/utils.h"
#include "mtl/matrix_traits.h"
#include "mtl/scaled1D.h"
#include "mtl/refcnt_ptr.h"

namespace mtl {

using std::lower_bound;
using std::find;


//: class for implementing a view into just the index array of a sparse
//   row/column
//!noindex:
template <class RepType>
class sp1D_index_array {
  typedef sp1D_index_array<RepType> self;
public:
  typedef int value_type;
  typedef int* pointer;
  typedef const int& reference;
  typedef const int& const_reference;

  typedef typename RepType::const_iterator rep_iter;

  /* JGS SGI compiler doesn't like this */
  class iterator : public RepType::const_iterator {
  public:
    typedef const int& reference;
    inline iterator() { }
    inline iterator(rep_iter i) : rep_iter(i) { }
    inline reference operator*() const { return (*this)->index; }
    inline reference operator[](int n) const { return ((*this)+n)->index; }

    /* if I don't implement these the return type is
       the superclass with egcs    
     */
    inline iterator operator+(int n) {
      return iterator(rep_iter::operator+(n));
    }
    inline iterator operator-(int n) {
      return iterator(rep_iter::operator-(n));
    }
  };

  typedef iterator const_iterator;

  typedef reverse_iter<iterator> reverse_iterator;
  typedef reverse_iter<const_iterator> const_reverse_iterator;

  sp1D_index_array(rep_iter s, rep_iter f) : start(s), finish(f) { }

  sp1D_index_array(const self& x) : start(x.start), finish(x.finish) { }

  inline const_iterator begin() { return start; }
  inline const_iterator end() { return finish; }
  inline const_reverse_iterator rbegin() {
    return const_reverse_iterator(finish); }
  inline const_reverse_iterator rend() {
    return const_reverse_iterator(start); }
  inline reference operator[](int n) {
    return start[n];
  }

  /* JGS 
    protected: 
  */

  rep_iter start;
  rep_iter finish;
};


//: Sparse 1-D Container Adaptor
//
// This is a sparse vector implementation that can use several
// different underlying containers, including <tt>std::vector</tt>,
// <tt>std::list</tt>, and <tt>std::set</tt>.
// <p>
// This adaptor is used in the implementation of the linked_list,
// tree, and sparse_pair OneD storage types (used with the array
// matrix storage type). This adaptor can also be used as a stand-alone
// Vector.
// <p>
// The value_type of the underlying containers must be entry1, which
// is just an index-value pair.
// <p>
// The elements are ordered by their index as they are inserted.
//
//!category: containers, adaptors
//!component: type
//!definition: sparse1D.h
//!tparam: RepType - The Container type used to store the index value pairs.
//!typereqs: The <TT>value_type</TT> of RepType must be of type <TT>entry1</TT>
//!models: ContainerRef?
//!example: gather_scatter.cc
template <class RepType>
class sparse1D {
  typedef sparse1D<RepType> self;
  typedef RepType rep_type;
  typedef rep_type ref_cont;
  typedef typename rep_type::iterator rep_type_iterator;
  typedef typename rep_type::const_iterator const_rep_type_iterator;
public:
  /**@name Type Definitions */

  enum { N = 0 };
  
  //: This is a sparse vector
  typedef sparse_tag sparsity;
  //: The index-value pair type
  typedef typename RepType::value_type entry_type;
  //: The value type
  typedef typename entry_type::PR PR;

  //: This is a 1D container
  typedef oned_tag dimension;

  //: The scaled type 
  typedef scaled1D< sparse1D< RepType > > scaled_type;
  //: The value type
  typedef typename entry_type::PR value_type;
  //: The type for pointers to the value type
  typedef value_type* pointer;
  //: The unsigned integral type for dimensions and indices
  typedef typename RepType::size_type size_type;
  //: The type for differences between iterators
  typedef typename RepType::difference_type difference_type;

  typedef entry1<PR> entry;
  //: The type for references to the value type
  typedef elt_ref<self> reference;
  //: The type for const references to the value type
  typedef PR const_reference;
  //: The iterator type
  typedef sparse_iterator<rep_type_iterator,value_type> iterator;
  //: The const iterator type
  typedef const_sparse_iterator<const_rep_type_iterator,value_type>
          const_iterator;
  //: The reverse iterator type
  typedef reverse_iter<iterator> reverse_iterator;
  //: The const reverse iterator type
  typedef reverse_iter<const_iterator> const_reverse_iterator;

  //: The type for the index array
  typedef sp1D_index_array<RepType> IndexArray;

  //: The reference type for the index array
  typedef sp1D_index_array<RepType> IndexArrayRef;

  //: The type for subrange vectors
  typedef self subrange;

  friend class elt_ref<self>;
  friend class const_elt_ref<self>;

protected:

  inline iterator __find(int i, std::set<entry_type>*) {
    rep_type_iterator pos = rep->lower_bound(entry1<PR>(i));
    return iterator(pos);
  }

  template <class R>
  inline iterator __find(int i, R*) {
    rep_type_iterator pos = lower_bound(rep->begin(), rep->end(),
					entry1<PR>(i));
    return iterator(rep->begin(), pos);
  }

  inline iterator find(int i) {
    return __find(i, rep.operator->());
  }

  /* const versions of find */

  inline const_iterator __find(int i, std::set<entry_type>*) const {
    const_rep_type_iterator pos = rep->lower_bound(entry1<PR>(i));
    return const_iterator(pos);
  }

  template <class R>
  inline const_iterator __find(int i, R*) const {
    const_rep_type_iterator pos = lower_bound(rep->begin(), rep->end(),
					entry1<PR>(i));
    return const_iterator(rep->begin(), pos);
  }

  inline const_iterator find(int i) const {
    return __find(i, rep.operator->());
  }

  inline iterator insert(iterator iter, int i, PR v) {
    rep_type_iterator pos = rep->insert(iter.base(), entry1<PR>(i, v));
    return iterator(rep->begin(), pos);
  }
  

public:

  /**@name Constructors */
  
  //: Default Constructor
  inline sparse1D() 
    : rep(new ref_cont()) { }

  //: Length N Constructor
  inline sparse1D(int n)
    : rep(new ref_cont()) { }

  //: Copy Constructor
  inline sparse1D(const self& x) 
    : rep(x.rep) { }

  //: Construct from index array
  template <class IndexArray>
  inline sparse1D(const IndexArray& x) : rep(new ref_cont()) {
      typename IndexArray::const_iterator i;
      for (i = x.begin(); i != x.end(); ++i)
	rep->push_back(entry_type(*i, PR(0)));
  }

  //: Assignment Operator
  inline self& operator=(const self& x) {
    rep = x.rep; return *this;
  }

  /**@name Access Methods */
  

  /**@name  Iterator Access Methods */
  
  //: Return an iterator pointing to the beginning of the vector
  //!wheredef: Container
  inline iterator begin() {
    return iterator(rep->begin()); 
  }
  //: Return an iterator pointing past the end of the vector
  //!wheredef: Container
  inline iterator end() {
    return iterator(rep->begin(), rep->end());
  }
  //: Return a const iterator pointing to the begining of the vector
  //!wheredef: Container
  inline const_iterator begin() const {
    return const_iterator(rep->begin()); 
  }
  //: Return a const iterator pointing past the end of the vector
  //!wheredef: Container
  inline const_iterator end() const {
    return const_iterator(rep->begin(), rep->end());
  }
  //: Return a reverse iterator pointing to the last element of the vector
  //!wheredef: Reversible Container
  inline reverse_iterator rbegin() {
    return reverse_iterator(end()); 
  }
  //: Return a reverse iterator pointing past the end of the vector
  //!wheredef: Reversible Container
  inline reverse_iterator rend() {
    return reverse_iterator(begin());
  }
  //: Return a const reverse iterator pointing to the last element of the vector
  //!wheredef: Reversible Container
  inline const_reverse_iterator rbegin() const {
    return const_reverse_iterator(end());
  }
  //: Return a const reverse iterator pointing past the end of the vector
  //!wheredef: Reversible Container
  inline const_reverse_iterator rend() const {
    return const_reverse_iterator(begin());
  }
  

  /**@name Element Access Methods */
  
  //: Element Access, return element with index i
  inline const_reference operator[](int i) const {
    const_iterator iter = find(i);
    if (iter != end() && iter.index() == i)
      return *iter;
    else
      return PR(0);
  }

  //: Element Access, return element with index i
  inline reference operator[](int i) {
    return reference(*this, i);
  }
  

  //: Insert the value at index i of the vector
  inline iterator insert(int i, const PR& value) {
    rep_type_iterator pos = lower_bound(rep->begin(), rep->end(), 
					 entry1<PR>(i));
    return rep->insert(pos, entry1<PR>(i, value));
  }

  /**@name Size Methods */
  
  //: Returns size of the vector
  inline int size() const {
    return rep->size();
  }
  //: Number of non-zero (stored) elements 
  inline int nnz() const {
    return rep->size();
  }

  inline void resize_imp(int n, std::set<entry_type>*) { }

  template <class R>
  inline void resize_imp(int n, R*) { 
    rep->resize(n);

  }

  //: Resizes the vector to size n
  inline void resize(int n) { 
    resize_imp(n, rep.operator->());
  }

  
  rep_type& get_rep() { return *rep; }

  
  inline void print() const {
    print_vector(*rep);
  }
  //: Return an array of indices describing the non-zero structure
  inline IndexArrayRef nz_struct() const { 
    return IndexArrayRef(rep->begin(), rep->end());
  }

protected:

  refcnt_ptr<rep_type> rep;

};

} /* namespace mtl */

#endif









  /* obsolete, not used anymore 
  inline void set(int i, PR val) {
    rep_type_iterator pos = lower_bound(rep->begin(), rep->end(),
					entry1<PR>(i));
    if (pos != rep->end())
      if ((*pos).index != i)
	rep->insert(pos, entry1<PR>(i, val));
      else
	(*pos).value = val;
    else
      rep->insert(pos, entry1<PR>(i, val));
  }

  inline PR get(int i) {
    PR ret;
    const_rep_type_iterator pos = std::find(rep->begin(), rep->end(), 
					    entry1<PR>(i));
    if (pos != rep->end())
      ret = (*pos).value;
    else
      ret = 0.0;
    return ret;
  }
  */
