//
// $COPYRIGHT$
//
//==================================================================

#ifndef MTL_SPARSE_ITERATOR_H
#define MTL_SPARSE_ITERATOR_H

#include "mtl/mtl_iterator.h"

namespace mtl {

using std::random_access_iterator_tag;
using std::iterator_traits;
using std::output_iterator_tag;
using std::advance;
using std::distance;

template <class Iterator, class T>
class const_sparse_iterator;

//: Sparse Vector Iterator
//
// This iterators is used to implement the sparse1D adaptor.  The base
// iterator returns a entry1 (an index-value pair) and this iterator
// makes it look like we are just dealing with the value for
// dereference, while the index() method return the index.
//
//!category: iterators, adaptors
//!component: type
//!tparam: Iterator - the underlying iterator type
//!tparam: T - the value type
template <class Iterator, class T>
class sparse_iterator {
  typedef sparse_iterator<Iterator,T> self;
  friend class const_sparse_iterator<Iterator,T>;
  typedef typename iterator_traits<Iterator>::iterator_category cat;
public:
  typedef T PR;
  //: The iterator category
  typedef cat iterator_category;
  //: The value type
  typedef PR value_type;
  //: The type for differences between iterators
  typedef int difference_type;
  //: The type for references to value type
  typedef PR& reference;
  //: The type for pointers to value type
  typedef PR* pointer;

public:
  //: Default Constructor
  inline sparse_iterator() { }
  //: Copy Constructor
  inline sparse_iterator(const sparse_iterator& x)
    : iter(x.iter), pos(x.pos) { }
  //: Constructor from underlying iterator
  inline sparse_iterator(const Iterator& iter_, int p = 0)
    : iter(iter_), pos(p) { }

  inline sparse_iterator(const Iterator& start, const Iterator& finish)
  { init(start, finish, cat()); }

  //: Assignment Operator
  inline self& operator=(const self& x) {
    iter = x.iter;
    pos = x.pos;
    return *this;
  }

  inline operator Iterator() { return convert(cat()); }

  //: Return the index of the element pointed to by this iterator
  //!wheredef: IndexedIterator
  inline int index() const { return index(cat()); }
  //: Return whether this iterator is not equal to iterator x
  inline bool operator!=(const self& x) const { return notequal(x, cat()); }
  //: Return whether this iterator is less than iterator x
  inline bool operator<(const self& x) const { return lessthan(x, cat()); }
  //: Deference, return the element pointed to by this iterator
  inline reference operator*() const { return deref(cat()); }
  //: Pre-increment operator
  inline self& operator++() { inc(cat()); return *this; }
  //: Post-increment operator
  inline self operator++(int) { self tmp = *this; inc(cat()); return tmp; }
  //: Pre-decrement operator
  inline self& operator--() { dec(cat()); return *this; }
  //: Add distance n to this iterator
  inline self& operator+=(int n) { add_assign(n, cat()); return *this; }
  //: Subtract distance n from this iterator
  inline self& operator-=(int n) { sub_assign(n, cat()); return *this; }
  //: Add this iterator and distance n
  inline self operator+(int n) const { return add(n, cat()); }
  //: Subtract this iterator and distance n
  inline self operator-(int n) const { return sub(n, cat()); }

  //: Return the difference between this iterator and iterator x
  inline int operator-(const self& x) const {
    return diff(x, cat());
  }

  inline Iterator base() const { return convert(cat()); }

protected:

  inline self add(int n, random_access_iterator_tag) const {
    return self(iter, pos + n);
  }
  template <class Iter>
  inline self add(int n, Iter) const {
    Iterator i = iter;
    std::advance(i, n);
    return self(i);
  }

  inline self sub(int n, random_access_iterator_tag) const {
    return self(iter, pos - n);
  }
  template <class Iter>
  inline self sub(int n, Iter) const {
    Iterator i = iter;
    std::advance(i, -n); // JGS ???
    return self(i);
  }

  inline int diff(const self& x, random_access_iterator_tag) const {
    return pos - x.pos;
  }
  template <class Iter>
  inline int diff(const self& x, Iter) const {
    return distance(iter, x.iter);
  }

  inline void init(const Iterator& start, const Iterator& finish,
		   random_access_iterator_tag) {
    iter = start;
    pos = finish - start;
  }
  template <class Iter>
  inline void init(const Iterator& start, const Iterator& finish,
		   Iter) { 
    iter = finish;
    pos = 0;
  }

  inline bool lessthan(const self& x, random_access_iterator_tag) const {
    return pos < x.pos; }
  template <class Iter>
  inline bool lessthan(const self& x, Iter) const {
    return iter < x.iter; }

  inline bool notequal(const self& x, random_access_iterator_tag) const {
    return pos != x.pos; }

  template <class Iter>
  inline bool notequal(const self& x, Iter) const {
    return iter != x.iter; }

  inline Iterator convert(random_access_iterator_tag) const {
    return iter + pos; }
  template <class Iter>
  inline Iterator convert(Iter) const {
    return iter; }

  inline int index(random_access_iterator_tag) const {
    return (*(iter + pos)).index; }

  template <class Iter>
  inline int index(Iter) const {
    return (*iter).index; }

  inline void inc(random_access_iterator_tag) { ++pos; }
  template <class Iter>
  inline void inc(Iter) { ++iter; }

  inline void dec(random_access_iterator_tag) { --pos; }
  template <class Iter>
  inline void dec(Iter) { --iter; }

  inline reference deref(random_access_iterator_tag) const { 
    return (*(iter + pos)).value; }

  template <class Iter>
  inline reference deref(Iter) const { 
    return reference((*iter).value); }

  inline void add_assign(int n, random_access_iterator_tag) {
    pos += n;
  }
  template <class Iter>
  inline void add_assign(int n, Iter) {
    advance(iter,n);
  }

  inline void sub_assign(int n, random_access_iterator_tag) {
    pos -= n;
  }
  template <class Iter>
  inline void sub_assign(int n, Iter) {
    advance(iter,n);
  }
public:
  Iterator iter;
  int pos;
};

template <class Iterator, class T>
inline bool
operator==(const sparse_iterator<Iterator, T>& a,
	   const sparse_iterator<Iterator, T>& b)
{
  return a.base() == b.base();
}




template <class Iterator, class T>
class const_sparse_iterator {
  typedef const_sparse_iterator<Iterator,T> self;
  typedef typename iterator_traits<Iterator>::iterator_category cat;
public:
  typedef T PR;
  typedef cat iterator_category;
  typedef PR value_type;
  typedef int difference_type;
  typedef const PR& reference;
  typedef PR* pointer;

public:

  inline const_sparse_iterator() { }

  inline const_sparse_iterator(const const_sparse_iterator& x)
    : iter(x.iter), pos(x.pos) { }

  inline const_sparse_iterator(const Iterator& iter_, int p = 0)
    : iter(iter_), pos(p) { }

  inline const_sparse_iterator(const Iterator& start, const Iterator& finish)
  { init(start, finish, cat()); }

  template <class Iter>
  inline const_sparse_iterator(const sparse_iterator<Iter,T>& si)
    : iter(si.iter), pos(si.pos) { }

  inline operator Iterator() { return convert(cat()); }

  inline int index() const { return index(cat()); }

  inline bool operator!=(const self& x) const { return notequal(x, cat()); }
  inline bool operator<(const self& x) const { return lessthan(x, cat()); }

  inline reference operator*() const { return deref(cat()); }

  inline int operator-(const self& x) const {
    return diff(x, cat());
  }

  inline self& operator++() { inc(cat()); return *this; }
  inline self operator++(int) { self tmp = *this; inc(cat()); return tmp; }
  inline self& operator--() { dec(cat()); return *this; }
  inline self& operator+=(int n) { add_assign(n, cat()); return *this; }
  inline self& operator-=(int n) { sub_assign(n, cat()); return *this; }
  inline Iterator base() const { return convert(cat()); }

  inline self operator+(int n) const { return add(n, cat()); }
  inline self operator-(int n) const { return sub(n, cat()); }

protected:


  inline self add(int n, random_access_iterator_tag) const {
    return self(iter, pos + n);
  }
  template <class Iter>
  inline self add(int n, Iter) const {
    Iterator i = iter;
    std::advance(i, n);
    return self(i);
  }

  inline self sub(int n, random_access_iterator_tag) const {
    return self(iter, pos - n);
  }
  template <class Iter>
  inline self sub(int n, Iter) const {
    Iterator i = iter;
    std::advance(i, -n);// JGS???
    return self(i);
  }

  inline int diff(const self& x, random_access_iterator_tag) const {
    return pos - x.pos;
  }

  template <class Iter>
  inline int diff(const self& x, Iter) const {
    return distance(iter, x.iter);
  }

  inline void init(const Iterator& start, const Iterator& finish,
		   random_access_iterator_tag) {
    iter = start;
    pos = finish - start;
  }

  template <class Iter>
  inline void init(const Iterator& start, const Iterator& finish, Iter) { 
    iter = finish;
    pos = 0;
  }

  inline bool lessthan(const self& x, random_access_iterator_tag) const {
    return pos < x.pos; }
  template <class Iter>
  inline bool lessthan(const self& x, Iter) const {
    return iter < x.iter; }

  inline bool notequal(const self& x, random_access_iterator_tag) const {
    return pos != x.pos; }
  template <class Iter>
  inline bool notequal(const self& x, Iter) const {
    return iter != x.iter; }

  inline Iterator convert(random_access_iterator_tag) const {
    return iter + pos; }
  template <class Iter>
  inline Iterator convert(Iter) const {
    return iter; }

  inline int index(random_access_iterator_tag) const {
    return (*(iter + pos)).index; }
  template <class Iter>
  inline int index(Iter) const {
    return (*iter).index; }

  inline void inc(random_access_iterator_tag) { ++pos; }
  template <class Iter>
  inline void inc(Iter) { ++iter; }

  inline void dec(random_access_iterator_tag) { --pos; }
  template <class Iter>
  inline void dec(Iter) { --iter; }

  inline reference deref(random_access_iterator_tag) const { 
    return (*(iter + pos)).value; }
  template <class Iter>
  inline reference deref(Iter) const { 
    return (*iter).value; }

  inline void add_assign(int n, random_access_iterator_tag) {
    pos += n;
  }
  template <class Iter>
  inline void add_assign(int n, Iter) {
    advance(iter,n);
  }

  inline void sub_assign(int n, random_access_iterator_tag) {
    pos -= n;
  }
  template <class Iter>
  inline void sub_assign(int n, Iter) {
    advance(iter,n);
  }

  Iterator iter;
  int pos;
};

template <class Iterator, class T>
inline bool
operator==(const const_sparse_iterator<Iterator, T>& a,
	   const const_sparse_iterator<Iterator, T>& b)
{
  return a.base() == b.base();
}





#if 0
template <class Iterator, class T>
class const_sparse_iterator {
  typedef const_sparse_iterator<Iterator,T> self;
public:
  typedef T PR;
  typedef forward_iterator_tag iterator_category;
  typedef PR value_type;
  typedef int difference_type;
  typedef const PR& reference; // JGS
  typedef const PR& const_reference;
  typedef PR* pointer;

  inline const_sparse_iterator() { }

  inline const_sparse_iterator(const const_sparse_iterator& x)
    : iter(x.iter), pos(x.pos) { }

  inline const_sparse_iterator(const Iterator& iter_, int p = 0)
    : iter(iter_), pos(p) { }

  template <class Iter>
  inline const_sparse_iterator(const sparse_iterator<Iter,T>& si)
    : iter(si.iter), pos(si.pos) { }

  inline operator const Iterator() const { return iter + pos; }

  inline int index() const {
    return (*(iter + pos)).index;
  }
  inline bool not_at(const self& x) const {
    return pos < x.pos;
  }
  inline bool operator!=(const self& x) const {
    return pos != x.pos;
  }
  inline bool operator<(const self& x) const {
    return pos < x.pos;
  }
  inline const_reference operator*() const {
    return (*(iter + pos)).value;
  }
  inline self& operator++() {
    ++pos;
    return *this;
  }
  inline self operator++(int) { self tmp = *this; ++pos; return tmp; }
  inline self& operator--() {
    --pos;
    return *this;
  }
  inline self& operator+=(int n) {
    pos += n;
    return *this;
  }
  inline self& operator-=(int n) {
    pos -= n;
    return *this;
  }
  inline Iterator base() const { return iter + pos; }

protected:
  Iterator iter;
  int pos;
};

template <class Iterator, class T>
inline bool
operator==(const const_sparse_iterator<Iterator, T>& a,
	   const const_sparse_iterator<Iterator, T>& b)
{
  return a.base() == b.base();
}
#endif


} /* namespace mtl */

#endif


