// -*- c++ -*-
//
// $COPYRIGHT$
//
//===========================================================================

#include "mtl/iterator_adaptor.h"


//: Tranforming Iterator
// This iterator adaptor applies some function during the dereference
//!category: iterators, adaptors
//!component: type
//!tparam: Iterator - The underlying iterator type
//!tparam: UnaryFunction - A function that takes one argument of value type
template <class Iterator, class UnaryFunction>
class transform_iterator
  : public iterator_adaptor< transform_iterator<Iterator,
                                                   UnaryFunction>,
                             Iterator >
{
  typedef iterator_adaptor< transform_iterator<Iterator,
                                                   UnaryFunction>,
                             Iterator > super;
public:
  /* for old broken compilers */
  //: The value type
  typedef typename super::value_type value_type;
  //: The difference type
  typedef typename super::difference_type difference_type;
  //: The iterator category
  typedef typename super::iterator_category iterator_category;
  //: The pointer type
  typedef typename super::pointer pointer;
  //: The reference type
  typedef typename super::reference reference;

  //: Normal Constructor
  inline transform_iterator(Iterator i, UnaryFunction op) 
    : super(*this, i), f(op) { }
  //: Copy Constructor
  inline transform_iterator(const transform_iterator& x)
    : super(*this, x.iter), f(x.f) { }
  //: Assignment Operator
  inline transform_iterator& operator=(const transform_iterator& x) {
    f = x.f; me = this; super::operator=(x); return *this; }
  //: Dereference Operator (applies the function here)  
  inline value_type operator*() { return f(*iter); }

protected:
  UnaryFunction f;
};

//: Helper function for creating a transforming iterator
//!category: iterators
//!component: function
template <class Iterator, class UnaryFunction> inline
transform_iterator<Iterator,UnaryFunction>
trans_iter(Iterator i, UnaryFunction op) {
  return transform_iterator<Iterator,UnaryFunction>(i, op);
}


/* this probably doesn't work, the me pointer all goofed up */
template <class Iterator, class T>
class scale_iterator 
  : public transform_iterator<Iterator, 
                                 std::binder1st< std::multiplies<T> > >
{
  typedef transform_iterator<Iterator,
                                std::binder1st< std::multiplies<T> > >
      super;
public:
  inline scale_iterator(Iterator i, T a) 
    : super(i, bind1st(multiplies<T>(),a)) { }
};



template <class RandomAccessIterator>
class constant_stride_generator {
  typedef constant_stride_generator<RandomAccessIterator> self;
  typedef std::iterator_traits<RandomAccessIterator>::difference_type
     Distance;
public:
  inline constant_stride_generator(Distance s) : stride_(s) { }
  inline constant_stride_generator(const self& x) : stride_(x.stride_) { }
  inline void inc(RandomAccessIterator& i) { i += stride_; }
  inline void dec(RandomAccessIterator& i) { i -= stride_; }
  inline void advance(RandomAccessIterator& i, int n) { i += n * stride_; }
  /* JGS this interfance to diff may need to be changed */
  inline Distance diff(const RandomAccessIterator& x,
		const RandomAccessIterator& y,
		const self& /* y_stride_gen */) const {
    return (x - y) / stride_;
  }
private:
  Distance stride_;
};

/*
  StrideGen is a stride generator!
  The simple version just creates constant strides
 */
template <class RandomAccessIterator, class StrideGen>
class general_stride_iterator 
  : public iterator_adaptor< general_stride_iterator<RandomAccessIterator,
                                                      StrideGen>,
			     RandomAccessIterator >
{
  typedef general_stride_iterator<RandomAccessIterator, StrideGen> self;
  typedef iterator_adaptor< self, RandomAccessIterator> super;
public:
  inline general_stride_iterator(RandomAccessIterator i, StrideGen s) 
    : super(*this, i), stride_gen(s) { }
  inline self& operator++() {
    stride_gen.inc(iter);
    return *this; 
  }
  inline self& operator--() { 
    stride_gen.dec(iter);
    return *this; 
  }
  inline self& operator+=(Distance n) { 
    stride_gen.advance(iter, n);
    return *this; 
  }
  inline self& operator-=(Distance n) {
    stride_gen.advance(iter, -n);
    return *this; 
  }

  inline Distance diff(const self& y) const {
    return stride_gen.diff(iter, y.iter, y.stride_gen);
  }

  inline friend Distance operator-(const self& x, const self& y) {
    return x.diff(y);
  }

protected:
  StrideGen stride_gen;
};


template <class RandomAccessIterator>
class stride_iterator 
  : public general_stride_iterator<RandomAccessIterator,
                        constant_stride_generator<RandomAccessIterator> >

{
  typedef general_stride_iterator<RandomAccessIterator,
                     constant_stride_generator<RandomAccessIterator> > super;
public:
  inline stride_iterator(RandomAccessIterator i, difference_type s)
    : super(i, constant_stride_generator<RandomAccessIterator>(s)) { }
};

