/* qop.h

Several quantum gates. (one bit and multi-controlled)

This file is part of the OpenQubit project.

Copyright (C) 1998-1999 OpenQubit.org
Yan Pritzker <skwp@thepentagon.com>

Please see the CREDITS file for contributors.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

//! author="Rafal Podeszwa, Christopher Dawson, Peter Belkner, Yan Pritzker"
//! lib="Quantum Operators and Gates"

#ifndef _QOP_H_
#define _QOP_H_

#include <math.h>
#include "utility.h"
#include "qstate.h"

class Q_Gate
{
public:
	Q_Gate(	const Complex& alpha, const Complex& beta,
		const Complex& delta, const Complex& theta ) :
		_a00(alpha), _a01(beta), _a10(delta), _a11(theta) {};

	void ApplyTo(Q_BitSet &q, int bit);

protected:
	Complex _a00, _a01, _a10, _a11;
};

class Gates
{
public:
	static Q_Gate Unitary(double, double, double, double);
	static Q_Gate RotQubit(double);
	static Q_Gate RotPhase(double);
	static Q_Gate PhaseShift(double);
	static Q_Gate Hadamard();
	static Q_Gate Not();
};

/*
class opFFT 
//: Fast Fourier Transform
{
public:
	opFFT() {};
	void operator() (Q_BitSet &q, int numbits=-1);
};

class opSPhaseShift 
//: Shor's phase shift operator (double-controlled phase shift);
{
// the matrix used for controlled single bit operation is
// (1             0             )
// (0   exp(i*Pi*delta/2^(k-j)) )

private:
	opUnitary<Q_GateMatrix::Controlled> CU;
public:
	opSPhaseShift() {};
	
	void operator() (Q_BitSet &q, int j, int k)
	{
		assert(j < k);    //see Shor's paper
		double delta = M_PI/(1 << (k-j));
		unsigned long mask;
		
		mask = (1 << j);
		
		//call general unitary controlled gate as suggested by Rafal
		CU.Param(delta,0,-delta/2,0);
		CU(q,mask,k);
	}
};

class ModExp
//: Modular Exponentiation
{
public:
	
	ModExp() {};
	void operator() (Q_BitSet &q, int a, int n, int b=-1) {
		//results go here first
		vector<Complex> _qArrayTmp(Q_StateVector::States()); 
		if(b==-1) b = q.Qubits();

		for(int i = 0; i < Q_StateVector::States(); i++)
			{
				if (Q_StateVector::Coef(i) != 0)
					_qArrayTmp[i + (modexp(a,i,n) << b)] = Q_StateVector::Coef(i);
			}

		for (int k = 0; k < Q_StateVector::States(); k++)
			Q_StateVector::Coef(k) = _qArrayTmp[k]; //copy results
	}
};

template <class OperatorType>
class DoAllBits
//: This class allows any operator to work on all bits of a state.
// This will only work for single bit operators requiring no parameters
// (i.e. this will not work with RotQubit which requires an angle theta)
{
private:
	OperatorType op;

public:
        void operator() (Q_BitSet &q) {
                for(int i=0; i < q.Qubits(); ++i) 
                        op(q,i);
	}
};

//: Unitary Operator
typedef opUnitary<Q_GateMatrix::SingleBit> 		Unitary;		

//: Controlled Unitary Operator
typedef opUnitary<Q_GateMatrix::Controlled>		CUnitary;

//: Qubit Rotation (Ry)
typedef opRotQubit<Q_GateMatrix::SingleBit>		RotQubit;

//: Controlled Qubit Rotation (Ry)
typedef opRotQubit<Q_GateMatrix::Controlled>	CRotQubit;

//: Phase Rotation (Rz)
typedef opRotPhase<Q_GateMatrix::SingleBit>		RotPhase;

//: Controlled Phase Rotation
typedef opRotPhase<Q_GateMatrix::Controlled>	CRotPhase;

//: Phase Shift (scalar multiplication of z axis)
typedef opPhaseShift<Q_GateMatrix::SingleBit>	PhaseShift;

//: Controlled Phase Shift
typedef opPhaseShift<Q_GateMatrix::Controlled>	CPhaseShift;

//: Shor Phase Shift
typedef opSPhaseShift				SPhaseShift;

//: Hadamard Operator
typedef opHadamard<Q_GateMatrix::SingleBit>		Hadamard;

//: Negation Operator
typedef opNOT<Q_GateMatrix::SingleBit>			Not;

//: Controlled Negation (CNot - a.k.a XOR)
typedef opNOT<Q_GateMatrix::Controlled>			CNot;

//: Fast Fourier Transform
typedef opFFT							FFT;

//: Hadamard on all bits == WalshHadamard
typedef DoAllBits<Hadamard> 		WalshHadamard;

*/
#endif
