/* qstate.h

Model of Quantum States and Registers.

This file is part of the OpenQubit project.

Copyright (C) 1998-1999 OpenQubit.org
Yan Pritzker <skwp@thepentagon.com>

Please see the CREDITS file for contributors.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

//! author="Yan Pritzker, Rafal Podeszwa, Christopher Dawson, Peter Belkner"
//! minor="Joe Nelson"
//! lib="Quantum State/Register [OpenQubit Core]"

#ifndef _QSTATE_H_
#define _QSTATE_H_

#include <iostream.h>
#include <vector>
#include <map>
#include <algo.h>

#include "complex.h"
#include "random.h"
#include "utility.h"
#include "debug.h"
#include "exceptions.h"

#define _RNGT_	double							
#define _RNG_ 	DblUniformRandGenerator		

#include "mtl/mtl.h"
#include "mtl/compressed1D.h"

using namespace mtl;

typedef compressed1D<Complex> amplitude_vector_t;
typedef amplitude_vector_t::iterator amp_iter;
typedef amplitude_vector_t::const_iterator amp_citer;

class Q_StateVector
//: Quantum State of machine including memory management facilities.
// This is the "Quantum Operating System"  This class will manage
// memory for the declaration of quantum registers. The Q_BitSet
// class will allow for relative indexing (e.g. bit 0 in a register
// may be bit 7 of the entire state). 
{
friend class Q_BitSet;
friend class Q_Gate;
public:

	//: Constructor to create state with specified number of bits.
	Q_StateVector(int nbits, int seedval=-1);

	//: Dump state to file.
	void Save(char[]);

	//: Read state from file.
	void Read(char[]);
	
	//: Display outcomes by coefficients.
	friend ostream& operator<<(ostream&, const Q_StateVector& V);
	
	//: Get number of total states.
	int States() const { return _nStates; }

	//: Get number of total qubits.
	int Qubits() const { return _nQubits; }

	//: Get Amplitude for viewing.
	Complex Amplitude(int) const throw (Q_Exception::RangeErr);

	//: Allocate a bit set of specified size.
	Q_BitSet* getBitSet(int) throw (Q_Exception::MallocErr);	

	//: Allocate a bit set of arbitrarily ordered bits.
	Q_BitSet* getBitSet(const vector<int>&) throw (Q_Exception::RangeErr);

private:
	amplitude_vector_t _AmplitudeVector;
	int _nQubits, _nStates;
	int _LastAllocated;
	
	//: Full reset.
	void _Reset() { 
		_AmplitudeVector.resize(0); 
		_AmplitudeVector[0]=Complex(1); 
	}

	//: Destructive measure of one bit
	int _Collapse(int) throw (Q_Exception::RangeErr);

	//: Collapse set specified by range
	int _Collapse(int,int) throw (Q_Exception::RangeErr);
	
	//: Get amplitude for modification.
	Complex& _Amplitude(int) throw (Q_Exception::RangeErr);
	
	//: Sum of normalized complex amplitudes.
	double _NormalizedSum();
	
	//: Random Number Generator
	static RandGenerator<_RNGT_> *_RNG;
};

class Q_BitSet
//: Quantum register, a reference to a part of the Q_StateVector.
{
friend class Q_Gate;
private:
	Q_StateVector *QSV;	//: Pointer to state this bitset belongs to.
	vector<int> _Bitset;	//: Which bits are represented.
	int _nQubits;		//: Size of register

public:
	//: Default constructor. For creation of registers to be assigned later.
	Q_BitSet() {};	

	//: Create a register addressing bits  from s to e.
	Q_BitSet(Q_StateVector *Q, int s, int e);

	//: Create a register with specified arbitrarily ordered qubits.
	Q_BitSet(Q_StateVector *Q, const vector<int>&);	

	//: Find out which qubit is represented by this qubit number.
	int getStateVectorBitNum(int i) { return _Bitset[i]; }

	//: Look at amplitude without modifying.
	Complex getAmplitude(int i) { return QSV->Amplitude(i); }
    
	//: Destructive bit measure.
	int Measure(int);

	//: Destructive measure of entire bit set.
	int Measure();

	//: Destructive measure of a range in bit set.
	int Measure(int,int);

	//: Number of Qubits.
	int numQubits() const { return _nQubits; }

	//: Number of possible states.
	int numStates() const { return 1 << _nQubits; }    
};

#endif
