/* qstate.cc

Implementation of Data Model of a Quantum State/Register
This file is part of the OpenQubit project.

Copyright (C) 1998-1999 OpenQubit.org
Yan Pritzker <skwp@thepentagon.com>

Please see the CREDITS file for contributors.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
#include "qstate.h"

extern int QState::_nQubits;
extern int QState::_nStates;
extern int QState::_nUsed;
extern array_t QState::_qArray;
extern RandGenerator<_RNGT_> *QState::_RNG;

int QState::_CollapseBit(int index) throw (QException::RangeErr)
//single qubit collapse
{
	//index is 0..nQubits-1, 0 being the most significant qubit,
	//e.g. in |0001>, the 1 is the least significant qubit and
	//its position is 0
	if (index < 0 || index > _nQubits-1) throw QException::RangeErr();

	double p0,p1;	//probabilities of measuring this bit as 0 and 1
	p0=p1=0.0;
	
	for(int i=0; i < _nStates; ++i) 
		if (IsBitSet(i,index))
			p1 += norm(_qArray[i]);
		else
			p0 += norm(_qArray[i]);

	D("Got probabilities...p0=%1.3f, p1=%1.3f\n", p0, p1);
   D("Sum of probabilities: %1.5f; Normalized amplitudes: %1.5f\n",
		 p0+p1, _Normalize());
	
	//sum of normalized amplitudes and sum of added probabilities
	//should equal..if not we messed up somewhere in the addition/normalization
	D("Normalized Sum + ROUND_ERR = %1.25f\n", _Normalize() + ROUND_ERR);
	D("norm(*this) - ROUND_ERR = %1.25f\n", _Normalize() - ROUND_ERR);
	D("p0+p1 should be between those and it is %1.10f\n", p0+p1);
	
	double rnd=_RNG->GetRandBetween(0,_Normalize());
	bool on  = p0 < rnd;
	
	for (int i=0;i < _nStates; i++) 
		if (IsBitSet(i,index))
			if(!on) _qArray[i] = 0.0;
			else _qArray[i] /= sqrt(p1);
		else
			if(on) _qArray[i] = 0.0;
			else _qArray[i] /= sqrt(p0);

	D("Set bit state to %d\n", on);
	return on;
}

int QState::_CollapseMask(unsigned long bits) throw (QException::RangeErr)
{
	int i; 
	int state=0;

	for (i = 0; i < _nQubits; i++)
		if ((1 << i) & bits) {
			state += _CollapseBit(i);
		}

	return state;
}

int QState::_CollapseRange(int begin, int end) throw (QException::RangeErr)
{
	int total=0;
	if(begin < 0 || end > _nQubits-1) throw QException::RangeErr();
	for(int i=begin; i<=end; i++) {
		total += _CollapseBit(i) * (int)pow(2,i);
	}
	return total;
}

void QState::_Reset(int s, int e) throw (QException::RangeErr)
	{ 
		if(s < 0 || e > _nQubits-1) throw QException::RangeErr();
		int k=0; for(k=s; k <= e; ++k) _qArray[k]=0; 
	}

void QState::_Reset()
	{ _Reset(0, _nQubits-1); }

void QState::Save(char filename[])
{
	FILE *FH;
	if ((FH=fopen(filename,"w"))==NULL)
		cerr << "ERROR: could not open file " << filename << endl;

	fprintf(FH,"QSTATE SIZE %d\n", _nStates);

	for(int i=0; i < _nStates; i++) {
		if(real(_qArray[i]) || imag(_qArray[i]))
			fprintf(FH, "%+1.17f \t %+1.17f \t |0x%X>\n",
					 real(_qArray[i]),
					 imag(_qArray[i]),
					 i);
	}
	fclose(FH);
}

double QState::_Normalize()
{
	double n=0.0;
	for(int i=0; i < _nStates; ++i) n += norm(_qArray[i]);
	return n;
}

void QState::Read(char filename[])
{
	FILE *FH;
	
	if((FH=fopen(filename,"r"))==NULL)
		cerr << "ERROR: could not open file " << filename << endl;

	double real,imag;
	int index;
	int size;

	fscanf(FH, "QSTATE SIZE %d\n", &size);
	assert(_nStates==size);
	
	while(FH)
	{
		int r=fscanf(FH,"%+1.17f \t %+1.17f \t |0x%X>\n", 
					 	 &real, &imag, &index);

		if(r==0) break;

		D("Scanned %f %f %d\n",real,imag,index);
		_qArray[index] = Complex(real,imag);
	}

	fclose(FH);
	D("Error: %f",_Normalize());
	assert( ApproxEqual(_Normalize(),1) );
}

int QState::_Allocate(int numbits) throw (QException::MallocErr) {
	int newused = _nUsed + numbits;
	int r=-1;

	D("Allocating %d bits and _nUsed is %d and _nQubits is %d.\n",
		numbits, _nUsed, _nQubits);
	if (newused <= _nQubits) {
		r=_nUsed;
		_nUsed = newused;
	}
	else
		throw QException::MallocErr();
	return r;
}
