/* qstate.h

Model of Quantum States and Registers.

This file is part of the OpenQubit project.

Copyright (C) 1998-1999 OpenQubit.org
Yan Pritzker <skwp@thepentagon.com>

Please see the CREDITS file for contributors.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#ifndef _QSTATE_H_
#define _QSTATE_H_

#include <iostream.h>
#include <vector>

#include "complex.h"
#include "random.h"
#include "utility.h"
#include "debug.h"
#include "exceptions.h"

#define _RNGT_	double							
#define _RNG_ 	DblUniformRandGenerator		

typedef std::vector<Complex> array_t;		//: Complex array type
//! author="Yan Pritzker, Rafal Podeszwa, Christopher Dawson, Peter Belkner"
//! minor="Joe Nelson"
//! lib="Quantum State/Register [OpenQubit Core]"

class QState
//: Quantum State of machine including memory management facilities. [Static]
// This is the "Quantum Operating System"  This class will manage
// memory for the declaration of quantum registers. The QRegister
// class will allow for relative indexing (e.g. bit 0 in a register
// may be bit 7 of the entire state). All the methods and members
// of this class are declared static because only one machine state
// can exist at any given time. 
{
public:

	//: Default initializer: machine with 1 Qubit.
	inline static void Initialize() { Initialize(1); }

	//: A state consisting of a specific number of Qubits.
	inline static void Initialize(int nQubits, int seedval=-1) 
	{
		_nQubits = nQubits;
		_nStates = 1 << nQubits;
		_qArray.resize(_nStates);
		_nUsed=0;
		_Reset();
		_qArray[0] = 1;
		if(seedval==-1)
			_RNG = new _RNG_(seedval);
		else
			_RNG = new _RNG_;
	}

	//: Dump state to file.
	static void Save(char[]);

	//: Read state from file.
	static void Read(char[]);
	
	//: Display outcomes by coefficients.
	static void Print();

	//: Get number of total states.
	static int States() { return _nStates; }

	//: Get number of total qubits.
	static int Qubits() { return _nQubits; }

	//: Get specified coefficient for modification.
	static Complex& Coef(int i) { return _qArray[i]; }

protected:
	//: Memory management facility.
	static int _Allocate(int) throw (QException::MallocErr);

	//: Destructive measure of entire machine state
	static int _Collapse();         
	
	//: Destructive measure of one bit
	static int _CollapseBit(int) throw (QException::RangeErr);

	//: Destructive set of bits (specified by bitmask) measure	
	static int _CollapseMask(unsigned long) throw (QException::RangeErr);

	//: Collapse set specified by range
	static int _CollapseRange(int,int) throw (QException::RangeErr);
													
	//: Reset certain part of the state (for QRegister).
	static void _Reset(int s, int e) throw (QException::RangeErr);
	
private:
	//: "cool" state to |psi> = |0>.
	static void _Reset();
	
	//: Sum of normalized complex amplitudes.
	static double _Normalize();
	
	//: RNG
	static RandGenerator<_RNGT_> *_RNG;

	static int _nQubits, _nStates, _nUsed;
	static array_t _qArray;
};

class QRegister : public QState
//: Quantum register, a reference to a part of the QState.
{
private:
   int  _Qubits;						//: Size of register
	int  _Offset;						//: Offset.

public:
	//: Register representing entire QState.
	QRegister()
		: _Qubits(QState::Qubits()), _Offset(0) {};

	//: Create a register with specified number of bits.
	QRegister(int bits) : _Qubits(bits)
	{	
		try { _Offset = _Allocate(bits); }
		catch (QException::MallocErr qe) {
			cerr << qe.msg();
			abort();
		}
		
   	D("Register Init. Offset: %d Numbits: %d.\n",_Offset,bits);
	}

	//: Destructive register measure.
	inline int Measure()
      { 	D("Measuring from %d to %d.\n",_Offset, _Offset+_Qubits-1);
			return _CollapseRange(_Offset, _Offset + _Qubits - 1); }

   //: Destructive bit measure.
   inline int Measure(int i)
      { return _CollapseBit(_Offset + i); }

	//: Measure bits in a given range (inclusive).
	inline int Measure(int begin, int end)
		{ return _CollapseRange(_Offset+begin, _Offset+end); }

	//: Returns specified bit + offset. (Relative indexing)
	int Bit(int i) const { return i + _Offset; }

	//: Get offset. Useful for operators.
	int Offset() const { return _Offset; }

	//: Number of Qubits.
	int Qubits() const { return _Qubits; }

	//: "cool" state of register to |psi> = 0.
   inline int Reset()
		{ _Reset(_Offset, _Offset + _Qubits - 1); }
};

#endif
