/* qstate.cc

Implementation of Data Model of a Quantum State/Register
This file is part of the OpenQubit project.

Copyright (C) 1998-1999 OpenQubit.org
Yan Pritzker <skwp@thepentagon.com>

Please see the CREDITS file for contributors.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
#include "qstate.h"

/* collapse code contributed by Rafal Podeszwa */

void QState::SetState(const Complex c[])
{
	double totalprob=0;
	
	for(int i=0; i<_nStates; i++)
		{ totalprob+= norm(c[i]); } 

	D("Sum of probabilities: %1.25f\n",totalprob);
	assert(1-ROUND_ERR <= totalprob && totalprob <= 1+ROUND_ERR );

	for(int j=0; j<_nStates; j++) 
		_qArray[j] = c[j];
}

int QState::_Collapse()
//collapse entire register
//this is an implementation of "Bernharrd's Collapse"
//as suggested by Peter Belkner
{
	double x=0.0;
	RandGenerator<int> *RNG = new IntUniformRandGenerator;
	unsigned long start = RNG->get_rand_between(0,_nStates);
	unsigned long i=start;
	double rnd = RNG->get_rand_between(0,norm(*this));
	
	D("Collapsing register. Got rnd %1.5f\n", rnd);
	
	while ((x += norm(_qArray[i])) < rnd) {
		if(++i == _nStates)
			i=0;
		
		if (start==i) {
			//this should never happen
			cerr << "Unexpected error #7891\n";
			i = (0 == i) ? _nStates - 1 : i - 1;
			break;
		}
	}
	_State=i;
	D("Set register state to %d\n",_State);

	/* added by Yan Pritzker -- set all other coefs to 0 since
		qubit is collapsed; set collapsed state to prob 1 */

	Reset(*this);
	_qArray[_State]=1;

	return i;
}

int QState::_Collapse(int index)
//single qubit collapse
{
	//index is 0..nQubits-1, 0 being the most significant qubit,
	//e.g. in |0001>, the 1 is the least significant qubit and
	//its position is nQubits-1
	assert(0 <= index && index < _nQubits);	

	double p0,p1;	//probabilities of measuring this bit as 0 and 1
	p0=p1=0;
	for (int i=0; i < _nStates; i++) {
		if (IsBitSet(i,index))
			p1 += norm(_qArray[i]);
		else
			p0 += norm(_qArray[i]);
	}

	D("Got probabilities...p0=%1.3f, p1=%1.3f\n", p0, p1);
   D("Deviation of the sum from 1: %f\n",(p0+p1)-1);
	
	int result;
	double rnd=RNG->get_rand_between(0,1);

	if (rnd <= p0)
		result = 0;
	else
		result = 1;

	double f0=sqrt(p0); //we will need it for renormalization
	double f1=sqrt(p1);

	for (int i=0;i < _nStates;i++) {
    if (IsBitSet(i,index))
        if (result == 0)
          _qArray[i]=0; // The bit is set but the measurement revealed 0 so
                			// the coefficient must be set to zero
        else
          _qArray[i]/=f1; //renormalization

    else  // same thing if the bit is 0
        if (result == 1)
          _qArray[i]=0;
        else
          _qArray[i]/=f0;
	}
	_State=result;

	D("Set bit state to %d\n",_State);
	return result;
}
void QState::Print() const
{
	static const char* withimag		= "(%1.6f,%1.6f) |%s>";
	static const char* withimag_last	= "(%1.6f,%1.6f) |%s>\n";
	static const char* noimag			= "%1.6f |%s>";
	static const char* noimag_last	= "%1.6f |%s>";
   
	int pad=count_bits(_nStates-1);

	for(int i=0; i < _nStates-1; i++) {
      if(!imag(_qArray[i]) && !real(_qArray[i])) { continue; }
		if(imag(_qArray[i])) {
         printf(	withimag,
               	real(_qArray[i]),
               	imag(_qArray[i]),
               	dtob(i,pad));
      } else {
         printf(noimag, real(_qArray[i]), dtob(i,pad));
   	}
		if(imag(_qArray[i+1]) || real(_qArray[i+1])) { printf(" + "); }
	}
 
   if(!imag(_qArray[_nStates-1]) && !real(_qArray[_nStates-1])) 
		{ printf("\n"); return; }
	if(imag(_qArray[_nStates-1])) {
		 printf ( withimag_last,
               real(_qArray[_nStates-1]),
               imag(_qArray[_nStates-1]),
               dtob(_nStates-1,pad));
   } else {
      printf ( noimag_last,
               real(_qArray[_nStates-1]),
               dtob(_nStates-1,pad));
	}
	printf("\n");
}

void QState::Dump(char filename[])
{
	FILE *FH;
	if ((FH=fopen(filename,"w"))==NULL)
		cerr << "ERROR: could not open file " << filename << endl;

	fprintf(FH,"QSTATE SIZE %d\n", _nStates);

	for(int i=0; i < _nStates; i++) {
		if(real(_qArray[i]) || imag(_qArray[i]))
			fprintf(FH, "%+1.17f \t %+1.17f \t |0x%X>\n",
					 real(_qArray[i]),
					 imag(_qArray[i]),
					 i);
	}
}

void QState::Read(char filename[])
{
	FILE *FH;
	
	if((FH=fopen(filename,"r"))==NULL)
		cerr << "ERROR: could not open file " << filename << endl;

	double real,imag;
	int index;
	int size;

	fscanf(FH, "QSTATE SIZE %d\n", &size);
	assert(this->_nStates==size);
	
	while(FH)
	{
		int r=fscanf(FH,"%+1.17f \t %+1.17f \t |0x%X>\n", 
					 	 &real, &imag, &index);

		if(r==0) break;

		D("Scanned %f %f %d\n",real,imag,index);
		_qArray[index] = Complex(real,imag);
	}

	assert(1-ROUND_ERR <= norm(*this) && norm(*this) <= 1+ROUND_ERR);
}
