/* random.h

RandLib -- Wrapper classes for several RNG algorithms.
This file is part of the OpenQubit project.

Copyright (C) 1998-1999 OpenQubit.org
Yan Pritzker <skwp@thepentagon.com>

Please see the CREDITS file for contributors.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
#ifndef _RANDOM_H_
#define _RANDOM_H_

#include <assert.h>
#include <time.h>
#include <stdlib.h>
#include <limits.h>
#include <float.h>

//! author="Joe Nelson" 

/**************************************************
**		random.h version 1.3
**		Written
**		by  Joe Nelson <3nelsons@pressenter.com>
**		on  01/15/99
**		for the OpenQubit project
**************************************************/

///////////////////////////////////////////////////
// The abstract base class from which all the rest
// of the random number generators in this library
// are derived
template<class T>
class RandGenerator
//: Abstract RNG Class
{
public:
	// seed() seeds the randomizer with a beginning value.
	// NOTE: in some implementations of RandGenerator, this
	// function may be meaningless
	// Also note: seedVal2 need not be used by all children
	virtual void seed(const unsigned int seedVal, const unsigned int seedVal2) = 0;
	// gets a rand between minVal and maxVal inclusive
	virtual T get_rand_between(const T minVal, const T maxVal)
	{
		assert(minVal <= maxVal);
		assert(minVal >= lowest_rand());
		assert(maxVal <= highest_rand());
		assert(rand_generation_range() >= (long double)maxVal - (long double)minVal);		

		if(minVal == maxVal)
			return minVal;
		return minVal + get_rand(maxVal-minVal);
	}
	// highest_rand and lowest_rand are implemented by children
	// of RandGenerator to return the highest and lowest values
	// that that particular child can possibly generate.
	// rand_generation_range returns the largest range of values
	// that get_rand can select from
	virtual const T highest_rand() const = 0;
	virtual const T lowest_rand() const = 0;
	virtual const T rand_generation_range() const = 0;
protected:
	// get_rand returns a random number from 0 to
	// maxVal inclusive. I decided to make this protected in order
	// to force the client programmer to be explicit.
	// The clients, from their code, must use get_rand_between(0, maxVal)
	// instead of the misinterpretable get_rand(maxVal)
	virtual T get_rand(const long double maxVal) = 0;
};

///////////////////////////////////////////////////
// Implementation of RandGenerator that uses
// standard library calls like rand() and srand()
class IntStdRandGenerator : public RandGenerator<int>
//: Integer RNG using rand()
{
public:
	IntStdRandGenerator(const unsigned int seedVal = 0, const unsigned int seedVal2 = 0)
	{ seed(seedVal, 0); }
	virtual void seed(const unsigned int seedVal, const unsigned int seedVal2)
	{
		if(seedVal == 0)
			srand(time(NULL));
		else
			srand(seedVal);
	}
	virtual const int highest_rand() const
	{ return RAND_MAX; }
	virtual const int lowest_rand() const
	{ return -RAND_MAX; }
	virtual const int rand_generation_range() const
	{ return RAND_MAX; }
protected:
	virtual int get_rand(const long double maxVal)
	{
		return static_cast<int>(maxVal*rand() / RAND_MAX);
	}
};

///////////////////////////////////////////////////
// Compiled and "slightly" modified by Paul Bourk
// http://www.mhri.edu.au/~pdb/unixsoftware/random/
// The author claims it is
// "THE BEST KNOWN RANDOM NUMBER GENERATOR AVAILABLE"
// It was integrated into random.h by Joe Nelson
class DblUniformRandGenerator : public RandGenerator<double>
//: Double RNG using uniform algorithm
{
public:
	DblUniformRandGenerator(const unsigned int seedVal = 0, const unsigned int seedVal2 = 0)
	{ seed(seedVal, seedVal2); }

	virtual void seed(unsigned int seedVal, unsigned int seedVal2)
	{
		double s,t;
		int ii,i,j,k,l,jj,m;

		/*
		Handle the seed range errors
		First random number seed must be between 0 and 31328
		Second seed must have a value between 0 and 30081
		*/
		assert(seedVal <= 31328);    // changed to assertions
		assert(seedVal2 <= 30081);

		if(seedVal == 0 || seedVal2 == 0) //added by Joe
		{
			seedVal  = time(NULL) % 31328;
			seedVal2 = time(NULL) % 30081;
		}

		i = (seedVal / 177) % 177 + 2;
		j = (seedVal % 177)       + 2;
		k = (seedVal2 / 169) % 178 + 1;
		l = (seedVal2 % 169);

		for (ii=0; ii<97; ii++) {
			s = 0.0;
			t = 0.5;
			for (jj=0; jj<24; jj++) {
				m = (((i * j) % 179) * k) % 179;
				i = j;
				j = k;
				k = m;
				l = (53 * l + 1) % 169;
				if (((l * m % 64)) >= 32)
					s += t;
				t *= 0.5;
			}
			u[ii] = s;
		}

		c    = 362436.0 / 16777216.0;
		cd   = 7654321.0 / 16777216.0;
		cm   = 16777213.0 / 16777216.0;
		i97  = 97;
		j97  = 33;
	}

	virtual const double highest_rand() const
	{ return DBL_MAX; }
	virtual const double lowest_rand() const
	{ return -DBL_MAX; }
	virtual const double rand_generation_range() const
	{ return DBL_MAX; }	
		
protected:
	double u[97],c,cd,cm;
	int i97,j97;

	virtual double get_rand(const long double maxVal)
	{ return maxVal * RandomUniform(); }

	double RandomUniform()
	{
		double uni;

		uni = u[i97-1] - u[j97-1];
		if (uni <= 0.0)
			uni++;
		u[i97-1] = uni;
		i97--;
		if (i97 == 0)
			i97 = 97;
		j97--;
		if (j97 == 0)
      		j97 = 97;
		c -= cd;
		if (c < 0.0)
			c += cm;
		uni -= c;
		if (uni < 0.0)
			uni++;

		return(uni);
	}
};

///////////////////////////////////////////////////
// This is a wrapper class for the
// DblUniformRandGenerator
class IntUniformRandGenerator : public RandGenerator<int>
//: Integer RNG using uniform algorithm
{
public:
	IntUniformRandGenerator(const unsigned int seedVal = 0, const unsigned int seedVal2 = 0)
	{
		_inner = new DblUniformRandGenerator(seedVal, seedVal2);
	}	
	virtual ~IntUniformRandGenerator()
	{ delete _inner; }

	virtual void seed(unsigned int seedVal, unsigned int seedVal2)
	{ _inner->seed(seedVal, seedVal2); }

	virtual const int highest_rand() const
	{ return INT_MAX; }
	virtual const int lowest_rand() const
	{ return -INT_MAX; }
	virtual const int rand_generation_range() const
	{ return INT_MAX; }	
			
protected:
	RandGenerator<double> *_inner;

	// get_rand is unnecessary because _inner does the work
	virtual int get_rand(const long double maxVal)
	{ return static_cast<int>(_inner->get_rand_between(0, maxVal)); }
};

#endif
