/*
Paros and its related class files.
Paros is an HTTP/HTTPS proxy for assessing web application security.
Copyright (C) 2003-2004 www.proofsecure.com

This program is free software; you can redistribute it and/or
modify it under the terms of the Clarified Artistic License
as published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
Clarified Artistic License for more details.

You should have received a copy of the Clarified Artistic License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/
package com.proofsecure.paros.network;
 
import java.io.IOException;
import java.net.Socket;
import java.net.SocketException;
import java.util.Vector;

public class HttpConnectionPool implements LastActiveAdapter {
	
	private static final int IDLE_CONNECTION_TIMEOUT_SEC	= 30;

	private Vector 		mPool				= new Vector();
	private String		mHostName			= "";
	private int			mHostPort			= 80;
	private boolean		mIsSecure			= false;
	private long		mLastActiveTime		= System.currentTimeMillis();
	private SSLConnector mSSL				= null;
	private ConnectorConfig mConfig			= null;
		
	public HttpConnectionPool(ConnectorConfig config, SSLConnector ssl) {
		mConfig = config;
		mSSL	= ssl;
	}

	protected void finalize() {
		close();
	}

	/**
	Get current Http input stream in top of queue
	*/
	//public HttpInputStream getCurrentHttpInputStream() {
	//}
	
	/**
	Get Http output stream basing on hostname + port + secure.
	*/
	public synchronized HttpConnection connect(String hostName, int port, boolean isSecure, boolean isUseClientCert) throws IOException {		
		HttpConnection conn = null;
		removeClosedSocket();
		
		for (int i=0; i<mPool.size(); i++) {
			conn = (HttpConnection) mPool.elementAt(i);
			if (hostName.equalsIgnoreCase(conn.mHostName) && port == conn.mPort && isSecure == conn.mIsSecure && !conn.isClosed()) {
				return conn;
			}
		}
		
		// not found matching connectin, create a new one.
		conn = createConnection(hostName, port, isSecure, isUseClientCert);
		mPool.add(conn);
		return conn;
	}

	private HttpConnection createConnection(String hostName, int hostPort, boolean isSecure, boolean isUseClientCert) throws IOException {
		
		Socket socket = null;
		if (isSecure) {
			if (checkAndUseProxy(hostName)) {
				try{
					socket = mSSL.clientViaProxy(hostName, hostPort, mConfig.getProxyChainName(), mConfig.getProxyChainPort(), isUseClientCert);
 				} catch (SocketException e) {
					socket = mSSL.client(hostName, hostPort, isUseClientCert);
 				}

			} else {
				socket = mSSL.client(hostName, hostPort, isUseClientCert);
			}

		} else {
			if (checkAndUseProxy(hostName)) {
				try {
					socket = new Socket(mConfig.getProxyChainName(), mConfig.getProxyChainPort());
	            }
	            catch (SocketException e){
	                socket = new Socket(hostName, hostPort);                		
	            }
	        } else {
				socket = new Socket(hostName, hostPort);
			}
		}
		socket.setTcpNoDelay(true);
		setLastActive();	// new connection, so last active
		HttpConnection conn = new HttpConnection(hostName, hostPort, isSecure, socket, this);
//		System.out.println("Connected. host=" + hostName + " port="+ hostPort + " isSecure=" + isSecure); 

		return conn;
	}
	
	
	public boolean checkAndUseProxy(String hostName) {
		if (mConfig.getProxyChainName() == null || mConfig.getProxyChainName().equals("") || mConfig.isSkipProxyChain(hostName)) {
			return false;
		} else {
			return true;
		}
	}
	
	public int available() {
		removeClosedSocket();
		for (int i=0; i < mPool.size(); i++) {
			HttpConnection conn = (HttpConnection) mPool.elementAt(i);
			int avail = 0;
			try {
				avail = conn.mHttpIn.available();
			} catch (Exception e) {
				conn.close();
			}
			if (avail > 0) {
				return avail;
			}
		}
		return 0;
	}
	
	private synchronized void removeClosedSocket() {
		HttpConnection conn = null;
		int i = 0;
		while (i < mPool.size()) {
			conn = (HttpConnection) mPool.elementAt(i);
			if (conn.isClosed()) {
				mPool.remove(i);
			} else {
				i++;
			}
		}
	}
	
	public synchronized boolean isClosed() {
		removeClosedSocket();
		return mPool.isEmpty();
	}
	
	/**
	Close all input and output stream in pool
	*/
	public synchronized void close() {
		HttpConnection conn = null;
		while (!mPool.isEmpty()) {
			conn = (HttpConnection) mPool.firstElement();
			conn.close();
			mPool.remove(conn);
		}

	}
	
	/**
	Get the last active time among the streams in the pool.
	*/
	public long lastActiveTimeMillis() {
		return mLastActiveTime;
	}
	
	/**
	Adapter for the streams to setup this pool.
	*/
	public void setLastActive() {
		mLastActiveTime = System.currentTimeMillis();
	}



	
}

	

