/*
Paros and its related class files.
Paros is an HTTP/HTTPS proxy for assessing web application security.
Copyright (C) 2003-2004 www.proofsecure.com

This program is free software; you can redistribute it and/or
modify it under the terms of the Clarified Artistic License
as published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
Clarified Artistic License for more details.

You should have received a copy of the Clarified Artistic License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/
package com.proofsecure.paros.network;

import java.util.Hashtable;
import java.util.Vector;
import java.util.StringTokenizer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;


abstract public class HttpHeader implements java.io.Serializable{


	public final static String CRLF 			= "\r\n";
	public final static String LF 				= "\n";
	public final static String CONTENT_LENGTH 	= "Content-length";
	public final static String TRANSFER_ENCODING = "Transfer-encoding";
	public final static String CONTENT_TYPE 	= "Content-Type";
	public final static String PROXY_CONNECTION = "Proxy-Connection";
	public final static String PROXY_AUTHENTICATE = "Proxy-authenticate";
	public final static String CONNECTION		= "Connection";
	public final static String AUTHORIZATION	= "Authorization";
	public final static String WWW_AUTHENTICATE = "WWW-Authenticate";
	public final static String LOCATION				= "Location";
	public final static String IF_MODIFIED_SINCE	= "If-Modified-Since";
	public final static String IF_NONE_MATCH		= "If-None-Match";
	public final static String USER_AGENT		= "User-Agent";
	public final static String ACCEPT_ENCODING = "Accept-Encoding";
	public final static String CACHE_CONTROL	= "Cache-control";
	public final static String PRAGMA			= "Pragma";

	public final static String VERSION_HTTP10 	= "HTTP/1.0";
	public final static String VERSION_HTTP11 	= "HTTP/1.1";
	public final static String _CLOSE 			= "Close";
	public final static String _KEEP_ALIVE 		= "Keep-alive";
	public final static String _CHUNKED			= "Chunked";
	
	public final static String SCHEME_HTTP		= "http://";
	public final static String SCHEME_HTTPS		= "https://";	

	public static Pattern patternCRLF			= Pattern.compile("\\r\\n", Pattern.MULTILINE);
	public static Pattern patternLF				= Pattern.compile("\\n", Pattern.MULTILINE);
	
	protected final static String p_TEXT		= "[^\\x00-\\x1f\\r\\n]*"; 
	protected final static String p_METHOD		= "(\\w+)";
	protected final static String p_SP			= " +";
	protected final static String p_URI			= "(\\S+)";
	protected final static String p_VERSION		= "(HTTP/\\d+\\.\\d+)";
	protected final static String p_STATUS_CODE	= "(\\d{3})";
	protected final static String p_REASON_PHRASE = "(" + p_TEXT + ")";
	
	protected String mStartLine = "";
	protected String mMsgHeader = "";
	protected boolean mMalformedHeader = false;
	protected Hashtable mHeaderFields = new Hashtable();
	protected String[] primeHeader = {"","",""};
	protected int content_length = -1;
	protected boolean isAbsoluteUriRequired = false;
	protected String mLineDelimiter = CRLF;
	protected String mVersion = "";

	public HttpHeader() {
		init();
	}

	public static HttpHeader getInstance(String data) throws HttpMalformedHeaderException {

		if (HttpResponseHeader.isStatusLine(data)) {
			return new HttpResponseHeader(data);
		} else if (HttpRequestHeader.isRequestLine(data)) {
			return new HttpRequestHeader(data);
		} else {
			throw new HttpMalformedHeaderException("Invalid header:\r\n" + data);
		}	
		
    }

	public HttpHeader(String data) throws HttpMalformedHeaderException {
		this();
		setMessage(data);
	}

	public String getMessage(){
    	return mMsgHeader;
  	}  	

	private void init() {
		mHeaderFields = new Hashtable();
		mStartLine = "";
		mMsgHeader = "";
		mMalformedHeader = false;
		content_length = -1;
		mLineDelimiter = CRLF;
	}
  	

	public void setMessage(String data) throws HttpMalformedHeaderException {
		init(); 

		mMsgHeader = data;
		try {
			if (!this.parseHeader(data)) {
				mMalformedHeader = true;
			}
		} catch (Exception e) {
			mMalformedHeader = true;
		}

		if (mMalformedHeader) {
			throw new HttpMalformedHeaderException();
		}

	}

	public String getHeader(String name) {
		Vector v = getHeaders(name);
		if (v == null) {
			return null;
		}
		
		return (String) (v.firstElement());
	}
    
    public Vector getHeaders(String name) {
    	Vector v = (Vector) mHeaderFields.get(name.toUpperCase());
    	return v;
    }
    
    public String getDelimiter() {
    	return mLineDelimiter;
    }

	public void addHeader(String name, String val) {
		mMsgHeader = mMsgHeader + name + ": " + val + mLineDelimiter;
		setInternalHeaderFields(name, val);

	}

    public void setHeader(String name, String value) {
		int pos = 0;
		int crlfpos = 0;
		StringBuffer sb = null;
		Pattern pattern = null;

		if (getHeaders(name) == null && value != null) {
			// header value not found, append to end
			addHeader(name, value);
		} else {
			try {
				pattern = getHeaderRegex(name);
				Matcher matcher = pattern.matcher(mMsgHeader);
				if (value == null) {
					// delete header
					mMsgHeader = matcher.replaceAll("");
				} else {
					// replace header
					String newString = name + ": " + value + mLineDelimiter;
					mMsgHeader = matcher.replaceAll(newString);
				}

				// set into hashtable
				setInternalHeaderFields(name, value);
				
			}
			catch (Exception e) {
			}
			
		}
    }
    
    private Pattern getHeaderRegex(String name) throws PatternSyntaxException
	{
		return Pattern.compile("^ *"+ name + " *: *[^\\r\\n]*" + mLineDelimiter, Pattern.CASE_INSENSITIVE | Pattern.MULTILINE);
	}

	public String getVersion() {
		return mVersion;
	}
	
	abstract public void setVersion(String version);


	public int getContentLength() {
        return content_length;
    }

	public void setContentLength(int len) {
		if (content_length != len) {
			setHeader(CONTENT_LENGTH, Integer.toString(len));
			content_length = len;
		}
	}

   	public boolean isConnectionClose() {
		boolean result = true;
		if (mMalformedHeader) {
			return true;
		}

		if (isHttp10()) {
			result = true;
			try {
				if (getHeader(CONNECTION).equalsIgnoreCase(_KEEP_ALIVE) || getHeader(PROXY_CONNECTION).equalsIgnoreCase(_KEEP_ALIVE)) {
					return false;
				}
			} catch (NullPointerException e) {
			}

		} else if (isHttp11()) {
			result = false;
			try {
				if (getHeader(CONNECTION).equalsIgnoreCase(_CLOSE)) {
					return true;
				} else if  (getHeader(PROXY_CONNECTION).equalsIgnoreCase(_CLOSE)) {
					return true;
				}
			} catch (NullPointerException e) {
			}
		}
		return result;
	}

	public boolean isHttp10() {
		if (mVersion.equals(VERSION_HTTP10)) {
			return true;
		}
		return false;
	}

	public boolean isHttp11() {
		if (mVersion.equals(VERSION_HTTP11)) {
			return true;
		}
		return false;
	}

    public boolean isTransferEncodingChunked() {
    	String transferEncoding = getHeader(TRANSFER_ENCODING);
    	if (transferEncoding != null && transferEncoding.equalsIgnoreCase(_CHUNKED)) {
    		return true;
    	}
    	return false;
    }

    public Hashtable getAllHeaderFields() {
        return mHeaderFields;
    }

    public boolean parseHeader(String data) throws Exception {
        StringTokenizer tokenizer = null;
        String 	token = null,
				name = null,
				value = null;
        int pos = 0;
        Pattern pattern = null;

        if(data == null || data.equals("")) {
            return true;
        }

        if ((pos = data.indexOf(CRLF)) < 0) {
        	if ((pos = data.indexOf(LF)) < 0) {
        		return false;
        	} else {
        		mLineDelimiter = LF;
        		pattern = patternLF;
        	}
        } else {
        	mLineDelimiter = CRLF;
        	pattern = patternCRLF;
        }
        
		String[] split = pattern.split(data);
		mStartLine = split[0];

		StringBuffer sb = new StringBuffer(2048);
		for (int i=1; i<split.length; i++)
		{
			token = split[i];
			if (token.equals("")) {
				continue;
			}
			
            if((pos = token.indexOf(":")) < 0) {
				mMalformedHeader = true;
                return false;
            }
            name  = token.substring(0, pos).trim();
            value = token.substring(pos +1).trim();

            if(name.equalsIgnoreCase(CONTENT_LENGTH)) {
            	try {
                	content_length = Integer.parseInt(value);
            	} catch (NumberFormatException nfe){}
            }

			
            if (name.equalsIgnoreCase(PROXY_CONNECTION)) {
            	//sb.append(name + ": " + _CLOSE + mLineDelimiter);
            } else if (name.equalsIgnoreCase(CONNECTION)) {
            	//sb.append(name + ": " + _CLOSE + mLineDelimiter);
            } else {
				sb.append(name + ": " + value + mLineDelimiter);
			}
			
			setInternalHeaderFields(name, value);
		}

        mMsgHeader = sb.toString();
		return true;
	}

	private void setInternalHeaderFields(String name, String value) {
		String key = name.toUpperCase();
		Vector v = getHeaders(key);
		if (v == null) {
			v = new Vector();
			mHeaderFields.put(key, v);
		}

		if (value != null) {
			v.add(value);
		} else {
			mHeaderFields.remove(key);
		}
	}
		
    public boolean isMalformedHeader() {
        return mMalformedHeader;
    }

    public String toStringNoAbsoluteUri() {
		return toStringAbsoluteUri();

    }

    public String toString() {
    	if (isAbsoluteUriRequired) {
    		return toStringAbsoluteUri();
    	} else {
    		return toStringNoAbsoluteUri();
    	}
    }
    


    public String toStringAbsoluteUri() {
		return getStartLine() + mLineDelimiter + mMsgHeader + mLineDelimiter;
    }

	public String getPrimeHeader() {
		return getStartLine();
	}
    
    public String getStartLine() {
    	return mStartLine;
    }

    public boolean isImage() {
    	return false;
    }

    public boolean isText() {
    	return true;
    }

	public void setAbsoluteUriRequired(boolean isRequired) {
		isAbsoluteUriRequired = isRequired;
	}
	
	/**
	For debugging purpose only.
	*/
	public static void showDebugMsg(String msg) {
		System.out.println(msg);
	}

}
