/*
Paros and its related class files.
Paros is an HTTP/HTTPS proxy for assessing web application security.
Copyright (C) 2003-2004 www.proofsecure.com

This program is free software; you can redistribute it and/or
modify it under the terms of the Clarified Artistic License
as published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
Clarified Artistic License for more details.

You should have received a copy of the Clarified Artistic License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/
package com.proofsecure.paros.scan;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.proofsecure.paros.network.HttpHeader;
import com.proofsecure.paros.network.HttpStatusCode;

public class TestCacheAndMisc extends AbstractAppTest {

	// pattern for testing

	// <form autocomplete=off>
	// <input type=password ...>
	// </form>

	public final static Pattern patternNoCache	= Pattern.compile("\\QNo-cache\\E|\\QNo-store\\E", PATTERN_PARAM);

	// <meta http-equiv="Pragma" content="no-cache">
	// <meta http-equiv="Cache-Control" content="no-cache">
	public final static Pattern patternHtmlNoCache = Pattern.compile("<META[^>]+(Pragma|\\QCache-Control\\E)[^>]+(\\QNo-cache\\E|\\QNo-store\\E)[^>]*>", PATTERN_PARAM);

	// check for autocomplete
	public final static Pattern patternAutocomplete	= Pattern.compile("AUTOCOMPLETE\\s*=[^>]*OFF[^>]*", PATTERN_PARAM);

	// used reluctant quantifer to make sure the same form and input element is referred 
	public final static Pattern patternForm = Pattern.compile("(<FORM\\s*[^>]+\\s*>).*?(<INPUT\\s*[^>]+type=[\"']?PASSWORD[\"']?[^>]+\\s*>).*?</FORM>", PATTERN_PARAM| Pattern.DOTALL);

	// check for private IP list
	public final static Pattern patternPrivateIP = Pattern.compile("(10\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}|172\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}|192\\.168\\.\\d{1,3}\\.\\d{1,3})", PATTERN_PARAM);

	// check for system information
	
	private final static String[] platforms = {
		// OS

		"Microsoft", "Windows ?\\d{4,4}", "Windows ?XP",
		"BeOS", "Redhat", "SUSE", "Mandrake", "Mandrakesoft", "Slackware",
		"BSD Unix", "FreeBSD", "HP-?UX", "Solaris", "AIX", "MVS", "OS.?390",
		"MacOS", "Debian", "RedFlag", 
		
		// web server list
		"Internet information server", "IIS ?\\d?", "Apache", "Stronghold", "Netscape enterprise", "iplanet", "Lotus Domino", "Domino",
		"IBM HTTP", "webstar", "rapidsite", "thttpd", "websitepro",
		
		// web application server
		// List from http://java.sun.com/j2ee/compatibility.html
		
		"Tomcat", "ATG Dynamo", "WebLogic", "Borland", 
		"Fujisu Interstage", "Cosminexus", "Websphere", "Orbix E2A",
		"Macromedia", "Jrun", "WebOTX", "Novell exteNd", "Pramati server",
		"Pramati studio",  "Oracle ?\\di", "SAS AppDev", "SeeBeyond ICAN", "SpiritSoft",
		"Sun ?ONE", "Sybase", "EAServer", "Tmax Soft JEUS", "Trifork Application",
		
		// web server
		"Internet information server", "IIS", "Apache", "Lotus Domino", "Domino",
		"IBM HTTP server", "ColdFusion", 
		
		//development tool
		"Frontpage", "Visual ?Basic", "Visual ?C", "dreamweaver", "VisualAge", "netbean", "perl", "php", "python", "powerbuilder",
		"Load runner", "JDeveloper", "JProbe", "PerformaSure", "Jbuilder", "Introscope",
		"OptimizeIT",
		
		// others
		"proxy server", "reverse proxy", "squid",
		
		// db
		"Microsoft SQL", "MS ?SQL", "Oracle", "Mysql", "msql", "foxbase", "sybase", "postgre ?sql",
		"powerbase",
		
		// firewall
		// http://www.infosyssec.org/infosyssec/firew1.htm
		
		"Cyberguard firewall", "Altavista firewall", "Ascend firewall",
		"Cisco PIX", "Eagle NT", "Elron firewall",
		"Check\\s*point", "Firewall-1",
		"Network-1 firewall", "GFX firewall", "GNAT box firewall",
		"NetGuardian", "Milkyway SecurIT", "Sonic firewall", "sonicwall", "Kane security analyst",
		"NetRoad firewall", "session wall-3", 
		"SUNScreen", "watchguard", "Nokia +\\d+", "Alteon", "Guardian firewall", 
		"NetScreen"
		
	};

//	private static Pattern[] patternPlatforms = null;

	private static Pattern patternComment	= Pattern.compile("<!--\\s*?(.*?)\\s*?-->");
	private static Pattern patternPlatforms = null;

	
	static {
//		patternPlatforms = new Pattern[platforms.length];
//		for (int i=0; i<platforms.length; i++) {
//			patternPlatforms[i] = Pattern.compile("<!--\\s*?.*?\\b(" + platforms[i] + ")\\s+*?\\s*?-->", PATTERN_PARAM);
//		}

		StringBuffer sb = new StringBuffer();
		
		sb.append("\\b(");
		for (int i=0; i<platforms.length; i++) {
			if (i > 0) {
				sb.append("|");
			}
			sb.append(platforms[i]);
		}
		sb.append(")\\s+");
		
		patternPlatforms = Pattern.compile(sb.toString(), PATTERN_PARAM);
	}
	
	TestCacheAndMisc() {
	}

    public String toString() {
        return "TestCacheAndMisc";
    }

	public String getTestName() {
		return "Browser Cache and Miscellaneous";
	}

	private void buildHeader() {
		// use original header	
	}

	protected void scan() throws Exception {
		buildHeader();
		writeStatus("BrowserCacheAndMisc: " + getRequestHeader().getURIHostPath());
		sendAndReceive();
		if (getResponseHeader().getStatusCode() != HttpStatusCode.OK) {
			return;
		}

		runChecks();
		
	}
	
	private void runChecks(){
		checkBrowserCache();
		checkFormAutocomplete();
		checkPrivateIPDisclosure();		
		checkComment();
		checkServerHeader();
		checkSetCookie();
		checkSessionIDGET();

		//checkSessionIDExposure();		This is placed inside checkSessionIDGET()
	}

	protected void scanOffline() throws Exception {
		if (getResponseHeader().getStatusCode() != HttpStatusCode.OK) {
			return;
		}		
		runChecks();
		
	}	
	private void checkBrowserCache() {

		boolean result = false;
				
		if (!getRequestHeader().getSecure()) {
			// no need to if non-secure page;
			return;
		} else if (getRequestHeader().isImage()) {
			// does not bother if image is cached
			return;
		} else if (getResponseBody().length() == 0) {
			return;
		}
		
		if (!matchHeaderPattern(HttpHeader.CACHE_CONTROL, patternNoCache)
			&& !matchHeaderPattern(HttpHeader.PRAGMA, patternNoCache)
			&& !matchBodyPattern(patternHtmlNoCache, null)) {
			
			result = true;
		}

		if (result) {
			bingo(10003, AlertItem.RISK_MEDIUM, AlertItem.WARNING, "",  "", "");
		}
	}
	
	private void checkFormAutocomplete() {

		String txtBody = getResponseBody().toString();
		String txtForm = null;
		String txtInput = null;
		Matcher matcherForm = patternForm.matcher(txtBody);
		Matcher matcherAutocomplete = null;
		boolean result = false;
		
		while (matcherForm.find()) {
			txtForm = matcherForm.group(1);	
			txtInput = matcherForm.group(2);

			//System.out.println(txtForm + txtInput);
			if (txtForm != null && txtInput != null) {
				matcherAutocomplete = patternAutocomplete.matcher(txtForm);
				if (matcherAutocomplete.find()) {
					continue;
				}
				matcherAutocomplete = patternAutocomplete.matcher(txtInput);
				if (matcherAutocomplete.find()) {
					continue;
				}
				
				bingo(10004, AlertItem.RISK_MEDIUM, AlertItem.WARNING, "", "", txtInput);
				
			}
				
		}

	}

	private void checkPrivateIPDisclosure() {
		
		String txtBody = getResponseBody().toString();
		String txtFound = null;
		Matcher matcher = patternPrivateIP.matcher(txtBody);
		while (matcher.find()) {
			txtFound = matcher.group();
			if (txtFound != null) {
				bingo(10005, AlertItem.RISK_LOW, AlertItem.WARNING, "", "", txtFound);
			}
		}				
	}
	
	private void checkComment() {
		Matcher commentMatcher = null;
		Matcher matcher = null;
		String txtBody = getResponseBody().toString();
		String txtComment = null;
		String txtFound = null;

/*
		for (int i=0; i<patternPlatforms.length; i++) {
			matcher = patternPlatforms[i].matcher(txtBody);
			while (matcher.find()) {
				txtFound = matcher.group(1);
				writeOutput("Warning -\tPlatform disclosed: " + getRequestHeader().getURIHostPath());
				writeOutput("\tComment = " + txtFound);
			}
		}
*/

		commentMatcher = patternComment.matcher(txtBody);
		while (commentMatcher.find()) {
			txtComment = commentMatcher.group(1);
			matcher = patternPlatforms.matcher(txtComment);
			while (matcher.find()) {
				txtFound = matcher.group(1);
				bingo(10006, AlertItem.RISK_INFO, AlertItem.NONE, "", "", txtFound);
			}
		}

	}
	
	/**
	Get server header.
	To avoid show up many times.  Should modify Scanner for addUniqueOtherInfo.
	*/
	private void checkServerHeader() {
		
		String serverHeader = getResponseHeader().getHeader("SERVER");
				
		bingo(10010, AlertItem.RISK_INFO, AlertItem.NONE, "", "", serverHeader);
	}

	private void checkSetCookie() {
		String setCookieHeader = getResponseHeader().getHeader("SET-COOKIE");
		
		if (setCookieHeader	== null || setCookieHeader.length() == 0) {
			setCookieHeader = getResponseHeader().getHeader("SET-COOKIE2");
		}
		
		if (setCookieHeader	== null || setCookieHeader.length() == 0) {
			return;
		}	

		bingo(10011, AlertItem.RISK_INFO, AlertItem.NONE, "", "", setCookieHeader);
	}
	
	/*
	private static Pattern staticSessionCookieNamePHP = Pattern("PHPSESSID", PATTERN.PARAM);
	private 
	
	ASP = ASPSESSIONIDxxxxx=xxxxxx
	PHP = PHPSESSID
	Cole fusion = CFID, CFTOKEN	(firmed, checked with Macromedia)
	Java (tomcat, jrun, websphere, sunone, weblogic )= JSESSIONID=xxxxx	
	
	*/
	
	private static Pattern staticSessionIDPHP1 = Pattern.compile("PHPSESSION=\\w+", PATTERN_PARAM);
	private static Pattern staticSessionIDPHP2 = Pattern.compile("PHPSESSID=\\w+", PATTERN_PARAM);
	private static Pattern staticSessionIDJava = Pattern.compile("JSESSIONID=\\w+", PATTERN_PARAM);
	private static Pattern staticSessionIDASP = Pattern.compile("ASPSESSIONID=\\w+", PATTERN_PARAM);
	private static Pattern staticSessionIDColdFusion = Pattern.compile("CFTOKEN=\\w+", PATTERN_PARAM);
	private static Pattern staticSessionIDApache = Pattern.compile("SESSIONID=\\w+", PATTERN_PARAM);
	private static Pattern staticSessionIDJW = Pattern.compile("JWSESSIONID=\\w+", PATTERN_PARAM);
	
	private static Pattern[] staticSessionIDList =
		{staticSessionIDPHP1, staticSessionIDPHP2, staticSessionIDJava, staticSessionIDColdFusion,
			staticSessionIDASP, staticSessionIDApache, staticSessionIDJW};
	
	private void checkSessionIDGET() {
		
		String uri = getRequestHeader().getURIPathQuery();
		Matcher matcher = null;
		String sessionID = null;
		
		for (int i=0; i<staticSessionIDList.length; i++) {
			matcher = staticSessionIDList[i].matcher(uri);
			if (matcher.find()) {
				sessionID = matcher.group(0);
				bingo(10012, AlertItem.RISK_LOW, AlertItem.WARNING, "", "", sessionID);
				checkSessionIDExposure();
				break;
			}
		}
		
	}
	
	private static final String paramHostHttp = "http://([\\w\\.\\-_]+)";
	private static final String paramHostHttps = "https://([\\w\\.\\-_]+)";
	private static final Pattern[] staticLinkCheck = {
		Pattern.compile("src\\s*=\\s*\"?" + paramHostHttp, PATTERN_PARAM),
		Pattern.compile("href\\s*=\\s*\"?" + paramHostHttp, PATTERN_PARAM),
		Pattern.compile("src\\s*=\\s*\"?" + paramHostHttps, PATTERN_PARAM),
		Pattern.compile("href\\s*=\\s*\"?" + paramHostHttps, PATTERN_PARAM),
		
	};
	
	private void checkSessionIDExposure() {

		String body = getResponseBody().toString();
		int risk = (getRequestHeader().getSecure()) ? AlertItem.RISK_MEDIUM : AlertItem.RISK_INFO;
		String linkHostName = null;
		Matcher matcher = null;
		
		for (int i=0; i<staticLinkCheck.length; i++) {
			matcher = staticLinkCheck[i].matcher(body);
		
			while (matcher.find()) {
				linkHostName = matcher.group(1);
				if (getRequestHeader().getHostName().compareToIgnoreCase(linkHostName) != 0) {
					bingo(10013, risk, AlertItem.WARNING, "", "", linkHostName);
				}
			}
		}
	}
}