/*
Paros and its related class files.
Paros is an HTTP/HTTPS proxy for assessing web application security.
Copyright (C) 2003-2004 www.proofsecure.com

This program is free software; you can redistribute it and/or
modify it under the terms of the Clarified Artistic License
as published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
Clarified Artistic License for more details.

You should have received a copy of the Clarified Artistic License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/
package com.proofsecure.paros.scan;

import java.io.IOException;
import java.util.regex.Pattern;

import com.proofsecure.paros.network.HttpStatusCode;
import com.proofsecure.paros.Global;

class TestSQLInjection extends TestAbstractParam {

	private static String SQL_OR_1 = "%20OR%201=1;--";
	private static String SQL_OR_2 = "'%20OR%201=1;--";
	private static String SQL_OR_3 = "'%20OR%20'1'='1";
	
	private static String SQL_DELAY_1 = "';waitfor%20delay%20'0:0:15';--";
	private static String SQL_DELAY_2 = ";waitfor%20delay%20'0:0:15';--";

	private static String SQL_BLIND_MS_INSERT = ");waitfor%20delay%20'0:0:15';--";
	private static String SQL_BLIND_INSERT = ");--";
	
	private static String SQL_AND_1 = "%20AND%201=1";		// true statement for comparison
	private static String SQL_AND_1_ERR = "%20AND%201=2";	// always false stmt for comparison
	
	private static String SQL_AND_2 = "'%20AND%20'1'='1";		// true statement
	private static String SQL_AND_2_ERR = "'%20AND%20'1'='2";	// always false statement for comparison
	
	private static String SQL_CHECK_ERR = "'INJECTED_PARAM";		// invalid statement to trigger SQL exception. Make sure the pattern below does not appear here
	
	private static int TIME_SPREAD = 15000;
	
	private static Pattern patternErrorODBC1 = Pattern.compile("Microsoft OLE DB Provider for ODBC Drivers.*error", PATTERN_PARAM);
	private static Pattern patternErrorODBC2 = Pattern.compile("ODBC.*Drivers.*error", PATTERN_PARAM);
	private static Pattern patternErrorGeneric = Pattern.compile("JDBC|ODBC|SQL", PATTERN_PARAM);
	
	private String mResBodyNormal 	= "";		// normal response for comparison
	private String mResBodyError	= "";	// error response for comparison

	TestSQLInjection() {
	}

    public String toString() {
        return "TestSQLInjection";
    }
    
	public String getTestName() {
		return "SQL Injection";
	}	
	
	protected void check(boolean isBody, String paramKey, String paramValue, String query, int insertPos) throws IOException {

		String bingoQuery = null;
		String displayURI = null;
		String newQuery = null;
		
		String resBodyAND = null;
		String resBodyANDErr = null;
		
		int pos = 0;
		long defaultTimeUsed = 0;
		long timeUsed = 0;
		long lastTime = 0;
		
		// always try normal query first
		newQuery = insertQuery(query, insertPos, paramKey + "=" + paramValue);
		createMessage(isBody, newQuery);
		lastTime = System.currentTimeMillis();
		sendAndReceive();
		defaultTimeUsed = System.currentTimeMillis() - lastTime;
		if (getResponseHeader().getStatusCode() != HttpStatusCode.OK) {
			return;
		}

		mResBodyNormal = getResponseBody().toString();
		
		// 2nd try an always error SQL query
		newQuery = insertQuery(query, insertPos, paramKey + "=" + paramValue + SQL_CHECK_ERR);
		createMessage(isBody, newQuery);
		lastTime = System.currentTimeMillis();
		sendAndReceive();
		defaultTimeUsed = System.currentTimeMillis() - lastTime;
		mResBodyError	= getResponseBody().toString();
		if (checkANDResult(newQuery)) {
			return;
		}

		// try blind SQL query using AND without quote
		bingoQuery = insertQuery(query, insertPos, paramKey + "=" + paramValue + SQL_AND_1);
		createMessage(isBody, bingoQuery);
		sendAndReceive();
		displayURI = getRequestHeader().getURIHostPathQuery();

		if (checkANDResult(bingoQuery)) {
			return;
		}
		if (getResponseHeader().getStatusCode() == HttpStatusCode.OK) {
			// try if 1st SQL AND looks like normal query
			resBodyAND = getResponseBody().toString().replaceAll(SQL_AND_1,"");

			if (resBodyAND.compareTo(mResBodyNormal) == 0) {
				newQuery = insertQuery(query, insertPos, paramKey + "=" + paramValue + SQL_AND_1_ERR);
				createMessage(isBody, newQuery);
				sendAndReceive();
				resBodyANDErr = getResponseBody().toString().replaceAll(SQL_AND_1_ERR,"");

				// build a always false AND query.  Result should be different to prove the SQL works.
				if (resBodyANDErr.compareTo(mResBodyNormal) != 0) {
					bingo(20001, AlertItem.RISK_HIGH, AlertItem.WARNING, displayURI, bingoQuery, "");
					return;
				}
			}

		}

		// try 2nd blind SQL query using AND with quote
		bingoQuery = insertQuery(query, insertPos, paramKey + "=" + paramValue + SQL_AND_2);
		createMessage(isBody, bingoQuery);
		sendAndReceive();
		displayURI = getRequestHeader().getURIHostPathQuery();

		if (checkANDResult(bingoQuery)) {
			return;
		}
		if (getResponseHeader().getStatusCode() == HttpStatusCode.OK) {
			// try if 1st SQL AND looks like normal query
			resBodyAND = getResponseBody().toString().replaceAll(SQL_AND_2,"");

			if (resBodyAND.compareTo(mResBodyNormal) == 0) {
				newQuery = insertQuery(query, insertPos, paramKey + "=" + paramValue + SQL_AND_2_ERR);
				createMessage(isBody, newQuery);
				sendAndReceive();
				resBodyANDErr = getResponseBody().toString().replaceAll(SQL_AND_2_ERR,"");
				// build a always false AND query.  Result should be different to prove the SQL works.
				if (resBodyANDErr.compareTo(mResBodyNormal) != 0) {
					bingo(20001, AlertItem.RISK_HIGH, AlertItem.WARNING, displayURI, bingoQuery, "");
					return;
				}
			}
		}


		// try BLIND SQL SELECT using timing 
		newQuery = insertQuery(query, insertPos, paramKey + "=" + paramValue + SQL_DELAY_1);
		createMessage(isBody, newQuery);
		lastTime = System.currentTimeMillis();
		sendAndReceive();
		timeUsed = System.currentTimeMillis() - lastTime;
				
		if (checkTimeResult(newQuery, defaultTimeUsed, timeUsed)) {
			return;
		}

		newQuery = insertQuery(query, insertPos, paramKey + "=" + paramValue + SQL_DELAY_2);
		createMessage(isBody, newQuery);
		lastTime = System.currentTimeMillis();
		sendAndReceive();
		timeUsed = System.currentTimeMillis() - lastTime;
				
		if (checkTimeResult(newQuery, defaultTimeUsed, timeUsed)) {
			return;
		}

		// try BLIND MSSQL INSERT using timing
		testBlindINSERT(isBody, paramKey, paramValue, query, insertPos);
	}
	
	private void testBlindINSERT(boolean isBody, String paramKey, String paramValue, String query, int insertPos) throws IOException {

		// only testing feature.
		if (!Global.isST) {
			return;
		}
		
		String bingoQuery = null;
		String displayURI = null;
		String newQuery = null;
		
		String resBody = null;
		String resBodyErr = null;
		
		int pos = 0;
		long defaultTimeUsed = 0;
		long timeUsed = 0;
		long lastTime = 0;

		int TRY_COUNT = 10;
		StringBuffer sbInsertValue = new StringBuffer();

		/*	below code is useless because insert can be detected by next section.
			If not, it's likely non MS-SQL and no multiple statement is allowed.
		// try insert using comparison

		for (int i=0; i<TRY_COUNT; i++) {
			// guess at most TRY_COUNT times.
			
			if (i>0) {
				sbInsertValue.append(",'0'");
			} else {
				
				// check if a known to be error response returned a page different from a normal response
				// if so, allow testing to proceed
				newQuery = insertQuery(query, insertPos, paramKey + "=" + paramValue
					+ "'" + SQL_BLIND_INSERT);
				createMessage(isBody, newQuery);
				sendAndReceive();
				resBody = getResponseBody().toString();
				if (resBody.compareTo(mResBodyNormal) == 0) {
					break;
				}
	
				try {
					long tmp = Long.parseLong(paramValue);
					newQuery = insertQuery(query, insertPos, paramKey + "=" + paramValue
						+ SQL_BLIND_INSERT);
					createMessage(isBody, newQuery);
					sendAndReceive();
					resBody = getResponseBody().toString();
					if (resBody.compareTo(mResBodyNormal) == 0) {
						break;
					}

				} catch (NumberFormatException e) {
				}
				
				// do not test for no length added case
				continue;
			}
				
			// try INSERT lengthened.  If for some return returned normal response,
			// that is a sign of success
			newQuery = insertQuery(query, insertPos, paramKey + "=" + paramValue
				+ "'" + sbInsertValue.toString() + SQL_BLIND_INSERT);
			createMessage(isBody, newQuery);
			lastTime = System.currentTimeMillis();
			System.out.println(newQuery);

			sendAndReceive();
			timeUsed = System.currentTimeMillis() - lastTime;

			resBody = getResponseBody().toString();
			if (resBody.compareTo(mResBodyNormal) == 0) {
				bingo(20001, AlertItem.RISK_HIGH, AlertItem.WARNING, displayURI, newQuery, "");
				return;
				
			}

			// no need to try following if not a value integer
			try {
				long tmp = Long.parseLong(paramValue);
			} catch (NumberFormatException e) {
				continue;
			}
			newQuery = insertQuery(query, insertPos, paramKey + "=" + paramValue
				+ sbInsertValue.toString() + SQL_BLIND_INSERT);
			System.out.println(newQuery);
			createMessage(isBody, newQuery);
			lastTime = System.currentTimeMillis();
			sendAndReceive();
			timeUsed = System.currentTimeMillis() - lastTime;
		
			if (resBody.compareTo(mResBodyNormal) == 0) {
				bingo(20001, AlertItem.RISK_HIGH, AlertItem.WARNING, displayURI, newQuery, "");
				return;
				
			}
			
		}
		*/		
		

		// try insert param using INSERT and timing
		sbInsertValue = new StringBuffer();
		for (int i=0; i<TRY_COUNT; i++) {
			// guess at most 10 parameters.
			
			if (i>0) {
				sbInsertValue.append(",'0'");
			}
			
			// try INSERT
			newQuery = insertQuery(query, insertPos, paramKey + "=" + paramValue
				+ "'" + sbInsertValue.toString() + SQL_BLIND_MS_INSERT);
			createMessage(isBody, newQuery);
			lastTime = System.currentTimeMillis();

			sendAndReceive();
			timeUsed = System.currentTimeMillis() - lastTime;

			if (checkTimeResult(newQuery, defaultTimeUsed, timeUsed)) {
				return;
			}

			// no need to try following if not a value integer
			try {
				long tmp = Long.parseLong(paramValue);
			} catch (NumberFormatException e) {
				continue;
			}
			newQuery = insertQuery(query, insertPos, paramKey + "=" + paramValue
				+ sbInsertValue.toString() + SQL_BLIND_MS_INSERT);
			createMessage(isBody, newQuery);
			lastTime = System.currentTimeMillis();

			sendAndReceive();
			timeUsed = System.currentTimeMillis() - lastTime;
		
			if (checkTimeResult(newQuery, defaultTimeUsed, timeUsed)) {
				return;
			}
			
		}

	}


	private boolean checkANDResult(String query) {

		StringBuffer sb = new StringBuffer();
		if (getResponseHeader().getStatusCode() != HttpStatusCode.OK
			&& !HttpStatusCode.isServerError(getResponseHeader().getStatusCode())) {
			return false;
		}
		
		if (matchBodyPattern(patternErrorODBC1, sb)
				|| matchBodyPattern(patternErrorODBC2, sb)) {
			// check for ODBC error.  Almost certain.
			bingo(20001, AlertItem.RISK_HIGH, AlertItem.WARNING, "", query, sb.toString());
			return true;
		} else if (matchBodyPattern(patternErrorGeneric, sb)) {
			// check for other sql error (JDBC) etc.  Suspicious.
			bingo(20001, AlertItem.RISK_HIGH, AlertItem.SUSPICIOUS, "", query, sb.toString());
			return true;
		}
		
		return false;
		
	}

	
	private boolean checkTimeResult(String query, long defaultTimeUsed, long timeUsed) {

		boolean result = checkANDResult(query);
		if (result) {
			return result;
		}


		if (timeUsed > defaultTimeUsed + TIME_SPREAD - 500) {		
			// allow 500ms discrepancy
			bingo(20001, AlertItem.RISK_HIGH, AlertItem.SUSPICIOUS, "", query, "");
			return true;
		}			
		return false;
	}
	
	protected void scan() throws Exception {
		boolean skip = (getEntity().mState == ParsedEntity.DO_NOT_SCAN);
			
		writeStatus("SQL Injection: " + (skip? "(skipped) " : "") + getRequestHeader().getURIHostPath());
		if (skip) {
			return;
		}
		init();
		checkUrlOrBody(false, myQuery);
		checkUrlOrBody(true, myReqBody.toString());
	}

}